/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmcommon
 * @short_description: Various enums and macros used for common tasks.
 *
 * Various enumerations and macros used by Pigment for common tasks.
 *
 * Last reviewed on 2007-04-12 (0.1.5)
 */

#ifndef __PGM_COMMON_H__
#define __PGM_COMMON_H__

#include <glib-object.h>
#include <math.h>        /* sin, cos, sincos */

G_BEGIN_DECLS

#define PGM_PADDING (4)

/* Windows exports */
#if defined(G_OS_WIN32) && (!defined(__MINGW32__))
#ifdef PGM_COMPILATION
#define PGM_EXPORT __declspec(dllexport)
#else
#define PGM_EXPORT __declspec(dllimport)
#endif /* PGM_COMPILATION */
#else
#define PGM_EXPORT
#endif /* defined(G_OS_WIN32) && (!defined(__MINGW32__)) */

/**
 * PGM_DEGREES_TO_RADIANS:
 * @theta: an angle in degrees.
 *
 * Converts from degrees to radians.
 */
#define PGM_DEGREES_TO_RADIANS(theta) ((theta) * (G_PI / 180.0f))

/**
 * PGM_RADIANS_TO_DEGREES:
 * @theta: an angle in radians.
 *
 * Converts from radians to degrees.
 */
#define PGM_RADIANS_TO_DEGREES(theta) ((theta) * (180.0f / G_PI))

/**
 * PGM_SINCOS:
 * @theta: an angle in radians.
 * @s: the address of the double where the sine value is going to be stored.
 * @c: the address of the double where the cosine value is going to be stored.
 *
 * Calculates sine and cosine of the angle @theta simultaneously.
 */
#if HAVE_SINCOS
#define PGM_SINCOS(theta,s,c) sincos ((theta), (s), (c))
#else
#define PGM_SINCOS(theta,s,c)          \
  G_STMT_START {                       \
    *(s) = sin (theta);                \
    *(c) = cos (theta);                \
  } G_STMT_END
#endif /* HAVE_SINCOS */

/**
 * PGM_FABSF:
 * @x: the floating-point number.
 *
 * Calculates the absolute value of float.
 */
#if HAVE_FABSF
#define PGM_FABSF(x) fabsf (x)
#else
#define PGM_FABSF(x) ((gfloat) fabs (x))
#endif /* HAVE_FABSF */

/**
 * PgmError:
 * @PGM_ERROR_X: Generic error code.
 * @PGM_ERROR_OK: No error occured.
 *
 * Every Pigment method returns an error code from that list.
 */
typedef enum {
  PGM_ERROR_X,
  PGM_ERROR_OK
} PgmError;

G_END_DECLS

#endif /* __PGM_COMMON_H__ */
