/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment media rendering library
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author(s): Loïc Molinari <loic@fluendo.com>
 *            Julien Moutte <julien@fluendo.com>
 */

/**
 * SECTION:pgmviewport
 * @short_description: An abstract class doing a visual projection of a canvas.
 * @see_also: #PgmCanvas, #PgmViewportFactory, #PgmDrawable.
 *
 * <refsect2>
 * <para>
 * A #PgmViewport object is used to do a visual projection of #PgmCanvas object.
 * Each #PgmDrawable added to the canvas is rendered on the viewport converting
 * canvas coordinates to display coordinates. Viewport is a base abstract class
 * from which implementation plugins inherit. You can instantiate several
 * viewport implementations through a #PgmViewportFactory. A viewport can handle
 * 0 or 1 canvas, if there's no canvas binded, nothing is projected except
 * the background color of the viewport.
 * </para>
 * <title>Implementation capacities</title>
 * <para>
 * Each viewport implementations are based on a dedicated graphical library such
 * as DirectFB, OpenGL or Direct3D to project the binded canvas. These libraries
 * can be completely different, some of them being 2d-based generating their
 * output through blit operations, other being 3d-based generating their output
 * through a 3d pipeline. Each of them can also optimize part of their rendering
 * paths thanks to dedicated hardware. These differences can lead to viewport
 * implementations with different capacities. The result being that some
 * properties of a drawable could not be correctly projected by some
 * implementations. You can retrieve the mask capacities handled by a Viewport
 * at runtime with the pgm_viewport_get_caps_mask() method, you can check that a
 * mask of capacities is handled thanks to the PGM_VIEWPORT_HAS_CAPS() macro,
 * and you can get the list of handled color spaces with the
 * pgm_viewport_get_pixel_formats() method.
 * </para>
 * <title>Event handling</title>
 * <para>
 * A viewport is responsible of the event handling. It catches the events
 * generated by the underlying backend, and converts them building Pigment
 * events, which are then dispatched through signals. You can connect functions
 * to these signals using g_signal_connect() , in UI programming these
 * functions are often called callbacks. Each time an event is generated, the
 * corresponding signal is emited and all the connected callbacks are called.
 * </para>
 * <para>
 * You are also responsible of the update of the viewport and you have to call
 * the pgm_viewport_update() method to force the refresh of the projection of
 * the canvas on the viewport. You can call this function after each bunch of
 * changes on your drawables, but you can also add a dedicated timeout to update
 * your viewport using g_timeout_add().
 * </para>
 * <title>Viewport size</title>
 * <para>
 * Practically speaking, the viewport is basically either a window on your
 * screen or your complete monitor/TV. When you define the size of the viewport,
 * the implementation will try to change the window size, or the video mode if
 * you set it to be fullscreen. Some viewport implementation are not able to
 * obtain the screen size in millimeters and you have to set it, so that the
 * viewport can calculate the appropriate projections taking in account non
 * square pixels.
 * </para>
 * <title>Recommended usage of viewport size</title>
 * <para>
 * Applications are strongly encouraged to use the viewport the following way:
 * <itemizedlist>
 * <listitem>
 * <para>
 * The application initializes Pigment using pgm_init and then try to get
 * the screen size in millimeters, the screen resolution and the viewport size
 * (window size or fullscreen size). Looking at those parameters the application
 * can calculate the pixel aspect ratio of the screen and figure out the visual
 * aspect ratio of the viewport (4:3, 16:9, etc).
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * When the application has defined the visual aspect ratio of the viewport, it
 * can decide the visual aspect ratio it's going to generate and set the canvas
 * size accordingly. Let's say for example that the viewport size is a 750x400
 * pixels window and that we have square pixels, this is a bit wider than 16:9.
 * The application decides to generate a 16:9 style user interface and sets the
 * canvas to 16.0x9.0. The viewport will draw black borders and project that
 * canvas as a 711x400 pixels large rectangle in the middle of that window.
 * </para>
 * </listitem>
 * <listitem>
 * <para>
 * If the viewport size changes (window resize), the application can decide to
 * use another aspect ratio or stick with the current one.
 * </para>
 * </listitem>
 * </itemizedlist>
 * </para>
 * </refsect2>
 *
 * Last reviewed on 2007-06-10 (0.1.5)
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <unistd.h>      /* pipe, close */
#include "pgmviewport.h"
#include "pgmimage.h"
#include "pgmmarshal.h"

/* Multi-press event constants */
#define MULTI_PRESS_DELAY    250 /* in milliseconds */
#define MULTI_PRESS_DISTANCE   5 /* in pixels */

GST_DEBUG_CATEGORY (pgm_viewport_debug);
#define GST_CAT_DEFAULT pgm_viewport_debug

/* Viewport signals */
enum {
  BUTTON_PRESS_EVENT,
  BUTTON_RELEASE_EVENT,
  KEY_PRESS_EVENT,
  KEY_RELEASE_EVENT,
  CONFIGURE_EVENT,
  DELETE_EVENT,
  SCROLL_EVENT,
  MOTION_NOTIFY_EVENT,
  EXPOSE_EVENT,
  LAST_SIGNAL
};

/* Mouse picking processing function as its defined in pgmdrawable.h */
typedef void (*DoPickingFunc) (PgmDrawable *drawable,
                               PgmEvent *event,
                               PgmVec3 *p1,
                               PgmVec3 *p2,
                               guint16 *emission_mask);

static GstObjectClass *parent_class = NULL;
static guint pgm_viewport_signals[LAST_SIGNAL] = { 0 };

/* Private functions */

/* Call a drawable picking function for each drawables in a layer */
static void
layer_foreach_do_picking (PgmCanvas *canvas,
                          GList *layer,
                          DoPickingFunc do_picking,
                          PgmEvent *event,
                          PgmVec3 *p1,
                          PgmVec3 *p2,
                          guint16 *emission_mask)
{
  PgmDrawable *drawable;
  GList *list, *walk;

  /* Get a copy of the layer */
  GST_OBJECT_LOCK (canvas);
  list = g_list_copy (layer);
  GST_OBJECT_UNLOCK (canvas);

  /* Reverse it so that we go from the nearest to the farthest */
  list = g_list_reverse (list);

  /* Iterate picking on the drawables. Since the user can return TRUE in its
   * signal handler to prevent further propagation of the drawable picking
   * signals, the signal emission mask allows to filter signals which still
   * need to be emitted. */
  walk = list;
  while (walk)
    {
      drawable = (PgmDrawable*) walk->data;
      do_picking (drawable, event, p1, p2, emission_mask);
      walk = walk->next;
    }

  /* And clean up */
  g_list_free (list);
  walk = NULL;
}

/* Popagate a mouse event to the canvas to process picking */
static void
propagate_picking_event (PgmViewport *viewport,
                         PgmEvent *event)
{
  PgmCanvas *canvas = viewport->canvas;
  PgmEventButton *button = (PgmEventButton*) event;
  guint16 emission_mask = PGM_DRAWABLE_PICKING_MASK;
  gfloat x1, y1, z1, x2, y2, z2;
  PgmVec3 p1, p2;

  /* Check if there's a canvas bound */
  if (G_UNLIKELY (!viewport->canvas))
    return;

  /* Create the picking line in canvas space */
  pgm_viewport_to_canvas (viewport, &x1, &y1, &z1, button->x, button->y, 0.0f);
  PGM_VEC3_INIT (p1, x1, y1, z1);
  pgm_viewport_to_canvas (viewport, &x2, &y2, &z2, button->x, button->y, 1.0f);
  PGM_VEC3_INIT (p2, x2, y2, z2);

  /* Propagate the picking to each drawables */
  switch (event->type)
    {
    case PGM_BUTTON_PRESS:
      layer_foreach_do_picking (canvas, canvas->near,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far,
                                (DoPickingFunc) _pgm_drawable_do_press_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_BUTTON_RELEASE:
      layer_foreach_do_picking (canvas, canvas->near,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far,
                                (DoPickingFunc) _pgm_drawable_do_release_event,
                                event, &p1, &p2, &emission_mask);
      break;
    case PGM_MOTION_NOTIFY:
      layer_foreach_do_picking (canvas, canvas->near,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->middle,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      layer_foreach_do_picking (canvas, canvas->far,
                                (DoPickingFunc) _pgm_drawable_do_motion_event,
                                event, &p1, &p2, &emission_mask);
      break;

    default:
      break;
    }
}

/* Emit double and triple button press event if needed */
static void
emit_multi_press_event (PgmViewport *viewport,
                        PgmEventButton *event)
{
  PgmEvent *new_event = NULL;
  /* Delay */
  static guint32 first_time = 0;
  static guint32 second_time = 0;
  guint32 current_time = event->time;
  /* Distance */
  static gint first_x = 0;
  static gint first_y = 0;
  gint current_x = event->x;
  gint current_y = event->y;

  /* Second click */
  if (current_time - first_time < MULTI_PRESS_DELAY
      && ABS (current_x - first_x) < MULTI_PRESS_DISTANCE
      && ABS (current_y - first_y) < MULTI_PRESS_DISTANCE)
    {
      new_event = pgm_event_copy ((PgmEvent *) event);
      new_event->type = PGM_DOUBLE_BUTTON_PRESS;
      g_signal_emit (G_OBJECT (viewport),
                     pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, new_event);
      pgm_event_free (new_event);
      second_time = current_time;
    }
  /* Third click */
  else if (current_time - second_time < MULTI_PRESS_DELAY
           && ABS (current_x - first_x) < MULTI_PRESS_DISTANCE
           && ABS (current_y - first_y) < MULTI_PRESS_DISTANCE)
    {
      new_event = pgm_event_copy ((PgmEvent *) event);
      new_event->type = PGM_TRIPLE_BUTTON_PRESS;
      g_signal_emit (G_OBJECT (viewport),
                     pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, new_event);
      pgm_event_free (new_event);
      first_time = 0;
      second_time = 0;
      first_x = 0;
      first_y = 0;
    }
  /* First click */
  else
    {
      first_time = current_time;
      first_x = current_x;
      first_y = current_y;
    }
}

/* IO watch callback for events. It dispatches the events from the queue,
 * emitting the correct signal for each of them. The queue is then freed. */
static gboolean
do_events (GIOChannel *source,
           GIOCondition condition,
           gpointer data)
{
  PgmViewport *viewport = PGM_VIEWPORT (data);
  GList *events, *walk;
  gchar buf;

  g_io_channel_read_chars (source, &buf, 1, NULL, NULL);

  /* Reverse the list since we prepend */
  g_mutex_lock (viewport->event_lock);
  events = g_list_reverse (viewport->event);
  viewport->event = NULL;
  g_mutex_unlock (viewport->event_lock);

  /* Dispatch all the events from the list */
  walk = events;
  while (walk)
    {
      PgmEvent *event = (PgmEvent *) walk->data;

      switch (event->type)
        {
        case PGM_MOTION_NOTIFY:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[MOTION_NOTIFY_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_KEY_PRESS:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[KEY_PRESS_EVENT], 0, event);
          break;

        case PGM_KEY_RELEASE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[KEY_RELEASE_EVENT], 0, event);
          break;

        case PGM_SCROLL:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[SCROLL_EVENT], 0, event);
          break;

        case PGM_BUTTON_PRESS:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[BUTTON_PRESS_EVENT], 0, event);
          emit_multi_press_event (viewport, (PgmEventButton *) event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_BUTTON_RELEASE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[BUTTON_RELEASE_EVENT], 0, event);
          propagate_picking_event (viewport, event);
          break;

        case PGM_EXPOSE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[EXPOSE_EVENT], 0, event);
          break;

        case PGM_CONFIGURE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[CONFIGURE_EVENT], 0, event);
          break;

        case PGM_DELETE:
          g_signal_emit (G_OBJECT (viewport),
                         pgm_viewport_signals[DELETE_EVENT], 0, event);
          break;

        default:
          break;
        }

      pgm_event_free (event);
      walk = walk->next;
    }

  g_list_free (events);
  events = NULL;

  return TRUE;
}

/* Canvas "size-changed" handler */
static void
canvas_size_changed_cb (PgmCanvas *canvas,
                        gpointer data)
{
  PgmViewport *viewport = PGM_VIEWPORT (data);

  pgm_viewport_update_projection (viewport);
}

/* GObject stuff */

G_DEFINE_TYPE (PgmViewport, pgm_viewport, GST_TYPE_OBJECT);

static void
pgm_viewport_dispose (GObject *object)
{
  PgmViewport *viewport = PGM_VIEWPORT (object);

  GST_OBJECT_LOCK (viewport);

  /* Event data cleaning */
  if (G_LIKELY (viewport->source))
    {
      g_source_remove (viewport->source);
      viewport->source = 0;
    }
  if (G_LIKELY (viewport->channel_out))
    {
      g_io_channel_unref (viewport->channel_out);
      viewport->channel_out = NULL;
      close (viewport->event_fd[0]);
    }
  if (G_LIKELY (viewport->channel_in))
    {
      g_io_channel_unref (viewport->channel_in);
      viewport->channel_in = NULL;
      close (viewport->event_fd[1]);
    }

  /* Frees remaining events */
  if (viewport->event)
    {
      GList *list = viewport->event;
      while (list)
        {
          pgm_event_free ((PgmEvent *) list->data);
          list = list->next;
        }
      g_list_free (viewport->event);
      viewport->event = NULL;
    }

  g_mutex_free (viewport->event_lock);

  if (viewport->icon)
    g_object_unref (viewport->icon);

  g_free (viewport->title);
  viewport->title = NULL;

  GST_OBJECT_UNLOCK (viewport);

  if (viewport->canvas)
    gst_object_unref (GST_OBJECT_CAST (viewport->canvas));

  gst_object_unref (GST_OBJECT_CAST (viewport->factory));

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_viewport_class_init (PgmViewportClass *klass)
{
  GObjectClass *gobject_class = (GObjectClass *) klass;

  parent_class = g_type_class_peek_parent (klass);

  /**
   * PgmViewport::button-press-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventButton
   *
   * Will be emitted on a mouse button press on the @viewport.
   */
  pgm_viewport_signals[BUTTON_PRESS_EVENT] =
    g_signal_new ("button-press-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, button_press_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::button-release-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventButton
   *
   * Will be emitted on a mouse button release on the @viewport.
   */
  pgm_viewport_signals[BUTTON_RELEASE_EVENT] =
    g_signal_new ("button-release-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, button_release_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::key-press-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventKey
   *
   * Will be emitted on a keyboard press on the @viewport.
   */
  pgm_viewport_signals[KEY_PRESS_EVENT] =
    g_signal_new ("key-press-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, key_press_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::key-release-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventKey
   *
   * Will be emitted on a keyboard release on the @viewport.
   */
  pgm_viewport_signals[KEY_RELEASE_EVENT] =
    g_signal_new ("key-release-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, key_release_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::configure-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventConfigure
   *
   * Will be emitted on a @viewport position or size change.
   */
  pgm_viewport_signals[CONFIGURE_EVENT] =
    g_signal_new ("configure-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, configure_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::delete-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventDelete
   *
   * Will be emitted on a @viewport close request (ie the user clicked the
   * close button of the window manager).
   */
  pgm_viewport_signals[DELETE_EVENT] =
    g_signal_new ("delete-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, delete_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::scroll-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventScroll
   *
   * Will be emitted on a mouse scroll on the @viewport.
   */
  pgm_viewport_signals[SCROLL_EVENT] =
    g_signal_new ("scroll-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, scroll_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::motion-notify-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventMotion
   *
   * Will be emitted on a mouse move on the @viewport.
   */
  pgm_viewport_signals[MOTION_NOTIFY_EVENT] =
    g_signal_new ("motion-notify-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, motion_notify_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);
  /**
   * PgmViewport::expose-event:
   * @viewport: the #PgmViewport
   * @event: the #PgmEventExpose
   *
   * Will be emitted when the @viewport visibility status has changed.
   */
  pgm_viewport_signals[EXPOSE_EVENT] =
    g_signal_new ("expose-event", G_TYPE_FROM_CLASS (klass),
                  G_SIGNAL_RUN_LAST,
                  G_STRUCT_OFFSET (PgmViewportClass, expose_event),
                  NULL, NULL, pgm_marshal_VOID__BOXED, G_TYPE_NONE,
                  1, PGM_TYPE_EVENT | G_SIGNAL_TYPE_STATIC_SCOPE);

  /* Default projection matrix */
  /* FIXME: How is this matrix computed? */
  PGM_MAT4X4_INIT (klass->default_projection,
                   763.94f, 0.0f,     0.0f,  0.0f,
                   0.0f,    763.94f,  0.0f,  0.0f,
                   0.0f,    0.0f,    -1.1f, -553.22f,
                   0.0f,    0.0f,    -1.0f,  0.0f);

  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_viewport_dispose);

  GST_DEBUG_CATEGORY_INIT (pgm_viewport_debug, "pgm_viewport", 0,
                           "viewport object");
}

static void
pgm_viewport_init (PgmViewport *viewport)
{
  PgmViewportClass *klass = PGM_VIEWPORT_GET_CLASS (viewport);

  /* Public data default values */
  viewport->canvas = NULL;
  viewport->factory = NULL;
  viewport->width = 800;
  viewport->height = 600;
  viewport->width_mm = -1;
  viewport->height_mm = -1;
  viewport->projected_width = 800;
  viewport->projected_height = 600;
  viewport->projected_x = 0;
  viewport->projected_y = 0;
  viewport->viewport_ratio = 4.0f/3.0f;
  viewport->canvas_ratio = 4.0f/3.0f;
  viewport->pixel_aspect_ratio = 1.0f;
  viewport->title = g_strdup ("");
  viewport->cursor = PGM_VIEWPORT_LEFT_ARROW;
  viewport->icon = NULL;
  viewport->fullscreen = FALSE;
  viewport->visible = FALSE;

  /* Projection matrix */
  PGM_MAT4X4_COPY (&viewport->projection, &klass->default_projection);

  /* Event dedicated lock */
  viewport->event_lock = g_mutex_new ();

  /* Event handling default values */
  viewport->event = NULL;
  viewport->event_fd[0] = -1;
  viewport->event_fd[1] = -1;
  viewport->channel_in = NULL;
  viewport->channel_out = NULL;
  viewport->source = 0;

  /* Create a pipe to silently watch for new events */
  if (G_UNLIKELY (pipe (viewport->event_fd) == -1))
    {
      GST_ERROR_OBJECT (viewport, "cannot create the event pipe");
      return;
    }

  /* Then build an IO channel for the writing */
  viewport->channel_in = g_io_channel_unix_new (viewport->event_fd[1]);
  if (G_UNLIKELY (!viewport->channel_in))
    {
      GST_ERROR_OBJECT (viewport, "cannot create the input channel");
      return;
    }

  /* Build an IO channel for the reading */
  viewport->channel_out = g_io_channel_unix_new (viewport->event_fd[0]);
  if (G_UNLIKELY (!viewport->channel_in))
    {
      GST_ERROR_OBJECT (viewport, "cannot create the output channel");
      return;
    }

  /* And add a watch to dispatch in the main loop (using the default
   * GMainContext) the events that have been pushed */
  viewport->source = g_io_add_watch (viewport->channel_out, G_IO_IN, do_events,
                                     viewport);
}

/**
 * pgm_viewport_update:
 * @viewport: a #PgmViewport object.
 *
 * Redraws all the #Drawable owned by the canvas on the @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_update (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->update)
    ret = klass->update (viewport);

  return ret;
}

/**
 * pgm_viewport_set_title:
 * @viewport: a #PgmViewport object.
 * @title: the title.
 *
 * Sets the title appearing in the title bar and in the iconified
 * window list. The string is copied and can be freed when the call returns.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_title (PgmViewport *viewport,
                        const gchar *title)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (G_UNLIKELY (viewport->title))
    g_free (viewport->title);
  viewport->title = g_strdup (title);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_title)
    ret = klass->set_title (viewport, title);

  return ret;
}

/**
 * pgm_viewport_get_title:
 * @viewport: a #PgmViewport object.
 * @title: a pointer to a pointer to a gchar where the title string will be
 * stored. g_free() after use.
 *
 * Retrieves the title of @viewport in @title.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_title (PgmViewport *viewport,
                        gchar **title)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (title != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *title = g_strdup (viewport->title);

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_show:
 * @viewport: a #PgmViewport object.
 *
 * Makes @viewport visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_show (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->visible = TRUE;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->show)
    ret = klass->show (viewport);

  return ret;
}

/**
 * pgm_viewport_hide:
 * @viewport: a #PgmViewport object.
 *
 * Makes @viewport invisible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_hide (PgmViewport *viewport)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->visible = FALSE;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->hide)
    ret = klass->hide (viewport);

  return ret;
}

/**
 * pgm_viewport_is_visible:
 * @viewport: a #PgmViewport object.
 * @visible: a pointer to a #gboolean where the visible state of the viewport
 * is going to be stored.
 *
 * Retrieves whether @viewport is visible.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_is_visible (PgmViewport *viewport,
                         gboolean *visible)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (visible != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *visible = viewport->visible;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_cursor:
 * @viewport: a #PgmViewport object.
 * @cursor: the cursor to set.
 *
 * Sets @cursor as the current cursor of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_cursor (PgmViewport *viewport,
                         PgmViewportCursor cursor)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->cursor == cursor)
    {
      GST_OBJECT_UNLOCK (viewport);
      return ret;
    }

  viewport->cursor = cursor;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_cursor)
    ret = klass->set_cursor (viewport, cursor);

  return ret;
}

/**
 * pgm_viewport_get_cursor:
 * @viewport: a #PgmViewport object.
 * @cursor: a pointer #PgmViewportCursor where the @viewport cursor is going
 * to be stored.
 *
 * Retrieves the @cursor of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_cursor (PgmViewport *viewport,
                         PgmViewportCursor *cursor)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (cursor != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *cursor = viewport->cursor;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_icon:
 * @viewport: a #PgmViewport object.
 * @icon: a #GdkPixbuf object to set as icon for @viewport.
 *
 * Sets @icon as the current icon of @viewport (shown by the window manager
 * e.g. when the @viewport window is minimised).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_icon (PgmViewport *viewport,
                       GdkPixbuf *icon)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->icon == icon)
    {
      GST_OBJECT_UNLOCK (viewport);
      return ret;
    }

  if (viewport->icon)
    g_object_unref (viewport->icon);

  if (icon)
    viewport->icon = g_object_ref (icon);
  else
    viewport->icon = NULL;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_icon)
    ret = klass->set_icon (viewport, icon);

  return ret;
}

/**
 * pgm_viewport_get_icon:
 * @viewport: a #PgmViewport object.
 * @icon: a double pointer #GdkPixbuf where the @viewport icon is going
 * to be stored.
 *
 * Retrieves the @icon of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_icon (PgmViewport *viewport,
                       GdkPixbuf **icon)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (icon != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *icon = viewport->icon;

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_size:
 * @viewport: a #PgmViewport object.
 * @width: the viewport width.
 * @height: the viewport height.
 *
 * Sets @viewport size in pixels to (@width,@height).
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_size (PgmViewport *viewport,
                       gint width,
                       gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_X;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (viewport->width == width && viewport->height == height)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_OK;
    }

  viewport->width = MAX (1, width);
  viewport->height = MAX (1, height);

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_size)
    ret = klass->set_size (viewport, viewport->width, viewport->height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_size:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the viewport width in pixels is
 * going to be stored.
 * @height: a pointer to a #gint where the viewport height in pixels is
 * going to be stored.
 *
 * Retrieves the size (@width,@height) of @viewport in pixels.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_size (PgmViewport *viewport,
                       gint *width,
                       gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *width = viewport->width;
  *height = viewport->height;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_size)
    ret = klass->get_size (viewport, width, height);

  return ret;
}

/**
 * pgm_viewport_set_fullscreen:
 * @viewport: a #PgmViewport object.
 * @fullscreen: the fullscreen state.
 *
 * Sets/unsets fullscreen mode of @viewport function of @fullscreen.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_fullscreen (PgmViewport *viewport,
                             gboolean fullscreen)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->fullscreen = fullscreen;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_fullscreen)
    ret = klass->set_fullscreen (viewport, fullscreen);

  return ret;
}

/**
 * pgm_viewport_get_fullscreen:
 * @viewport: a #PgmViewport object.
 * @fullscreen: a pointer to a #gboolean where the fullscreen state is going
 * to be stored.
 *
 * Retrieves the fullscreen state of @viewport in @fullscreen.
 *
 * MT safe.
 *
 * Returns: A #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_fullscreen (PgmViewport *viewport,
                             gboolean *fullscreen)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (fullscreen != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  *fullscreen = viewport->fullscreen;

  GST_OBJECT_UNLOCK (viewport);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_fullscreen)
    ret = klass->get_fullscreen (viewport, fullscreen);

  return ret;
}

/**
 * pgm_viewport_set_screen_resolution:
 * @viewport: a #PgmViewport object.
 * @width: the screen width in pixels.
 * @height: the screen height in pixels.
 *
 * Sets a new resolution (@width,@height) of @viewport in pixels. This is
 * changing the video mode or the display resolution so you are strongly
 * encouraged to restore it to original value when exiting.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_screen_resolution (PgmViewport *viewport,
                                    gint width,
                                    gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_screen_resolution)
    ret = klass->set_screen_resolution (viewport, width, height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_screen_resolution:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the screen width in pixels is going to be
 * stored.
 * @height: a pointer to a #gint where the screen height in pixels is going to
 * be stored.
 *
 * Retrieves the resolution (@width,@height) of @viewport in pixels.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_screen_resolution (PgmViewport *viewport,
                                    gint *width,
                                    gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_screen_resolution)
    ret = klass->get_screen_resolution (viewport, width, height);

  return ret;
}

/**
 * pgm_viewport_set_screen_size_mm:
 * @viewport: a #PgmViewport object.
 * @width: the screen width in millimeters.
 * @height: the screen height in millimeters.
 *
 * Sets the physical screen size in millimeters. This is used if the @viewport
 * implementation cannot retrieve that information by itself. You have to
 * define the physical ratio of the screen so that @viewport can calculate the
 * pixel aspect ratio.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_screen_size_mm (PgmViewport *viewport,
                                 gint width,
                                 gint height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  viewport->width_mm = width;
  viewport->height_mm = height;

  GST_OBJECT_UNLOCK (viewport);

  GST_DEBUG_OBJECT (viewport, "forcing screen size to %d mm x %d mm", width,
                    height);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->set_screen_size_mm)
    ret = klass->set_screen_size_mm (viewport, width, height);

  pgm_viewport_update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_get_screen_size_mm:
 * @viewport: a #PgmViewport object.
 * @width: a pointer to a #gint where the screen width in millimeters is going
 * to be stored.
 * @height: a pointer to a #gint where the screen height in millimeters is going
 * to be stored.
 *
 * Retrieves the physical (@width,@height) size of @viewport in millimeters.
 * If the plugin can not provide that information it will return -1 as width
 * and height. In that case the application should provide the screen size
 * through pgm_viewport_set_screen_size_mm().
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_screen_size_mm (PgmViewport *viewport,
                                 gint *width,
                                 gint *height)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (width != NULL, PGM_ERROR_X);
  g_return_val_if_fail (height != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  /* If we have a screen size defined already (cached or forced), we use it */
  if (viewport->width_mm != -1 && viewport->height_mm != -1)
    {
      *width = viewport->width_mm;
      *height = viewport->height_mm;
      GST_OBJECT_UNLOCK (viewport);
      GST_DEBUG_OBJECT (viewport, "screen size %dx%d mm (from cache)",
                        *width, *height);
    }
  else
    {
      GST_OBJECT_UNLOCK (viewport);

      klass = PGM_VIEWPORT_GET_CLASS (viewport);

      /* Otherwise we get it from plugin */
      if (klass->get_screen_size_mm)
        ret = klass->get_screen_size_mm (viewport, width, height);

      /* If it's a valid size we cache it */
      if (*width != -1 && *height != -1)
        {
          GST_OBJECT_LOCK (viewport);

          viewport->width_mm = *width;
          viewport->height_mm = *height;

          GST_OBJECT_UNLOCK (viewport);

          GST_DEBUG_OBJECT (viewport, "caching screen size %dx%d mm", *width,
                            *height);
        }
    }

  return ret;
}

/**
 * pgm_viewport_push_event:
 * @viewport: a #PgmViewport object.
 * @event: the #PgmEvent to push.
 *
 * Push an event in the list.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_push_event (PgmViewport *viewport,
                         PgmEvent *event)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (event != NULL, PGM_ERROR_X);

  g_mutex_lock (viewport->event_lock);
  viewport->event = g_list_prepend (viewport->event, event);
  g_mutex_unlock (viewport->event_lock);

  if (G_LIKELY (viewport->channel_in))
    {
      g_io_channel_write_chars (viewport->channel_in, "#", 1, NULL, NULL);
      g_io_channel_flush (viewport->channel_in, NULL);
    }

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_set_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas: the #PgmCanvas to bind.
 *
 * Sets the @canvas to be projected by the @viewport. This function increases
 * the refcount on the canvas. Any previously set canvas on @viewport is
 * unreffed. If @canvas is NULL, the previously set canvas is unreffed.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_set_canvas (PgmViewport *viewport,
                         PgmCanvas *canvas)
{
  PgmViewportClass *klass;
  PgmCanvas *tmp = NULL;
  PgmError ret = PGM_ERROR_OK;
  gulong pixel_formats = 0;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  GST_OBJECT_LOCK (viewport);

  /* Is it the canvas already binded? If yes just return. */
  if (viewport->canvas == canvas)
    {
      GST_OBJECT_UNLOCK (viewport);
      return PGM_ERROR_X;
    }

  /* Get the viewport pixel formats */
  if (klass->get_pixel_formats)
    {
      GST_OBJECT_UNLOCK (viewport);
      klass->get_pixel_formats (viewport, &pixel_formats);
      GST_OBJECT_LOCK (viewport);
    }

  /* If there was a previous canvas */
  if (viewport->canvas)
    {
      /* Backup the pointer */
      tmp = viewport->canvas;
      /* And update the previous canvas pixel formats mask */
      _pgm_canvas_remove_pixel_formats (tmp, pixel_formats);
      /* Disconnect "size-changed" handler */
      g_signal_handler_disconnect (tmp, viewport->canvas_size_handler);
    }

  /* If there's a new canvas */
  if (canvas)
    {
      /* Bind it taking a refcount */
      viewport->canvas = gst_object_ref (GST_OBJECT_CAST (canvas));
      /* Update the new canvas pixel formats mask */
      _pgm_canvas_add_pixel_formats (canvas, pixel_formats);
      /* Connect "size-changed" handler */
      viewport->canvas_size_handler =
        g_signal_connect (G_OBJECT (canvas), "size-changed",
                          G_CALLBACK (canvas_size_changed_cb),
                          (gpointer) viewport);
    }
  else
    viewport->canvas = NULL;

  GST_OBJECT_UNLOCK (viewport);

  /* Let the plugin do what it needs with the canvas */
  if (klass->set_canvas)
    ret = klass->set_canvas (viewport, canvas);

  /* And then, unref the previous canvas */
  if (tmp)
    gst_object_unref (GST_OBJECT_CAST (tmp));

  pgm_viewport_update_projection (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas: a pointer to #PgmCanvas pointer in which the canvas bound to
 * @viewport is going to be stored. The refcount of canvas is increased.
 *
 * Retrieves the current canvas or NULL if there's no canvas bound.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_canvas (PgmViewport *viewport,
                         PgmCanvas **canvas)
{
  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (canvas != NULL, PGM_ERROR_X);

  GST_OBJECT_LOCK (viewport);

  if (!viewport->canvas)
    {
      GST_DEBUG_OBJECT (viewport, "no canvas binded");
      *canvas = NULL;
    }
  else
    *canvas = gst_object_ref (GST_OBJECT_CAST (viewport->canvas));

  GST_OBJECT_UNLOCK (viewport);

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_update_projection:
 * @viewport: a #PgmViewport object.
 *
 * Update the projection. The projection update is done automatically by
 * Pigment, this function is only useful for plugin that have changed for
 * instance the size of the viewport, and want to adapt the projection
 * accordingly. 
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_update_projection (PgmViewport *viewport)
{
  PgmCanvas *canvas = viewport->canvas;
  PgmError ret = PGM_ERROR_X;
  gint resolution_width;
  gint resolution_height;
  gint screen_width;
  gint screen_height;
  gfloat tmp;
  PgmVec3 v;

  PgmViewportClass *klass = PGM_VIEWPORT_GET_CLASS (viewport);

  /* Get resolution and screen size */
  pgm_viewport_get_screen_resolution (viewport, &resolution_width,
                                      &resolution_height);
  pgm_viewport_get_screen_size_mm (viewport, &screen_width, &screen_height);

  GST_OBJECT_LOCK (viewport);

  /* Get pixel-aspect-ratio function of the resolution and screen size */
  viewport->pixel_aspect_ratio = ((gfloat) resolution_width / resolution_height)
    * ((gfloat) screen_height / screen_width);

  /* Viewport ratio */
  viewport->viewport_ratio = (gfloat) viewport->width / viewport->height;

  /* Canvas ratio */
  if (canvas)
    {
      tmp = canvas->width * viewport->pixel_aspect_ratio;
      viewport->canvas_ratio = tmp / canvas->height;
    }
  else
    viewport->canvas_ratio = 4.0f / 3.0f * viewport->pixel_aspect_ratio;

  /* Projection position and size */
  if (viewport->viewport_ratio >= viewport->canvas_ratio)
    {
      tmp = viewport->width - viewport->height * viewport->canvas_ratio;
      viewport->projected_x = tmp * 0.5f;
      viewport->projected_y = 0;
      viewport->projected_width = viewport->width - tmp;
      viewport->projected_height = viewport->height;
    }
  else
    {
      tmp = viewport->height - viewport->width / viewport->canvas_ratio;
      viewport->projected_x = 0;
      viewport->projected_y = tmp * 0.5f;
      viewport->projected_width = viewport->width;
      viewport->projected_height = viewport->height - tmp;
    }

  /* Update projection matrix */
  PGM_MAT4X4_COPY (&viewport->projection, &klass->default_projection);
  if (canvas)
    {
      PGM_VEC3_INIT (v, 2.0f / canvas->width, -2.0f / canvas->height, 1.0f);
      pgm_mat4x4_scale (&viewport->projection, &v);
      PGM_VEC3_INIT (v, -canvas->width / 2.0f, -canvas->height / 2.0f,
                     -763.9432905087f);
      pgm_mat4x4_translate (&viewport->projection, &v);
      pgm_mat4x4_inverse (&viewport->inv_projection, &viewport->projection);
    }

  GST_OBJECT_UNLOCK (viewport);

  /* Let the plugin apply the parameters */
  if (klass->update_projection)
    ret = klass->update_projection (viewport);

  return ret;
}

/**
 * pgm_viewport_from_canvas:
 * @viewport: a #PgmViewport object.
 * @viewport_x: a gfloat address to store the projected viewport x component.
 * @viewport_y: a gfloat address to store the projected viewport y component.
 * @viewport_z: a gfloat address to store the projected viewport z component.
 * @canvas_x: the canvas x component to project.
 * @canvas_y: the canvas y component to project.
 * @canvas_z: the canvas z component to project.
 *
 * Retrieves the projection of a 3-components vector in canvas coordinates on
 * the viewport in a 3-components vector in viewport coordinates.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_from_canvas (PgmViewport *viewport,
                          gfloat *viewport_x,
                          gfloat *viewport_y,
                          gfloat *viewport_z,
                          gfloat canvas_x,
                          gfloat canvas_y,
                          gfloat canvas_z)
{
  PgmVec4 viewport_vec, canvas_vec;
  PgmMat4x4 tmp;

  pgm_mat4x4_transpose (&tmp, &viewport->projection);

  /* Convert from world (canvas) coordinates to clipping coordinates */
  PGM_VEC4_INIT (canvas_vec, canvas_x, canvas_y, canvas_z, 1.0f);
  pgm_vec4_mult_mat4x4_vec4 (&viewport_vec, &tmp, &canvas_vec);

  /* Perspective divide, convert from clipping coordinates to NDC */
  viewport_vec.v[0] /= viewport_vec.v[3];
  viewport_vec.v[1] /= viewport_vec.v[3];
  viewport_vec.v[2] /= viewport_vec.v[3];

  /* Convert from NDC to viewport coordinates */
  *viewport_x = (viewport_vec.v[0] + 1.0f) / 2.0f * viewport->projected_width;
  *viewport_y = (1.0f - viewport_vec.v[1]) / 2.0f * viewport->projected_height;
  *viewport_z = (viewport_vec.v[2] + 1.0f) / 2.0f;

  /* And add the position offset */
  *viewport_x += viewport->projected_x;
  *viewport_y += viewport->projected_y;

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_to_canvas:
 * @viewport: a #PgmViewport object.
 * @canvas_x: a gfloat address to store the unprojected canvas x component.
 * @canvas_y: a gfloat address to store the unprojected canvas y component.
 * @canvas_z: a gfloat address to store the unprojected canvas z component.
 * @viewport_x: the viewport x component to project.
 * @viewport_y: the viewport y component to project.
 * @viewport_z: the viewport z component to project.
 *
 * Retrieves the projection of a 3-components vector in viewport coordinates on
 * the viewport in a 3-components vector in canvas coordinates. The z component
 * of the viewport being clamped in the range [0.0, 1.0].
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_to_canvas (PgmViewport *viewport,
                        gfloat *canvas_x,
                        gfloat *canvas_y,
                        gfloat *canvas_z,
                        gfloat viewport_x,
                        gfloat viewport_y,
                        gfloat viewport_z)
{
  PgmVec4 canvas_vec, viewport_vec;
  PgmMat4x4 tmp;

  pgm_mat4x4_transpose (&tmp, &viewport->inv_projection);

  /* Remove the position offset */
  PGM_VEC4_INIT (viewport_vec, viewport_x, viewport_y, viewport_z, 1.0f);
  viewport_vec.v[0] -= viewport->projected_x;
  viewport_vec.v[1] -= viewport->projected_y;

  /* Convert from viewport coordinates to clipping coordinates */
  viewport_vec.v[0] =
    ((2.0f * viewport_vec.v[0]) / viewport->projected_width) - 1.0f;
  viewport_vec.v[1] =
    1.0f - ((2.0f * viewport_vec.v[1]) / viewport->projected_height);
  viewport_vec.v[2] = (2.0f * viewport_vec.v[2]) - 1.0f;

  /* Convert from clipping coordinates to world (canvas) coordinates */
  pgm_vec4_mult_mat4x4_vec4 (&canvas_vec, &tmp, &viewport_vec);

  /* No guarantee that w is 1, so rescale appropriately */
  *canvas_x = canvas_vec.v[0] / canvas_vec.v[3];
  *canvas_y = canvas_vec.v[1] / canvas_vec.v[3];
  *canvas_z = canvas_vec.v[2] / canvas_vec.v[3];

  return PGM_ERROR_OK;
}

/**
 * pgm_viewport_get_pixel_formats:
 * @viewport: a #PgmViewport object.
 * @formats_mask: a pointer to a #gulong where the mask of supported
 * #PgmImagePixelFormat is going to be stored.
 *
 * Gets the list of supported pixel formats of @viewport.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_pixel_formats (PgmViewport *viewport,
                                gulong *formats_mask)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (formats_mask != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_pixel_formats)
    ret = klass->get_pixel_formats (viewport, formats_mask);

  return ret;
}

/**
 * pgm_viewport_get_caps_mask:
 * @viewport: a #PgmViewport object.
 * @caps_mask: a pointer to a #gulong where the mask of #PgmViewportCapacity is
 * going to be stored.
 *
 * Retrieves the mask of supported #PgmViewportCapacity.
 *
 * MT safe.
 *
 * Returns: a #PgmError indicating success/failure.
 */
PgmError
pgm_viewport_get_caps_mask (PgmViewport *viewport,
                            gulong *caps_mask)
{
  PgmViewportClass *klass;
  PgmError ret = PGM_ERROR_OK;

  g_return_val_if_fail (PGM_IS_VIEWPORT (viewport), PGM_ERROR_X);
  g_return_val_if_fail (caps_mask != NULL, PGM_ERROR_X);

  klass = PGM_VIEWPORT_GET_CLASS (viewport);

  if (klass->get_caps_mask)
    ret = klass->get_caps_mask (viewport, caps_mask);

  return ret;
}
