/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

/*
 * OpenGL backend for XWindow and GLX. It tries to respect the "Extended
 * Window Manager Hints" (EWMH) specification of the X Desktop Group.
 *
 * Events are handled and dispatched in a dedicated thread using a GMainLoop
 * with its own GMainContext. A GSource event is added to the loop to watch the
 * file descriptor returned by XWindow, each time XWindow writes in this one to
 * signal for new available events, the dispatcher is called.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */

#include <string.h>         /* memset */
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <X11/cursorfont.h>
#include <gdk-pixbuf-xlib/gdk-pixbuf-xlib.h>
#include "pgmglxbackend.h"
#include "pgmglviewport.h"

GST_DEBUG_CATEGORY_EXTERN (pgm_gl_debug);
#define GST_CAT_DEFAULT pgm_gl_debug

static PgmBackendClass *parent_class = NULL;

/* Prototypes */
static gboolean event_prepare  (GSource *source,
                                gint *timeout);
static gboolean event_check    (GSource *source);
static gboolean event_dispatch (GSource *source,
                                GSourceFunc callback,
                                gpointer data);

/* Event handling functions */
static GSourceFuncs event_funcs = {
  event_prepare, event_check, event_dispatch, NULL
};

/* Map GLX extension names and feature bits */
typedef struct {
  const gchar *name;
  gint         feature_mask;
} ExtensionMap;

/* GLX extensions checked at runtime */
static ExtensionMap glx_extensions_map[] = {
  { "GLX_SGIX_fbconfig",    PGM_GLX_FEAT_FBCONFIG     },
  { "GLX_SGI_video_sync",   PGM_GLX_FEAT_VIDEO_SYNC   },
  { "GLX_SGI_swap_control", PGM_GLX_FEAT_SWAP_CONTROL },
  { NULL,                   0                         }
};

/* GLX functions binding */
static PgmGlxBackendProcAddress glx_proc_address = {
  (pgm_glx_choose_fbconfig)          NULL,
  (pgm_glx_get_fbconfigs)            NULL,
  (pgm_glx_get_fbconfig_attrib)      NULL,
  (pgm_glx_get_visual_from_fbconfig) NULL,
  (pgm_glx_create_new_context)       NULL,
  (pgm_glx_get_video_sync)           NULL,
  (pgm_glx_wait_video_sync)          NULL,
  (pgm_glx_swap_interval)            NULL
};

/* Private functions */

/* Retrieve a function pointer given a procedure name */
static inline gpointer
get_proc_address (PgmGlxBackend *glxbackend,
                  const gchar *proc_name)
{
  /* FIXME: glXGetProcAddress should be loaded with dlopen. */
  return glXGetProcAddress ((const guchar *) proc_name);
}

/* Check whether an extension is supported by the GLX implementation given
 * the extension name and the list of supported extensions */
static gboolean
has_glx_extension (const gchar *extensions,
                   const gchar *extension_name)
{
  gchar *end;
  gint ext_name_len = strlen (extension_name);
  gchar *p = (gchar*) extensions;
  size_t size;

  if (!extensions)
    return FALSE;

  end = p + strlen (p);
  while (p < end)
    {
      size = strcspn (p, " ");
      if ((ext_name_len == size)  && (strncmp (extension_name, p, size) == 0))
        return TRUE;
      p += size + 1;
    }

  return FALSE;
}

/* Bind the GLX extension proc addresses depending on the features available
 * at run-time and the GLX version. */
static void
bind_glx_extensions (PgmGlxBackend *glxbackend)
{
  PgmGlxBackendProcAddress *glx = glxbackend->glx;

  /* FBConfig */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_FBCONFIG)
    {
      if (glxbackend->version >= 1.3f)
        {
          glx->choose_fbconfig = (pgm_glx_choose_fbconfig)
            get_proc_address (glxbackend, "glXChooseFBConfig");
          glx->get_fbconfigs = (pgm_glx_get_fbconfigs)
            get_proc_address (glxbackend, "glXGetFBConfigs");
          glx->get_fbconfig_attrib = (pgm_glx_get_fbconfig_attrib)
            get_proc_address (glxbackend, "glXGetFBConfigAttrib");
          glx->get_visual_from_fbconfig = (pgm_glx_get_visual_from_fbconfig)
            get_proc_address (glxbackend, "glXGetVisualFromFBConfig");
          glx->create_new_context = (pgm_glx_create_new_context)
            get_proc_address (glxbackend, "glXCreateNewContext");
        }
      else
        {
          glx->choose_fbconfig = (pgm_glx_choose_fbconfig)
            get_proc_address (glxbackend, "glXChooseFBConfigSGIX");
          glx->get_fbconfigs = (pgm_glx_get_fbconfigs)
            get_proc_address (glxbackend, "glXGetFBConfigsSGIX");
          glx->get_fbconfig_attrib = (pgm_glx_get_fbconfig_attrib)
            get_proc_address (glxbackend, "glXGetFBConfigAttribSGIX");
          glx->get_visual_from_fbconfig = (pgm_glx_get_visual_from_fbconfig)
            get_proc_address (glxbackend, "glXGetVisualFromFBConfigSGIX");
          glx->create_new_context = (pgm_glx_create_new_context)
            get_proc_address (glxbackend, "glXCreateContextWithConfigSGIX");
        }

      if ((!glx->choose_fbconfig) || (!glx->get_fbconfigs)
          || (!glx->get_fbconfig_attrib) || (!glx->get_visual_from_fbconfig)
          || (!glx->create_new_context))
        {
          glxbackend->feature_mask &= ~PGM_GLX_FEAT_FBCONFIG;
        }
    }

  /* Video sync */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
    {
      glx->get_video_sync = (pgm_glx_get_video_sync)
        get_proc_address (glxbackend, "glXGetVideoSyncSGI");
      glx->wait_video_sync = (pgm_glx_wait_video_sync)
        get_proc_address (glxbackend, "glXWaitVideoSyncSGI");

      if ((!glx->get_video_sync) || (!glx->wait_video_sync))
        glxbackend->feature_mask &= ~PGM_GLX_FEAT_VIDEO_SYNC;
    }

  /* Swap control */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
    {
      glx->swap_interval = (pgm_glx_swap_interval)
        get_proc_address (glxbackend, "glXSwapIntervalSGI");

      if (!glx->swap_interval)
        glxbackend->feature_mask &= ~PGM_GLX_FEAT_SWAP_CONTROL;
    }
}

/* Get the GLX informations and load the extensions */
static gboolean
load_glx_extensions (PgmGlxBackend *glxbackend)
{
  gint error_base, event_base;
  gint major, minor;
  gint i = 0;

  /* Are GLX extensions supported? */
  if (!glXQueryExtension (glxbackend->dpy, &error_base, &event_base))
    {
      GST_ERROR_OBJECT (glxbackend, "GLX extensions not supported");
      return FALSE;
    }

  /* Get the GLX server version */
  if (!glXQueryVersion (glxbackend->dpy, &major, &minor))
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't query GLX version");
      return FALSE;
    }

  /* Check minimum GLX version */
  glxbackend->version = major + minor / 10.0f;
  if (major < 1 || (major == 1 && minor < 2))
    {
      GST_ERROR_OBJECT (glxbackend, "GLX version %.1f (1.2 minimum required)",
                        glxbackend->version);
      return FALSE;
    }

  glxbackend->vendor = (const gchar *)
    glXGetClientString (glxbackend->dpy, GLX_VENDOR);

  /* ATI's driver emulates GLX 1.3 support */
  if (glxbackend->vendor)
    if (glxbackend->version < 1.3f)
      if (!strncmp ("ATI", glxbackend->vendor, 3))
        glxbackend->version = 1.3f;

  /* Retrieve and bind supported extensions */
  glxbackend->extensions = (const gchar *)
    glXQueryExtensionsString (glxbackend->dpy, glxbackend->screen);

  glxbackend->glx = &glx_proc_address;

  while (glx_extensions_map[i].name)
    {
      if (has_glx_extension (glxbackend->extensions,
                             glx_extensions_map[i].name))
        glxbackend->feature_mask |= glx_extensions_map[i].feature_mask;
      i++;
    }

  bind_glx_extensions (glxbackend);

  return TRUE;
}

/* Create a context the old way */
static gboolean
create_context (PgmGlxBackend *glxbackend)
{
  gint attrib[] = { GLX_RGBA, GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1,
                    GLX_BLUE_SIZE, 1, GLX_DOUBLEBUFFER, 1, GLX_DEPTH_SIZE, 1,
                    None };

  glxbackend->vi = glXChooseVisual (glxbackend->dpy, glxbackend->screen, attrib);
  if (!glxbackend->vi)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't retrieve an RGBA config");
      return FALSE;
    }

  glxbackend->ctx = glXCreateContext (glxbackend->dpy, glxbackend->vi,
                                      NULL, PGM_GL_TRUE);
  if (!glxbackend->ctx)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't create OpenGL context");
      return FALSE;
    }

  return TRUE;
}

/* Create a context the new (using fbconfig) */
static gboolean
create_context_using_fbconfig (PgmGlxBackend *glxbackend)
{
  PgmGlxBackendProcAddress *glx = glxbackend->glx;
  gint attrib[] = { GLX_DRAWABLE_TYPE, GLX_WINDOW_BIT, GLX_RENDER_TYPE,
                    GLX_RGBA_BIT, GLX_RED_SIZE, 1, GLX_GREEN_SIZE, 1,
                    GLX_BLUE_SIZE, 1, GLX_DOUBLEBUFFER, 1, GLX_DEPTH_SIZE, 1,
                    None };

  glxbackend->fbconfig = glx->choose_fbconfig (glxbackend->dpy,
                                               glxbackend->screen, attrib,
                                               &glxbackend->nbconfigs);
  if (!glxbackend->fbconfig)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't retrieve an RGBA config");
      return FALSE;
    }

  glxbackend->vi = glx->get_visual_from_fbconfig (glxbackend->dpy,
                                                  glxbackend->fbconfig[0]);

  /* FIXME: Should use glXCreateNewContext but doesn't seem to work */
  glxbackend->ctx = glXCreateContext (glxbackend->dpy, glxbackend->vi, NULL,
                                      PGM_GL_TRUE);
  if (!glxbackend->ctx)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't create OpenGL context");
      return FALSE;
    }

  return TRUE;
}

/* Setup the method used for the VBlank syncing */
static void
setup_vblank (PgmGlxBackend *glxbackend)
{
  gchar *variable;

  /* NVIDIA proprietary driver has an environment variable to automatically
   * enable VBlank syncing, in the case this variable is defined we just
   * disable it on Pigment side. */
  if (getenv ("__GL_SYNC_TO_VBLANK"))
    {
      GST_DEBUG_OBJECT (glxbackend, "__GL_SYNC_TO_VBLANK defined in the "
                        "environment, disabling vblank on pigment side");
      glxbackend->vblank_mode = PGM_VBLANK_NONE;

      return;
    }

  /* The PGM_GL_VBLANK environment variable overrides the automatic VBlank
   * method choice done by Pigment. */
  variable = getenv ("PGM_GL_VBLANK");
  if (variable)
    {
      /* VBlank disabled */
      if (variable[0] == '0')
        {
          GST_DEBUG_OBJECT (glxbackend, "PGM_GL_VBLANK disables vblank syncing");
          glxbackend->vblank_mode = PGM_VBLANK_NONE;
        }

      /* VBlank using the GLX_SGI_video_sync extension */
      else if (variable[0] == '1')
        {
          if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
            {
              GST_DEBUG_OBJECT (glxbackend, "PGM_GL_VBLANK forces use of the "
                                "video_sync extension for vblank syncing");
              glxbackend->vblank_mode = PGM_VBLANK_VIDEO_SYNC;
            }
          else
            GST_DEBUG_OBJECT (glxbackend, "PGM_GL_VBLANK mode not supported by "
                              "the GLX implementation");
        }

      /* VBlank using the GLX_SGI_swap_control extension */
      else if (variable[0] == '2')
        {
          if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
            {
              GST_DEBUG_OBJECT (glxbackend, "PGM_GL_VBLANK forces use of the "
                                "swap_control extension for vblank syncing");
              glxbackend->vblank_mode = PGM_VBLANK_SWAP_CONTROL;
              glxbackend->glx->swap_interval (1);
            }
          else
            GST_DEBUG_OBJECT (glxbackend, "PGM_GL_VBLANK mode not supported by "
                              "the GLX implementation");
        }

      return;
    }

  /* Use the swap_control extension by default */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_SWAP_CONTROL)
    {
      GST_DEBUG_OBJECT (glxbackend, "Using the swap_control extension for "
                        "vblank syncing");
      glxbackend->vblank_mode = PGM_VBLANK_SWAP_CONTROL;
      glxbackend->glx->swap_interval (1);
      return;
    }

  /* Use the video_sync extension if swap_control's not available */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_VIDEO_SYNC)
    {
      GST_DEBUG_OBJECT (glxbackend, "Using the video_sync extension for "
                        "vblank syncing");
      glxbackend->vblank_mode = PGM_VBLANK_VIDEO_SYNC;
      return;
    }

  /* Fallback to ... nothing */
  GST_DEBUG_OBJECT (glxbackend, "No extensions found to enable vblank syncing");
  glxbackend->vblank_mode = PGM_VBLANK_NONE;
}

/* Specifies the WM_PROTOCOLS property of the window */
static void
set_wm_protocols (PgmGlxBackend *glxbackend)
{
  Atom protocols[2];

  protocols[0] = XInternAtom (glxbackend->dpy, "WM_DELETE_WINDOW", False);
  protocols[1] = XInternAtom (glxbackend->dpy, "_NET_WM_PING", False);

  XSetWMProtocols (glxbackend->dpy, glxbackend->win, protocols, 2);
}

/* Updates the _NET_WM_USER_TIME property of the window */
static void
set_wm_user_time (PgmGlxBackend *glxbackend,
                  glong timestamp)
{
  if (timestamp)
    {
      Atom atom_WM_USER_TIME;

      atom_WM_USER_TIME = XInternAtom (glxbackend->dpy, "_NET_WM_USER_TIME",
                                       False);

      XChangeProperty (glxbackend->dpy, glxbackend->win, atom_WM_USER_TIME,
                       XA_CARDINAL, 32, PropModeReplace, (guchar *) &timestamp,
                       1);
    }
}

/* Handles the protocols event and returns an event to push if any */
static PgmEvent *
handle_wm_protocols_event (PgmGlxBackend *glxbackend,
                           XEvent *xevent)
{
  Atom atom = (Atom) xevent->xclient.data.l[0];
  PgmEvent *pgmevent = NULL;
  Atom atom_WM_DELETE_WINDOW;
  Atom atom_NET_WM_PING;

  atom_WM_DELETE_WINDOW = XInternAtom (glxbackend->dpy, "WM_DELETE_WINDOW",
                                       False);
  atom_NET_WM_PING = XInternAtom (glxbackend->dpy, "_NET_WM_PING", False);

  /* Delete request */
  if (atom == atom_WM_DELETE_WINDOW && xevent->xany.window == glxbackend->win)
    {
      pgmevent = pgm_event_new (PGM_DELETE);

      ((PgmEventDelete *) pgmevent)->time = xevent->xclient.data.l[1];
      set_wm_user_time (glxbackend, xevent->xclient.data.l[1]);
    }

  /* Ping request */
  else if (atom == atom_NET_WM_PING && xevent->xany.window == glxbackend->win)
    {
      xevent->xclient.window = glxbackend->root;

      XSendEvent (glxbackend->dpy, glxbackend->root, False,
                  SubstructureRedirectMask | SubstructureNotifyMask,
                  (XEvent *) &xevent->xclient);
    }

  return pgmevent;
}

/* Creates and returns a void cursor from X */
static Cursor
get_none_cursor (PgmGlxBackend *backend)
{
  Cursor none = None;
  gchar none_bits[32];
  Pixmap none_pixmap;
  XColor color;

  memset (none_bits, 0, sizeof (none_bits));
  memset (&color, 0, sizeof (color));

  none_pixmap = XCreateBitmapFromData (backend->dpy, backend->win, none_bits,
                                       16, 16);
  if (none_pixmap != None)
    {
      none = XCreatePixmapCursor (backend->dpy, none_pixmap, none_pixmap,
                                  &color, &color, 0, 0);
      XFreePixmap (backend->dpy, none_pixmap);
    }

  return none;
}

/* Frees the current X icon */
static gboolean
free_icon (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XWMHints *old_wm_hints = NULL;

  old_wm_hints = XGetWMHints (glxbackend->dpy, glxbackend->win);

  if (old_wm_hints)
    {
      if ((old_wm_hints->flags & IconPixmapHint) && old_wm_hints->icon_pixmap)
        XFreePixmap (glxbackend->dpy, old_wm_hints->icon_pixmap);
      if ((old_wm_hints->flags & IconMaskHint) && old_wm_hints->icon_mask)
        XFreePixmap (glxbackend->dpy, old_wm_hints->icon_mask);
      XFree (old_wm_hints);
    }

  return TRUE;
}

/* Removes duplicated Expose events from the queue */
static void
compress_expose_events (PgmGlxBackend *glxbackend,
                        XEvent *xevent)
{
  XEvent event;

  while (XCheckTypedWindowEvent (glxbackend->dpy, xevent->xexpose.window,
                                 Expose, &event));
}

/* Update the viewport size and the projection */
static void
update_viewport_size (PgmGlxBackend *glxbackend,
                      gint width,
                      gint height)
{
  PgmBackend *backend = PGM_BACKEND (glxbackend);
  PgmGlViewport *glviewport = backend->context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);

  GST_OBJECT_LOCK (viewport);

  if (viewport->width != width || viewport->height != height)
    {
      viewport->width = width;
      viewport->height = height;

      GST_OBJECT_UNLOCK (viewport);
      pgm_viewport_update_projection (viewport);

      return;
    }

  GST_OBJECT_UNLOCK (viewport);
}

/* Fill a PgmEventKey giving the corresponding keyboard xevent */
static void
translate_key_event (PgmGlxBackend *glxbackend,
                     PgmEventKey *event,
                     XEvent *xevent)
{
  gint index;

  event->time = xevent->xkey.time;
  event->modifier = (PgmModifierType) xevent->xkey.state;

  /* FIXME: This is a really basic support, a lot of keys are not handled */
  index = (event->modifier & (PGM_SHIFT_MASK | PGM_CAPSLOCK_MASK)) ? 1 : 0;
  event->keyval = XKeycodeToKeysym (glxbackend->dpy, xevent->xkey.keycode, index);

  event->hardware_keycode = xevent->xkey.keycode;
}

/* Fill a PgmEventMotion giving the corresponding motion xevent */
static void
translate_motion_event (PgmGlxBackend *glxbackend,
                        PgmEventMotion *event,
                        XEvent *xevent)
{
  event->time = xevent->xmotion.time;
  event->x = xevent->xmotion.x;
  event->y = xevent->xmotion.y;
}

/* Fill a PgmEventButton giving the corresponding button xevent */
static void
translate_button_press_event (PgmGlxBackend *glxbackend,
                              PgmEventButton *event,
                              XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Fill a PgmEventScroll giving the corresponding button xevent */
static void
translate_scroll_event (PgmGlxBackend *glxbackend,
                        PgmEventScroll *event,
                        XEvent *xevent)
{
  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;
}

/* Fill a PgmEventConfigure giving the corresponding configure xevent */
static void
translate_configure_event (PgmGlxBackend *glxbackend,
                           PgmEventConfigure *event,
                           XEvent *xevent)
{
  event->x = xevent->xconfigure.x;
  event->y = xevent->xconfigure.y;
  event->width = xevent->xconfigure.width;
  event->height = xevent->xconfigure.height;
}

/* Create a button event (Release) giving the corresponding button xevent */
static PgmEvent *
create_button_release_event (PgmGlxBackend *glxbackend,
                             XEvent *xevent)
{
  PgmEventButton *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_LEFT;
      break;

    case 2:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_MIDDLE;
      break;

    case 3:
      event = (PgmEventButton *) pgm_event_new (PGM_BUTTON_RELEASE);
      event->button = PGM_BUTTON_RIGHT;
      break;

    /* No release */
    default:
      return NULL;
    }

  event->time = xevent->xbutton.time;
  event->x = xevent->xbutton.x;
  event->y = xevent->xbutton.y;

  return (PgmEvent *) event;
}

/* Create a button event (Scroll, Press) given an XEvent */
static PgmEvent *
create_button_event (PgmGlxBackend *glxbackend,
                     XEvent *xevent)
{
  PgmEvent *event;

  switch (xevent->xbutton.button)
    {
    case 1:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_LEFT;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 2:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_MIDDLE;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 3:
      event = pgm_event_new (PGM_BUTTON_PRESS);
      ((PgmEventButton *) event)->button = PGM_BUTTON_RIGHT;
      translate_button_press_event (glxbackend, (PgmEventButton *) event,
                                    xevent);
      break;

    case 4:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_DOWN;
      translate_scroll_event (glxbackend, (PgmEventScroll *) event, xevent);
      break;

    case 5:
      event = pgm_event_new (PGM_SCROLL);
      ((PgmEventScroll *) event)->direction = PGM_SCROLL_UP;
      translate_scroll_event (glxbackend, (PgmEventScroll *) event, xevent);
      break;

    default:
      event = NULL;
      break;
    }

  return event;
}

/* Dispatches events pushing them on the GlViewport queue */
static gboolean
dispatch_glx_events (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmGlViewport *glviewport = backend->context->glviewport;
  PgmViewport *viewport = PGM_VIEWPORT (glviewport);

  GST_DEBUG_OBJECT (glxbackend, "dispatch_events");

  if (XPending (glxbackend->dpy))
    {
      PgmEvent *pgmevent = NULL;
      XEvent xevent;

      XNextEvent (glxbackend->dpy, &xevent);

      switch (xevent.type)
        {
        case Expose:
          compress_expose_events (glxbackend, &xevent);
          pgmevent = pgm_event_new (PGM_EXPOSE);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

        case KeyPress:
          pgmevent = pgm_event_new (PGM_KEY_PRESS);
          translate_key_event (glxbackend, (PgmEventKey *) pgmevent, &xevent);
          set_wm_user_time (glxbackend, xevent.xkey.time);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

        case KeyRelease:
          pgmevent = pgm_event_new (PGM_KEY_RELEASE);
          translate_key_event (glxbackend, (PgmEventKey *) pgmevent, &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

        case ButtonPress:
          pgmevent = create_button_event (glxbackend, &xevent);
          if (pgmevent)
            {
              set_wm_user_time (glxbackend, xevent.xbutton.time);
              pgm_viewport_push_event (viewport, pgmevent);
            }
          break;

        case ButtonRelease:
          pgmevent = create_button_release_event (glxbackend, &xevent);
          if (pgmevent)
            pgm_viewport_push_event (viewport, pgmevent);
          break;

        case MotionNotify:
          pgmevent = pgm_event_new (PGM_MOTION_NOTIFY);
          translate_motion_event (glxbackend, (PgmEventMotion *) pgmevent,
                                  &xevent);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

        case ConfigureNotify:
          pgmevent = pgm_event_new (PGM_CONFIGURE);
          translate_configure_event (glxbackend, (PgmEventConfigure *) pgmevent,
                                     &xevent);
          update_viewport_size (glxbackend,
                                ((PgmEventConfigure *) pgmevent)->width,
                                ((PgmEventConfigure *) pgmevent)->height);
          pgm_viewport_push_event (viewport, pgmevent);
          break;

        case ClientMessage:
          pgmevent = handle_wm_protocols_event (glxbackend, &xevent);
          if (pgmevent)
            pgm_viewport_push_event (viewport, pgmevent);
          break;

        default:
          break;
        }
    }

  return TRUE;
}

/* Events preparing */
static gboolean
event_prepare (GSource *source,
               gint *timeout)
{
  *timeout = -1;

  return FALSE;
}

/* Events checking */
static gboolean
event_check (GSource *source)
{
  PgmGlxBackendSource *glxbackend_source = (PgmGlxBackendSource *) source;

  return (glxbackend_source->poll_fd.revents & G_IO_IN) != 0;
}

/* Events dispatching */
static gboolean
event_dispatch (GSource *source,
                GSourceFunc callback,
                gpointer data)
{
  if (callback)
    return callback (data);

  return FALSE;
}

/* Event handling thread entry point */
static gpointer
event_loop (gpointer data)
{
  PgmGlxBackend *glxbackend = (PgmGlxBackend *) data;
  PgmGlxBackendSource *source;

  /* Context and loop creation. Note that we are not using the default context
   * to avoid conflicts with applications. */
  glxbackend->event_context = g_main_context_new ();
  glxbackend->event_loop = g_main_loop_new (glxbackend->event_context, FALSE);

  /* Create and initialize the event handling dedicated source */
  source = (PgmGlxBackendSource *)
    g_source_new (&event_funcs, sizeof (PgmGlxBackendSource));
  source->poll_fd.fd = ConnectionNumber (glxbackend->dpy);
  source->poll_fd.events = G_IO_IN;
  source->glxbackend = glxbackend;
  g_source_add_poll ((GSource *) source, &source->poll_fd);

  /* Attach it */
  g_source_set_callback ((GSource *) source, (GSourceFunc) dispatch_glx_events,
                         glxbackend, NULL);
  glxbackend->event_id = g_source_attach ((GSource *) source,
                                          glxbackend->event_context);
  g_source_unref ((GSource *) source);

  /* And enter the main loop */
  g_main_loop_run (glxbackend->event_loop);

  return NULL;
}

/* Switches the window to fullscreen depending on the fullscreen value */
static void
set_fullscreen (PgmGlxBackend *glxbackend,
                gboolean fullscreen)
{
  PgmBackend *backend = PGM_BACKEND (glxbackend);
  PgmViewport *viewport = PGM_VIEWPORT (backend->context->glviewport);
  XClientMessageEvent xclient;
  Atom atom_WM_STATE_FULLSCREEN;
  Atom atom_WM_STATE;

  atom_WM_STATE = XInternAtom (glxbackend->dpy, "_NET_WM_STATE", False);
  atom_WM_STATE_FULLSCREEN = XInternAtom (glxbackend->dpy,
                                          "_NET_WM_STATE_FULLSCREEN", False);

  /* Adapt the viewport size */
  if (glxbackend->fullscreen != fullscreen)
    {
      if (fullscreen)
        {
          glxbackend->windowed_width = viewport->width;
          glxbackend->windowed_height = viewport->height;
          update_viewport_size (glxbackend, glxbackend->resolution_width,
                                glxbackend->resolution_height);
        }
      else
        update_viewport_size (glxbackend, glxbackend->windowed_width,
                              glxbackend->windowed_height);
    }

  /* And switch the fullscreen state */
  xclient.type = ClientMessage;
  xclient.window = glxbackend->win;
  xclient.display = glxbackend->dpy;
  xclient.message_type = atom_WM_STATE;
  xclient.format = 32;

  xclient.data.l[0] = fullscreen;
  xclient.data.l[1] = atom_WM_STATE_FULLSCREEN;
  xclient.data.l[2] = None;
  xclient.data.l[3] = 0;
  xclient.data.l[4] = 0;

  XSendEvent (glxbackend->dpy, glxbackend->root, False,
              SubstructureRedirectMask | SubstructureNotifyMask,
              (XEvent *) &xclient);
  XSync (glxbackend->dpy, True);
}

/* PgmBackend methods */

static gboolean
pgm_glx_backend_create_window (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  GError *error = NULL;
  PgmViewport *viewport;
  gint width, height;

  GST_DEBUG_OBJECT (glxbackend, "create_window");

  viewport = PGM_VIEWPORT (backend->context->glviewport);

  /* Events dispatching is done in a different thread */
  if (!XInitThreads ())
    {
      GST_ERROR_OBJECT (glxbackend,
                        "unable to init Xlib support for concurrent threads");
      return FALSE;
    }

  glxbackend->dpy = XOpenDisplay (NULL);
  if (!glxbackend->dpy)
    {
      GST_ERROR_OBJECT (glxbackend, "couldn't open default display");
      return FALSE;
    }

  glxbackend->screen = DefaultScreen (glxbackend->dpy);
  glxbackend->root = DefaultRootWindow (glxbackend->dpy);

  if (!load_glx_extensions (glxbackend))
    return FALSE;

  pgm_viewport_get_size (viewport, &width, &height);

  glxbackend->win = XCreateSimpleWindow (glxbackend->dpy, glxbackend->root,
                                         0, 0, width, height, 0, 0,
                                         BlackPixel (glxbackend->dpy,
                                                     glxbackend->screen));

  XSelectInput (glxbackend->dpy, glxbackend->win,
                KeyPressMask | KeyReleaseMask | ButtonPressMask
                | ButtonReleaseMask | StructureNotifyMask | ExposureMask
                | PointerMotionMask | PropertyChangeMask);

  set_wm_protocols (glxbackend);

  /* Create the context */
  if (glxbackend->feature_mask & PGM_GLX_FEAT_FBCONFIG)
    {
      if (!create_context_using_fbconfig (glxbackend))
        return FALSE;
    }
  else if (!create_context (glxbackend))
    return FALSE;

  glXMakeCurrent (glxbackend->dpy, glxbackend->win, glxbackend->ctx);

  setup_vblank (glxbackend);

  if (glXIsDirect (glxbackend->dpy, glxbackend->ctx))
    glxbackend->feature_mask |= PGM_GLX_FEAT_DIRECT_RENDERING;

  /* Create event handling thread */
  glxbackend->event_thread = g_thread_create (event_loop, glxbackend, TRUE,
                                              &error);
  if (error != NULL)
    {
      GST_ERROR_OBJECT ("couldn't create event handling thread: %s",
                        error->message);
      return FALSE;
    }

  glxbackend->resolution_width =
    DisplayWidth (glxbackend->dpy, glxbackend->screen);
  glxbackend->resolution_height =
    DisplayHeight (glxbackend->dpy, glxbackend->screen);
  glxbackend->size_mm_width =
    DisplayWidthMM (glxbackend->dpy, glxbackend->screen);
  glxbackend->size_mm_height =
    DisplayHeightMM (glxbackend->dpy, glxbackend->screen);

  glxbackend->none_cursor = get_none_cursor (glxbackend);

  gdk_pixbuf_xlib_init (glxbackend->dpy, glxbackend->screen);

  glxbackend->created = TRUE;

  return TRUE;
}

static gboolean
pgm_glx_backend_destroy_window (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "destroy_window");

  if (glxbackend->created)
    {
      /* Leave event handling thread */
      g_source_remove (glxbackend->event_id);
      g_main_loop_quit (glxbackend->event_loop);
      g_main_loop_unref (glxbackend->event_loop);
      g_main_context_unref (glxbackend->event_context);
      g_thread_join (glxbackend->event_thread);

      /* Clean up X related data */
      XFreeCursor (glxbackend->dpy, glxbackend->none_cursor);
      XFree (glxbackend->fbconfig);
      glXDestroyContext (glxbackend->dpy, glxbackend->ctx);
      XDestroyWindow (glxbackend->dpy, glxbackend->win);
      XCloseDisplay (glxbackend->dpy);
      glxbackend->created = FALSE;
    }

  return TRUE;
}

static void
pgm_glx_backend_set_title (PgmBackend *backend,
                           const gchar *title)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  Atom atom_WM_NAME;
  Atom atom_UTF8_STRING;

  GST_DEBUG_OBJECT (glxbackend, "set_title");

  atom_WM_NAME = XInternAtom (glxbackend->dpy, "_NET_WM_NAME", False);
  atom_UTF8_STRING = XInternAtom (glxbackend->dpy, "UTF8_STRING", False);

  XChangeProperty (glxbackend->dpy, glxbackend->win, atom_WM_NAME,
                   atom_UTF8_STRING, 8, PropModeReplace, (guchar *) title,
                   strlen (title));

  XSync (glxbackend->dpy, True);
}

static void
pgm_glx_backend_hide (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "hide");

  XUnmapWindow (glxbackend->dpy, glxbackend->win);
  XSync (glxbackend->dpy, True);
}

static void
pgm_glx_backend_show (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmViewport *viewport = PGM_VIEWPORT (backend->context->glviewport);

  GST_DEBUG_OBJECT (glxbackend, "show");

  XMapWindow (glxbackend->dpy, glxbackend->win);

  /* If the fullscreen state has changed to FALSE during the hidden period */

  /* we need to resize the window to avoid showing it on the whole screen */
  if (!glxbackend->fullscreen)
    {
      gint width, height;
      pgm_viewport_get_size (viewport, &width, &height);
      XResizeWindow (glxbackend->dpy, glxbackend->win, width, height);
    }

  /* and then change accordingly the fullscreen state */
  set_fullscreen (glxbackend, glxbackend->fullscreen);
}

static void
pgm_glx_backend_swap_buffers (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "swap_buffers");

  glXSwapBuffers (glxbackend->dpy, glxbackend->win);
}

static gpointer
pgm_glx_backend_get_proc_address (PgmBackend *backend,
                                  const gchar *proc_name)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "get_proc_address");

  return get_proc_address (glxbackend, proc_name);
}

static gboolean
pgm_glx_backend_set_size (PgmBackend *backend,
                          gint width,
                          gint height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_size");

  XResizeWindow (glxbackend->dpy, glxbackend->win, width, height);
  XSync (glxbackend->dpy, True);

  return FALSE;
}

static gboolean
pgm_glx_backend_set_fullscreen (PgmBackend *backend,
                                gboolean fullscreen)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_fullscreen");

  set_fullscreen (glxbackend, fullscreen);
  glxbackend->fullscreen = fullscreen;

  return TRUE;
}

static void
pgm_glx_backend_get_screen_size_mm (PgmBackend *backend,
                                    gint *width,
                                    gint *height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "get_screen_size_mm");

  *width = glxbackend->size_mm_width;
  *height = glxbackend->size_mm_height;
}

static gboolean
pgm_glx_backend_set_screen_resolution (PgmBackend *backend,
                                       gint width,
                                       gint height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_screen_resolution");

  /* FIXME */

  return FALSE;
}

static void
pgm_glx_backend_get_screen_resolution (PgmBackend *backend,
                                       gint *width,
                                       gint *height)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "get_screen_resolution");

  *width = glxbackend->resolution_width;
  *height = glxbackend->resolution_height;
}

static gboolean
pgm_glx_backend_build_text_lists (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;
  XFontStruct *font;
  gint first;
  gint last;

  GST_DEBUG_OBJECT (glxbackend, "build_text_lists");

  glxbackend->text_lists = gl->gen_lists (256);
  if (!gl->is_list (glxbackend->text_lists))
    {
      GST_ERROR_OBJECT (glxbackend, "unable to build text display lists\n");
      return FALSE;
    }

  font = XLoadQueryFont (glxbackend->dpy, "7x13bold");
  if (!font)
    {
      GST_WARNING_OBJECT (glxbackend, "unable to load X font \"7x13bold\"\n");
      font = XLoadQueryFont (glxbackend->dpy, "fixed");
      if (!font)
        {
          GST_ERROR_OBJECT (glxbackend, "unable to load X font \"fixed\"\n");
          return FALSE;
        }
    }

  first = font->min_char_or_byte2;
  last = font->max_char_or_byte2;

  /* FIXME: Set up an error handler */
  glXUseXFont (font->fid, first, last - first + 1,
               glxbackend->text_lists + first);

  return TRUE;
}

static gboolean
pgm_glx_backend_destroy_text_lists (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (glxbackend, "destroy_text_lists");

  if (gl->is_list (glxbackend->text_lists))
    gl->delete_lists (glxbackend->text_lists, 256);

  return TRUE;
}

static void
pgm_glx_backend_raster_text (PgmBackend *backend,
                             const gchar *text,
                             gfloat x,
                             gfloat y,
                             gfloat r,
                             gfloat g,
                             gfloat b)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  PgmContextProcAddress *gl = backend->context->gl;

  GST_DEBUG_OBJECT (glxbackend, "raster_text");

  gl->load_identity ();
  gl->bind_texture (PGM_GL_TEXTURE_2D, 0);
  gl->push_attrib (PGM_GL_LIST_BIT);
  gl->color_4f (r, g, b, 1.0f);
  gl->raster_pos_2f (x, y);
  gl->list_base (glxbackend->text_lists);
  gl->call_lists (strlen (text), PGM_GL_UNSIGNED_BYTE, (PgmGlUbyte *) text);
  gl->pop_attrib ();
}

static void
pgm_glx_backend_wait_for_vblank (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "wait_for_vblank");

  if (glxbackend->vblank_mode == PGM_VBLANK_VIDEO_SYNC)
    {
      PgmGlxBackendProcAddress *glx = glxbackend->glx;
      guint retrace_count;

      glx->get_video_sync (&retrace_count);
      glx->wait_video_sync (2, (retrace_count + 1) % 2, &retrace_count);
    }
}

static gboolean
pgm_glx_backend_set_cursor (PgmBackend *backend,
                            PgmViewportCursor cursor)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (glxbackend, "set_cursor");

  switch (cursor)
    {
    case PGM_VIEWPORT_LEFT_ARROW:
      XDefineCursor (glxbackend->dpy, glxbackend->win,
                     XCreateFontCursor (glxbackend->dpy, XC_top_left_arrow));
      break;

    case PGM_VIEWPORT_INHERIT:
      XDefineCursor (glxbackend->dpy, glxbackend->win,
                     XCreateFontCursor (glxbackend->dpy, XC_top_left_arrow));
      break;

    case PGM_VIEWPORT_NONE:
      XDefineCursor (glxbackend->dpy, glxbackend->win, glxbackend->none_cursor);
      break;

    default:
      break;
    }

  XSync (glxbackend->dpy, True);

  return TRUE;
}

static gboolean
pgm_glx_backend_set_icon (PgmBackend *backend,
                          GdkPixbuf *icon)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);
  XWMHints wm_hints = { 0 };
  Pixmap icon_pixmap = None;
  Pixmap icon_mask = None;

  GST_DEBUG_OBJECT (glxbackend, "set_icon");

  if (icon)
    gdk_pixbuf_xlib_render_pixmap_and_mask (icon, &icon_pixmap, &icon_mask, 128);
  else
    {
      icon_pixmap = None;
      icon_mask = None;
    }

  free_icon (backend);

  wm_hints.flags = IconPixmapHint | IconMaskHint;
  wm_hints.icon_pixmap = icon_pixmap;
  wm_hints.icon_mask = icon_mask;

  XSetWMHints (glxbackend->dpy, glxbackend->win, &wm_hints);
  XSync (glxbackend->dpy, True);

  return TRUE;
}

static gboolean
pgm_glx_backend_is_accelerated (PgmBackend *backend)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (backend);

  GST_DEBUG_OBJECT (backend, "is_accelerated");

  return glxbackend->feature_mask & PGM_GLX_FEAT_DIRECT_RENDERING;
}

/* GObject stuff */

G_DEFINE_TYPE (PgmGlxBackend, pgm_glx_backend, PGM_TYPE_BACKEND);

static void
pgm_glx_backend_dispose (GObject *object)
{
  PgmGlxBackend *glxbackend = PGM_GLX_BACKEND (object);

  GST_DEBUG_OBJECT (glxbackend, "dispose");

  pgm_glx_backend_destroy_text_lists (PGM_BACKEND (glxbackend));

  if (glxbackend->created)
    {
      free_icon (PGM_BACKEND (glxbackend));
      pgm_glx_backend_destroy_window (PGM_BACKEND (glxbackend));
    }

  GST_CALL_PARENT (G_OBJECT_CLASS, dispose, (object));
}

static void
pgm_glx_backend_class_init (PgmGlxBackendClass *klass)
{
  GObjectClass *gobject_class;
  PgmBackendClass *backend_class;

  parent_class = g_type_class_peek_parent (klass);

  gobject_class = G_OBJECT_CLASS (klass);
  backend_class = PGM_BACKEND_CLASS (klass);

  /* GObject virtual table */
  gobject_class->dispose = GST_DEBUG_FUNCPTR (pgm_glx_backend_dispose);

  /* PgmBackend virtual table */
  backend_class->create_window =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_create_window);
  backend_class->destroy_window =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_destroy_window);
  backend_class->set_title = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_title);
  backend_class->hide = GST_DEBUG_FUNCPTR (pgm_glx_backend_hide);
  backend_class->show = GST_DEBUG_FUNCPTR (pgm_glx_backend_show);
  backend_class->swap_buffers =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_swap_buffers);
  backend_class->get_proc_address =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_proc_address);
  backend_class->set_size = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_size);
  backend_class->set_fullscreen =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_fullscreen);
  backend_class->get_screen_size_mm =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_screen_size_mm);
  backend_class->set_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_set_screen_resolution);
  backend_class->get_screen_resolution =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_get_screen_resolution);
  backend_class->build_text_lists =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_build_text_lists);
  backend_class->destroy_text_lists =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_destroy_text_lists);
  backend_class->raster_text = GST_DEBUG_FUNCPTR (pgm_glx_backend_raster_text);
  backend_class->wait_for_vblank =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_wait_for_vblank);
  backend_class->set_cursor = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_cursor);
  backend_class->set_icon = GST_DEBUG_FUNCPTR (pgm_glx_backend_set_icon);
  backend_class->is_accelerated =
    GST_DEBUG_FUNCPTR (pgm_glx_backend_is_accelerated);
}

static void
pgm_glx_backend_init (PgmGlxBackend *glxbackend)
{
  GST_DEBUG_OBJECT (glxbackend, "init");

  glxbackend->dpy = NULL;
  glxbackend->glx = NULL;
  glxbackend->created = FALSE;
  glxbackend->fullscreen = FALSE;
  glxbackend->windowed_width = 800;
  glxbackend->windowed_height = 600;
  glxbackend->vendor = 0;
  glxbackend->version = 0.0f;
  glxbackend->extensions = 0;
  glxbackend->feature_mask = 0;
  glxbackend->vblank_mode = PGM_VBLANK_NONE;
}

/* Public methods */

PgmBackend *
pgm_glx_backend_new (PgmContext *context)
{
  PgmBackend *backend;

  backend = g_object_new (PGM_TYPE_GLX_BACKEND, NULL);
  GST_DEBUG_OBJECT (PGM_GLX_BACKEND (backend), "created new glxbackend");

  backend->context = context;

  return backend;
}
