/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment OpenGL plugin
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 *
 * Author: Loïc Molinari <loic@fluendo.com>
 */

#ifndef __PGM_TEXTURE_H__
#define __PGM_TEXTURE_H__

#include <pgm/pgmimage.h>
#include "pgmcontext.h"
#include "pgmgldefs.h"

G_BEGIN_DECLS

/* Texture locking */
#define PGM_TEXTURE_LOCK(texture)   (g_mutex_lock (texture->mutex))
#define PGM_TEXTURE_UNLOCK(texture) (g_mutex_unlock (texture->mutex))

/* Type casting */
#define PGM_TEXTURE(obj) ((PgmTexture*) (obj))

typedef struct _PgmTexture PgmTexture;

/* Function pointer types */
typedef void (*PgmTextureUploadFunc) (PgmTexture*, void*);
typedef void (*PgmTextureCreateFunc) (PgmTexture*);
typedef void (*PgmTextureBindFunc)   (PgmTexture*);
typedef void (*PgmTextureUnbindFunc) (PgmTexture*);

/* Data storage types */
typedef enum {
  PGM_TEXTURE_CLEAN,
  PGM_TEXTURE_BUFFER,
  PGM_TEXTURE_GST_BUFFER,
  PGM_TEXTURE_PIXBUF,
} PgmTextureStorageType;

struct _PgmTexture {
  GMutex *mutex;

  PgmTextureStorageType storage;

  union {
    gpointer  buffer;
    GstBuffer *gstbuffer;
    GdkPixbuf *pixbuf;
  } data;

  PgmImagePixelFormat csp;
  gint width;
  gint height;
  gint width_pot;
  gint height_pot;
  gint stride;
  gint size;

  /* Parameters */
  PgmGlEnum filter;
  PgmGlEnum wrap;

  /* Number of identifiant */
  gint count;

  /* Identifiant(s) */
  guint *id;

  /* I420 and YV12 planar program values */
  guint cbcr_height;
  guint y_stride, cbcr_stride;
  guint cb_offset, cr_offset;
};

/* create a new texture */
PgmTexture *pgm_texture_new               (PgmContext *context);

/* Free a texture */
void        pgm_texture_free              (PgmTexture *texture);

/* Set a raw buffer */
void        pgm_texture_set_buffer        (PgmTexture *texture,
                                           guchar *buffer,
                                           PgmImagePixelFormat csp,
                                           guint width,
                                           guint height,
                                           guint size,
                                           guint stride);

/* Set a GdkPixbuf */
void        pgm_texture_set_pixbuf        (PgmTexture *texture,
                                           GdkPixbuf *pixbuf);

/* Set a GstBuffer */
void        pgm_texture_set_gst_buffer    (PgmTexture *texture,
                                           GstBuffer *gstbuffer,
                                           PgmImagePixelFormat csp,
                                           guint width,
                                           guint height,
                                           guint stride);

/* Update a GstBuffer */
void        pgm_texture_update_gst_buffer (PgmTexture *texture,
                                           GstBuffer *gstbuffer);

/* Bind the texture */
void        pgm_texture_bind              (PgmTexture *texture);

/* Unbind the texture */
void        pgm_texture_unbind            (PgmTexture *texture);

/* Generate the OpenGL texture */
void        pgm_texture_generate          (PgmTexture *texture);

/* Clean up the OpenGL texture */
void        pgm_texture_clean             (PgmTexture *texture);

/* Upload the OpenGL texture */
void        pgm_texture_upload            (PgmTexture *texture);

/* Update the OpenGL texture */
void        pgm_texture_update            (PgmTexture *texture);

G_END_DECLS

#endif /* __PGM_TEXTURE_H__ */
