/* -*- Mode: C; c-basic-offset: 2; indent-tabs-mode: nil -*-
 *
 * Pigment unit test for pgmimage.c
 *
 * Copyright © 2006, 2007 Fluendo Embedded S.L.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <gst/check/gstcheck.h>
#include <pgm/pgm.h>
#include <string.h>
#include <stdlib.h>

/* test image master/slave delete */
GST_START_TEST (test_image_mater_slave_del)
{
  PgmError ret;
  PgmImage *img1, *img2;

  pgm_init (NULL, NULL);

  /* create 2 floating images with refcount 1 */
  img1 = PGM_IMAGE (pgm_image_new ());
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  img2 = PGM_IMAGE (pgm_image_new ());
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* make img1 a slave of img2, increasing the refcount of the two images */
  ret = pgm_image_set_from_image (img1, img2);
  fail_if (ret != PGM_ERROR_OK, "image_set_from_image error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 2);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 2);

  /* del master image, the refcount of the two images are still the same */
  ret = pgm_image_clear (img2);
  fail_if (ret != PGM_ERROR_OK, "image_clear error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 2);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 2);

  /* stop the slavery, decreasing the refcount of the two images */
  ret = pgm_image_clear (img1);
  fail_if (ret != PGM_ERROR_OK, "image_clear error");
  ASSERT_OBJECT_REFCOUNT (img1, "img1", 1);
  ASSERT_OBJECT_REFCOUNT (img2, "img2", 1);

  /* clean up our references */
  gst_object_unref (GST_OBJECT_CAST (img1));
  gst_object_unref (GST_OBJECT_CAST (img2));

  pgm_deinit ();
}
GST_END_TEST;


/* test default values */
GST_START_TEST (test_image_default_values)
{
  PgmError ret;
  PgmImage *image;
  PgmImageAlignment alignment;
  PgmImageLayoutType layout;
  PgmImageInterpType interp;
  PgmImageStorageType storage;
  guint numerator, denominator;

  pgm_init (NULL, NULL);
  image = PGM_IMAGE (pgm_image_new ());

  /* alignment */
  ret = pgm_image_get_alignment (image, &alignment);
  fail_if (ret != PGM_ERROR_OK, "image_get_alignment error");
  fail_if (alignment != PGM_IMAGE_CENTER, "bad default alignment");

  /* aspect style */
  ret = pgm_image_get_layout (image, &layout);
  fail_if (ret != PGM_ERROR_OK, "image_get_layout error");
  fail_if (layout != PGM_IMAGE_SCALED, "bad default drawable style");

  /* interp */
  ret = pgm_image_get_interp (image, &interp);
  fail_if (ret != PGM_ERROR_OK, "image_get_interp error");
  fail_if (interp != PGM_IMAGE_BILINEAR, "bad default interp");

  /* aspect-ratio */
  ret = pgm_image_get_aspect_ratio (image, &numerator, &denominator);
  fail_if (ret != PGM_ERROR_OK, "image_get_aspect_ratio error");
  fail_if (numerator != 0 || denominator != 1, "bad default aspect-ratio");

  /* storage type */
  ret = pgm_image_get_storage_type (image, &storage);
  fail_if (ret != PGM_ERROR_OK, "image_get_storage_type error");
  fail_if (storage != PGM_IMAGE_EMPTY, "bad default storage type");

  gst_object_unref (GST_OBJECT_CAST (image));
  pgm_deinit ();
}
GST_END_TEST;


/* test set/get correctness */
GST_START_TEST (test_image_set_get_correctness)
{
  PgmError ret;
  PgmImage *image;
  PgmImageAlignment alignment;
  PgmImageLayoutType layout;
  PgmImageInterpType interp;
  guint numerator, denominator;

  pgm_init (NULL, NULL);
  image = PGM_IMAGE (pgm_image_new ());

  /* alignment */
  ret = pgm_image_set_alignment (image, PGM_IMAGE_BOTTOM);
  fail_if (ret != PGM_ERROR_OK, "image_set_alignment error");
  ret = pgm_image_get_alignment (image, &alignment);
  fail_if (ret != PGM_ERROR_OK, "image_get_alignment error");
  fail_if (alignment != PGM_IMAGE_BOTTOM, "alignment not set");

  /* layout type */
  ret = pgm_image_set_layout (image, PGM_IMAGE_TILED);
  fail_if (ret != PGM_ERROR_OK, "image_set_layout error");
  ret = pgm_image_get_layout (image, &layout);
  fail_if (ret != PGM_ERROR_OK, "image_get_layout error");
  fail_if (layout != PGM_IMAGE_TILED, "laytout not set");

  /* interp */
  ret = pgm_image_set_interp (image, PGM_IMAGE_NEAREST);
  fail_if (ret != PGM_ERROR_OK, "image_set_interp error");
  ret = pgm_image_get_interp (image, &interp);
  fail_if (ret != PGM_ERROR_OK, "image_get_interp error");
  fail_if (interp != PGM_IMAGE_NEAREST, "interp not set");

  /* aspect-ratio */
  ret = pgm_image_set_aspect_ratio (image, 16, 9);
  fail_if (ret != PGM_ERROR_OK, "image_set_aspect_ratio error");
  ret = pgm_image_get_aspect_ratio (image, &numerator, &denominator);
  fail_if (ret != PGM_ERROR_OK, "image_get_aspect_ratio error");
  fail_if (numerator != 16 || denominator != 9, "aspect-ratio not set");

  gst_object_unref (GST_OBJECT_CAST (image));
  pgm_deinit ();
}
GST_END_TEST;


#define IMAGE_SIZE 1267 /* PNG image data size */
#define MOTIF_SIZE   20 /* Filename motif size */

/* Called whenever the image from fd is loaded */
static void
fd_loaded_cb (PgmImage *image,
              gpointer data)
{
  GMainLoop *loop = (GMainLoop*) data;

  /* Signal the completion of loading stopping the main loop */
  g_main_loop_quit (loop);
}

/* test set_from_fd */
GST_START_TEST (test_image_set_from_fd)
{
  /* PNG image data, 8 x 8, 8-bit/color RGB, non-interlaced */
  const guint8 png_image_data[IMAGE_SIZE] = {
    0x89, 0x50, 0x4e, 0x47, 0x0d, 0x0a, 0x1a, 0x0a, 0x00, 0x00, 0x00, 0x0d, 0x49,
    0x48, 0x44, 0x52, 0x00, 0x00, 0x00, 0x18, 0x00, 0x00, 0x00, 0x18, 0x08, 0x02,
    0x00, 0x00, 0x00, 0x6f, 0x15, 0xaa, 0xaf, 0x00, 0x00, 0x00, 0x01, 0x73, 0x52,
    0x47, 0x42, 0x00, 0xae, 0xce, 0x1c, 0xe9, 0x00, 0x00, 0x00, 0x06, 0x62, 0x4b,
    0x47, 0x44, 0x00, 0xff, 0x00, 0xff, 0x00, 0xff, 0xa0, 0xbd, 0xa7, 0x93, 0x00,
    0x00, 0x00, 0x09, 0x70, 0x48, 0x59, 0x73, 0x00, 0x00, 0x0b, 0x13, 0x00, 0x00,
    0x0b, 0x13, 0x01, 0x00, 0x9a, 0x9c, 0x18, 0x00, 0x00, 0x00, 0x07, 0x74, 0x49,
    0x4d, 0x45, 0x07, 0xd7, 0x09, 0x1a, 0x00, 0x2e, 0x0b, 0x6d, 0x8e, 0x83, 0x19,
    0x00, 0x00, 0x00, 0x19, 0x74, 0x45, 0x58, 0x74, 0x43, 0x6f, 0x6d, 0x6d, 0x65,
    0x6e, 0x74, 0x00, 0x43, 0x72, 0x65, 0x61, 0x74, 0x65, 0x64, 0x20, 0x77, 0x69,
    0x74, 0x68, 0x20, 0x47, 0x49, 0x4d, 0x50, 0x57, 0x81, 0x0e, 0x17, 0x00, 0x00,
    0x04, 0x4e, 0x49, 0x44, 0x41, 0x54, 0x38, 0xcb, 0x9d, 0xd5, 0x49, 0x6f, 0x5c,
    0x45, 0x10, 0x07, 0xf0, 0x7f, 0x55, 0x2f, 0x6f, 0xf6, 0x19, 0x7b, 0xc6, 0x36,
    0x8e, 0xd7, 0xb0, 0x25, 0x10, 0x13, 0x2b, 0x40, 0x1c, 0x01, 0x02, 0x84, 0x40,
    0x41, 0x1c, 0x01, 0x21, 0x71, 0xe2, 0x4b, 0xf0, 0x11, 0x38, 0x72, 0xe7, 0x0b,
    0x20, 0x4e, 0x9c, 0xb8, 0x92, 0x03, 0x4b, 0x2e, 0x91, 0x10, 0x52, 0x58, 0x84,
    0x51, 0x24, 0x48, 0x62, 0x27, 0xe0, 0x18, 0x3b, 0xde, 0x66, 0xe6, 0xbd, 0xe9,
    0xd7, 0x5d, 0xc5, 0x21, 0x10, 0x39, 0x03, 0x07, 0x44, 0xab, 0x0f, 0xdd, 0xd5,
    0xad, 0x9f, 0xaa, 0xaa, 0x0f, 0x0d, 0x40, 0xff, 0x9e, 0x5f, 0x00, 0x1d, 0xfc,
    0xaf, 0x31, 0xd7, 0x74, 0xf8, 0x0f, 0x0a, 0x01, 0x6d, 0xc0, 0x3c, 0x18, 0xf4,
    0x80, 0xff, 0x6b, 0x95, 0xd9, 0xd7, 0xdf, 0x5a, 0xb5, 0x00, 0x80, 0x2f, 0x81,
    0x37, 0x81, 0xfd, 0x7b, 0x07, 0x0d, 0xe7, 0x1e, 0x69, 0xb5, 0xce, 0x75, 0xbb,
    0x0f, 0xb7, 0xda, 0x6d, 0xd7, 0xec, 0x47, 0x77, 0x75, 0xf7, 0xc6, 0xe5, 0xdf,
    0x37, 0xb6, 0xf2, 0xfc, 0x18, 0xd4, 0xb4, 0xfc, 0xde, 0xfc, 0xd4, 0xc7, 0x0b,
    0x67, 0x26, 0xa6, 0x4e, 0xf5, 0x16, 0x57, 0x66, 0x09, 0xb8, 0x0c, 0xbc, 0x09,
    0xec, 0xdc, 0xbf, 0xb2, 0xda, 0xed, 0xbe, 0xbd, 0x7c, 0xf2, 0xfc, 0xf4, 0x94,
    0x63, 0x26, 0x00, 0x40, 0x10, 0xf9, 0x66, 0x7b, 0xfb, 0xa3, 0x9f, 0x7e, 0xea,
    0x95, 0xe7, 0x5e, 0xce, 0xde, 0x89, 0x5a, 0xac, 0x9b, 0x4f, 0xe7, 0xd7, 0x5e,
    0xad, 0xaf, 0xdc, 0xd6, 0x87, 0x0e, 0x9a, 0xb3, 0x8d, 0xc9, 0xee, 0x84, 0x05,
    0x3e, 0x38, 0xae, 0x00, 0x78, 0xa4, 0x35, 0xb3, 0xda, 0x9d, 0xf3, 0x1c, 0xef,
    0x47, 0x1c, 0xd1, 0x6a, 0x7a, 0xf7, 0xc3, 0xc6, 0x2b, 0x65, 0x4a, 0xc1, 0x5f,
    0xdb, 0xad, 0x7c, 0xd5, 0x3b, 0xbb, 0xdc, 0x7f, 0x66, 0xbf, 0xba, 0x54, 0x97,
    0x5a, 0x2d, 0x20, 0x98, 0x5a, 0xd5, 0x02, 0x97, 0x1e, 0x2c, 0xde, 0x01, 0xaf,
    0x03, 0xeb, 0xf7, 0x9a, 0xa2, 0xaa, 0x45, 0x4a, 0x7f, 0x6c, 0x3f, 0x7b, 0xa7,
    0x3d, 0xa2, 0x85, 0xcf, 0xe0, 0x07, 0x92, 0xb6, 0x05, 0x9b, 0xbd, 0xf3, 0x27,
    0x30, 0x5d, 0xa8, 0x45, 0xd3, 0x1a, 0xf8, 0x96, 0x35, 0x6d, 0xf3, 0x8f, 0xd6,
    0x9e, 0xf1, 0xe6, 0xea, 0x64, 0x86, 0xb9, 0x7a, 0x1d, 0xc0, 0x5e, 0x2c, 0xbe,
    0xef, 0xcd, 0x6f, 0x5e, 0xc0, 0xe1, 0xea, 0x6e, 0x7e, 0xba, 0x1f, 0x4f, 0x49,
    0xbe, 0xa8, 0xbf, 0xe9, 0xd1, 0xde, 0xed, 0x1b, 0x53, 0x59, 0xe3, 0xf6, 0xd5,
    0x9b, 0x3f, 0x7f, 0xfd, 0xed, 0xcc, 0xf2, 0x89, 0x4e, 0xc7, 0x8d, 0x41, 0x06,
    0x08, 0x83, 0xb8, 0xbb, 0x95, 0xe7, 0x4d, 0xef, 0x2b, 0x8e, 0xae, 0x3d, 0x35,
    0xbb, 0x7f, 0xa6, 0x82, 0x93, 0x85, 0xed, 0xa9, 0x36, 0x22, 0x35, 0x21, 0x35,
    0xc3, 0x53, 0x9d, 0xbe, 0x64, 0x37, 0xaf, 0xfc, 0x18, 0xef, 0xe4, 0x28, 0x12,
    0x55, 0xfc, 0xb5, 0xeb, 0xbf, 0x9a, 0x07, 0x9f, 0xf9, 0x51, 0xe0, 0x56, 0x10,
    0xb9, 0x93, 0xe7, 0xd7, 0xf6, 0xf7, 0x6a, 0x6b, 0x2b, 0xe1, 0xc9, 0x76, 0xec,
    0xe6, 0xd3, 0x93, 0xf5, 0x7a, 0xa5, 0xc2, 0xc6, 0x78, 0xef, 0x8d, 0xab, 0x64,
    0xd5, 0xee, 0xf4, 0xe2, 0x53, 0x07, 0x1b, 0x77, 0x07, 0x3f, 0xfc, 0x5c, 0x39,
    0xc2, 0xfa, 0x95, 0xeb, 0x3b, 0x37, 0xef, 0xf2, 0x31, 0x88, 0x81, 0x1b, 0xf7,
    0x37, 0xda, 0xb6, 0xe5, 0x62, 0x1b, 0x13, 0x79, 0xb3, 0x61, 0x58, 0xa3, 0x55,
    0x6d, 0x65, 0xb5, 0xaa, 0xa9, 0xb6, 0xfc, 0x44, 0xaf, 0xb9, 0x30, 0x37, 0xb5,
    0x58, 0x9b, 0x6e, 0x2f, 0xbd, 0x76, 0xbe, 0xe8, 0x78, 0xa1, 0xd1, 0x93, 0x2f,
    0x9d, 0xb6, 0xc7, 0xa0, 0x04, 0xa4, 0x69, 0x5e, 0x58, 0xb1, 0xcf, 0xcf, 0x98,
    0xc5, 0x99, 0x57, 0x0a, 0x9d, 0x3d, 0xa2, 0x6c, 0x64, 0x8c, 0x4d, 0x48, 0x2c,
    0xa5, 0x4b, 0x31, 0xb3, 0x2d, 0xe7, 0x7c, 0x31, 0xda, 0xbf, 0xbe, 0xf1, 0xdd,
    0xfc, 0xd9, 0xea, 0xec, 0xe4, 0xda, 0xd6, 0xee, 0x4e, 0xff, 0xd6, 0xef, 0xdc,
    0xf2, 0xe3, 0xcd, 0xbe, 0xe0, 0xde, 0x78, 0xa7, 0xfa, 0xfe, 0x6c, 0x7d, 0x6e,
    0xf4, 0xe2, 0x06, 0x2d, 0x1d, 0xd8, 0x4a, 0xa4, 0x24, 0xc3, 0xc3, 0x41, 0xca,
    0x83, 0x86, 0x88, 0x14, 0xca, 0x90, 0xa7, 0xbd, 0xd1, 0x95, 0x4f, 0x2e, 0x6d,
    0xac, 0xdf, 0x7c, 0xfc, 0xfc, 0xc2, 0xd2, 0x85, 0x53, 0xdd, 0xe5, 0x87, 0xd6,
    0xbf, 0xb9, 0x3a, 0x0e, 0x9d, 0x34, 0x2b, 0xa7, 0xdd, 0xda, 0xe6, 0xfc, 0x65,
    0xff, 0xe2, 0x40, 0x27, 0x8f, 0x60, 0x35, 0x12, 0x95, 0x30, 0x0a, 0xca, 0xbc,
    0xf3, 0x96, 0x2b, 0xde, 0xd4, 0x27, 0x6a, 0x2b, 0x2f, 0x3c, 0xb7, 0xfd, 0xcb,
    0xee, 0xe7, 0x1f, 0x5f, 0xea, 0x76, 0x4d, 0xef, 0x64, 0x7d, 0xe1, 0xe9, 0xc5,
    0x71, 0x68, 0xa0, 0x87, 0x5b, 0xfa, 0x8b, 0x79, 0x7e, 0xaf, 0x73, 0x96, 0x7c,
    0x93, 0x8d, 0xab, 0x28, 0x1b, 0x61, 0x24, 0x63, 0x12, 0x54, 0x42, 0x90, 0x61,
    0xbf, 0xec, 0x1f, 0xe5, 0xfd, 0x9d, 0xd9, 0xa5, 0x5e, 0xa3, 0x82, 0xd1, 0x41,
    0x1c, 0xe5, 0x36, 0xdc, 0x0d, 0xf7, 0x7a, 0xd4, 0x06, 0x3c, 0x20, 0x40, 0xd8,
    0x96, 0xcd, 0xd4, 0xd8, 0x39, 0xf1, 0xe8, 0x4b, 0xa8, 0x65, 0x86, 0xbd, 0x26,
    0x0d, 0x21, 0x26, 0xa8, 0x65, 0x24, 0xa5, 0x7e, 0x11, 0x0f, 0x0f, 0x72, 0x8c,
    0xc4, 0xaa, 0x34, 0xa7, 0x3a, 0x33, 0xcf, 0x2d, 0x8d, 0xd8, 0x10, 0xdb, 0x51,
    0x31, 0xb4, 0x80, 0x1e, 0x4b, 0xe8, 0xc6, 0x4c, 0xe5, 0xe9, 0xb5, 0xd3, 0xbd,
    0xc7, 0xe6, 0xbb, 0x9d, 0x56, 0xf3, 0x28, 0x3f, 0x3a, 0xca, 0xf3, 0x42, 0xc9,
    0xb0, 0xc2, 0x31, 0xa9, 0x94, 0x8e, 0x52, 0xb7, 0x0a, 0x62, 0xb1, 0x76, 0x9b,
    0x88, 0x0d, 0x0c, 0x4b, 0xd2, 0x32, 0x39, 0xb6, 0x0f, 0x56, 0xb6, 0x7c, 0x71,
    0xe1, 0xdc, 0x13, 0x27, 0x52, 0xa8, 0xdb, 0x28, 0x3a, 0x1c, 0x86, 0xa2, 0x28,
    0x8d, 0x33, 0xce, 0x58, 0x4e, 0x14, 0xa3, 0xc4, 0xc4, 0x49, 0xa0, 0x84, 0xa4,
    0x22, 0x8c, 0xcc, 0xd9, 0x12, 0xea, 0xbd, 0xa9, 0x5b, 0x3f, 0x06, 0xe1, 0xe2,
    0x7c, 0xe7, 0xc7, 0x99, 0x43, 0x46, 0x11, 0x06, 0x71, 0x14, 0x46, 0x8e, 0x8d,
    0x2a, 0x24, 0x44, 0x15, 0x4a, 0x62, 0x52, 0x99, 0x9c, 0x77, 0x89, 0x39, 0xa9,
    0x2a, 0x34, 0x26, 0x49, 0x10, 0x25, 0x61, 0x93, 0x8d, 0x43, 0x44, 0xf5, 0x83,
    0xe6, 0xa0, 0x9a, 0xe7, 0xa9, 0x10, 0x21, 0x4f, 0x0c, 0xc3, 0x50, 0x65, 0x85,
    0x97, 0x04, 0x26, 0x2b, 0xb0, 0x4c, 0xac, 0xcc, 0xce, 0x03, 0x26, 0x69, 0x39,
    0xb4, 0x9c, 0x89, 0x12, 0x8f, 0x41, 0x9f, 0x6f, 0x5e, 0x4c, 0x12, 0x35, 0x2f,
    0x39, 0x20, 0x23, 0xf6, 0x86, 0x55, 0x2c, 0x53, 0xcd, 0x24, 0x32, 0x49, 0x9d,
    0x71, 0x04, 0x90, 0xc2, 0x80, 0x0d, 0x79, 0x88, 0xb1, 0x36, 0x53, 0xd5, 0x3c,
    0x1f, 0x8d, 0x43, 0x97, 0x6e, 0xbd, 0xca, 0x0c, 0x89, 0x91, 0xc9, 0x1a, 0x32,
    0x5a, 0x92, 0xe5, 0xaa, 0x21, 0x27, 0x4a, 0xd6, 0x39, 0x89, 0x82, 0x52, 0x35,
    0xaa, 0x55, 0xa1, 0x24, 0x9c, 0x04, 0x31, 0x42, 0x60, 0xc9, 0x8d, 0x97, 0xb6,
    0x55, 0x4c, 0x71, 0xd2, 0x24, 0xec, 0xbc, 0x4f, 0x64, 0xac, 0xf3, 0x4c, 0x95,
    0x14, 0x95, 0x12, 0x25, 0x11, 0xc3, 0x20, 0x02, 0x98, 0xa0, 0xa2, 0xa9, 0x0c,
    0x12, 0x92, 0x8e, 0x3c, 0xf9, 0x5a, 0xa5, 0x3e, 0x9e, 0x11, 0x53, 0x42, 0x90,
    0x18, 0xa5, 0x0c, 0x81, 0xc8, 0x10, 0x2c, 0x00, 0x66, 0x58, 0x67, 0x1d, 0x33,
    0x09, 0x34, 0x29, 0x04, 0x10, 0xd2, 0xb2, 0x0c, 0xc3, 0x81, 0x08, 0x14, 0x2c,
    0xa2, 0xf6, 0x5f, 0x7e, 0x8d, 0x51, 0xc9, 0xea, 0xc2, 0x60, 0x48, 0x6a, 0x12,
    0x47, 0x97, 0x35, 0x00, 0x82, 0x6a, 0x12, 0x12, 0x51, 0x81, 0x68, 0x28, 0x25,
    0x0a, 0x73, 0x8a, 0xc3, 0x61, 0x66, 0x6b, 0x24, 0x14, 0x63, 0xf8, 0x13, 0x82,
    0x80, 0x25, 0xa5, 0x08, 0xca, 0xb2, 0x07, 0x00, 0x00, 0x00, 0x00, 0x49, 0x45,
    0x4e, 0x44, 0xae, 0x42, 0x60, 0x82
  };
  char motif[MOTIF_SIZE];
  GMainLoop *loop;
  PgmImage *image;
  PgmError ret;
  ssize_t size;
  off_t offset;
  int error;
  int fd;

  pgm_init (NULL, NULL);
  image = PGM_IMAGE (pgm_image_new ());
  loop = g_main_loop_new (NULL, FALSE);

  /* Create a unique file and open it */
  strncpy (motif, "/tmp/XXXXXX", MOTIF_SIZE);
  fd = mkstemp (motif);
  fail_if (fd < 0, "mkstemp failed");

  /* Automatically delete the file at the end of the process */
  unlink (motif);

  /* Write image data */
  size = write (fd, png_image_data, IMAGE_SIZE);
  fail_if (size != IMAGE_SIZE, "write failed");

  /* Reposition offset */
  offset = lseek (fd, 0, SEEK_SET);
  fail_if (offset != 0, "lseek failed");

  /* Connect a callback to signal loading completion */
  g_signal_connect (G_OBJECT (image), "fd-loaded",
                    G_CALLBACK (fd_loaded_cb), loop);

  /* Start loading from fd */
  ret = pgm_image_set_from_fd (image, fd, 0);
  fail_if (ret != PGM_ERROR_OK, "image_set_from_fd error");

  /* Wait for the loading completion */
  g_main_loop_run (loop);

  /* The fd should be closed, let's see trying to close it */
  error = close (fd);
  fail_unless (error == -1 && errno == EBADF,
               "image_set_from_fd didn't close the fd");

  g_main_loop_unref (loop);
  gst_object_unref (GST_OBJECT_CAST (image));
  pgm_deinit ();
}
GST_END_TEST;

Suite *
pgm_image_suite (void)
{
  Suite *s = suite_create ("PgmImage");
  TCase *tc_chain = tcase_create ("pgmimage tests");

  suite_add_tcase (s, tc_chain);
  tcase_add_test (tc_chain, test_image_mater_slave_del);
  tcase_add_test (tc_chain, test_image_default_values);
  tcase_add_test (tc_chain, test_image_set_get_correctness);
  tcase_add_test (tc_chain, test_image_set_from_fd);

  return s;
}

GST_CHECK_MAIN (pgm_image);
