/***************************************************************************
 *            qof-datebook.h
 *
 *  Thu Oct 21 15:38:58 2004
 *  Copyright  2004-2005  Neil Williams
 *  linux@codehelp.co.uk
 ****************************************************************************/

/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
/** @addtogroup QOF
	@{ */
/** @addtogroup DB Datebooks and Appointments

A note on TIME:
	- QOF uses UTC time throughout
	- QSF uses UTC time in the XML, using the Z timezone-neutral designation.
	- pilot-link and the Palm use localtime.

The pilot-link QOF objects therefore expect UTC time from QOF and pass
\b localtime to the pilot-link code.

	@{ */
/** @file  qof-datebook.h
	@brief QOF datebook definitions for pilot-link
	@author Copyright (c) 2004 Neil Williams <linux@codehelp.co.uk>
*/

#ifndef _QOF_DATEBOOK_H
#define _QOF_DATEBOOK_H

#include <qof.h>
#include "pilot-qof.h"
#include "pi-datebook.h"
#include "qof-main.h"

/** \name QOF object declaration
@{
*/
#define PILOT_LINK_QOF_DATEBOOK "pilot_datebook"
#define QOF_DATEBOOK_DESC       "Pilot-link QOF datebook"
#define DATEBOOK_DESCRIPTION    "description"
#define DATEBOOK_DURATION       "duration_hours"

/** @} */

/** \brief Is this a transient repeat object?

To make it easier to query repeat appointments, transient
events are created for each incidence of a repeat. This
parameter allows such transient events to be ignored in
certain queries. */
#define DATEBOOK_REPEATER          "transient_repeat"

/** Wraps QOF around the libpisock object. */
typedef struct {
	QofInstance inst;
	Appointment_t wrap; //0.12
	char *category;
	gboolean repeater;   /**< \brief Repeat appointment marker
	
	Set to TRUE if this is a transient instance created
	from the repeat data of another instance. This is done to
	make it easier to query repeat appointments. Sufficient
	transient repeaters are created to fill the time between
	start_time and repeat_end, up to a maximum of a year and a
	day - exceptions in the original appointment are not created.
	Unlike qof_instance_gemini, the repeater is created in the
	SAME book as the original. Repeater entities are ignored when
	packing a book into a Palm database. */
}qof_datebook;

/** \brief see QOF or alioth for information on this macro.

To document enums generated using the macro, look at the
doxygen.cfg file in this package, under "PREPROCESSOR".

Note the use of the NON_TYPEDEF versions of the macro here. enum values
cannot be defined twice so if the original definition in libpisock is
NOT a typedef, use the NON_TYPEDEF version of the macro.

http://qof.sourceforge.net/doxy/group__Utilities.html

http://alioth.debian.org/snippet/detail.php?type=snippet&id=13
*/

/** \enum repeatTypes

Control the repeating of events.
*/
/** \enum repeatTypes::repeatNone

Event does not repeat. Default value.
*/
/** \enum repeatTypes::repeatDaily

Event repeats every day. Many of these events
will have exceptions.
*/

#define ENUM_REPEAT_LIST(_) \
	_(repeatNone,) \
	_(repeatDaily,) \
	_(repeatWeekly,) \
	_(repeatMonthlyByDay,) \
	_(repeatMonthlyByDate,) \
	_(repeatYearly,)

AS_STRING_DEC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)
FROM_STRING_DEC_NON_TYPEDEF (repeatTypes, ENUM_REPEAT_LIST)

/** \enum alarmTypes

Units to measure the digit entered in alarm_advance.
*/

#define ENUM_ALARM_UNIT(_) \
	_(advMinutes,) _(advHours,) _(advDays,)

AS_STRING_DEC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)
FROM_STRING_DEC_NON_TYPEDEF (alarmTypes, ENUM_ALARM_UNIT)

#define ENUM_MONTH_DAYQ(_) \
	_(dom1stSun,) _(dom1stMon,) _(dom1stTue,) _(dom1stWen,) _(dom1stThu,) \
	_(dom1stFri,) _(dom1stSat,) _(dom2ndSun,) _(dom2ndMon,) _(dom2ndTue,) \
	_(dom2ndWen,) _(dom2ndThu,) _(dom2ndFri,) _(dom2ndSat,) _(dom3rdSun,) \
	_(dom3rdMon,) _(dom3rdTue,) _(dom3rdWen,) _(dom3rdThu,) _(dom3rdFri,) \
	_(dom3rdSat,) _(dom4thSun,) _(dom4thMon,) _(dom4thTue,) _(dom4thWen,) \
	_(dom4thThu,) _(dom4thFri,) _(dom4thSat,) \
	_(domLastSun,) _(domLastMon,) _(domLastTue,) _(domLastWen,) \
	_(domLastThu,) _(domLastFri,) _(domLastSat,)

/** \enum DayOfMonthType

Possible settings for repeats on certain days of each month.
dom == day of month in each case.
*/

AS_STRING_DEC_NON_TYPEDEF(DayOfMonthType, ENUM_MONTH_DAYQ)
FROM_STRING_DEC_NON_TYPEDEF(DayOfMonthType, ENUM_MONTH_DAYQ)

/** \brief Create a datebook object in QOF

Sets a default time for start and end as the current time.
Creates a KVP frame ready for any exceptions which is
an array of struct tm. The length of array = datebook_getExceptions()
*/
qof_datebook* datebook_create(QofBook*);

/** Register the datebook with QOF */
gboolean DateBookRegister (void);
/** unpack the Palm datebook record into the entity. */
int datebook_unpack(QofEntity *ent, gpointer user_data);
/** \brief pack the entity into a Palm datebook record */
int datebook_pack(QofEntity *ent, gpointer user_data);
/** free all memory related to the datebook record. */
int qof_datebook_free(QofEntity *ent, gpointer user_data);
/** \brief unpack the application information.

The AppInfo contains the category list (each record only holds
the index value of this list as the category value) and other
preferences.
*/
int datebook_appinfo_unpack(QofEntity *ent, gpointer user_data);

#endif /* _QOF_DATEBOOK_H */
/** @} */
/** @} */
