/* Copyright (c) 2001-2012, David A. Clunie DBA Pixelmed Publishing. All rights reserved. */

package com.pixelmed.display;

import com.pixelmed.geometry.GeometryOfSlice;
import com.pixelmed.geometry.GeometryOfVolume;

import java.awt.Graphics;
import java.awt.image.BufferedImage;

/**
 * <p>A class that supports matching the geometry of a superimposed image
 * and an underlying images, and creating BufferedImages suitable for
 * drawing on an underlying image.</p>
 *
 * @see com.pixelmed.display.SingleImagePanel
 *
 * @author	dclunie
 */

public class SuperimposedImage {

	private static final String identString = "@(#) $Header: /userland/cvs/pixelmed/imgbook/com/pixelmed/display/SuperimposedImage.java,v 1.2 2012/11/21 18:36:22 dclunie Exp $";
	
	public static final double DEFAULT_CLOSEST_SLICE_TOLERANCE_DISTANCE = 0.01d;	// mm
	
	// these are protected so as to be accessible to SuperimposedDicomImage or other potential sub-classes for other formats
	protected SourceImage superimposedSourceImage;
	protected GeometryOfVolume superimposedGeometry;
	
	/**
	 * <p>Is the superimposed slice close enough to the underlying slice to superimpose?</p>
	 *
	 * @param	geometryOfSuperimposedSlice
	 * @param	geometryOfUnderlyingSlice
	 * @param	toleranceDistance			difference in distance along normal to orientation for underlying and superimposed frames to be close enough to superimpose, in mm
	 */
	public static boolean isSliceCloseEnoughToSuperimpose(GeometryOfSlice geometryOfSuperimposedSlice,GeometryOfSlice geometryOfUnderlyingSlice,double toleranceDistance) {
		double superimposedDistanceAlongNormal = geometryOfSuperimposedSlice.getDistanceAlongNormalFromOrigin();
		double underlyingDistanceAlongNormal = geometryOfUnderlyingSlice.getDistanceAlongNormalFromOrigin();
		double signedDifference = superimposedDistanceAlongNormal - underlyingDistanceAlongNormal;
		double difference = Math.abs(signedDifference);
		boolean result = difference < toleranceDistance;
System.err.println("SuperimposedImage.isSliceCloseEnoughToSuperimpose(): distance along normal superimposed = "+superimposedDistanceAlongNormal+" underlying = "+underlyingDistanceAlongNormal+" difference = "+difference+" toleranceDistance = "+toleranceDistance+" is "+(result ? "" : "NOT ")+"close enough");
		if (!result) {
			double[] normal = geometryOfSuperimposedSlice.getNormalArray();
			double[] correctionRequired = new double[3];
			correctionRequired[0] = normal[0] * signedDifference;
			correctionRequired[1] = normal[1] * signedDifference;
			correctionRequired[2] = normal[2] * signedDifference;
System.err.println("SuperimposedImage.isSliceCloseEnoughToSuperimpose(): correction to superimposed TLHC required = ("+correctionRequired[0]+","+correctionRequired[1]+","+correctionRequired[2]+")");
		}
		return result;
	}
	
	/**
	 * <p>Is the superimposed slice close enough to the underlying slice to superimpose?</p>
	 *
	 * <p>Assumes a default tolerance factor that is close to zero but allows for floating point rounding error.</p>
	 *
	 * @param	geometryOfSuperimposedSlice
	 * @param	geometryOfUnderlyingSlice
	 */
	public static boolean isSliceCloseEnoughToSuperimpose(GeometryOfSlice geometryOfSuperimposedSlice,GeometryOfSlice geometryOfUnderlyingSlice) {
		return isSliceCloseEnoughToSuperimpose(geometryOfSuperimposedSlice,geometryOfUnderlyingSlice,DEFAULT_CLOSEST_SLICE_TOLERANCE_DISTANCE);
	}
	
	protected SuperimposedImage() {
		this.superimposedSourceImage = null;
		this.superimposedGeometry = null;
	}

	/**
	 * <p>A class that supports matching the geometry of a superimposed image
	 * and a specified underlying image, and creating a BufferedImage suitable for
	 * drawing on that underlying image.</p>
	 */
	public class AppliedToUnderlyingImage {
		private BufferedImage bufferedImage;
		private double columnOrigin;
		private double rowOrigin;
	
		/**
		 * @return	a BufferedImage if a superimposed frame that is close enough can be found, otherwise null
		 */
		public BufferedImage getBufferedImage() { return bufferedImage; }

		public double getColumnOrigin() { return columnOrigin; }

		public double getRowOrigin()    { return rowOrigin; }
	
		/**
		 * @param	underlyingGeometry
		 * @param	underlyingFrame			numbered from 0
		 * @param	toleranceDistance		difference in distance along normal to orientation for underlying and superimposed frames to be close enough to superimpose, in mm
		 */
		private AppliedToUnderlyingImage(GeometryOfVolume underlyingGeometry,int underlyingFrame,double toleranceDistance) {
			bufferedImage = null;
			columnOrigin = 0;
			rowOrigin = 0;
			
			if (underlyingGeometry != null) {
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): underlyingFrame = "+underlyingFrame);
				GeometryOfSlice geometryOfUnderlyingSlice = underlyingGeometry.getGeometryOfSlice(underlyingFrame);
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): geometryOfUnderlyingSlice = "+geometryOfUnderlyingSlice);
				if (geometryOfUnderlyingSlice != null && superimposedGeometry != null) {
					int superimposedFrame = superimposedGeometry.findClosestSliceInSamePlane(geometryOfUnderlyingSlice);
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): closest superimposed frame = "+superimposedFrame);
					GeometryOfSlice geometryOfSuperimposedSlice = superimposedGeometry.getGeometryOfSlice(superimposedFrame);
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): geometryOfSuperimposedSlice = "+geometryOfSuperimposedSlice);
					// closest slice may not be "close enough", so check that normal distance is (near) zero (e.g., Z positions are the same in the axial case)
					if (isSliceCloseEnoughToSuperimpose(geometryOfSuperimposedSlice,geometryOfUnderlyingSlice,toleranceDistance)) {
						double[] tlhcSuperimposedIn3DSpace = geometryOfSuperimposedSlice.getTLHCArray();
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): tlhc of superimposed slice in 3D space = "+java.util.Arrays.toString(tlhcSuperimposedIn3DSpace));
						if (tlhcSuperimposedIn3DSpace != null && tlhcSuperimposedIn3DSpace.length == 3) {
							double[] tlhcSuperimposedInUnderlyingImageSpace = geometryOfUnderlyingSlice.lookupImageCoordinate(tlhcSuperimposedIn3DSpace);
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): tlhc of superimposed slice in underlying image space = "+java.util.Arrays.toString(tlhcSuperimposedInUnderlyingImageSpace));
							if (tlhcSuperimposedInUnderlyingImageSpace != null && tlhcSuperimposedInUnderlyingImageSpace.length == 2) {
								columnOrigin  = tlhcSuperimposedInUnderlyingImageSpace[0];
								rowOrigin     = tlhcSuperimposedInUnderlyingImageSpace[1];
								if (superimposedSourceImage != null) {
									BufferedImage originalBufferedImage = superimposedSourceImage.getBufferedImage(superimposedFrame);
//System.err.println("SuperimposedImage.AppliedToUnderlyingImage(): originalBufferedImage = "+originalBufferedImage);
									if (originalBufferedImage != null) {
										// http://docs.oracle.com/javase/tutorial/2d/images/examples/SeeThroughImageApplet.java
										bufferedImage = new BufferedImage(originalBufferedImage.getWidth(),originalBufferedImage.getHeight(),BufferedImage.TYPE_INT_ARGB);
										Graphics g = bufferedImage.getGraphics();
										g.drawImage(originalBufferedImage,0,0,null);
									}
								}
							}
						}
					}
				}
			}
		}
		
		public String toString() {
			return "(bufferedImage="+(bufferedImage == null ? "null" : bufferedImage.toString())+",columnOrigin="+columnOrigin+",rowOrigin="+rowOrigin+")";
		}
	}
	
	/**
	 * @param	underlyingGeometry
	 * @param	underlyingFrame				numbered from 0
	 * @param	toleranceDistance			difference in distance along normal to orientation for underlying and superimposed frames to be close enough to superimpose, in mm
	 * @return								an instance of AppliedToUnderlyingImage, which will contain a BufferedImage if a superimposed frame that is close enough can be found
	 */
	public AppliedToUnderlyingImage getAppliedToUnderlyingImage(GeometryOfVolume underlyingGeometry,int underlyingFrame,double toleranceDistance) {
		return new AppliedToUnderlyingImage(underlyingGeometry,underlyingFrame,toleranceDistance);
	}
	
	/**
	 * @param	underlyingGeometry
	 * @param	underlyingFrame				numbered from 0
	 * @return								an instance of AppliedToUnderlyingImage, which will contain a BufferedImage if a superimposed frame that is close enough can be found
	 */
	public AppliedToUnderlyingImage getAppliedToUnderlyingImage(GeometryOfVolume underlyingGeometry,int underlyingFrame) {
		return new AppliedToUnderlyingImage(underlyingGeometry,underlyingFrame,DEFAULT_CLOSEST_SLICE_TOLERANCE_DISTANCE);
	}

	/**
	 * @param	superimposedSourceImage
	 * @param	superimposedGeometry
	 */
	public SuperimposedImage(SourceImage superimposedSourceImage,GeometryOfVolume superimposedGeometry) {
		this.superimposedSourceImage = superimposedSourceImage;
		this.superimposedGeometry = superimposedGeometry;
	}
}

