#include "htmlexportwizard.h"
#include "border.h"
#include "browser.h"
#include <qlayout.h>
#include <qgroupbox.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <qhbox.h>
#include <klocale.h>
#include <kglobal.h>
#include <kconfig.h>

HTMLExportWizard::HTMLExportWizard(const QStringList &fileList,
                                   QWidget *parent, const char *name)
    : KWizard(parent, name, true)
{
    // TODO Checkbox slots and config

    KConfig *config = KGlobal::config();
    QString oldGrp = config->group();
    config->setGroup("HTMLGallery");

    setCaption(i18n("HTML Image Gallery Wizard"));

    count = fileList.count();

    // Page one
    QWidget *page = new QWidget(this);
    QVBoxLayout *layout = new QVBoxLayout(page, 10);
    QLabel *lbl = new QLabel(i18n("<QT>This wizard allows you to configure many different options\n\
for your HTML image gallery. On this first page you can\n\
configure the number of rows and columns of thumbnail previews\n\
for each page, as well as the thumbnail size.\n\n\
<B>Tip:</B> If you use a standard thumbnail size Pixie can use existing\n\
thumbnails on your hard drive, which is faster!</QT>"), page);
    layout->addWidget(lbl);

    QGroupBox *groupBox = new QGroupBox(2, Qt::Vertical,
                                        i18n("Rows and columns per page"), page);
    rowInput = new KIntNumInput(config->readNumEntry("Rows", 5), groupBox);
    rowInput->setRange(1, 32);
    rowInput->setLabel(i18n("Rows per page"), AlignVCenter);
    colInput = new KIntNumInput(config->readNumEntry("Cols", 5), groupBox);
    colInput->setRange(1, 32);
    colInput->setLabel(i18n("Columns per page"), AlignVCenter);
    layout->addWidget(groupBox);

    groupBox = new QGroupBox(3, Qt::Vertical,
                             i18n("Thumbnail size"), page);
    QHBox *container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Standard sizes:"), container);
    stdSizeCombo = new QComboBox(container);
    stdSizeCombo->insertItem(i18n("Small"), 0);
    stdSizeCombo->insertItem(i18n("Medium"), 1);
    stdSizeCombo->insertItem(i18n("Large"), 2);
    stdSizeCombo->insertItem(i18n("Huge"), 3);
    stdSizeCombo->insertItem(i18n("Nonstandard size"), 4);
    connect(stdSizeCombo, SIGNAL(activated(int)), this,
            SLOT(slotStdSizeCombo(int)));
    container->setStretchFactor(stdSizeCombo, 1);
    wInput = new KIntNumInput(config->readNumEntry("ThumbWidth", 90), groupBox);
    wInput->setRange(32, 128);
    wInput->setLabel(i18n("Width"), AlignVCenter);
    connect(wInput, SIGNAL(valueChanged(int)), this,
            SLOT(slotThumbSizeChanged(int)));
    hInput = new KIntNumInput(config->readNumEntry("ThumbHeight", 90), groupBox);
    hInput->setRange(32, 128);
    hInput->setLabel(i18n("Height"), AlignVCenter);
    connect(hInput, SIGNAL(valueChanged(int)), this,
            SLOT(slotThumbSizeChanged(int)));
    slotThumbSizeChanged(0);
    layout->addWidget(groupBox);
    layout->addStretch(1);

    addPage(page, i18n("Thumbnail and grid size options"));
    setFinishEnabled(page, true);

    // page two
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    container = new QHBox(page);
    QGroupBox *previewBox = new QGroupBox(1, Qt::Horizontal,
                                          i18n("Preview"), container);
    preview = new QLabel(previewBox);
    lbl = new QLabel(i18n("On this page you may choose if you wish\n\
the thumbnails to have a custom frame, and if\n\
so which border style, width, and color to use."), container);
    container->setStretchFactor(lbl, 1);
    layout->addWidget(container);

    frameGrp = new QButtonGroup(5, Qt::Vertical, i18n("Thumbnail frame style"), page);

    QRadioButton *btn = new QRadioButton(i18n("None"), frameGrp);
    (void)new QRadioButton(i18n("Solid"), frameGrp);
    (void)new QRadioButton(i18n("Beveled"), frameGrp);
    (void)new QRadioButton(i18n("Liquid"), frameGrp);
    (void)new QRadioButton(i18n("Round Corners"), frameGrp);
    frameGrp->setButton(config->readNumEntry("BorderStyle", 0));
    connect(frameGrp, SIGNAL(clicked(int)), this, SLOT(slotBorderType(int)));
    layout->addWidget(frameGrp);

    QGroupBox *colorBox = new QGroupBox(4, Qt::Horizontal,
                                        i18n("Thumbnail frame colors"), page);
    fgLbl = new QLabel(i18n("Foreground"), colorBox);
    frameFgBtn = new KColorButton(config->readColorEntry("FrameFg", &Qt::gray), colorBox);
    bgLbl = new QLabel(i18n("Background"), colorBox);
    frameBgBtn = new KColorButton(config->readColorEntry("FrameBg", &Qt::white), colorBox);
    fillLbl = new QLabel(i18n("Button Fill"), colorBox);
    frameFillBtn = new KColorButton(config->readColorEntry("FrameFill", &Qt::lightGray), colorBox);
    layout->addWidget(colorBox);
    connect(frameFgBtn, SIGNAL(changed(const QColor &)), this,
            SLOT(slotBorderColorChanged(const QColor &)));
    connect(frameBgBtn, SIGNAL(changed(const QColor &)), this,
            SLOT(slotBorderColorChanged(const QColor &)));
    connect(frameFillBtn, SIGNAL(changed(const QColor &)), this,
            SLOT(slotBorderColorChanged(const QColor &)));

    thumbFrameWidthInput = new KIntNumInput(config->readNumEntry("FrameWidth", 1), page);
    thumbFrameWidthInput->setRange(1, 32);
    thumbFrameWidthInput->setLabel(i18n("Thumbnail frame width:"), AlignVCenter);
    connect(thumbFrameWidthInput, SIGNAL(valueChanged(int)), this,
            SLOT(slotBorderWidth(int)));
    layout->addWidget(thumbFrameWidthInput);

    layout->addStretch(1);

    addPage(page, i18n("Thumbnail style options"));
    setFinishEnabled(page, true);

    // page three
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    lbl = new QLabel(i18n("Here you may select the general style for how you want\n\
the HTML to look. You can select if you want a grid of thumbnails\n\
or a column on the left and a frame for viewing on the right, if\n\
you want images to be shown in a new window, and other various\n\
options like table border width and if the thumbnail previews\n\
should display filenames."), page);
    layout->addWidget(lbl);

    groupBox = new QGroupBox(i18n("HTML Style"), page);
    styleGrp = new QButtonGroup(page);
    styleGrp->hide();
    QGridLayout *styleLayout = new QGridLayout(groupBox, 1, 1, 5);
    btn = new QRadioButton(groupBox);
    styleGrp->insert(btn, Grid);
    styleLayout->addWidget(btn, 1, 0);
    btn = new QRadioButton(groupBox);
    styleGrp->insert(btn, GridNewWindow);
    styleLayout->addWidget(btn, 2, 0);
    btn = new QRadioButton(groupBox);
    styleGrp->insert(btn, FrameThumbnail);
    styleLayout->addWidget(btn, 3, 0);
    btn = new QRadioButton(groupBox);
    styleGrp->insert(btn, FrameList);
    styleLayout->addWidget(btn, 4, 0);
    styleGrp->setButton(config->readNumEntry("Style", 0));
 
    connect(styleGrp, SIGNAL(clicked(int)), this,
            SLOT(slotStyleChanged(int)));

    lbl = new QLabel(groupBox);
    //QPixmap pix1(PixieBrowser::uicImage("thumbgrid.png"));
    QPixmap pix1(PixieBrowser::uicImage("thumbgrid.png"));
    lbl->setPixmap(pix1);
    styleLayout->addWidget(lbl, 1, 1);
    lbl = new QLabel(groupBox);
    QPixmap pix2(PixieBrowser::uicImage("thumbgrid-newwindow.png"));
    lbl->setPixmap(pix2);
    styleLayout->addWidget(lbl, 2, 1);
    lbl = new QLabel(groupBox);
    QPixmap pix3(PixieBrowser::uicImage("thumbframe.png"));
    lbl->setPixmap(pix3);
    styleLayout->addWidget(lbl, 3, 1);
    lbl = new QLabel(groupBox);
    QPixmap pix4(PixieBrowser::uicImage("nothumbframe.png"));
    lbl->setPixmap(pix4);
    styleLayout->addWidget(lbl, 4, 1);

    lbl = new QLabel(i18n("Grid layout"), groupBox);
    styleLayout->addWidget(lbl, 1, 2);
    lbl = new QLabel(i18n("Grid layout, image in new window"), groupBox);
    styleLayout->addWidget(lbl, 2, 2);
    lbl = new QLabel(i18n("Thumnail column frame and view frame"), groupBox);
    styleLayout->addWidget(lbl, 3, 2);
    lbl = new QLabel(i18n("Image list frame and view frame (no previews)"), groupBox);
    styleLayout->addWidget(lbl, 4, 2);
    styleLayout->addRowSpacing(0, 15);
    styleLayout->setColStretch(3, 1);

    layout->addWidget(groupBox);

    tableBorderInput = new KIntNumInput(config->readNumEntry("HTMLBorderWidth", 0), page);
    tableBorderInput->setRange(0, 6);
    tableBorderInput->setLabel(i18n("HTML table border:"), AlignVCenter);
    layout->addWidget(tableBorderInput);

    imageNameCB = new QCheckBox(i18n("Include image filenames"), page);
    imageNameCB->setChecked(config->readBoolEntry("IncludeName", false));
    layout->addWidget(imageNameCB);
    layout->addStretch(1);

    addPage(page, i18n("HTML style options"));
    setFinishEnabled(page, true);

    // page four
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    lbl = new QLabel(i18n("You may configure the colors of various HTML items here.\n"),
                     page);
    layout->addWidget(lbl);

    groupBox = new QGroupBox(i18n("HTML colors"), page);
    QGridLayout *colorLayout = new QGridLayout(groupBox, 1, 1, 5);
    lbl = new QLabel(i18n("Background:"), groupBox);
    colorLayout->addWidget(lbl, 1, 0);
    bgBtn = new KColorButton(config->readColorEntry("HTMLBg", &Qt::white), groupBox);
    connect(bgBtn, SIGNAL(changed(const QColor &)), this,
            SLOT(slotBorderColorChanged(const QColor &))); // update preview
    colorLayout->addWidget(bgBtn, 1, 1);
    lbl = new QLabel(i18n("Text:"), groupBox);
    colorLayout->addWidget(lbl, 2, 0);
    textBtn = new KColorButton(config->readColorEntry("HTMLText", &Qt::black), groupBox);
    colorLayout->addWidget(textBtn, 2, 1);
    lbl = new QLabel(i18n("Link:"), groupBox);
    colorLayout->addWidget(lbl, 3, 0);
    linkBtn = new KColorButton(config->readColorEntry("HTMLLink", &Qt::blue), groupBox);
    colorLayout->addWidget(linkBtn, 3, 1);
    lbl = new QLabel(i18n("Followed link:"), groupBox);
    colorLayout->addWidget(lbl, 4, 0);
    followedLinkBtn = new KColorButton(config->readColorEntry("HTMLFollowedLink", &Qt::darkMagenta),groupBox);
    colorLayout->addWidget(followedLinkBtn, 4, 1);
    colorLayout->addRowSpacing(0, 15);
    colorLayout->setColStretch(2, 1);

    layout->addWidget(groupBox);
    layout->addStretch(1);

    addPage(page, i18n("HTML color options"));
    setFinishEnabled(page, true);

    // gotta do this after loading HTML colors...
    updateBorderPreview();

    // page five
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    lbl = new QLabel(i18n("On this page you can configure various navigation buttons and\n\
images. You also can select if to use a \"home\" button, what\n\
text to use for all the buttons, or if to use existing images\n\
on your website."), page);
    layout->addWidget(lbl);

    groupBox = new QGroupBox(5, Qt::Vertical,
                             i18n("Home button"), page);
    useHomeCB = new QCheckBox(i18n("Use a \"home\" button"), groupBox);
    connect(useHomeCB, SIGNAL(toggled(bool)), this,
            SLOT(slotUseHomeCB(bool)));
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Home button URL:"), container);
    homeURLEdit = new QLineEdit(config->readEntry("HomeURL", i18n("http://")), container);
    container->setStretchFactor(homeURLEdit, 1);
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Home button text:"), container);
    homeURLText = new QLineEdit(config->readEntry("HomeBtnText", i18n("home")), container);
    container->setStretchFactor(homeURLText, 1);
    useHomeImageCB = new QCheckBox(i18n("Use an existing image on your website"), groupBox);
    connect(useHomeImageCB, SIGNAL(toggled(bool)), this,
            SLOT(slotUseHomeImageCB(bool)));
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Home button image URL:"), container);
    homeImageEdit = new QLineEdit(config->readEntry("HomeImageURL", i18n("http://")), container);
    container->setStretchFactor(homeImageEdit, 1);
    layout->addWidget(groupBox);
    useHomeImageCB->setChecked(config->readBoolEntry("UseHomeImage", false));
    slotUseHomeImageCB(useHomeImageCB->isOn());
    useHomeCB->setChecked(config->readBoolEntry("UseHome", false));
    slotUseHomeCB(useHomeCB->isOn());

    groupBox = new QGroupBox(5, Qt::Vertical,
                             i18n("Next button"), page);
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Next button text:"), container);
    nextURLText = new QLineEdit(config->readEntry("NextBtnText", i18n("Next >>")), container);
    container->setStretchFactor(nextURLText, 1);
    useNextImageCB = new QCheckBox(i18n("Use an existing image on your website"), groupBox);
    connect(useNextImageCB, SIGNAL(toggled(bool)), this,
            SLOT(slotUseNextImageCB(bool)));
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Next button image URL:"), container);
    nextImageEdit = new QLineEdit(config->readEntry("NextImageURL", i18n("http://")), container);
    container->setStretchFactor(nextImageEdit, 1);
    layout->addWidget(groupBox);
    useNextImageCB->setChecked(config->readBoolEntry("UseNextImage", false));
    slotUseNextImageCB(useNextImageCB->isOn());

    groupBox = new QGroupBox(5, Qt::Vertical,
                             i18n("Back button"), page);
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Back button text:"), container);
    backURLText = new QLineEdit(config->readEntry("BackBtnText", i18n("<< Back")), container);
    container->setStretchFactor(backURLText, 1);
    useBackImageCB = new QCheckBox(i18n("Use an existing image on your website"), groupBox);
    connect(useBackImageCB, SIGNAL(toggled(bool)), this,
            SLOT(slotUseBackImageCB(bool)));
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Back button image URL:"), container);
    backImageEdit = new QLineEdit(config->readEntry("BackImageURL", i18n("http://")), container);
    container->setStretchFactor(backImageEdit, 1);
    layout->addWidget(groupBox);
    useBackImageCB->setChecked(config->readBoolEntry("UseBackImage", false));
    slotUseBackImageCB(useBackImageCB->isOn());

    layout->addStretch(1);
    addPage(page, i18n("Navigation buttons"));
    setFinishEnabled(page, true);

    // page six
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    lbl = new QLabel(i18n("Here you can specify an image and URL to use for a\n\
banner ad."), page);
    layout->addWidget(lbl);
    groupBox = new QGroupBox(5, Qt::Vertical,
                             i18n("Banner ad"), page);
    useBannerCB = new QCheckBox(i18n("Use a banner ad"), groupBox);
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Banner points to:"), container);
    bannerURLEdit = new QLineEdit(config->readEntry("BannerURL", i18n("http://")), container);
    container->setStretchFactor(bannerURLEdit, 1);
    container = new QHBox(groupBox);
    lbl = new QLabel(i18n("Banner image location:"), container);
    bannerImageEdit = new QLineEdit(config->readEntry("BannerImageURL", i18n("http://")), container);
    container->setStretchFactor(bannerImageEdit, 1);
    bannerHInput = new KIntNumInput(config->readNumEntry("BannerHeight", 90), groupBox);
    bannerHInput->setRange(16, 256);
    bannerHInput->setLabel(i18n("Banner frame height:"), AlignVCenter);
    slotStyleChanged(styleGrp->id(styleGrp->selected()));
    useBannerCB->setChecked(config->readBoolEntry("UseBanner", false));
    slotUseBannerCB(useBannerCB->isOn());
    connect(useBannerCB, SIGNAL(toggled(bool)), this,
            SLOT(slotUseBannerCB(bool)));
    layout->addWidget(groupBox);
    layout->addStretch(1);
    addPage(page, i18n("Banner Ad"));
    setFinishEnabled(page, true);

    // page seven
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);

    lbl = new QLabel(i18n("Here you can set the text for the web page title. This will\n\
be shown both on the top of the page and in the titlebar."), page);
    layout->addWidget(lbl);

    groupBox = new QGroupBox(2, Qt::Vertical,
                             i18n("Title text:"), page);
    lbl = new QLabel(i18n("Title:"), groupBox);
    headerTextEdit = new QLineEdit(config->readEntry("Title", i18n("Web Image Gallery")), groupBox);
    layout->addWidget(groupBox);
    layout->addStretch(1);
    addPage(page, i18n("Web page title"));
    setFinishEnabled(page, true);

    // page eight
    page = new QWidget(this);
    layout = new QVBoxLayout(page, 10);
    lbl = new QLabel(i18n("And finally, here you may select the output filenames for the\n\
HTML image gallery. For example, using the default filename of\n\
\"thumbnails\" if you had one page of output it would be named\n\
\"thumbnails.html\". For more than one page of output you would\n\
get \"thumbnails.html\", \"thumbnails2.html\", \"thumbnails3.html\",\n\
etc... There is no \"thumbnails1.html\" page so you can use the\n\
same link no matter how many pages there are."), page);
    layout->addWidget(lbl);
    groupBox = new QGroupBox(1, Qt::Vertical,
                             i18n("Output filename prefix"), page);
    outputEdit = new QLineEdit(config->readEntry("FilenamePrefix", "thumbnails"),
                               groupBox);
    layout->addWidget(groupBox);
    layout->addStretch(1);
    addPage(page, i18n("Output filename"));
    setFinishEnabled(page, true);

    config->setGroup(oldGrp);
}

void HTMLExportWizard::slotStyleChanged(int type)
{
    if(type == 0 || type == 1)
        bannerHInput->setEnabled(false);
    else
        bannerHInput->setEnabled(true);
}


void HTMLExportWizard::slotBorderType(int type)
{
    type -= 1;
    if(type == -1 || type == RoundCorner){
        frameFgBtn->setEnabled(false);
        frameBgBtn->setEnabled(false);
        frameFillBtn->setEnabled(false);
        fgLbl->setEnabled(false);
        bgLbl->setEnabled(false);
        fillLbl->setEnabled(false);
        thumbFrameWidthInput->setEnabled(false);
    }
    else if(type == Solid){
        frameFgBtn->setEnabled(true);
        frameBgBtn->setEnabled(false);
        frameFillBtn->setEnabled(true);
        fgLbl->setEnabled(true);
        bgLbl->setEnabled(false);
        fillLbl->setEnabled(true);
        thumbFrameWidthInput->setEnabled(true);
    }
    else if(type == Liquid){
        frameFgBtn->setEnabled(true);
        frameBgBtn->setEnabled(false);
        frameFillBtn->setEnabled(false);
        fgLbl->setEnabled(true);
        bgLbl->setEnabled(false);
        fillLbl->setEnabled(false);
        thumbFrameWidthInput->setEnabled(true);
    }
    else{
        frameFgBtn->setEnabled(true);
        frameBgBtn->setEnabled(true);
        frameFillBtn->setEnabled(true);
        fgLbl->setEnabled(true);
        bgLbl->setEnabled(true);
        fillLbl->setEnabled(true);
        thumbFrameWidthInput->setEnabled(true);
    }

    if(type == Bevel){
        fgLbl->setText(i18n("Upper left"));
        bgLbl->setText(i18n("Lower right"));
    }
    else{
        fgLbl->setText(i18n("Foreground"));
        bgLbl->setText(i18n("Background"));
    }

    if(type == Liquid)
        thumbFrameWidthInput->setRange(7, 24);
    else
        thumbFrameWidthInput->setRange(1, 24);
    updateBorderPreview();
}

void HTMLExportWizard::slotBorderWidth(int)
{
    updateBorderPreview();
}

void HTMLExportWizard::slotBorderColorChanged(const QColor &)
{
    updateBorderPreview();

}

void HTMLExportWizard::updateBorderPreview()
{
    QImage src;
    QImage dest;
    int type = frameGrp->id(frameGrp->selected())-1;
    if(type != RoundCorner && type != -1)
        src.create(64-thumbFrameWidthInput->value()*2,
                   64-thumbFrameWidthInput->value()*2, 32);
    else
        src.create(64, 64, 32);
    if(type == -1){
        src.fill(bgBtn->color().rgb());
        QPixmap pix;
        pix.convertFromImage(src);
        preview->setPixmap(pix);
        return;
    }

    if(type == Solid){
        src.fill(frameFillBtn->color().rgb());
        KIFBorderEffect::solid(src, dest, frameFgBtn->color(),
                               thumbFrameWidthInput->value());
    }
    else if(type == Bevel){
        src.fill(frameFillBtn->color().rgb());
        KIFBorderEffect::bevel(src, dest, frameFgBtn->color(), frameBgBtn->color(),
                               thumbFrameWidthInput->value());
    }
    else if(type == Liquid){
        src.fill(frameFgBtn->color().rgb());
        KIFBorderEffect::liquid(src, dest, frameFgBtn->color(), bgBtn->color(),
                                thumbFrameWidthInput->value());
    }
    else if(type == RoundCorner){
        src.fill(bgBtn->color().rgb());
        KIFBorderEffect::roundCorner(src, dest, bgBtn->color());
    }
    QPixmap pix;
    pix.convertFromImage(dest);
    preview->setPixmap(pix);
}

void HTMLExportWizard::slotUseNextImageCB(bool val)
{
    nextImageEdit->setEnabled(val);
    nextURLText->setEnabled(!val);
}

void HTMLExportWizard::slotUseBackImageCB(bool val)
{
    backImageEdit->setEnabled(val);
    backURLText->setEnabled(!val);
}

void HTMLExportWizard::slotUseHomeImageCB(bool val)
{
    homeImageEdit->setEnabled(val);
    homeURLText->setEnabled(!val);
}

void HTMLExportWizard::slotUseHomeCB(bool val)
{
    useHomeImageCB->setEnabled(val);
    homeURLEdit->setEnabled(val);
    if(val){
        val = useHomeImageCB->isOn();
        homeImageEdit->setEnabled(val);
        homeURLText->setEnabled(!val);
    }
    else{
        homeImageEdit->setEnabled(false);
        homeURLText->setEnabled(false);
    }
}

void HTMLExportWizard::slotUseBannerCB(bool val)
{
    bannerURLEdit->setEnabled(val);
    bannerImageEdit->setEnabled(val);
}

void HTMLExportWizard::slotThumbSizeChanged(int /* val */)
{
    if(hInput->value() == 48 && wInput->value() == 48)
        stdSizeCombo->setCurrentItem(0); // Small
    else if(hInput->value() == 64 && wInput->value() == 64)
        stdSizeCombo->setCurrentItem(1); // Med
    else if(hInput->value() == 90 && wInput->value() == 90)
        stdSizeCombo->setCurrentItem(2); // Large
    else if(hInput->value() == 120 && wInput->value() == 120)
        stdSizeCombo->setCurrentItem(3); // Huge
    else
        stdSizeCombo->setCurrentItem(4); // Nonstandard
}

void HTMLExportWizard::slotStdSizeCombo(int id)
{
    if(id == 0){ // Small
        wInput->setValue(48);
        hInput->setValue(48);
    }
    else if(id == 1){ // Med
        wInput->setValue(64);
        hInput->setValue(64);
    }
    else if(id == 2){ // Large
        wInput->setValue(90);
        hInput->setValue(90);
    }
    else if(id == 3){ // Large
        wInput->setValue(120);
        hInput->setValue(120);
    }
}


void HTMLExportWizard::writeSettings()
{
    KConfig *config = KGlobal::config();
    QString oldGrp = config->group();
    config->setGroup("HTMLGallery");

    config->writeEntry("Rows", rowInput->value());
    config->writeEntry("Cols", colInput->value());
    config->writeEntry("ThumbWidth", wInput->value());
    config->writeEntry("ThumbHeight", hInput->value());

    config->writeEntry("BorderStyle", frameGrp->id(frameGrp->selected()));
    config->writeEntry("FrameFg", frameFgBtn->color());
    config->writeEntry("FrameBg", frameBgBtn->color());
    config->writeEntry("FrameFill", frameFillBtn->color());
    config->writeEntry("FrameWidth", thumbFrameWidthInput->value());

    config->writeEntry("Style", styleGrp->id(styleGrp->selected()));
    config->writeEntry("HTMLBorderWidth", tableBorderInput->value());
    config->writeEntry("IncludeName", imageNameCB->isOn());

    config->writeEntry("HTMLBg", bgBtn->color());
    config->writeEntry("HTMLText", textBtn->color());
    config->writeEntry("HTMLLink", linkBtn->color());
    config->writeEntry("HTMLFollowedLink", followedLinkBtn->color());

    config->writeEntry("HomeURL", homeURLEdit->text());
    config->writeEntry("HomeBtnText", homeURLText->text());
    config->writeEntry("HomeImageURL", homeImageEdit->text());
    config->writeEntry("UseHomeImage", useHomeImageCB->isOn());
    config->writeEntry("UseHome", useHomeCB->isOn());

    config->writeEntry("NextBtnText", nextURLText->text());
    config->writeEntry("NextImageURL", nextImageEdit->text());
    config->writeEntry("UseNextImage", useNextImageCB->isOn());

    config->writeEntry("BackBtnText", backURLText->text());
    config->writeEntry("BackImageURL", backImageEdit->text());
    config->writeEntry("UseBackImage", useBackImageCB->isOn());

    config->writeEntry("BannerURL", bannerURLEdit->text());
    config->writeEntry("BannerImageURL", bannerImageEdit->text());
    config->writeEntry("BannerHeight", bannerHInput->value());
    config->writeEntry("UseBanner", useBannerCB->isOn());

    config->writeEntry("Title", headerTextEdit->text());

    config->writeEntry("FilenamePrefix", outputEdit->text());

    config->sync();
    config->setGroup(oldGrp);
}

void HTMLExportWizard::accept()
{
    writeSettings();
    KWizard::accept();
}

