/*
 *   Copyright (C) 2007 Zack Rusin <zack@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include "sampleitem.h"

#include <QGraphicsSceneMouseEvent>
#include <kstandarddirs.h>

#include <qdebug.h>
#include <QtNetwork>
#include <math.h>

SampleItem::SampleItem(QObject *parent, const QVariantList &args)
    : Plasma::GLApplet(parent, args)
{
    setHasConfigurationInterface(false);
    setAcceptsHoverEvents(true);
    carousel_rot = 0.0f;
    y_rot = -3.0f;
    scale = 1.0f;

    mouse_down = false;

    select_anim = false;
    anim_scale = 1.0f;
    anim_alpha = 1.0f;

    zoom_in = false;
    zoom_out = false;
    zoomed_in = false;

    QString imageLocation = KStandardDirs::locate("wallpaper",
                                                  "soft-green.jpg");
    qt_logo = bindTexture(QImage(imageLocation));
    current_image = qt_logo;

    http = new QHttp(this);
    connect(http, SIGNAL(requestFinished(int, bool)),
            SLOT(processResults(int, bool)));

    startTimer(50);
    prepareTextures("kde");
}

SampleItem:: ~SampleItem()
{
    for (int i=0; i<textures.size(); ++i)
        deleteTexture(textures.at(i));
    delete http;
}

void SampleItem::timerEvent(QTimerEvent *)
{
    if (!mouse_down) {
        carousel_rot += 0.5;
        update();
    }
    if (zoom_in) {
        scale += scale*0.1f;
        if (scale >= 2.4f) {
            scale = 2.4f;
            zoom_in = false;
            zoomed_in = true;
        }
    }
    if (zoom_out) {
        scale -= scale*0.1f;
        if (scale <= 1.0f) {
            scale = 1.0f;
            zoom_out = false;
            zoomed_in = false;
        }
    }
}

void SampleItem::mousePressEvent(QGraphicsSceneMouseEvent *e)
{
    mouse_down = true;
    current_at_press = current_blade;
    anchor = e->pos().toPoint();
    GLApplet::mousePressEvent(e);
}

void SampleItem::mouseReleaseEvent(QGraphicsSceneMouseEvent *e)
{
    mouse_down = false;
    if (e->button() == Qt::LeftButton) {
        if (textures.size() > 0 && current_blade < textures.size()) {
            if (current_blade != -1)
                current_image = textures.at(current_blade);
            else
                current_blade = qt_logo;
            if (current_blade != -1 && current_at_press == current_blade) {
                select_anim = true;
                if (zoomed_in)
                    zoom_out = true;
            }
        }
    } else if (e->button() == Qt::RightButton) {
        if (!zoom_out && !zoom_in) {
            if (zoomed_in)
                zoom_out = true;
            else
                zoom_in = true;
        }
    }
    GLApplet::mouseReleaseEvent(e);
}

void SampleItem::mouseMoveEvent(QGraphicsSceneMouseEvent *e)
{
    QPoint diff = e->pos().toPoint() - anchor;
    if (e->buttons() & Qt::LeftButton)
        y_rot += diff.x()/5.0f;
    anchor = e->pos().toPoint();
    GLApplet::mouseMoveEvent(e);
    update();
}

int SampleItem::bladeAtPos(const QPoint &p)
{
    const int max_buf = 512;
    GLuint buffer[max_buf];
    GLint viewport[4];
    glGetIntegerv(GL_VIEWPORT, viewport);
    glSelectBuffer(max_buf, buffer);
    glRenderMode(GL_SELECT);

    glInitNames();
    glPushName(-1);

    glMatrixMode(GL_PROJECTION);
    glPushMatrix();
    glLoadIdentity();
    gluPickMatrix(GLdouble(p.x()), GLdouble(viewport[3] - p.y()), 1.0, 1.0, viewport);

    drawBlades(true);
    glMatrixMode(GL_PROJECTION);
    glPopMatrix();

    int records = glRenderMode(GL_RENDER);
    int blade = -1;
    GLuint min_z = 0xffffffff;
    if (records > 1) {
        for (int i=0; i<records; ++i) {
            // check for the min z coordinate on multiple hit records
            if (min_z > (buffer[i*4+1])) {
                min_z = (buffer[i*4+1]);
                blade = buffer[i*4+3];
            }
        }
    } else if (records > 0) {
        blade = buffer[3];
    }
    return blade;
}

void SampleItem::wheelEvent(QGraphicsSceneWheelEvent *e)
{
    e->delta() > 0 ? scale += scale*0.2f : scale -= scale*0.2f;
    if (scale > 2.4f)
        scale = 2.4f;
    else if (scale < 1.0f)
        scale = 1.0f;
    update();
    //GLApplet::wheelEvent(e);
}

void SampleItem::paintGLInterface(QPainter *,
                                  const QStyleOptionGraphicsItem *)
{
    // clear is done before paintEvent() is called
    glViewport(0, 0, int(width()), int(height()));
    glColor4f(1.0f, 1.0, 1.0f, 1.0f);
    glLineWidth(2.0f);

    glEnable(GL_BLEND);
    glBlendFunc(GL_ONE, GL_ONE_MINUS_SRC_ALPHA);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    glPushMatrix();
    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();
    glPushMatrix();

    glDisable(GL_DEPTH_TEST);
    glDisable(GL_TEXTURE_2D);
#if 0
    glBegin(GL_QUADS);
    {
        glColor4f(0.0f, 0.0f, 0.0f, 0.0f);
        glVertex2f(1.0f, 0.5f);
        glVertex2f(-1.0f, 0.5f);

        glColor4f(0.68f, 0.68f, 1.0f, 1.0f);
        glVertex2f(-1.0f, -1.0f);
        glVertex2f(1.0f, -1.0f);
    }
    glEnd();
#endif

    glEnable(GL_DEPTH_TEST);
    glEnable(GL_TEXTURE_2D);

    glMatrixMode(GL_PROJECTION);
    glLoadIdentity();
    drawBlades();

    glMatrixMode(GL_PROJECTION);
    glPopMatrix();
    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    glDisable(GL_DEPTH_TEST);
    GLfloat sw, sh;

    if (width() > height()) {
        sw = 1.0f;
        sh = width()/GLfloat(height());
    } else {
        sw = height()/GLfloat(width());
        sh = 1.0f;
    }

    glTranslatef(-0.3f*scale - scale/8.0f, -0.1f*scale - scale/8.0f, 0.0f);
    glScalef(0.5f*sw, 0.5f*sh, 0.5f);

    GLfloat alpha = 1.0f - (scale-1.0f)/1.4f;
    glDisable(GL_TEXTURE_2D);
    glColor4f(0.5f*alpha, 0.5f*alpha, 0.5f*alpha, 0.5f*alpha);
    glRectf(-1.1f, -1.1f, 1.1f, 1.1f);

    glEnable(GL_TEXTURE_2D);
    glColor4f(1.0f*alpha, 1.0f*alpha, 1.0f*alpha, 1.0f*alpha);
    glBindTexture(GL_TEXTURE_2D, current_image);
    glBegin(GL_QUADS);
    {
        glTexCoord2f(0.0f, 0.0f); glVertex2f(-1.0f,-1.0f);
        glTexCoord2f(0.0f, 1.0f); glVertex2f(-1.0f,1.0f);
        glTexCoord2f(1.0f, 1.0f); glVertex2f(1.0f,1.0f);
        glTexCoord2f(1.0f, 0.0f); glVertex2f(1.0f,-1.0f);
    }
    glEnd();
}

void SampleItem::drawBlades(bool select_mode)
{
    glTranslatef((((2.4f-scale) - 0.2857f*(2.4f-scale))*(0.43f))/scale,
                 ((2.4f-scale)*0.43f)/scale, 0.0f);
    glScalef(scale*0.43f, scale*0.43f, scale*0.43f);

    gluPerspective(90.0f, 1.0f, 1.0f, 26.0f);
    glTranslatef(0.0f, 0.0f, -4.0f);
    glScalef(0.43f, 0.43f, 0.43f);
    glRotatef(36.4f, 1.0f, 0.0f, 0.0f);
    glRotatef(y_rot, 0.0f, 1.0f, 0.0f);

    glMatrixMode(GL_MODELVIEW);
    glLoadIdentity();

    glRotatef(carousel_rot, 0, 1, 0);
    int num_blades = images.size() != 0 ? images.size() : 10;
    GLfloat blade_rotation = 360.0f / num_blades;
    for (int x = 0; x < num_blades; ++x) {
        if (x > textures.size() - 1)
            glBindTexture(GL_TEXTURE_2D, qt_logo);
        else
            glBindTexture(GL_TEXTURE_2D, textures.at(x));

        glRotatef(blade_rotation, 0.0f, 1.0f, 0.0f);
        glTranslatef(0.0f, 0.0f, -4.0f);
        if (select_anim && current_blade == x) {
            glMatrixMode(GL_MODELVIEW);
            glPushMatrix();
            glScalef(anim_scale, anim_scale, anim_scale);
            glColor4f(anim_alpha, anim_alpha, anim_alpha, anim_alpha);
            anim_scale += 0.3f;
            anim_alpha -= 0.1f;
        } else {
            if (current_blade == x)
                glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
            else
                glColor4f(0.9f, 0.9f, 0.9f, 1.0f);
        }
        glLoadName(x);
        glBegin(GL_QUADS);
        {
            glTexCoord2f(1.0f, 0.0f); glVertex3f(0.0f, -2.0f,2.0f);
            glTexCoord2f(1.0f, 1.0f); glVertex3f(0.0f, 2.0f,2.0f);
            glTexCoord2f(0.0f, 1.0f); glVertex3f(0.0f, 2.0f,-2.0f);
            glTexCoord2f(0.0f, 0.0f); glVertex3f(0.0f, -2.0f,-2.0f);
        }
        glEnd();
        if (current_blade == x) {
            glDisable(GL_TEXTURE_2D);
            glColor4f(1.0f, 1.0f, 1.0f, 1.0f);
            glBegin(GL_LINE_LOOP);
            {
                glVertex3f(0.0f, -2.1f,2.1f);
                glVertex3f(0.0f, 2.1f,2.1f);
                glVertex3f(0.0f, 2.1f,-2.1f);
                glVertex3f(0.0f, -2.1f,-2.1f);
            }
            glEnd();
            glEnable(GL_TEXTURE_2D);
        }

        if (!select_mode) {
            glTranslatef(0.0f, -4.2f, 0.0f);
            glBegin(GL_QUADS);
            {
                glColor4f(0.0f, 0.0f, 0.0f, 0.0f);
                glTexCoord2f(0.0f, 1.0f); glVertex3f(0.0f, -2.0f,-2.0f);
                glTexCoord2f(1.0f, 1.0f); glVertex3f(0.0f, -2.0f,2.0f);
                glColor4f(0.2f, 0.2f, 0.2f, 0.2f);
                glTexCoord2f(1.0f, 0.0f); glVertex3f(0.0f, 2.0f,2.0f);
                glTexCoord2f(0.0f, 0.0f); glVertex3f(0.0f, 2.0f,-2.0f);
            }
            glEnd();
            if (current_blade == x) {
                glDisable(GL_TEXTURE_2D);
                glBegin(GL_LINE_LOOP);
                {
                    glColor4f(0.0f, 0.0f, 0.0f, 0.0f);
                    glVertex3f(0.0f, -2.1f,2.1f);
                    glVertex3f(0.0f, 2.1f,2.1f);
                    glColor4f(0.2f, 0.2f, 0.2f, 0.2f);
                    glVertex3f(0.0f, 2.1f,-2.1f);
                    glVertex3f(0.0f, -2.1f,-2.1f);
                }
                glEnd();
                glEnable(GL_TEXTURE_2D);
            }
            glTranslatef(0.0f, 4.2f, 0.0f);
        }
        if (select_anim && current_blade == x) {
            glPopMatrix();
            if (anim_alpha <= 0.0f) {
                select_anim = false;
                anim_scale = 1.0f;
                anim_alpha = 1.0f;
            }
        }
        glTranslatef(0.0f, 0.0f, 4.0f);
    }
}

void SampleItem::prepareTextures(const QString &search)
{
    for (int i=0; i<textures.size(); ++i)
        deleteTexture(textures.at(i));
    current_image = qt_logo;
    images.clear();
    textures.clear();
    http->setHost("www.flickr.com");
    search_id = http->get(QString("/search/?q=%1").arg(search));
    emit statusChanged("Starting image search..");
}

void SampleItem::processResults(int id, bool error)
{
    if (error) {
        emit statusChanged("An error occured while searching for images.");
        return;
    }

    if (id == search_id) {
        QString host;
        if (http->bytesAvailable() > 0) {
            QByteArray ba = http->readAll();
            const char *data = ba.constData();
            int i = 0;
            while (i < ba.size()-5) {
                if (data[i] == '.') {
                    if (data[i+1] == 'j' && data[i+2] == 'p'
                        && data[i+3] == 'g' && data[i+4] == '\"') {
                        int j = i;
                        while (j > 0 && data[j] != '\"') --j;
                        QByteArray addr(ba.mid(j+1, i-j+3));
                        if (!addr.startsWith("http")) {
                            ++i;
                            continue;
                        }
                        QStringList list = QString::fromLatin1(addr).split('/');
                        host = list.at(2);
                        images << "/" + list.at(3) + "/" + list.at(4);
                    }
                }
                ++i;
            }
        }

        if (images.size() > 0) {
            http->setHost(host);
            image_id = http->get(images.at(0));
            emit statusChanged(QString("Found %1 images. Getting image 1 of %1..").arg(images.size()));
        }
    }
    if (id == image_id) {
        if (http->bytesAvailable() > 0) {
            QByteArray ba = http->readAll();
            QImage im(QImage::fromData(ba));
            if (!im.isNull())
                textures.append(bindTexture(im));
            else
                images.removeAt(textures.size());
            if (images.size() > textures.size()) {
                emit statusChanged(QString("Found %1 images. Getting image %2 of %1..")
                                   .arg(images.size()).arg(textures.size()));
                image_id = http->get(images.at(textures.size()));
            } else {
                emit statusChanged("Ready.");
            }
        }
    }
}

QRectF SampleItem::boundingRect() const
{
    return QRectF(0, 0, 340, 240);
}

void SampleItem::hoverMoveEvent(QGraphicsSceneHoverEvent *e)
{
    current_blade = bladeAtPos(e->pos().toPoint());
}

#include "sampleitem.moc"
