/*
 *   Copyright © 2007 Fredrik Höglund <fredrik@kde.org>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU Library General Public License version 2 as
 *   published by the Free Software Foundation
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details
 *
 *   You should have received a copy of the GNU Library General Public
 *   License along with this program; if not, write to the
 *   Free Software Foundation, Inc.,
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
 */

#include <QWidget>
#include <QX11Info>

#include "clipboardengine.h"
#include "clipboardengine.moc"

#include "clipboardpoll.h"

#include <X11/Xlib.h>
#include <X11/Xatom.h>
#include <fixx11h.h>


// QClipboard uses qt_x_time as the timestamp for selection operations.
// It is updated mainly from user actions, but Klipper polls the clipboard
// without any user action triggering it, so qt_x_time may be old,
// which could possibly lead to QClipboard reporting empty clipboard.
// Therefore, qt_x_time needs to be updated to current X server timestamp.

// Call KApplication::updateUserTime() only from functions that are
// called from outside (DCOP), or from QTimer timeout !

static Time next_x_time;

static Bool update_x_time_predicate(Display*, XEvent* event, XPointer)
{
    if (next_x_time != CurrentTime)
        return False;

    // from qapplication_x11.cpp
    switch (event->type)
    {
        case ButtonPress:
        case ButtonRelease:
            next_x_time = event->xbutton.time;
            break;

        case MotionNotify:
            next_x_time = event->xmotion.time;
            break;

        case KeyPress:
        case KeyRelease:
            next_x_time = event->xkey.time;
            break;

        case PropertyNotify:
            next_x_time = event->xproperty.time;
            break;

        case EnterNotify:
        case LeaveNotify:
            next_x_time = event->xcrossing.time;
            break;

        case SelectionClear:
            next_x_time = event->xselectionclear.time;
            break;

        default:
            break;
    }

    return False;
}


K_EXPORT_PLASMA_DATAENGINE(clipboard, ClipboardEngine)


ClipboardEngine::ClipboardEngine(QObject* parent, const QVariantList& args)
    : Plasma::DataEngine(parent)
{
    Q_UNUSED(args)

    updateTimestamp();

    poll = new ClipboardPoll();
    connect(poll, SIGNAL(clipboardChanged(bool)), SLOT(clipboardChanged(bool)));
}


ClipboardEngine::~ClipboardEngine()
{
    delete poll;
}


QStringList ClipboardEngine::sources() const
{
    return QStringList();
}


void ClipboardEngine::init()
{
}


void ClipboardEngine::clipboardChanged(bool selectionMode)
{
    Q_UNUSED(selectionMode)
}


void ClipboardEngine::updateTimestamp()
{
    static QWidget* w = 0;
    if ( !w )
        w = new QWidget;

    unsigned char data[ 1 ];

    XChangeProperty( QX11Info::display(), w->winId(), XA_ATOM, XA_ATOM, 8, PropModeAppend, data, 1 );
    next_x_time = CurrentTime;

    XEvent dummy;
    XCheckIfEvent( QX11Info::display(), &dummy, update_x_time_predicate, NULL );
    if( next_x_time == CurrentTime )
    {
        XSync( QX11Info::display(), False );
        XCheckIfEvent( QX11Info::display(), &dummy, update_x_time_predicate, NULL );
    }

    Q_ASSERT( next_x_time != CurrentTime );

    QX11Info::setAppTime(next_x_time);

    XEvent ev; // remove the PropertyNotify event from the events queue
    XWindowEvent( QX11Info::display(), w->winId(), PropertyChangeMask, &ev );
}

