
/* Copyright 2009  Jan Gerrit Marker <jangerrit@weiler-marker.com>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

//Project includes
#include "amarokrunner.h"

//Qt includes
#include <QtDBus>
#include <QMessageBox>

//KDE includes
#include <KDebug>
#include <KIcon>
#include <KProcess>

AmarokRunner::AmarokRunner(QObject *parent, const QVariantList& args)
        : Plasma::AbstractRunner(parent, args)
{
    Q_UNUSED(args);

    KGlobal::locale()->insertCatalog("plasma_runner_amarokrunner");
    //Adding syntax
    Plasma::RunnerSyntax syntax1(i18n("<command>"), i18n("Let's Amarok do <command>, you can change the commands in the settings dialog"));
    Plasma::RunnerSyntax syntax2(i18n("<search term>"), i18n("Searchs in Amarok's current playlist for <search term>"));
    syntax2.addExampleQuery(i18nc("A song title of a birthday song (example query)", "Happy birthday"));
    QList<Plasma::RunnerSyntax> syns;
    syns << syntax1 << syntax2;
    setSyntaxes(syns);

    setSpeed(SlowSpeed);
    setObjectName("Amarok Runner");
    reloadConfiguration();

    searching = false;
}

AmarokRunner::~AmarokRunner()
{
}

void AmarokRunner::match(Plasma::RunnerContext &context)
{
    kDebug() << "Setting 'searching' to 'false'";
    searching = false;
    bool prefixWasCalled = false;
    const QString term = context.query();
    if (term.length() < 2)   //String is too short: no match
        return;

    if (equalsComplete(term, prefix)) {
        prefixWasCalled = true;
    } else if (noprefix) {//Prefix is only used for command listing
        prefix = "";
    } else if (!noprefix && !term.startsWith(prefix, val_caseSens)) {
        return;
    }

    QStringList playcontrol;
    playcontrol << "org.kde.amarok" << "/Player" << "org.freedesktop.MediaPlayer";
    if (equals(term, prefix + com_play) || prefixWasCalled) {
        QStringList temp = playcontrol;
        temp << "Play";
        addMatch(context, this, term, i18n("Start playing"), i18n("Amarok"), KIcon("media-playback-start"), temp, 1.0);
    }
    if (!context.isValid()) {
        return;
    }
    if (amarokRunning()) {
        if (!context.isValid()) {
            return;
        }
        if (!playlistEmpty()) {
            kDebug() << "playlist is not empty";
            if ((equals(term, prefix + com_next) || prefixWasCalled) && nextSongAvailable()) {
                QStringList temp = playcontrol;
                temp << "Next";
                addMatch(context, this, term, i18n("Play next song"), i18n("Amarok"), KIcon("media-skip-forward"), temp, 1.0);
            }
            if ((equals(term, prefix + com_prev) || prefixWasCalled) && prevSongAvailable()) {
                QStringList temp = playcontrol;
                temp << "Prev";
                addMatch(context, this, term, i18n("Play previous song"), i18n("Amarok") , KIcon("media-skip-backward"), temp, 1.0);
            }
        }// !playlistEmpty()
        if (!context.isValid()) {
            return;
        }
        if (equals(term, prefix + com_pause) || prefixWasCalled) {
            QStringList temp = playcontrol;
            temp << "Pause";
            addMatch(context, this, term, i18n("Pause playing"), i18n("Amarok"), KIcon("media-playback-pause"), temp, 1.0);
        }
        if (equals(term, prefix + com_stop) || prefixWasCalled) {
            QStringList temp = playcontrol;
            temp << "Stop";
            addMatch(context, this, term, i18n("Stop playing"), i18n("Amarok"), KIcon("media-playback-stop"), temp, 1.0);
        }

        if (equals(term, prefix + com_up) || prefixWasCalled) {
            QStringList temp = playcontrol;
            temp << "VolumeUp" << QString("%1").arg(val_up) ;
            addMatch(context, this, term, i18n("Incrase volume by %1" , val_up), i18n("Amarok"), KIcon("audio-volume-high"), temp, 1.0);
        } else if (equals(term, QRegExp(prefix + com_up + " \\d{1,2}0{0,1}"))) {
            int volumeChange = getNumberAsText(term, ' ').toInt();
            QStringList temp = playcontrol;
            temp << "VolumeUp" << QString("%1").arg(volumeChange) ;
            addMatch(context, this, term, i18n("Incrase volume by %1" , volumeChange), i18n("Amarok"), KIcon("audio-volume-high"), temp, 1.0);
        }

        if (equals(term, prefix +  com_down) || prefixWasCalled) {
            QStringList temp = playcontrol;
            temp << "VolumeDown" << QString("%1").arg(val_down);
            addMatch(context, this, term, i18n("Reduce volume by %1", val_down), i18n("Amarok"), KIcon("audio-volume-low"), temp, 1.0);
        } else if (equals(term, QRegExp(prefix + com_down + " \\d{1,2}0{0,1}"))) {
            int volumeChange = getNumberAsText(term, ' ').toInt();
            QStringList temp = playcontrol;
            temp << "VolumeDown" << QString("%1").arg(volumeChange);
            addMatch(context, this, term, i18n("Reduce volume by %1", volumeChange), i18n("Amarok"), KIcon("audio-volume-low"), temp, 1.0);
        }

        if (equals(term, prefix +  com_mute) || prefixWasCalled) {
            QStringList temp = playcontrol;
            temp << "Mute";
            addMatch(context, this, term, i18n("Mute"), i18n("Amarok"), KIcon("audio-volume-muted"), temp, 1.0);
        }
        if (equals(term, QRegExp(prefix + com_volume + "\\d{1,2}0{0,1}"))) {
            QStringList temp = playcontrol;
            QString newVolumeAsText = getNumberAsText(term , '=');
            temp << "VolumeSet" << newVolumeAsText;
            addMatch(context, this, term, i18n("Set volume to %1%" , newVolumeAsText.toInt()), i18n("Amarok"), KIcon("audio-volume-medium"), temp, 1.0);
        }
        if (equals(term, prefix +  com_quit) || prefixWasCalled) {
            QStringList temp;
            temp << playcontrol[0] << "/" << "org.freedesktop.MediaPlayer" << "Quit";
            addMatch(context, this, term, i18n("Quit Amarok"), KIcon("application-exit"), temp, 1.0);
        }
        if (context.isValid() && term.length() > 4 && !prefixWasCalled) {
            QString temp = term;
            searching = true;
            QList<QVariantMap> tracks = findSongs(temp.remove(prefix, val_caseSens));
            searching = false; //TEST
            kDebug() << "search should be finished -> setting 'searching' to 'false'";
            for (int i = 0; i < tracks.length(); ++i) {
                if (! context.isValid()) {
                    return;
                }

                QStringList data;
                data << playcontrol[0] << "/TrackList" << "org.freedesktop.MediaPlayer";
                data << "AddTrack" << tracks.at(i)["inList"].toString();
                if (data.last().toInt() == currentSong()) {
                    QStringList temp = playcontrol;
                    temp << "Stop";
                    addMatch(context, this, term, i18n("Stop playing"), i18nc("<song> from <artist>", "%1 from %2 is already played", tracks.at(i)["title"].toString(), tracks.at(i)["artist"].toString()), KIcon("media-playback-stop"), temp, 1.0);
                } else {
                    addMatch(context, this, term, i18nc("<song> from <artist> in <album>", "%1 from %2 in %3", tracks.at(i)["title"].toString(), tracks.at(i)["artist"].toString(), tracks.at(i)["album"].toString()), i18n("play song"), KIcon("media-playback-start"), data, 1.0);
                }
                data << "DelTrack" << tracks.at(i)[ "inList"].toString();
                addMatch(context, this, term, i18nc("<song> from <artist> in <album>", "%1 from %2 in %3", tracks.at(i)["title"].toString(), tracks.at(i)["artist"].toString(), tracks.at(i)["album"].toString()), i18n("remove song from playlist") , KIcon("list-remove"), data, 1.0);
            }
        }
    }//amarokRunning()
    prefix = config().readEntry("prefix", i18n("Amarok"));
}

void AmarokRunner::run(const Plasma::RunnerContext &context, const Plasma::QueryMatch &match)
{
    Q_UNUSED(context);
    const QStringList data = match.data().toStringList();

    if (!interfaceAvailable(data[0] , data[1])) {//Interface isn't available
        if (!startAmarok()) {//We couldn't start Amarok
            QMessageBox::critical(0, i18n("Amarok not found"), i18n("Amarok 2.x wasn't found so the Amarok Runner isn't able to work."));
            return;
        }
        while (!interfaceAvailable(data[0], data[1])) {//Waiting for the interface to appear
            ;
        }
    }


    QDBusMessage m = QDBusMessage::createMethodCall(data[0], data[1], data[2], data[3]);

    QList<QVariant> args;
    if (equalsComplete(data[3], "VolumeDown") || equalsComplete(data[3], "VolumeUp") || equalsComplete(data[3], "VolumeSet")) {// The volume has to be changed, so we add an argument
        args.append(data[4].toInt());
        m.setArguments(args);
    } else if (equalsComplete(data[3], "AddTrack")) {
        QDBusInterface amarok("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
        QDBusReply<QVariantMap > urltemp1 = amarok.call("GetMetadata", data[4].toInt());
        QVariantMap track = urltemp1.value();
        QString url = track["location"].toString();
        m = QDBusMessage::createMethodCall(data[0], data[1], data[2], "DelTrack");
        args.append(data[4].toInt());
        m.setArguments(args);
        QDBusConnection::sessionBus().send(m);
        args.clear();

        m = QDBusMessage::createMethodCall(data[0], data[1], data[2], data[3]);
        args << url;
        args << true;
    } else if (equalsComplete(data[3], "DelTrack")) {
        args.append(data[4].toInt());
    }
    //Arguments have to be added here as else if
    else if ((data.length() > 4) && (data[4].compare("") != 0)) {//Theres any other argument
        args.append(data[4]);
    }
    m.setArguments(args);
    QDBusConnection::sessionBus().send(m);
}

bool AmarokRunner::equals(const QString &str1, const QString &str2)
{
    return (str2.contains(str1, val_caseSens));
}

bool AmarokRunner::equals(const QString& str, QRegExp exp)
{
    exp.setCaseSensitivity(val_caseSens);
    return exp.exactMatch(str);
}

bool AmarokRunner::equalsComplete(const QString &str1, const QString &str2)
{
    return (str1.compare(str2 , val_caseSens) == 0);
}

QString AmarokRunner::getNumberAsText(const QString& str, const QChar& separator)
{
    return str.section(separator, 1, 1);
}

void AmarokRunner::reloadConfiguration()
{
    KConfigGroup grp = config();
    val_caseSens = (grp.readEntry("val_caseSens" , (int) Qt::CaseInsensitive) == (int) Qt::CaseSensitive) ? Qt::CaseSensitive : Qt::CaseInsensitive;
    noprefix = grp.readEntry("prefixuse", true);
    searchPlaylist = grp.readEntry("searchPlaylist" , true);
    prefix = grp.readEntry("prefix", i18n("Amarok"));
    com_play = grp.readEntry("com_play" , i18n("Play"));
    com_pause = grp.readEntry("com_pause", i18n("Pause")) ;
    com_next = grp.readEntry("com_next" , i18n("Next"));
    com_mute = grp.readEntry("com_mute" , i18n("Mute"));
    com_up = grp.readEntry("com_up" , i18n("Up"));
    val_up = grp.readEntry("val_up" , 15);
    com_down = grp.readEntry("com_down" , i18n("Down"));
    val_down = grp.readEntry("val_down" , 15) ;
    com_prev = grp.readEntry("com_prev" , i18n("Prev"));
    com_stop = grp.readEntry("com_stop" , i18n("Stop"));
    com_volume = grp.readEntry("com_volume" , i18n("Volume="));
    com_quit = grp.readEntry("com_quit" , i18n("Quit"));
}

bool AmarokRunner::startAmarok()
{
    KProcess* amarok = new KProcess();
    amarok->setWorkingDirectory(QDir::homePath());
    return amarok->startDetached("amarok");
}

bool AmarokRunner::interfaceAvailable(const QString &interface, const QString &part)
{
    QDBusInterface* intf = new QDBusInterface(interface, part);
    return intf->isValid();
}

void AmarokRunner::addMatch(Plasma::RunnerContext& context, AmarokRunner* runner, const QString &term, const QString &title, const KIcon &icon, const QStringList &data, const float &relevance)
{
    Plasma::QueryMatch match(runner);
    match.setText(title);
    match.setIcon(icon);
    match.setData(data);
    match.setRelevance(relevance);
    context.addMatch(term, match);
}

void AmarokRunner::addMatch(Plasma::RunnerContext& context , AmarokRunner* runner, const QString &term, const QString &title, const QString &subtext, const KIcon &icon, const QStringList &data, const float &relevance)
{
    Plasma::QueryMatch match(runner);
    match.setText(title);
    match.setSubtext(subtext);
    match.setIcon(icon);
    match.setData(data);
    match.setRelevance(relevance);
    context.addMatch(term, match);
}


QList<QVariantMap> AmarokRunner::findSongs(const QString &term)
{
    QDBusInterface amarok("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
    QDBusReply<int> length = amarok.call("GetLength");

    QList<QVariantMap> ret;
    kDebug() << searching;
    for (int i = 0; (i < length) && searching; i++) {
        kDebug() << "searching:" << i << ":" << searching;
        QDBusReply<QVariantMap > temp = amarok.call("GetMetadata", i);
        QVariantMap track = temp.value();
        if ((term.compare("") == 0) || trackMatchs(track, term)) {
            track.insert("inList", i);
            ret << track;
        }
    }

    if (!searching) {
        kDebug() << "abort: return";
        return QList<QVariantMap>(); //Search was aborted, so we return nothing
    }

    return ret;
}

bool AmarokRunner::trackMatchs(const QVariantMap &track, const QString &term)
{
    return (track["title"].toString().contains(term, Qt::CaseInsensitive)
            || track["album"].toString().contains(term, Qt::CaseInsensitive)
            || track["artist"].toString().contains(term, Qt::CaseInsensitive));
    /* If 'comment' should be used add
     * || track["comment"].toString().contains ( term,Qt::CaseInsensitive )
     * to the return (...)
     */
}

bool AmarokRunner::amarokRunning()
{
    return interfaceAvailable("org.kde.amarok", "/");
}

bool AmarokRunner::playlistEmpty()
{
    const int numberOfSongs = songsInPlaylist();
    kDebug() << "Number of songs in Playlist" << numberOfSongs; 
    return numberOfSongs;
}

int AmarokRunner::songsInPlaylist()
{
    return findSongs("").count();
}

bool AmarokRunner::nextSongAvailable()
{
    QDBusInterface amarok("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
    QDBusReply<int> length = amarok.call("GetLength");
    QDBusReply<int> current = amarok.call("GetCurrentTrack");
    return !((length - 1) == current);
}

bool AmarokRunner::prevSongAvailable()
{
    QDBusInterface amarok("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
    QDBusReply<int> current = amarok.call("GetCurrentTrack");
    return current > 0;
}

int AmarokRunner::currentSong()
{
    QDBusInterface amarok("org.kde.amarok", "/TrackList", "org.freedesktop.MediaPlayer");
    QDBusReply<int> current = amarok.call("GetCurrentTrack");
    return current;
}
#include "amarokrunner.moc"
