/***********************************************************************************
* Fancy Tasks: Plasmoid providing a fancy representation of your tasks and launchers.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "FancyTasksTask.h"
#include "FancyTasksIcon.h"

#include <QTimer>
#include <QVarLengthArray>

#include <KLocale>
#include <NETRootInfo>
#include <KWindowSystem>

FancyTasksTask::FancyTasksTask(TaskManager::AbstractGroupableItem *abstractItem, TaskManager::GroupManager *groupManager, FancyTasksIcon *parent) : QObject(parent),
    m_taskIcon(parent),
    m_abstractItem(NULL),
    m_groupManager(groupManager),
    m_taskType(Invalid)
{
    setTask(abstractItem);
}

void FancyTasksTask::setTask(TaskManager::AbstractGroupableItem *abstractItem)
{
    m_abstractItem = abstractItem;

    if (m_abstractItem->isGroupItem())
    {
        m_group = static_cast<TaskManager::TaskGroup*>(abstractItem);

        m_taskType = Group;

        QList<WId> windowList = windows();

        for (int i = 0; i < windowList.count(); ++i)
        {
            emit windowAdded(windowList.at(i));
        }

        connect(m_group, SIGNAL(changed(::TaskManager::TaskChanges)), this, SLOT(taskChanged(::TaskManager::TaskChanges)));
        connect(m_group, SIGNAL(groupEditRequest()), this, SLOT(showPropertiesDialog()));
        connect(m_group, SIGNAL(itemAdded(AbstractGroupableItem*)), this, SLOT(addItem(AbstractGroupableItem*)));
        connect(m_group, SIGNAL(itemRemoved(AbstractGroupableItem*)), this, SLOT(removeItem(AbstractGroupableItem*)));
    }
    else
    {
        m_task = static_cast<TaskManager::TaskItem*>(abstractItem);

        m_taskType = (m_task->task().isNull()?Startup:Task);

        if (m_taskType == Task)
        {
            emit windowAdded(windows().at(0));
        }

        connect(m_task, SIGNAL(changed(::TaskManager::TaskChanges)), this, SLOT(taskChanged(::TaskManager::TaskChanges)));
    }

    if (m_taskType == Startup)
    {
        connect(m_task, SIGNAL(gotTaskPointer()), this, SLOT(setTaskPointer()));
    }
    else
    {
        QTimer::singleShot(1000, this, SLOT(publishIconGeometry()));
    }

    emit changed(FancyTasksApplet::EveythingChanged);
}

void FancyTasksTask::setTaskPointer()
{
    setTask(m_abstractItem);
}

void FancyTasksTask::close()
{
    m_abstractItem->close();
}

void FancyTasksTask::activate()
{
    if (m_taskType == Task)
    {
        if (!m_task->task()->isIconified() && (!m_taskIcon || m_taskIcon->activeWindow() == m_task->task()->window()))
        {
            m_task->task()->setIconified(true);
        }
        else if (!m_task->task()->isActive() || m_task->task()->isIconified())
        {
            m_task->task()->activate();
        }
        else if (!m_task->task()->isActive() && !m_task->task()->isOnTop())
        {
            m_task->task()->raise();
        }
        else
        {
            m_task->task()->setIconified(true);
        }
    }
}

void FancyTasksTask::activateWindow(bool force)
{
    if (m_taskType == Task && m_task->task() && (!m_taskIcon || m_taskIcon->isUnderMouse() || force))
    {
        m_task->task()->activateRaiseOrIconify();
    }
}

void FancyTasksTask::publishIconGeometry()
{
    if (m_taskType == Task && m_task->task() && m_taskIcon)
    {
        m_task->task()->publishIconGeometry(m_taskIcon->iconGeometry());
    }
}

void FancyTasksTask::dropItems(TaskManager::ItemList items)
{
    if (m_taskType == Startup || m_groupManager->groupingStrategy() != TaskManager::GroupManager::ManualGrouping)
    {
        return;
    }

    if (m_taskType == Task)
    {
        items.append(m_abstractItem);
    }
    else
    {
        items.append(m_group->members());
    }

    m_groupManager->manualGroupingRequest(items);
}

void FancyTasksTask::showPropertiesDialog()
{
    if (m_taskType == Group && m_groupManager->taskGrouper()->editableGroupProperties() & TaskManager::AbstractGroupingStrategy::Name)
    {
        QWidget *groupWidget = new QWidget;

        m_groupUi.setupUi(groupWidget);
        m_groupUi.icon->setIcon(m_group->icon());
        m_groupUi.name->setText(m_group->name());
        m_groupUi.color->setColor(m_group->color());

        KDialog *groupDialog = new KDialog;
        groupDialog->setMainWidget(groupWidget);
        groupDialog->setButtons(KDialog::Cancel | KDialog::Ok);

        connect(groupDialog, SIGNAL(okClicked()), this, SLOT(setProperties()));

        groupDialog->setWindowTitle(i18n("%1 Settings", m_group->name()));
        groupDialog->show();
    }
}

void FancyTasksTask::setProperties()
{
    m_group->setIcon(KIcon(m_groupUi.icon->icon()));
    m_group->setName(m_groupUi.name->text());
    m_group->setColor(m_groupUi.color->color());
}

void FancyTasksTask::taskChanged(::TaskManager::TaskChanges changes)
{
    FancyTasksApplet::Changes taskChanges;

    if (changes & TaskManager::NameChanged || changes & TaskManager::DesktopChanged)
    {
        taskChanges |= FancyTasksApplet::TextChanged;
    }

    if (changes & TaskManager::IconChanged)
    {
        taskChanges |= FancyTasksApplet::IconChanged;
    }

    if (changes & TaskManager::StateChanged)
    {
        taskChanges |= FancyTasksApplet::StateChanged;
    }

    if (changes & TaskManager::GeometryChanged || changes & TaskManager::WindowTypeChanged || changes & TaskManager::ActionsChanged || changes & TaskManager::TransientsChanged)
    {
        taskChanges |= FancyTasksApplet::WindowsChanged;
    }

    if (changes & TaskManager::ColorChanged)
    {
        taskChanges |= FancyTasksApplet::OtherChanges;
    }

    emit changed(taskChanges);
}

void FancyTasksTask::addItem(AbstractGroupableItem *abstractItem)
{
    if (!abstractItem->isGroupItem())
    {
        TaskManager::TaskItem *task = static_cast<TaskManager::TaskItem*>(abstractItem);

        if (!task->task().isNull())
        {
            emit windowAdded(task->task()->window());
        }
    }

    emit changed(FancyTasksApplet::WindowsChanged);
}

void FancyTasksTask::removeItem(AbstractGroupableItem *abstractItem)
{
    if (!abstractItem->isGroupItem())
    {
        TaskManager::TaskItem *task = static_cast<TaskManager::TaskItem*>(abstractItem);

        if (!task->task().isNull())
        {
            emit windowRemoved(task->task()->window());
        }
    }
    emit changed(FancyTasksApplet::WindowsChanged);
}

KMenu* FancyTasksTask::contextMenu()
{
    KMenu *menu = new KMenu;
    TaskManager::BasicMenu *taskMenu;
    QList<QAction*> actions;

    if (m_taskType == Group)
    {
        taskMenu = new TaskManager::BasicMenu(NULL, m_group, m_groupManager, actions);
    }
    else
    {
        taskMenu = new TaskManager::BasicMenu(NULL, m_task, m_groupManager, actions);
    }

    menu->addActions(taskMenu->actions());

    connect(menu, SIGNAL(destroyed()), taskMenu, SLOT(deleteLater()));

    return menu;
}

FancyTasksTask::TaskType FancyTasksTask::taskType() const
{
    return m_taskType;
}

AbstractGroupableItem* FancyTasksTask::abstractItem()
{
    return m_abstractItem;
}

TaskItem* FancyTasksTask::task()
{
    return m_task;
}

TaskGroup* FancyTasksTask::group()
{
    return m_group;
}

KIcon FancyTasksTask::icon()
{
    switch (m_taskType)
    {
        case Startup:
            return KIcon(m_task->startup()->icon());
        break;
        case Task:
            return KIcon(m_task->task()->icon());
        break;
        case Group:
            return KIcon(m_group->icon());
        break;
        default:
            return KIcon();
        break;
    }
}

QColor FancyTasksTask::color() const
{
    return ((m_taskType == Group)?m_group->color():QColor());
}

QString FancyTasksTask::title() const
{
    QString title = ((m_taskType == Group)?m_group->name():((m_taskType == Task)?m_task->task()->visibleName():m_task->startup()->text()));

    if (title.isEmpty())
    {
        if (m_taskType == Group)
        {
            title = static_cast<TaskManager::TaskItem*>(m_group->members().at(0))->task()->visibleName();

            m_group->setName(title);
        }
        else
        {
            title = i18n("Application");
        }
    }

    return title;
}

QString FancyTasksTask::description() const
{
    return ((m_taskType == Startup)?i18n("Starting application..."):(m_abstractItem->isOnAllDesktops()?i18n("On all desktops"):i18nc("Which virtual desktop a window is currently on", "On %1", KWindowSystem::desktopName(m_abstractItem->desktop()))));
}

QList<WId> FancyTasksTask::windows()
{
    QList<WId> windows;

    if (m_taskType == Task)
    {
        windows.append(m_task->task()->window());
    }
    else if (m_taskType == Group)
    {
        TaskManager::ItemList tasks = m_group->members();

        for (int i = 0; i < tasks.count(); ++i)
        {
            windows.append(static_cast<TaskManager::TaskItem*>(tasks.at(i))->task()->window());
        }
    }

    return windows;
}

bool FancyTasksTask::isActive() const
{
    return m_abstractItem->isActive();
}

bool FancyTasksTask::demandsAttention() const
{
    return m_abstractItem->demandsAttention();
}
