/***************************************************************************
 *   Copyright (C) 2009 by Kai Dombrowe <just89@gmx.de>                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA .        *
 ***************************************************************************/


/// Qt
#include <QFontMetrics>
#include <QTimer>
#include <QPainter>
#include <QDrag>

/// KDE
#include <KIcon>
#include <Plasma/Theme>
#include <KDebug>

/// own
#include "gamelist.h"
#include "gameitem.h"



GameList::GameList( QGraphicsWidget *parent )
	: QGraphicsWidget( parent )
{

	setItemDelegate( new QItemDelegate( 0 ) );
	setFlag( QGraphicsItem::ItemIsFocusable );
	setAcceptHoverEvents( true );
        setSizePolicy( QSizePolicy::Preferred, QSizePolicy::Preferred );

	_model = NULL;
	firstItem = -1;
	lastItem = -1;
	_font = Plasma::Theme::defaultTheme()->font( Plasma::Theme::DesktopFont );
	_color = Plasma::Theme::defaultTheme()->color( Plasma::Theme::TextColor );
	hover = new Plasma::FrameSvg( this );
	hover->setImagePath( "widgets/viewitem" );
	hover->setCacheAllRenderedFrames( true );
	hover->setElementPrefix( "hover" );

	listFrame = new Plasma::FrameSvg( this );
	listFrame->setImagePath( "widgets/frame" );
	listFrame->setCacheAllRenderedFrames( true );
	listFrame->setElementPrefix( "sunken" );
	frameRect == QRectF();
	_frame = true;

	rectId = -1;
	buttonRects << QRect() << QRect() << QRect();

	dragIndex = QModelIndex();
	dragWidget = 0;
	timer = new QTimer;
	timer->setSingleShot( true );
	connect( timer, SIGNAL( timeout() ), this, SLOT( startDrag() ) );

	selectionFrame = new Plasma::FrameSvg( this );
	selectionFrame->setImagePath( "widgets/viewitem" );
	selectionFrame->setCacheAllRenderedFrames( true );
	selectionFrame->setElementPrefix( "selected" );

	add = new KIcon( "list-add" );
	edit = new KIcon( "edit-rename" );
	remove = new KIcon( "list-remove" );

	verticalScrollBar = new Plasma::ScrollBar( this );
	verticalScrollBar->setValue( 0 );

	connect( verticalScrollBar, SIGNAL( valueChanged(int) ), this, SLOT( scrollChanged(int) ) );

}



GameList::~GameList()
{

	delete verticalScrollBar;
	delete add;
	delete edit;
	delete remove;
	delete selectionFrame;
	delete listFrame;
	delete hover;
	delete timer;

}


void GameList::setModel( QAbstractItemModel *m )
{

	if( m == _model )
            return;

	if( _model )
		disconnect( _model, SIGNAL( dataChanged(QModelIndex, QModelIndex)), this, SLOT( dataChanged(QModelIndex, QModelIndex) ) );

        _model = m;

	Q_ASSERT_X( _model->index(0,0) == _model->index(0,0),
		"QAbstractItemView::setModel",
		"A model should return the exact same index "
		"(including its internal id/pointer) when asked for it twice in a row." );

        if( _model )
		connect( _model, SIGNAL( dataChanged(QModelIndex, QModelIndex) ), this, SLOT( dataChanged(QModelIndex, QModelIndex) ) );

	setSelectionModel( new QItemSelectionModel( _model ) );
        if( _selectionModel )
	{
		connect( _selectionModel, SIGNAL( selectionChanged(const QItemSelection&, const QItemSelection&) ),
			this, SLOT( selectionChanged(const QItemSelection&, const QItemSelection&) ) );
	}
	initPaintRange();

}


void GameList::initPaintRange()
{

	scrollChanged( 0 );

}


void GameList::dataChanged( const QModelIndex &topLeft, const QModelIndex &bottomRight )
{

	Q_UNUSED(topLeft)
	Q_UNUSED(bottomRight)
	calcSize();
	scrollChanged( verticalScrollBar->value() );

}


void GameList::selectionChanged( const QItemSelection &topLeft, const QItemSelection &bottomRight )
{

	  Q_UNUSED(topLeft)
	  Q_UNUSED(bottomRight)
	  update();

}


void GameList::scrollChanged( int value )
{

	if( _model == NULL )
		return;

	int count = _model->rowCount();
	qreal fontH = (_font.pointSize()*2);
	qreal itemH = 0;
	qreal listH = 0;

	listH = (frameRect.height()-10);

	for( int i = 0; i < count; i++ )
		itemH += fontH;

	qreal tmp = 0;
	firstItem = 0;
	for( int i = 0; i < count; i++ )
	{

		if( tmp > value || value == 0 )
			break;

		tmp += fontH;
		firstItem++;
	}

	int vItems = (listH/(itemH/count)); 
	lastItem = ((firstItem+vItems)-1);
	if( lastItem > count )
		lastItem = count;

	verticalScrollBar->setRange( 0, (itemH-listH) );
	verticalScrollBar->setPageStep( fontH );
	verticalScrollBar->setSingleStep( fontH );

	update();

}


void GameList::mousePressEvent( QGraphicsSceneMouseEvent *event )
{

	if( event->button() & Qt::LeftButton )
	{
		dragIndex = indexAt( event->pos() );
		dragWidget = event->widget();
		timer->start( 2000 );
		event->accept();
	}

}


void GameList::mouseReleaseEvent( QGraphicsSceneMouseEvent *event )
{

	timer->stop();
	QModelIndex clickedIndex = dragIndex;

	if( contentsRect().intersected( buttonRects.at( 0 ) ).contains( event->pos() ) )
	{
		emit addClicked();
		return;
	}
	else if( contentsRect().intersected( buttonRects.at( 1 ) ).contains( event->pos() ) )
	{
		emit editClicked();
		return;
	}
	else if( contentsRect().intersected( buttonRects.at( 2 ) ).contains( event->pos() ) )
	{
		emit removeClicked();
		return;
	}

	if( !clickedIndex.isValid() )
	{
		_selectionModel->clearSelection();
		update();
		return;
	}

	foreach( const QRectF& rect, itemRects )
	{
		if( contentsRect().intersected( rect ).contains( event->pos() ) )
		{
			if( event->button() & Qt::LeftButton &! _selectionModel->isSelected( clickedIndex ) )
				_selectionModel->select( clickedIndex, QItemSelectionModel::ClearAndSelect );
			else if( event->button() & Qt::LeftButton )
				_selectionModel->select( clickedIndex, QItemSelectionModel::Toggle );

			_selectionModel->setCurrentIndex( clickedIndex, QItemSelectionModel::NoUpdate );
			emit clicked( clickedIndex, event->button() );
			break;
		}
	}
	dragIndex = QModelIndex();
	dragWidget = 0;

}


void GameList::startDrag()
{

	if( !dragIndex.isValid() || !dragWidget )
	{
		_selectionModel->clearSelection();
		update();
		return;
	}

	QString icon = _model->data( dragIndex, 1 ).toString();

	QMimeData *data = new QMimeData;
	data->setImageData( icon );
	data->setData( "row", QString::number( dragIndex.row() ).toLatin1() );

	QDrag *drag = new QDrag( dragWidget );
	drag->setMimeData( data );
	drag->setPixmap( KIcon( icon ).pixmap( 36, 36 ) );

	drag->exec();
	dragIndex = QModelIndex();
	dragWidget = 0;

}


void GameList::mouseDoubleClickEvent( QGraphicsSceneMouseEvent *event )
{

	foreach( QRectF rect, itemRects )
	{
		if( contentsRect().intersected( rect ).contains( event->pos() ) )
		{
			event->accept();
			QModelIndex clickedIndex = indexAt( event->pos() );
			if( clickedIndex.isValid() )
				emit doubleClick( clickedIndex );
			break;
		}
	}

}


/// TODO
void GameList::keyPressEvent( QKeyEvent *event )
{

	Q_UNUSED( event )

}


void GameList::wheelEvent( QGraphicsSceneWheelEvent *event )
{

	int value = 0;
	if( event->delta() > 0 )
	{
		value = (verticalScrollBar->value()-verticalScrollBar->pageStep());
		verticalScrollBar->setValue( value );
	}
	else
	{
		value = (verticalScrollBar->value()+verticalScrollBar->pageStep());
		verticalScrollBar->setValue( value );
	}

}


void GameList::paint( QPainter *painter,  const QStyleOptionGraphicsItem *options,  QWidget *widget )
{

	if( firstItem == -1 || lastItem == -1 )
		return;

//	kDebug() << "Painted items: " << firstItem << " to " << lastItem << endl;

	if( _frame )
	{
		listFrame->resizeFrame( frameRect.size() );
		listFrame->paintFrame( painter, frameRect.topLeft() );
	}

	itemRects.clear();
	for( int i = firstItem; i <= lastItem; i++ )
	{

		if( !_model->hasIndex( i, 0 ) ) 
			continue;

		QModelIndex index = _model->index( i, 0 );
		QString text = _model->data( index, 0 ).toString();
		QString kicon = _model->data( index, 1 ).toString();

		qreal fsize = (_font.pointSize()*2);

		QRectF hRect = QRect( QPoint( ((fsize*2)+verticalScrollBar->size().width()), ((i-firstItem)*fsize)+5), QSize( (size().width()), fsize ) ); 
		hRect.setWidth( (frameRect.width()-fsize) );

		KIcon *icon = new KIcon( kicon );
		QRect iconRect = QRect( QPoint( (fsize+verticalScrollBar->size().width()), ((i-firstItem)*fsize)+5 ), QSize( fsize, fsize ) );

		icon->paint( painter, iconRect );

		QRectF shRect = hRect;

		shRect.moveLeft( iconRect.left() );

		painter->setFont( _font );
		painter->setPen( _color );

		if( _selectionModel->isSelected( index ) )
		{
			selectionFrame->resizeFrame( shRect.size() );
			selectionFrame->paintFrame( painter, shRect.topLeft() );
		}
		painter->drawText( hRect, text, Qt::AlignLeft|Qt::AlignVCenter );

		itemRects << shRect;
	}

	if( hstate == Add )
	{
		hover->resizeFrame( buttonRects.at( 0 ).size() );
		hover->paintFrame( painter, buttonRects.at( 0 ).topLeft() );
	}
	else if( hstate == Remove )
	{
		hover->resizeFrame( buttonRects.at( 2 ).size() );
		hover->paintFrame( painter, buttonRects.at( 2 ).topLeft() );
	}
	else if( hstate == Edit )
	{
		hover->resizeFrame( buttonRects.at( 1 ).size() );
		hover->paintFrame( painter, buttonRects.at( 1 ).topLeft() );
	}
	else if( hstate == Item && rectId != -1 && itemRects.size() > rectId )
	{
		hover->resizeFrame( itemRects.at( rectId ).size() );
		hover->paintFrame( painter, itemRects.at( rectId ).topLeft() );
	}

	add->paint( painter, buttonRects.at( 0 ) );
	edit->paint( painter, buttonRects.at( 1 ) );
	remove->paint( painter, buttonRects.at( 2 ) );

	if( firstItem == 0 && lastItem > (_model->rowCount()-2) )
	{
		if( !verticalScrollBar->nativeWidget()->isHidden() )
		{
			verticalScrollBar->hide();
			verticalScrollBar->resize( 0, 0 );
		}
	}
	else
	{
		if( verticalScrollBar->nativeWidget()->isHidden() )
			verticalScrollBar->show();
		verticalScrollBar->resize( verticalScrollBar->preferredSize().width(), size().height() );
		verticalScrollBar->paint( painter, options, widget );
	}

}


QModelIndex GameList::indexAt( const QPointF &p ) const
{

	for( int i = 0; i < itemRects.size(); i++ )
	{
		if( contentsRect().intersected( itemRects.at( i ) ).contains( p ) )
		{
			QModelIndex index = _model->index( (firstItem+i), 0 ); 
			if( index.isValid() )
				return index;
			else
				return QModelIndex();
		}
	}
	return QModelIndex();

}


int GameList::row( const QGraphicsItem *item, const QPointF& point )
{

	return indexAt( mapFromItem( item, point ) ).row();

}


void GameList::setColor( const QColor& c )
{

	if( _color == c )
		return;

	_color = c;
	update();

}


void GameList::setFont( const QFont& f )
{

	if( _font == f )
		return;

	_font = f;
	calcSize();
	scrollChanged( verticalScrollBar->value() );

}


void GameList::hoverMoveEvent( QGraphicsSceneHoverEvent *event )
{

	if( contentsRect().intersected( buttonRects.at( 0 ) ).contains( event->pos() ) )
	{
		if( hstate != Add )
		{
			hstate = Add;
			update();
		}
	}
	else if( contentsRect().intersected( buttonRects.at( 1 ) ).contains( event->pos() ) )
	{
		if( hstate != Edit )
		{
			hstate = Edit;
			update();
		}
	}
	else if( contentsRect().intersected( buttonRects.at( 2 ) ).contains( event->pos() ) )
	{
		if( hstate != Remove )
		{
			hstate = Remove;
			update();
		}
	}
	else if( (!contentsRect().intersected( buttonRects.at( 0 ) ).contains( event->pos() ) & Add) ||
		(!contentsRect().intersected( buttonRects.at( 2 ) ).contains( event->pos() ) & Remove) ||
		(!contentsRect().intersected( buttonRects.at( 1 ) ).contains( event->pos() ) & Edit) )
	{
		hstate = Nothing;
		update();
	}

	bool itemHover = false;
	foreach( const QRectF& rect, itemRects )
	{
		if( contentsRect().intersected( rect ).contains( event->pos() ) )
		{
			hstate = Item;
			itemHover = true;
			rectId = itemRects.indexOf( rect );
			update();
			break;
		}
	}

	if( itemHover == false && hstate == Item )
	{
		hstate = Nothing;
		rectId = -1;
		update();
	}

}


void GameList::hoverLeaveEvent( QGraphicsSceneHoverEvent *event )
{

	QGraphicsWidget::hoverLeaveEvent( event );
	if( hstate == Add || hstate == Remove || hstate == Edit || hstate == Item )
	{
		hstate = Nothing;
		rectId = -1;
		update();
	}
  
}


QModelIndexList GameList::selectedRows()
{

	return _selectionModel->selectedRows();

}


void GameList::resizeEvent( QGraphicsSceneResizeEvent *event )
{

	QGraphicsWidget::resizeEvent( event );
	calcSize();
	qupdate();

}


void GameList::calcFrameRect()
{

	frameRect = contentsRect();
	qreal frameW = ((frameRect.width()-(verticalScrollBar->size().width()+_font.pointSize()))-(_font.pointSize()*2))-_font.pointSize();
	frameRect.setWidth( frameW );
	frameRect.moveLeft( (verticalScrollBar->size().width()+_font.pointSize()) );

}


void GameList::calcButtonRects()
{

	buttonRects.clear();
	QRect addRect = contentsRect().toRect();
	addRect.setSize( QSize( (_font.pointSize()*2), (_font.pointSize()*2) ) );
	addRect.moveTopRight( contentsRect().topRight().toPoint() );
	addRect.moveRight( (addRect.right()-(_font.pointSize()/2)) );

	QRect editRect = contentsRect().toRect();
	editRect.setSize( QSize( (_font.pointSize()*2), (_font.pointSize()*2) ) );
	editRect.moveCenter( contentsRect().center().toPoint() );
	editRect.moveRight( (contentsRect().right()-(_font.pointSize()/2)) );

	QRect removeRect = contentsRect().toRect();
	removeRect.setSize( QSize( (_font.pointSize()*2), (_font.pointSize()*2) ) );
	removeRect.moveBottomRight( contentsRect().bottomRight().toPoint() );
	removeRect.moveRight( (removeRect.right()-(_font.pointSize()/2)) );

	buttonRects << addRect << editRect << removeRect;

}


void GameList::calcSize()
{

	qreal h = 0;
	qreal w = 0;

	QFontMetrics fm( _font );
	for( int i = 0; i < _model->rowCount(); i++ )
	{
		if( !_model->hasIndex( i, 0 ) ) 
			continue;

		QModelIndex index = _model->index( i, 0 );
		QString text = _model->data( index, 0 ).toString();
		if( fm.width( text ) > w )
			w = fm.width( text );
	}
	qreal tmp = 0;
	foreach( const QRectF& rect, buttonRects )
	{
		if( rect.width() > tmp )
			tmp = rect.width();

		h+= rect.height();
	}
	w += (tmp+(_font.pointSize()*2)+verticalScrollBar->size().width());
	w += (_font.pointSize()*4);
	h += _font.pointSize();

	if( w < 64 )
		w = 64;

	if( h < 64 )
		h = 64;

	setMinimumSize( QSizeF( w, h ) );
	setPreferredSize( minimumSize() );
	calcFrameRect();
	calcButtonRects();

}


#include "gamelist.moc"

