/***********************************************************************************
* Windows List: Plasmoid to show list of opened windows.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "plasma-windowslist.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsLinearLayout>

#include <Plasma/Theme>
#include <Plasma/ToolTipManager>

#include <KConfigDialog>
#include <KLocale>
#include <KRun>
#include <KMenu>
#include <KIcon>
#include <KIconLoader>
#include <KWindowSystem>
#include <NETRootInfo>
#include <taskmanager/taskmanager.h>
#include <taskmanager/groupmanager.h>
#include <taskmanager/taskitem.h>
#include <taskmanager/taskactions.h>

void WindowsListMenu::contextMenuEvent(QContextMenuEvent* event)
{
    if (activeAction() && activeAction()->data().type() == QVariant::ULongLong)
    {
        QList<QAction*> actionList;
        TaskManager::TaskManager *tasks = new TaskManager::TaskManager;

        KWindowInfo *window = new KWindowInfo(activeAction()->data().toULongLong(), NET::WMGeometry | NET::WMFrameExtents | NET::WMWindowType | NET::WMDesktop | NET::WMState | NET::XAWMState | NET::WMVisibleName);

        TaskManager::BasicMenu *taskMenu = new TaskManager::BasicMenu(NULL, new TaskManager::TaskItem(NULL, tasks->findTask(window->win())), new TaskManager::GroupManager(NULL), actionList);

        taskMenu->popup(mapToGlobal(actionGeometry(activeAction()).center()));
    }

    event->accept();
}


WindowsList::WindowsList(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args)
{
    setAspectRatioMode(Plasma::ConstrainedSquare);

    int iconSize = IconSize(KIconLoader::Desktop);

    resize((iconSize * 2), (iconSize * 2));
}

void WindowsList::init()
{
    KConfigGroup configuration = config();

    m_showManagementActions = configuration.readEntry("showManagementActions", true);
    m_groupWindows = configuration.readEntry("groupWindows", true);
    m_useArrowIcon = configuration.readEntry("useArrowIcon", false);

    m_icon = new Plasma::IconWidget(KIcon("preferences-system-windows"), QString(), this);

    registerAsDragHandle(m_icon);

    QGraphicsLinearLayout *layout = new QGraphicsLinearLayout(this);
    layout->setContentsMargins(0, 0, 0, 0);
    layout->setSpacing(0);
    layout->addItem(m_icon);

    Plasma::ToolTipContent toolTipData(i18n("Windows list"), i18n("Show list of opened windows"), KIcon("preferences-system-windows").pixmap(IconSize(KIconLoader::Desktop)));

    Plasma::ToolTipManager::self()->setContent(this, toolTipData);

    constraintsEvent(0);

    connect(this, SIGNAL(activate()), this, SLOT(showMenu()));
    connect(m_icon, SIGNAL(clicked()), this, SLOT(showMenu()));
}

void WindowsList::constraintsEvent(Plasma::Constraints constraints)
{
    QString icon;

    Q_UNUSED(constraints);

    if (m_useArrowIcon)
    {
        switch (location())
        {
            case Plasma::LeftEdge:
                icon = "arrow-left";
            break;
            case Plasma::RightEdge:
                icon = "arrow-right";
            break;
            case Plasma::TopEdge:
                icon = "arrow-down";
            break;
            default:
                icon = "arrow-up";
        }
    }
    else
    {
        icon = "preferences-system-windows";
    }

    m_icon->setIcon(KIcon(icon));
}

void WindowsList::createConfigurationInterface(KConfigDialog *parent)
{
    KConfigGroup configuration = config();
    QWidget *widget = new QWidget();

    m_ui.setupUi(widget);

    m_ui.showManagementActions->setChecked(m_showManagementActions);
    m_ui.groupWindows->setChecked(m_groupWindows);
    m_ui.useArrowIcon->setChecked(m_useArrowIcon);

    parent->addPage(widget, i18n("Settings"), icon());

    connect(parent, SIGNAL(applyClicked()), this, SLOT(configAccepted()));
    connect(parent, SIGNAL(okClicked()), this, SLOT(configAccepted()));
}

void WindowsList::configAccepted()
{
    KConfigGroup configuration = config();

    m_showManagementActions = m_ui.showManagementActions->isChecked();
    m_groupWindows = m_ui.groupWindows->isChecked();
    m_useArrowIcon = m_ui.useArrowIcon->isChecked();

    configuration.writeEntry("showManagementActions", m_showManagementActions);
    configuration.writeEntry("groupWindows", m_groupWindows);
    configuration.writeEntry("useArrowIcon", m_useArrowIcon);

    constraintsEvent(0);

    emit configNeedsSaving();
}

void WindowsList::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->buttons() != Qt::MidButton)
    {
        Applet::mousePressEvent(event);

        return;
    }

    WindowsListMenu *menu = new WindowsListMenu;
    QList<WId> windows = KWindowSystem::windows();

    menu->addTitle("Current desktop");

    for (int i = 0; i < windows.count(); ++i)
    {
        KWindowInfo *window = new KWindowInfo(windows.at(i), NET::WMGeometry | NET::WMFrameExtents | NET::WMWindowType | NET::WMDesktop | NET::WMState | NET::XAWMState | NET::WMVisibleName);
        NET::WindowType type = window->windowType(NET::NormalMask | NET::DialogMask | NET::OverrideMask | NET::UtilityMask | NET::DesktopMask | NET::DockMask | NET::TopMenuMask | NET::SplashMask | NET::ToolbarMask | NET::MenuMask);

        if (!window->isOnDesktop(KWindowSystem::currentDesktop()) || type == NET::Desktop || type == NET::Dock || type == NET::TopMenu || type == NET::Splash || type == NET::Menu || type == NET::Toolbar || window->hasState(NET::SkipPager))
        {
            continue;
        }

        menu->addAction(prepareAction(window));
    }

    connect(menu, SIGNAL(triggered(QAction*)), this, SLOT(triggerAction(QAction*)));

    if (!menu->isEmpty())
    {
        menu->exec(QCursor::pos());
    }

    delete menu;

    event->setAccepted(true);
}

void WindowsList::showMenu()
{
    WindowsListMenu *menu = new WindowsListMenu;
    QAction *action;
    QMenu *contextMenu;
    QList<WId> windows = KWindowSystem::windows();
    QList< QList<QAction*> > list;
    int number;

    contextMenu = menu->contextMenu();
    contextMenu->addAction("test");

    if (m_showManagementActions)
    {
        menu->addTitle("Actions");

        action = menu->addAction("Unclutter windows");

        connect(action, SIGNAL(triggered()), menu, SLOT(slotUnclutterWindows()));

        action = menu->addAction("Cascade windows");

        connect(action, SIGNAL(triggered()), menu, SLOT(slotCascadeWindows()));
    }

    for (int i = 0; i <= KWindowSystem::numberOfDesktops(); ++i)
    {
        list.append(QList<QAction*>());
    }

    for (int i = 0; i < windows.count(); ++i)
    {
        KWindowInfo *window = new KWindowInfo(windows.at(i), NET::WMGeometry | NET::WMFrameExtents | NET::WMWindowType | NET::WMDesktop | NET::WMState | NET::XAWMState | NET::WMVisibleName);
        NET::WindowType type = window->windowType(NET::NormalMask | NET::DialogMask | NET::OverrideMask | NET::UtilityMask | NET::DesktopMask | NET::DockMask | NET::TopMenuMask | NET::SplashMask | NET::ToolbarMask | NET::MenuMask);

        if (type == NET::Desktop || type == NET::Dock || type == NET::TopMenu || type == NET::Splash || type == NET::Menu || type == NET::Toolbar || window->hasState(NET::SkipPager))
        {
            windows.removeAt(i);

            --i;

            continue;
        }

        action = prepareAction(window);

        number = ((!m_groupWindows || window->onAllDesktops())?0:window->desktop());

        QList<QAction*> subList = list.value(number);
        subList.append(action);

        list.replace(number, subList);
    }

    number = 0;

    for (int i = 0; i <= KWindowSystem::numberOfDesktops(); ++i)
    {
        if (list.value(i).isEmpty())
        {
            continue;
        }

        menu->addTitle(i?KWindowSystem::desktopName(i):(m_groupWindows?i18n("On all desktops"):i18n("Windows")));

        for (int j = 0; j < list.value(i).count(); ++j)
        {
            ++number;

            menu->addAction(list.value(i).value(j));
        }
    }

    connect(menu, SIGNAL(triggered(QAction*)), this, SLOT(triggerAction(QAction*)));

    if (number)
    {
        menu->exec(QCursor::pos());
    }

    delete menu;
}

QAction* WindowsList::prepareAction(KWindowInfo *window)
{
    QAction *action;
    QList<QAction*> actionList;

    action = new QAction(QIcon(KWindowSystem::icon(window->win())), window->visibleName(), this);
    action->setData((unsigned long long) window->win());

    QFont font = QFont(action->font());

    if (window->isMinimized())
    {
        font.setItalic(true);
    }
    else if (KWindowSystem::activeWindow() == window->win())
    {
        font.setUnderline(true);
        font.setBold(true);
    }

    action->setFont(font);

    return action;
}

void WindowsList::triggerAction(QAction *action)
{
    if (action->data().type() == QVariant::ULongLong)
    {
        if (KWindowSystem::activeWindow() == action->data().toULongLong())
        {
            KWindowSystem::minimizeWindow(action->data().toULongLong());
        }
        else
        {
            KWindowSystem::activateWindow(action->data().toULongLong());
        }
    }
}

#include "plasma-windowslist.moc"
