/*
 *  Copyright 2008 by Alessandro Diaferia <alediaferia@gmail.com>

 *  This program is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU General Public License as
 *  published by the Free Software Foundation; either version 2 of 
 *  the License, or (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "plasma-previewer.h"

// Qt
#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QGraphicsProxyWidget>
#include <QGraphicsSceneDragDropEvent>
#include <QGraphicsScene>
#include <QGraphicsView>
#include <QGraphicsWidget>
#include <QLabel>
#include <QAction>
#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QToolButton>

// KDE
#include <KIcon>
#include <KLocale>
#include <KIconLoader>
#include <KParts/ReadOnlyPart>
#include <KMimeTypeTrader>
#include <KIconLoader>
#include <KVBox>
#include <KFileDialog>
#include <KMenu>
#include <KAction>
#include <KUrl>
#include <KMimeType>
#include <KPushButton>

// Plasma
#include <plasma/widgets/icon.h>
#include <plasma/widgets/label.h>
#include <plasma/dialog.h>

Previewer::Previewer(QObject *parent, const QVariantList &args) : Plasma::Applet(parent, args),
                                                                close_i(0),
                                                                base(0),
                                                                m_dialog(0),
                                                                m_part(0),
                                                                previewing(false),
                                                                m_label(0),
                                                                mime_icon(0)

{
  setAspectRatioMode(Plasma::IgnoreAspectRatio);
  setBackgroundHints(TranslucentBackground);

}

Previewer::~Previewer()
{
 delete m_dialog;
 if(!hasFailedToLaunch()){
   // TODO save settings..
  }
}

void Previewer::init()
{
    //if(m_icon.isNull()) setFailedToLaunch(true, "Unable to create plasmoid");
 
  m_dialog = new Plasma::Dialog();
  m_dialog->setWindowFlags(Qt::X11BypassWindowManagerHint);
  ResizeFilter *filter = new ResizeFilter();
  //connect(filter, SIGNAL(resized(const QSize&)), this, SLOT(rePopupForSize(const QSize&)));
  m_dialog->installEventFilter(filter);

  d_lay = new QVBoxLayout(m_dialog);
  d_lay->setSpacing(1);
  d_lay->setMargin(0);
  base = new KVBox();
  base->setPalette(m_dialog->palette());

  m_label = new Plasma::Label;
  m_label->hide();
  m_label->nativeWidget()->show();
  m_label->nativeWidget()->setAlignment(Qt::AlignLeft | Qt::AlignVCenter);
  mime_icon = new QLabel();
  mime_icon->setSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed);
  close_i = new QToolButton(m_dialog);
  close_i->setAutoRaise(true);
  close_i->setIcon(KIcon("dialog-close"));

  QHBoxLayout *n_lay = new QHBoxLayout();
  n_lay->addWidget(mime_icon);
  n_lay->addWidget(m_label->nativeWidget());
  n_lay->addWidget(close_i);

  d_lay->addLayout(n_lay);
  d_lay->addWidget(base);
  //view->hide();

  lay = new QGraphicsLinearLayout(this);
  lay->setOrientation(Qt::Vertical);

  setLayout(lay);

  m_icon = new Plasma::Icon(KIcon("document-preview"), i18n("Preview"), this);
  m_icon->setNumDisplayLines(2);
  m_icon->setDrawBackground(true);
  lay->addItem(m_icon);

  connect(close_i, SIGNAL(clicked()), this, SLOT(closeFile()));
  connect(m_icon, SIGNAL(clicked()), this, SLOT(showHide()));

  setAcceptDrops( true ); 
  m_menu = new KMenu(i18n("Viewer"));
  m_menu->addTitle(i18n("Viewer"));
  KAction *open = new KAction(KIcon("document-preview"), i18n("&Open"), this);
  connect(open, SIGNAL(triggered()), this, SLOT(openFile()));

  QAction *menu = m_menu->menuAction();
  menu->setIcon(KIcon("arrow-up-double"));
  m_icon->addIconAction(menu);
  connect(menu, SIGNAL(triggered()), this, SLOT(popup()));
  m_menu->addAction(open);

  KAction *onTop = new KAction(i18n("Stay always on top"), this);
  onTop->setCheckable(true);
  onTop->setChecked(true);
  connect(onTop, SIGNAL(toggled(bool)), this, SLOT(stayOnTop(bool)));
  actions<<onTop;

  QAction *sep = new QAction(this);
  sep->setSeparator(true);
  actions<<sep;

  registerAsDragHandle(m_icon);
  resize(m_icon->sizeFromIconSize(IconSize(KIconLoader::Desktop)));


}

void Previewer::stayOnTop(bool top)
{
 if(top){
  m_dialog->setWindowFlags(Qt::X11BypassWindowManagerHint);
 }else{
  m_dialog->setWindowFlags(Qt::FramelessWindowHint);
 }
}

void Previewer::openFile(KUrl u)
{
 if(!u.isValid())
    u = KFileDialog::getOpenUrl();

 if(!u.isValid())
   return;

 previewing = true;

 KMimeType::Ptr mimeType = KMimeType::findByUrl(u, 0, true);
 mime_icon->setPixmap(KIconLoader::global()->loadIcon(mimeType->iconName(), KIconLoader::Toolbar));

 if(m_part != 0) delete m_part;
 m_part = KMimeTypeTrader::createPartInstanceFromQuery<KParts::ReadOnlyPart>(mimeType->name(), base);
 if(!m_part) return;

 if(m_part->openUrl(u)){
   m_label->nativeWidget()->setText(u.fileName());
   m_dialog->resize(400,500);
   m_dialog->move(popupPosition(m_dialog->size()));
   m_dialog->show();  
   base->setFocus(Qt::TabFocusReason);

   if(!previewHistory().contains(u)){
         addPreview(u);
   }
 }

}

void Previewer::rePopupForSize(const QSize& size)
{
 m_dialog->move(popupPosition(size)); // let's reposition the widget since its size changed
}

void Previewer::popup()
{
  if(m_menu->isVisible()){
     m_menu->hide();
     return;
    }
  m_menu->popup(popupPosition(m_menu->size()));

}

void Previewer::closeFile()
{
 //previewing = false;
 m_dialog->hide();
}

void Previewer::dropEvent(QGraphicsSceneDragDropEvent *event)
{

 if (!KUrl::List::canDecode(event->mimeData())){
   //event->reject();
   return;
  }

  const KUrl::List urls = KUrl::List::fromMimeData(event->mimeData());

  if(previewing){
       closeFile();
  }

  if(urls.count() == 1){
    //event->accept();
    openFile(urls[0]);
  }
}

void Previewer::showHide()
{
  if(m_dialog->isVisible()){

   m_dialog->hide();

  }
  else
  {
   if(m_part) m_dialog->show();
  }
}

void Previewer::addPreview(const QUrl& url)
{
 KMimeType::Ptr mimeType = KMimeType::findByUrl(KUrl(url), 0, true);
 KAction *a = new KAction(KIcon(mimeType->iconName()), KUrl(url).fileName(), this);
 a->setData(url);
 connect(a, SIGNAL(triggered()), this, SLOT(reopenPreview()));
 actions.append(a);
}

void Previewer::reopenPreview()
{
 KAction *s = qobject_cast<KAction*>(sender());
 if(s){
  openFile(s->data().toUrl());
 }
}

QList<QAction*> Previewer::contextualActions()
{
 return actions;
}

QList<QUrl> Previewer::previewHistory()
{
 QList<QUrl> history;
 for(int i = 0; i < actions.count(); i++){
  if(actions[i]->data().canConvert(QVariant::Url))
    history<<actions[i]->data().toUrl();
 }
 return history;
}

#include "plasma-previewer.moc"

