#include "dt.h"

/* Area for storing handler values */
Bit32u r3h_DS;
Bit32u r3h_ESP;
Bit32u r3h_target_EIP;
Bit32u r3h_ESP_empty;

/* Area for storing guest values */
Bit32u guest_SS;
Bit32u guest_ESP;
Bit32u guest_EIP;

unsigned r3h_request;
Bit32u   r3h_data;

unsigned char r3h_stack[R3H_STACK_SIZE];


asm (
  ".text \n\t"
  ".globl __r3h_branch \n\t"
  "__r3h_branch:       \n\t"
  /* Save guest context modified by the handler */
  "  pushl %ds \n\t" /* Save guest DS/ES since handler needs to use these */
  "  pushl %es \n\t" /* and sets them to its own segment. */
  "  pushal    \n\t" /* Save guest general registers. */
  "  pushfl    \n\t" /* Save guest eflags */

  "  mov  %ss, %ax   \n\t" /* Copy handler SS to {DS,ES}. */
  "  mov  %ax, %ds   \n\t"
  "  mov  %ax, %es   \n\t"
  "  cld             \n\t"
  "  call r3h_branch \n\t" /* Call C handler */

  ".globl __r3h_prime \n\t"
  "__r3h_prime:       \n\t"
  /* Restore guest context modified by the handler */
  "  popfl    \n\t" /* Restore guest eflags */
  "  popal    \n\t" /* Restore guest general registers */
  "  popl %es \n\t" /* Restore guest DS/ES segments which were modified */
  "  popl %ds \n\t" /* for use by handler. */

  "  cs; mov guest_SS,  %ss         \n\t"
  "  cs; mov guest_ESP, %esp        \n\t"
  "  cs; jmp *(r3h_target_EIP)  \n\t"
  );

asm (
  ".text \n\t"
  ".globl __r3h_ret \n\t"
  "__r3h_ret:       \n\t"
  /* Save guest context modified by the handler */
  "  pushl %ds \n\t" /* Save guest DS/ES since handler needs to use these */
  "  pushl %es \n\t" /* and sets them to its own segment. */
  "  pushal    \n\t" /* Save guest general registers. */
  "  pushfl    \n\t" /* Save guest eflags */

  "  mov  %ss, %ax   \n\t" /* Copy handler SS to {DS,ES}. */
  "  mov  %ax, %ds   \n\t"
  "  mov  %ax, %es   \n\t"
  "  cld             \n\t"
  "  call r3h_ret    \n\t" /* Call C handler */

  /* Restore guest context modified by the handler */
  "  popfl    \n\t" /* Restore guest eflags */
  "  popal    \n\t" /* Restore guest general registers */
  "  popl %es \n\t" /* Restore guest DS/ES segments which were modified */
  "  popl %ds \n\t" /* for use by handler. */

  "  cs; mov guest_SS,  %ss         \n\t"
  "  cs; mov guest_ESP, %esp        \n\t"
  "  cs; jmp *(r3h_target_EIP)  \n\t"
  );


/* This represents the C component of the dynamic branch convenience
 * routine.  I just pretend the targets of the branches are already
 * translated in the cache, and use a simple switch statement to
 * get the translated code address.  Real table lookup logic would
 * go here.  This should be coded extremely efficiently, as it will
 * be executed a lot.
 *
 * If we opened the code cache pages and related tables up for write
 * permissions from ring3 (and thus the ring3 convenience handlers),
 * conceivably we could code the translation here, kicking in upon
 * cache misses.  Otherwise, we would defer to the monitor at ring0.
 */

  void
r3h_branch(gc_t gc, unsigned gOff)
{
  unsigned hashRow, i;

  /* Search the G2T table first, ideally the instruction will have
   * been translated already, and the translation address is in there.
   */
/* +++ what about addr==0 */
/* similar code as in dtTranslateG2T() */
  hashRow = DT_G2THashSelect(gOff);
  for (i=0; i<DT_G2THashWidth; i++) {
    if ( dtG2THash[hashRow][i].gOff == gOff ) {
      r3h_target_EIP = dtG2THash[hashRow][i].tOff;
      return;
      }
    }

  /* Guest to Tcode address pair not in G2T hash table. */
  /* +++ Should look in L2M hash table first here */
  r3h_target_EIP = r3hToMonRequest(R3HToMonRequestG2T, gOff);
}

  void
r3h_ret(gc_t gc)
{
  r3hToMonRequest(R3HToMonRequestTerminate, 0);
}

  Bit32u
r3hToMonRequest(unsigned req, Bit32u data)
{
  switch (req) {
    case R3HToMonRequestG2T:
      r3h_request = req;
      r3h_data = data;
      __r3h2mon();
      return(r3h_data);

    case R3HToMonRequestTerminate:
      r3h_request = R3HToMonRequestTerminate;
      r3h_data    = 0;
      __r3h2mon();
      return(0);

    default:
      r3h_request = R3HToMonRequestPanic;
      r3h_data    = 0;
      return(0);
    }
}
