/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001  Kevin P. Lawton
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */



#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <signal.h>

#include "dt.h"


dtL2MHash_t dtL2MHash;
dtG2THash_t dtG2THash;
dtMetaEntry_t dtMetaTable[DT_MetaTableElements];
unsigned dtMetaFreeIndex;

#if DT_DEBUG
unsigned instrG2THit[DT_G2THashWidth];
unsigned instrG2TMiss;
#endif


  void
dtInitFragCache(void)
{
  /* Initialize the fragment cache hash tables */
  memset(dtL2MHash, 0, sizeof(dtL2MHash));
  memset(dtG2THash, 0, sizeof(dtG2THash));

  /* Initialize the fragment cache meta table */
  memset(dtMetaTable, 0, sizeof(dtMetaTable));
  dtMetaFreeIndex = 0;
}


  static void
dtSetG2TPair(unsigned hashRow, Bit32u gOff, Bit32u tOff)
{
  /* Add a {guest offset, tcode offset} pairing to the G2T hash table */

  unsigned i;

/*
 * +++ handle:
 *   - no entries yet
 *   - swapping current 1st entry
 *   - zeroing downstream entries
 */

  for (i = 0; i < DT_G2THashWidth; i++) {
    if (dtG2THash[hashRow][i].tOff == 0) {
      dtG2THash[hashRow][i].gOff = gOff;
      dtG2THash[hashRow][i].tOff = tOff;
      return;
      }
    }

  /* +++ For now, bump first entry. */
  dtG2THash[hashRow][0].gOff = gOff;
  dtG2THash[hashRow][0].tOff = tOff;
}


  static unsigned
dtTranslateLPAToMI(Bit32u lpa)
{
  /* Translate from a Linear Page Address to a Meta Index */

  unsigned i, hashRow, tag;

  /* Note: lpa is the upper 20 bits, the 32bit linear address >> 12 */
  hashRow = DT_LPAToMIHash(lpa);
  tag = DT_LPAToMITag(lpa);

  for (i = 0; i < DT_L2MHashWidth; i++) {
    if (dtL2MHash[hashRow][i].tag == tag)
      return dtL2MHash[hashRow][i].metai;
    }

  /*
   *  LPA not in L2M hash table.  See if there is a Meta element
   *  which matches this page.
   */

  /* +++ For now, perform a brute-force lookup in the Meta table. */
  for (i = 0; i < DT_MetaTableElements; i++) {
    if (dtMetaTable[i].lpa == lpa) {
      /*
       *  Found the Meta element corresponding to this
       *  Linear Page Address.  We should cache this
       *  pairing in the LPA to Mi hash table so next
       *  time an efficient lookup will occur.
       */

      /* +++ Do this more intelligently */
      dtL2MHash[hashRow][0].tag = tag;
      dtL2MHash[hashRow][0].metai = i;
      return i;
      }
    }

  /*
   *  LPA not in either hash table or meta table.  We need to create
   *  a new meta table entry for this page.
   */

  /* +++ For now, simple allocation scheme */
  if (dtMetaFreeIndex >= DT_MetaTableElements) {
    printf("dtTranslateLPAToMI: free_index > table size\n");
    exit(1);
    }

  i = dtMetaFreeIndex++;
  dtMetaTable[i].lpa = lpa;
  dtMetaTable[i].offsetLookupFreeIndex = 0;
  dtMetaTable[i].tcodeBufferFreeIndex = 0;

  return i;
}


  Bit32u
dtTranslateG2T(Bit32u gOff)
{
  /* Translate from a guest offset to tcode offset */

  unsigned hashRow, i, metaIndex;
  Bit32u tAddr;
  Bit32u gla;

  /*
   *  Search the G2T table first, ideally the instruction will have
   *  been translated already, and the translation address in there.
   */

/* +++ what about addr==0 */
  hashRow = DT_G2THashSelect(gOff);
  for (i = 0; i < DT_G2THashWidth; i++) {
    if (dtG2THash[hashRow][i].gOff == gOff) {
      return dtG2THash[hashRow][i].tOff;
      }
    }

  gla = CS.base + gOff;
  metaIndex = dtTranslateLPAToMI(gla >> 12);
  tAddr = dtMetaLookupTCode(metaIndex, gla);
  if (tAddr == 0) {
    /* Instruction does not have associated tcode; we must translate. */
    tAddr = dtTranslateSequence(metaIndex, gOff, gla);
    dtSetG2TPair(hashRow, gOff, tAddr);
    }

  return tAddr;
}


  Bit32u
dtMetaLookupTCode(unsigned metaIndex, Bit32u gla)
{
  /* Lookup a tcode offset associated with the guest linear address */

  unsigned i;
  Bit32u pOff;

  pOff = gla & 0x00000fff;

  for (i = 0; i < dtMetaTable[metaIndex].offsetLookupFreeIndex; i++) {
    if (dtMetaTable[metaIndex].offsetLookup[i].pOff == pOff)
      return dtMetaTable[metaIndex].offsetLookup[i].tcode;
    }

  return 0; /* not found */
}


  Bit8u *
dtCreateFragment(Bit8u * tcode, unsigned tcodeLen, Bit32u pOff,
                 unsigned metaIndex)
{
  unsigned freeIndex;
  int tcodeBufRemain;
  Bit8u *tcodePtr;

  /* Add tcode sequence to tcode buffer. */
  /* +++ For now it's stored more simply in the meta array. */

  /* Do we have the room? */
  freeIndex = dtMetaTable[metaIndex].tcodeBufferFreeIndex;
  tcodeBufRemain = (DT_TcodeBufferMax - 1) - freeIndex;
  if (((int) tcodeLen) > tcodeBufRemain) {
    printf("dtTS: tcode buffer full\n");
    exit(1);
    }

  /*
   *  Reposition pointer to the tcode buffer where we will copy
   *  tcode.  Copy the tcode and advance the free index.
   */

  tcodePtr = &dtMetaTable[metaIndex].tcodeBuffer[freeIndex];
  memcpy(tcodePtr, tcode, tcodeLen);
  dtMetaTable[metaIndex].tcodeBufferFreeIndex += tcodeLen;

  /* Add page offset -> tcode lookup entry in page-oriented tables */
  freeIndex = dtMetaTable[metaIndex].offsetLookupFreeIndex;
  if (freeIndex >= (DT_OffsetLookupMax - 1)) {
    printf("dtTS: pOff -> tcode table full\n");
    exit(1);
    }

  dtMetaTable[metaIndex].offsetLookup[freeIndex].pOff = pOff;
  dtMetaTable[metaIndex].offsetLookup[freeIndex].tcode = (Bit32u) tcodePtr;
  dtMetaTable[metaIndex].offsetLookupFreeIndex++;

  return tcodePtr;
}
