/*
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 1999-2001 Kevin P. Lawton
 *
 *  arith8.c:  8-bit arithmetic instructions
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */


#include "plex86.h"
#include "monitor.h"



  void
CMPXCHG_EbGb(vm_t *vm)
{
  Bit8u  op2, op1, diff;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subb %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (diff)
    : "mq" (op1), "1" (G_AL(vm))
    : "memory", "cc"
    );
 
  if (diff == 0) {
    /* dest <-- src */
    op2 = ReadReg8(vm, vm->i.nnn);
 
    /* +++ docs say CMPXCHG causes a write cycle regardless */
    if (vm->i.mod == 0xc0) {
      WriteReg8(vm, vm->i.rm, op2);
      }
    else {
      write_RMW_virtual_byte(vm, &op2);
      }
    }
  else {
    /* accumulator <-- dest */
    G_AL(vm) = op1;
    }

  SetOSZAPC(vm, eflags);
}

  void
ADD_EbGb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_GbEb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "addb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  WriteReg8(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
ADD_ALIb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = G_AL(vm);
  op2 = vm->i.Ib;
 
  asm volatile (
    "addb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  G_AL(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
NEG_Eb(vm_t *vm)
{
  Bit8u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "negb %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=mq" (op1) 
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
INC_Eb(vm_t *vm)
{
  Bit8u op1;
  Bit32u eflags;

  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }

  asm volatile (
    "incb %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=mq" (op1) 
    : "1" (op1)
    : "memory", "cc"
    );

  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }

  SetOSZAP(vm, eflags);
}

  void
DEC_Eb(vm_t *vm)
{
  Bit8u op1;
  Bit32u eflags;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "decb %1 \n\t"
    "pushfl  \n\t"
    "popl   %0"
    : "=g" (eflags), "=mq" (op1)
    : "1" (op1)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAP(vm, eflags);
}

  void
ADD_EbIb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Ib;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "addb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EbIb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Ib;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_EbGb(vm_t *vm)
{
  Bit8u  op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_GbEb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg8(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
ADC_ALIb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = G_AL(vm);
  op2 = vm->i.Ib;
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "adcb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  G_AL(vm) = op1;
  SetOSZAPC(vm, eflags);
}

  void
SBB_EbGb(vm_t *vm)
{
  Bit8u  op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_GbEb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;

  op1 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc         \n\t"
      "sbbb %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc         \n\t"
      "sbbb %3, %1 \n\t"
      "pushfl      \n\t"
      "popl   %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  WriteReg8(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_ALIb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = G_AL(vm);
  op2 = vm->i.Ib;

  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  G_AL(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
SBB_EbIb(vm_t *vm)
{
  Bit8u  op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Ib;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  if (G_GetCF(vm)) {
    asm volatile (
      "stc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
  else {
    asm volatile (
      "clc        \n\t"
      "sbbb %3, %1\n\t"
      "pushfl     \n\t"
      "popl %0"
      : "=g" (eflags), "=q" (op1)
      : "1" (op1), "mq" (op2)
      : "memory", "cc"
      );
    }
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EbIb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Ib;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_EbIb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = vm->i.Ib;
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "cmpb %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "q" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_EbGb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "cmpb %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "q" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_ALIb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op1 = G_AL(vm);
  op2 = vm->i.Ib;
 
  asm volatile (
    "cmpb %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "q" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
CMP_GbEb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "cmpb %2, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags)
    : "q" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_GbEb(vm_t *vm)
{
  Bit8u op1, op2;
  Bit32u eflags;
 
  op1 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op2 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op2);
    }
 
  asm volatile (
    "subb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  WriteReg8(vm, vm->i.nnn, op1);
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_EbGb(vm_t *vm)
{
  Bit8u op2, op1;
  Bit32u eflags;
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
 
  asm volatile (
    "subb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.rm, op1);
    }
  else {
    write_RMW_virtual_byte(vm, &op1);
    }
 
  SetOSZAPC(vm, eflags);
}

  void
SUB_ALIb(vm_t *vm)
{
  Bit8u  op1, op2;
  Bit32u eflags;
 
  op1 = G_AL(vm);
  op2 = vm->i.Ib;
 
  asm volatile (
    "subb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (op1)
    : "1" (op1), "mq" (op2)
    : "memory", "cc"
    );
 
  G_AL(vm) = op1;
 
  SetOSZAPC(vm, eflags);
}

  void
XADD_EbGb(vm_t *vm)
{
  Bit8u op2, op1, sum;
  Bit32u eflags;
 
  /* XADD dst(r/m8), src(r8)
   * temp <-- src + dst         | sum = op2 + op1
   * src  <-- dst               | op2 = op1
   * dst  <-- tmp               | op1 = sum
   */
 
  op2 = ReadReg8(vm, vm->i.nnn);
 
  if (vm->i.mod == 0xc0) {
    op1 = ReadReg8(vm, vm->i.rm);
    }
  else {
    read_RMW_virtual_byte(vm, vm->i.seg, vm->i.rm_addr, &op1);
    }
  sum = op1;
 
  asm volatile (
    "addb %3, %1\n\t"
    "pushfl     \n\t"
    "popl %0"
    : "=g" (eflags), "=q" (sum)
    : "1" (sum), "mq" (op2)
    : "memory", "cc"
    );
 
  if (vm->i.mod == 0xc0) {
    WriteReg8(vm, vm->i.nnn, op1);
    WriteReg8(vm, vm->i.rm, sum);
    }
  else {
    write_RMW_virtual_byte(vm, &sum);
    WriteReg8(vm, vm->i.nnn, op1);
    }
 
  SetOSZAPC(vm, eflags);
}
