/************************************************************************
 * $Id: iosniffer.c,v 1.3 2002/07/21 23:14:55 DemonLord Exp $
 ************************************************************************
 *
 *  plex86: run multiple x86 operating systems concurrently
 *  Copyright (C) 2002  Kevin P. Lawton
 *
 *  iosniffer.c: Intercepts and prints IO calls on ports.
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 */

/*
 *  Slows down a lot the IO calls. I should use something else than
 *  linked lists. :-P
 *
 *  Eric Laberge <e.laberge@videotron.ca>
 */

#include <stdio.h>
#include <string.h>

#include "iosniffer.h"
#include "plex86.h"
#include "user.h"
#include "plugin.h"

#define IO_OP_READ      1
#define IO_OP_WRITE     2

int iosnifferRegisterIOReadHandler(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len);
int iosnifferRegisterIOWriteHandler(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len);

int (*realRegisterIOReadHandler)(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len);
int (*realRegisterIOWriteHandler)(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len);

static void port_write(void *this_ptr, Bit32u address, Bit32u value, unsigned int io_len);
static Bit32u port_read(void *this_ptr, Bit32u address, unsigned int io_len);

typedef struct _io_handler_lnk
{
  struct _io_handler_lnk *next;
  int port;
  union
  {
    ioReadHandler_t read;
    ioWriteHandler_t write;
  } handler;
  char *comment;
} io_handler_lnk;

static void last_op(int op_type, Bit32u address, unsigned int io_len, Bit32u value, io_handler_lnk *hand);

io_handler_lnk *ioread = NULL;
io_handler_lnk *iowrite = NULL;

int plugin_init(plugin_t *plugin, int argc, char *argv[])
{
  printf("Loading IO Sniffer plugin...\n");
  realRegisterIOReadHandler = pluginRegisterIOReadHandler;
  realRegisterIOWriteHandler = pluginRegisterIOWriteHandler;

  pluginRegisterIOReadHandler = iosnifferRegisterIOReadHandler;
  pluginRegisterIOWriteHandler = iosnifferRegisterIOWriteHandler;
  return 0;
}

void plugin_fini(void)
{
  last_op(0, 0, 0, 0, NULL);    /* Prints out any message left */
  printf("Unloading IO Sniffer plugin...\n");
  pluginRegisterIOReadHandler = realRegisterIOReadHandler;
  pluginRegisterIOWriteHandler = realRegisterIOWriteHandler;
}
      

int iosnifferRegisterIOReadHandler(void *thisPtr, ioReadHandler_t callback, unsigned base, const char *name, unsigned len)
{
  int i;
  printf("Registering IO Read Handler: base 0x%x, len %d\t%s (%s)\n", base, len, name, getString(base, 0));
  for (i = 0; i < len; i++)
  {
    io_handler_lnk *newhandler = (io_handler_lnk*) malloc(sizeof(io_handler_lnk));
    newhandler->port = base + i;
    newhandler->handler.read = callback;
    newhandler->next = ioread;
    newhandler->comment = getString(base + i, 0);
    ioread = newhandler;
  }
  return realRegisterIOReadHandler(thisPtr, port_read, base, name, len);
}

int iosnifferRegisterIOWriteHandler(void *thisPtr, ioWriteHandler_t callback, unsigned base, const char *name, unsigned len)
{
  int i;
  printf("Registering IO Write Handler: base 0x%x, len %d\t%s (%s)\n", base, len, name, getString(base, 1));
  for (i = 0; i < len; i++)
  {
    io_handler_lnk *newhandler = (io_handler_lnk*) malloc(sizeof(io_handler_lnk));
    newhandler->port = base + i;
    newhandler->handler.write = callback;
    newhandler->next = iowrite;
    newhandler->comment = getString(base + i, 1);
    iowrite = newhandler;
  }
  return realRegisterIOWriteHandler(thisPtr, port_write, base, name, len);
}

static void port_write(void *this_ptr, Bit32u address, Bit32u value, unsigned int io_len)
{
  static io_handler_lnk *old_iter = NULL;
  io_handler_lnk *iter = iowrite;
  if (old_iter && old_iter->port == address)
    iter = old_iter;
  else
  {
    while (iter && iter->port != address)
      iter = iter->next;
  }
  
  if (!iter)
  {
    printf("Unknown port 0x%x\n", address);
    return;
  }
  
  last_op(IO_OP_WRITE, address, io_len, value, iter);
  old_iter = iter;

  iter->handler.write(this_ptr, address, value, io_len);
}

static Bit32u port_read(void *this_ptr, Bit32u address, unsigned int io_len)
{
  static io_handler_lnk *old_iter = NULL;
  Bit32u result;
  io_handler_lnk *iter = ioread;
  if (old_iter && old_iter->port == address)
    iter = old_iter;
  else
  {
    while (iter && iter->port != address)
      iter = iter->next;
  }

  if (!iter)
  {
    printf("Unknown port 0x%x\n", address);
    return 0;
  }

  result = iter->handler.read(this_ptr, address, io_len);
  last_op(IO_OP_READ, address, io_len, result, iter);
  old_iter = iter;
  return result;
}

static void last_op(int op_type, Bit32u address, unsigned int io_len, Bit32u value, io_handler_lnk *hand)
{
  static Bit32u old_address = 0, old_value = 0, old_io_len = 0, count = 0;
  static int old_op_type = 0;
  static io_handler_lnk *old_hand = NULL;

  if (op_type != old_op_type || old_address != address || old_value != value || old_io_len != io_len)
  {
    if (old_address != 0)
    {
      if (old_op_type == IO_OP_READ)
        printf("Reading %d bytes from port 0x%x (%d times):\t0x%x\t%s\n", old_io_len, old_address, count, old_value, old_hand->comment);
      else if (old_op_type == IO_OP_WRITE)
        printf("Writing %d bytes to port 0x%x   (%d times):\t0x%x\t%s\n", old_io_len, old_address, count, old_value, old_hand->comment);
    }
    old_address = address;
    old_value = value;
    old_io_len = io_len;
    old_hand = hand;
    old_op_type = op_type;
    count = 0;
  }
  count++;
}
