/*
    This file is part of pload - a program to monitor ppp activity for X
    Copyright (C) 1999-2000  Matt Smith <mdsmith@engr.utk.edu>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

*/

/*
 proc_stat.c - get stats from linux proc filesystem

   example /proc/net/dev contents:

Inter-|   Receive                                                |  Transmit
 face |bytes    packets errs drop fifo frame compressed multicast|bytes    packets errs drop fifo colls carrier compressed
    lo:32894796   21239    0    0    0     0          0         0 32894796   21239    0    0    0     0       0          0
  ppp0: 3880205    5577    0    0    0     0          0         0   368360    5192    0    0    0     0       0          0


 thanks to Mathieu Clabaut <clabault@multimania.com> for catching
 the potentially missing space after the device name

*/

#ifdef LINUXPROC

#include <stdio.h>	/* fopen, rewind, fgets */
#include <stdlib.h>	/* exit, strtoul */
#include <string.h>	/* strtok, strstr */
#include "pload.h"

void proc_stat(if_data *ifd)
{
	const char sep[] = " :";   /* spaces or colons for field separators */
	char *s;
	char buf[256];
	int i;
	int found = 0;
	
	/* if it's not open, open it */
	if (ifd->file == NULL)
	{
		ifd->file = fopen("/proc/net/dev", "r");
		if (ifd->file == NULL) die("couldn't fopen /proc/net/dev");
	}
	
	/* make sure we are at the start of file */
	rewind(ifd->file);
	
	/* skip over first two lines */
	(void *)fgets(buf, sizeof(buf), ifd->file);
	(void *)fgets(buf, sizeof(buf), ifd->file);
	
	/* loop till we find the specified device */
	while ( (found == 0) && (fgets(buf, sizeof(buf), ifd->file) != NULL) )
	{
		s = strtok(buf, sep);
		/* see if the first field is the device */
		if (strstr(s, ifd->device) != NULL)
		{
			/* yup, there it is */
			/* now read in the next field IN */
			s = strtok(NULL, sep);
			ifd->in_bytes = strtoul(s, (char **)NULL, 10);
			
			/* skip the next seven fields */
			for (i=0;i<7;i++)
				s = strtok(NULL, sep);
			
			/* now read in the OUT bytes field */
			s = strtok(NULL, sep);
			ifd->out_bytes = strtoul(s, (char **)NULL, 10);
			/* done */
			
			/* exit out of while loop */
			found = 1;
			
		}
		/* nope, can't find it */
	}
	
	if (found == 0)	/* EOF caught, non-existant or not connected */
	{
		ifd->in_bytes = 0UL;
		ifd->out_bytes = 0UL;
	}
	else
		/* finish to EOF to get new stats next time */
		while (fgets(buf, sizeof(buf), ifd->file) != NULL);
	return;
}

#endif
