/*
 * $Id: font.c,v 1.25 2003/09/24 13:02:03 prussar Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2002, Mark Ian Lillywhite and Michael Nordstrom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "debug.h"
#include "hires.h"
#include "list.h"
#include "os.h"
#include "prefsdata.h"
#include "util.h"
#include "const.h"
#include "grayfont.h"

#include "font.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define EOL 0xFF


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
typedef struct {
    FontModeType  fontMode;
    FontID*       style;
} StyleType;


typedef struct {
    DmResType     resourceType;
    DmResID       resourceID;
    FontID        fontID;
    FontID        substituteID;
} CustomFontType;


typedef struct
{
    UInt16     cardNo;
    LocalID    dbID;
    Char       name[ dmDBNameLength ];
}
DBEntryType;


/* PalmOS 2 Fonts */
static FontID os2StandardStyles[ MAXSTYLES ] = {
    stdFont, boldFont, boldFont, boldFont,
    boldFont, stdFont, stdFont, stdFont, stdFont,
    stdFont, stdFont, stdFont
};

static FontID os2BoldStyles[ MAXSTYLES ] = {
    boldFont, boldFont, boldFont, boldFont,
    boldFont, boldFont, boldFont, boldFont, stdFont,
    boldFont, boldFont, boldFont
};

/* PalmOS 3+ Fonts */
static FontID os3NarrowStyles[ MAXSTYLES ] = {
    narrowFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, narrowBoldFont, narrowBoldFont, narrowBoldFont, narrowFixedFont,
    narrowFont, narrowFont, narrowFont
};

static FontID os3StandardStyles[ MAXSTYLES ] = {
    stdFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, boldFont, boldFont, boldFont, narrowFixedFont,
    stdFont, narrowFont, narrowFont
};

static FontID os3BoldStyles[ MAXSTYLES ] = {
    boldFont, largeBoldFont, largeBoldFont, largeBoldFont,
    largeBoldFont, boldFont, boldFont, boldFont, narrowFixedFont,
    stdFont, narrowBoldFont, narrowBoldFont
};

static FontID os3LargeStyles[ MAXSTYLES ] = {
    largeFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, largeBoldFont, largeBoldFont, largeBoldFont, narrowFixedFont,
    stdFont, stdFont, stdFont
};

static FontID os3LargeBoldStyles[ MAXSTYLES ] = {
    largeBoldFont, largeBoldFont, largeBoldFont, largeBoldFont,
    largeBoldFont, largeBoldFont, largeBoldFont, largeBoldFont, narrowFixedFont,
    boldFont, boldFont, boldFont
};

static FontID os3UserStyles[ MAXSTYLES ] = {
    userStdFont, userLargeBoldFont, userLargeBoldFont, userLargeFont,
    userLargeFont, userBoldFont, userBoldFont, userBoldFont, userFixedFont,
    userStdFont, userNarrowFont, userNarrowFont
};

/* Palm HiRes Fonts */
#ifdef HAVE_HIRES
static FontID palmHiResTinyStyles[ MAXSTYLES ] = {
    tinyFont_palm, boldFont, boldFont, smallBoldFont_palm,
    smallBoldFont_palm, tinyBoldFont_palm, tinyBoldFont_palm, tinyBoldFont_palm,
    stdFixedFont_palm, tinyFont_palm, tinyFont_palm, tinyFont_palm
};

static FontID palmHiResSmallStyles[ MAXSTYLES ] = {
    smallFont_palm, largeBoldFont, largeBoldFont, boldFont,
    boldFont, smallBoldFont_palm, smallBoldFont_palm, smallBoldFont_palm,
    stdFixedFont_palm, tinyFont_palm, tinyFont_palm, tinyFont_palm
};

static FontID palmHiResNarrowStyles[ MAXSTYLES ] = {
    narrowFont, largeBoldFont, largeBoldFont, boldFont,
    boldFont, narrowBoldFont, narrowBoldFont, narrowBoldFont,
    stdFixedFont_palm, narrowFont, narrowFont, narrowFont
};

static FontID palmHiResStandardStyles[ MAXSTYLES ] = {
    stdFont, largeBoldFont, largeBoldFont, boldFont,
    boldFont, boldFont, smallBoldFont_palm, boldFont,
    stdFixedFont_palm, stdFont, narrowFont, narrowFont
};

static FontID palmHiResLargeStyles[ MAXSTYLES ] = {
    largeFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, boldFont, boldFont, largeBoldFont, stdFixedFont_palm,
    stdFont, stdFont, stdFont
};

static FontID palmHiResUserStyles[ MAXSTYLES ] = {
    userStdFont_palm, userLargeBoldFont_palm, userLargeBoldFont_palm, 
    userLargeFont_palm,
    userLargeFont_palm, userBoldFont_palm, userBoldFont_palm, userBoldFont_palm,
    userFixedFont_palm, userStdFont_palm, userNarrowFont_palm,
    userNarrowFont_palm
};

/* Sony HiRes Fonts */
#ifdef HAVE_SONY_SDK
static FontID sonyHiResTinyStyles[ MAXSTYLES ] = {
    hrTinyFont, hrSmallBoldFont, hrSmallBoldFont, hrTinyBoldFont,
    hrTinyBoldFont, hrTinyBoldFont, hrTinyBoldFont, hrTinyBoldFont,
    narrowFixedFont, narrowFont, narrowFont, narrowFont
};

static FontID sonyHiResSmallStyles[ MAXSTYLES ] = {
    hrSmallFont, hrBoldFont, hrBoldFont, hrSmallBoldFont,
    hrSmallBoldFont, hrTinyBoldFont, hrTinyBoldFont, hrSmallBoldFont,
    narrowFixedFont, hrTinyFont, hrTinyFont, hrTinyFont
};

static FontID sonyHiResNarrowStyles[ MAXSTYLES ] = {
    narrowFont, largeBoldFont, largeBoldFont, largeFont,
    largeFont, narrowBoldFont, narrowBoldFont, narrowBoldFont, narrowFixedFont,
    narrowFont, narrowFont, narrowFont
};

static FontID sonyHiResStandardStyles[ MAXSTYLES ] = {
    hrStdFont, hrLargeBoldFont, hrLargeBoldFont, hrBoldFont,
    hrBoldFont, hrSmallBoldFont, hrSmallBoldFont, hrBoldFont,
    stdFixedFont_sony, hrSmallFont, hrSmallFont, hrSmallFont
};

static FontID sonyHiResLargeStyles[ MAXSTYLES ] = {
    hrLargeFont, hrLargeBoldFont, hrLargeBoldFont, hrLargeBoldFont,
    hrLargeBoldFont, hrBoldFont, hrBoldFont, hrLargeBoldFont,
    stdFixedFont_sony, hrStdFont, hrStdFont, hrStdFont
};

static FontID sonyHiResUserStyles[ MAXSTYLES ] = {
    userStdFont_sony, userLargeBoldFont_sony, userLargeBoldFont_sony,
    userLargeFont_sony,
    userLargeFont_sony, userBoldFont_sony, userBoldFont_sony,
    userBoldFont_sony, userFixedFont_sony,
    userStdFont_sony, userNarrowFont_sony, userNarrowFont_sony
};
#endif /* HAVE_SONY_SDK */

/* Handera HiRes Fonts */
#ifdef HAVE_HANDERA_SDK
/* These need to be built at run-time by HanderaConvertFontList() */
static FontID handeraHiResStandardStyles[ MAXSTYLES ] = {};
static FontID handeraHiResLargeStyles[ MAXSTYLES ] = {};
#endif /* HAVE_HANDERA_SDK */

#endif /* HAVE_HIRES */


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static LinkedList   customFontList      = NULL;
static FontID*      StylesLibrary;
static FontID*      StylesMain;
static Int16        currentStyle        = DEFAULTSTYLE;
static Int16        currentStyleFont    = stdFont;
static Int16        prevFontHeight      = -1;


static CustomFontType ListOfCustomStandardFonts[] = {
    { 'NFNT', narrowFontID, narrowFont, stdFont },
    { 'NFNT', narrowFixedFontID, narrowFixedFont, boldFont },
    { 'NFNT', narrowBoldFontID, narrowBoldFont, boldFont },
    { 'NFNT', userStdFontID, userStdFont, stdFont },
    { 'NFNT', userBoldFontID, userBoldFont, boldFont },
    { 'NFNT', userLargeFontID, userLargeFont, largeFont },
    { 'NFNT', userLargeBoldFontID, userLargeBoldFont, largeBoldFont },
    { 'NFNT', userFixedFontID, userFixedFont, boldFont },
    { 'NFNT', userNarrowFontID, userNarrowFont, stdFont },
    { EOL, NULL, NULL }
};

#ifdef HAVE_HIRES
static CustomFontType ListOfCustomPalmHiResFonts[] = {
    { 'nfnt', tinyFontID_palm, tinyFont_palm, stdFont },
    { 'nfnt', tinyBoldFontID_palm, tinyBoldFont_palm, stdFont },
    { 'nfnt', smallFontID_palm, smallFont_palm, stdFont },
    { 'nfnt', smallBoldFontID_palm, smallBoldFont_palm, boldFont },
    { 'nfnt', stdFixedFontID_palm, stdFixedFont_palm, boldFont },
    { 'nfnt', userStdFontID_palm, userStdFont_palm, stdFont },
    { 'nfnt', userBoldFontID_palm, userBoldFont_palm, boldFont },
    { 'nfnt', userLargeFontID_palm, userLargeFont_palm, largeFont },
    { 'nfnt', userLargeBoldFontID_palm, userLargeBoldFont_palm, largeBoldFont },
    { 'nfnt', userFixedFontID_palm, userFixedFont_palm, boldFont },
    { 'nfnt', userNarrowFontID_palm, userNarrowFont_palm, stdFont },
    { EOL, NULL, NULL }
};

#ifdef HAVE_SONY_SDK
static CustomFontType ListOfCustomSonyHiResFonts[] = {
    { 'NFNT', stdFixedFontID_sony, stdFixedFont_sony, hrBoldFont },
    { 'NFNT', userStdFontID_sony, userStdFont_sony, hrStdFont },
    { 'NFNT', userBoldFontID_sony, userBoldFont_sony, hrBoldFont },
    { 'NFNT', userLargeFontID_sony, userLargeFont_sony, hrLargeFont },
    { 'NFNT', userLargeBoldFontID_sony, userLargeBoldFont_sony, hrLargeBoldFont },
    { 'NFNT', userFixedFontID_sony, userFixedFont_sony, hrBoldFont },
    { 'NFNT', userNarrowFontID_sony, userNarrowFont_sony, hrStdFont },
    { EOL, NULL, NULL }
};
#endif /* HAVE_SONY_SDK */

#endif /* HAVE_HIRES */

static StyleType ListOfFontStylesOS2[] = {
    { FONT_DEFAULT, os2StandardStyles },
    { FONT_BOLD, os2BoldStyles },
    { EOL, NULL }
};

static StyleType ListOfFontStylesOS3[] = {
    { FONT_NARROW, os3NarrowStyles },
    { FONT_DEFAULT, os3StandardStyles },
    { FONT_BOLD, os3BoldStyles },
    { FONT_LARGE, os3LargeStyles },
    { FONT_LARGEBOLD, os3LargeBoldStyles },
    { FONT_USER, os3UserStyles },
    { EOL, NULL }
};

#ifdef HAVE_HIRES
static StyleType ListOfFontStylesPalm[] = {
    { FONT_TINY, palmHiResTinyStyles },
    { FONT_SMALL, palmHiResSmallStyles },
    { FONT_NARROW, palmHiResNarrowStyles },
    { FONT_DEFAULT, palmHiResStandardStyles },
    { FONT_LARGE, palmHiResLargeStyles },
    { FONT_USER, palmHiResUserStyles },
    { EOL, NULL }
};

#ifdef HAVE_SONY_SDK
static StyleType ListOfFontStylesSony[] = {
    { FONT_TINY, sonyHiResTinyStyles },
    { FONT_SMALL, sonyHiResSmallStyles },
    { FONT_NARROW, sonyHiResNarrowStyles },
    { FONT_DEFAULT, sonyHiResStandardStyles },
    { FONT_LARGE, sonyHiResLargeStyles },
    { FONT_USER, sonyHiResUserStyles },
    { EOL, NULL }
};
#endif /* HAVE_SONY_SDK */

#ifdef HAVE_HANDERA_SDK
static StyleType ListOfFontStylesHandera[] = {
    { FONT_TINY, os3StandardStyles },
    { FONT_SMALL, os3LargeStyles },
    { FONT_NARROW, os3NarrowStyles },
    { FONT_DEFAULT, handeraHiResStandardStyles },
    { FONT_LARGE, handeraHiResLargeStyles },
    { EOL, NULL }
};
#endif /* HAVE_HANDERA_SDK */

#endif /* HAVE_HIRES */

static LinkedList userFontDBList          = NULL;
static Int16      numberOfUserFonts       = 0;
static Char**     userFontNames           = NULL;
static Char*      emptyUserFontNameList[] = { 
    "--" 
};
static DmOpenRef  currentUserFontDBRef;
static Int16      currentUserFontNumber   = NO_SUCH_USER_FONT;
static DeviceFontType  currentDeviceFont;


/* Load custom fonts into memory */
void LoadCustomFonts
    (
    DeviceFontType  deviceFont
    )
{
    CustomFontType* fontList;
    MemHandle*      handleNode;
    FontType*       font;

    currentDeviceFont = deviceFont;
    
    GrayFntClearSubstitutionList();

    if ( customFontList == NULL )
        customFontList = ListCreate();

    switch ( deviceFont ) {
        case STANDARD_FONTS:
            fontList = ListOfCustomStandardFonts;
            break;
#ifdef HAVE_HIRES
        case PALM_FONTS:
            fontList = ListOfCustomPalmHiResFonts;
            break;
#ifdef HAVE_SONY_SDK
        case SONY_FONTS:
            fontList = ListOfCustomSonyHiResFonts;
            break;
#endif
#endif
        default:
            return;
    }

    for ( ; fontList->resourceType != EOL; fontList++ ) {
        handleNode = SafeMemPtrNew( sizeof( MemHandle ) );
        *handleNode = DmGetResource( fontList->resourceType,
                                     fontList->resourceID );
        if ( *handleNode != NULL ) {
            font   = MemHandleLock( *handleNode );
            FntDefineFont( fontList->fontID, font );
            ListAppend( customFontList, handleNode );
        }
        else {
            SafeMemPtrFree( handleNode );
            GrayFntSubstitute( fontList->fontID, fontList->substituteID );
        }
    }
}



/* Release custom fonts from memory */
void ReleaseCustomFonts( void )
{
    MemHandle* handleNode;
    MemHandle* nextHandleNode;

    if ( customFontList == NULL )
        return;

    handleNode = ListFirst( customFontList );
    while ( handleNode != NULL ) {
        if ( *handleNode != NULL ) {
            MemHandleUnlock( *handleNode );
            DmReleaseResource( *handleNode );
        }
        nextHandleNode = ListNext( customFontList, handleNode );
        ListTakeOut( customFontList, handleNode );
        SafeMemPtrFree( handleNode );
        handleNode = nextHandleNode;
    }
    ListRelease( customFontList );
    customFontList = NULL;
}



/* Refresh custom fonts in memory */
void RefreshCustomFonts( void )
{
    if ( ! Support30() )
        return;
    ReleaseCustomFonts();
    LoadCustomFonts( currentDeviceFont );
}


/* Return the current font style */
Int16 GetCurrentStyle( void )
{
    return currentStyle;
}



/* Return the previous font style */
Int16 GetPrevFontHeight( void )
{
    if ( prevFontHeight < 0 )
        return FntCharHeight(); /* FIXME! */
    else
        return prevFontHeight;
}



/* Set the previous font height */
void SetPrevFontHeight
      (
      Int16  height
      )
{
      prevFontHeight = height;
}



/* Return main style font */
FontID GetMainStyleFont
    (
    const Int16 style   /* numerical representation for font style */
    )
{
    return StylesMain[ style ];
}



/* Get the default style font height */
Int16 GetDefaultMainStyleHeight( void )
{
    FontID oldFont;
    Int16  height;
    UInt16 prevCoordSys;

    prevCoordSys = PalmSetCoordinateSystem( NATIVE );

    oldFont = FntGetFont();
    FntSetFont( StylesMain[ DEFAULTSTYLE ] );
    height = FntLineHeight() + Prefs()->lineSpacing;
    FntSetFont( oldFont );

    PalmSetCoordinateSystem( prevCoordSys );
    
    return height;
}



/* Set and return main style font */
FontID SetMainStyleFont
    (
    const Int16 style   /* numerical representation for font style */
    )
{
    FntSetFont ( StylesMain[ currentStyle ] );
    prevFontHeight   = FntCharHeight();
    currentStyle     = style;
    currentStyleFont = StylesMain[ currentStyle ];
    FntSetFont( currentStyleFont );
    return currentStyleFont;
}



/* Return library style font */
FontID GetLibraryStyleFont
    (
    const Int16 style   /* numerical representation for font style */
    )
{
    return StylesLibrary[ style ];
}



/* Set the general pointer to its arch-dependant function */
void SetFontFunctions( void )
{
    if ( Support30() )
        SetFontStyles = SetFontStylesOS3;
    else
        SetFontStyles = SetFontStylesOS2;

#ifdef HAVE_HIRES
    if ( IsHiResTypePalm( HiResType() ) )
        SetFontStyles = SetFontStylesPalm;
#endif
#ifdef HAVE_SONY_SDK
    if ( IsHiResTypeSony( HiResType() ) )
        SetFontStyles = SetFontStylesSony;
#endif
#ifdef HAVE_HANDERA_SDK
    if ( IsHiResTypeHandera( HiResType() ) )
        SetFontStyles = SetFontStylesHandera;
#endif

    SetFontStyles();
}



/* Set selected font style */
void SetFontStylesOS2( void )
{
    FontModeType prefsFont;
    StyleType*   list;

    prefsFont = Prefs()->fontModeMain;
    for ( list = ListOfFontStylesOS2; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesMain = list->style;
            break;
        }
    }
    prefsFont = Prefs()->fontModeLibrary;
    for ( list = ListOfFontStylesOS2; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesLibrary = list->style;
            break;
        }
    }
}



/* Set selected font style */
void SetFontStylesOS3( void )
{
    FontModeType prefsFont;
    StyleType*   list;

    prefsFont = Prefs()->fontModeMain;
    for ( list = ListOfFontStylesOS3; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesMain = list->style;
            break;
        }
    }
    prefsFont = Prefs()->fontModeLibrary;
    for ( list = ListOfFontStylesOS3; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesLibrary = list->style;
            break;
        }
    }
}



#ifdef HAVE_HIRES
/* Set selected font style */
void SetFontStylesPalm( void )
{
    FontModeType prefsFont;
    StyleType*   list;

    prefsFont = Prefs()->fontModeMain;
    for ( list = ListOfFontStylesPalm; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesMain = list->style;
            break;
        }
    }
    prefsFont = Prefs()->fontModeLibrary;
    for ( list = ListOfFontStylesPalm; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesLibrary = list->style;
            break;
        }
    }
}
#endif



#if defined( HAVE_HIRES ) && defined( HAVE_SONY_SDK )
/* Set selected font style */
void SetFontStylesSony( void )
{
    FontModeType prefsFont;
    StyleType*   list;

    prefsFont = Prefs()->fontModeMain;
    for ( list = ListOfFontStylesSony; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesMain = list->style;
            break;
        }
    }
    prefsFont = Prefs()->fontModeLibrary;
    for ( list = ListOfFontStylesSony; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesLibrary = list->style;
            break;
        }
    }
}
#endif



#if defined( HAVE_HIRES ) && defined( HAVE_HANDERA_SDK )
/* Set selected font style */
void SetFontStylesHandera( void )
{
    FontModeType prefsFont;
    StyleType*   list;

    /* Convert font list into VGA-specific versions of font */
    HanderaConvertFontList( os3StandardStyles, handeraHiResStandardStyles );
    HanderaConvertFontList( os3LargeStyles, handeraHiResLargeStyles );

    prefsFont = Prefs()->fontModeMain;
    for ( list = ListOfFontStylesHandera; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesMain = list->style;
            break;
        }
    }
    prefsFont = Prefs()->fontModeLibrary;
    for ( list = ListOfFontStylesHandera; list->fontMode != EOL; list++ ) {
        if ( list->fontMode == prefsFont ) {
            StylesLibrary = list->style;
            break;
        }
    }
}
#endif



/* Display the example font character centred in the given object ID */
void DisplayChar
    (
    FontID      fontID, /* font to print */
    const char  letter, /* letter to display */
    FormType*   form,   /* pointer form to print to */
    UInt16      objID   /* object to print in form */
    )
{
    RectangleType bounds;
    Coord         x;
    Coord         y;
#ifdef HAVE_GRAY_FONT
    Char          orientation;
    orientation   = GrayFntSetOrientation( GRAY_FONT_NORMAL );
#endif

    FrmGetObjectBounds( form, FrmGetObjectIndex( form, objID ), &bounds );
    FntSetFont( fontID );

    /* double our input values for sony */
    HiResAdjustBounds( &bounds, sonyHiRes );

    /* find approximate centered position for character */
    x = bounds.topLeft.x +
        ( bounds.extent.x / 2 ) -
        ( FntCharWidth( letter ) / 2 );
    y = bounds.topLeft.y +
        ( bounds.extent.y / 2 ) -
        ( FntCharHeight() / 2 );

    WinDrawChars( &letter, 1, x, y );

#ifdef HAVE_GRAY_FONT
    GrayFntSetOrientation( orientation );
#endif
}



/* Replacement for FntLineHeight() that takes into account subscript and
   superscript text */
extern Int16 FontLineHeight( void )
{
    Int16 height;

    if ( currentStyle == SUPSTYLE ) {
        height = FntLineHeight() / 4 + GetPrevFontHeight();
    }
    else
        height = FntLineHeight();

    return height;
}


/* Initialize internal lists of user font prcs */
void InitializeUserFontDBs( void )
{
    DmSearchStateType   stateInfo;
    UInt16              cardNo;
    LocalID             dbID;
    Err                 err;
    DBEntryType*        dbListEntry;
    Int16               i;

    userFontDBList = ListCreate();

    if ( NULL == userFontDBList )
        return;

    numberOfUserFonts = 0;

    err = DmGetNextDatabaseByTypeCreator( true, &stateInfo,
            (UInt32) UserFontResourceType, (UInt32) ViewerAppID,
            false, &cardNo, &dbID );

    while ( err == errNone ) {
        dbListEntry = SafeMemPtrNew( sizeof( DBEntryType ) );
        err = DmDatabaseInfo( cardNo, dbID, dbListEntry->name, NULL, NULL,
                NULL, NULL, NULL, NULL, NULL, NULL, NULL, NULL );
        if ( err == errNone ) {
            dbListEntry->cardNo = cardNo;
            dbListEntry->dbID   = dbID;
            ListAppend( userFontDBList, dbListEntry );
            numberOfUserFonts++;
        }
        else
            SafeMemPtrFree( dbListEntry );
        err = DmGetNextDatabaseByTypeCreator( false, &stateInfo,
                (UInt32) UserFontResourceType, (UInt32) ViewerAppID, false,
                &cardNo, &dbID );
    }

    if ( 0 < numberOfUserFonts ) {
        dbListEntry = ListFirst( userFontDBList );
        userFontNames = SafeMemPtrNew( numberOfUserFonts * sizeof( Char* ) );
        for ( i = 0; i < numberOfUserFonts && NULL != dbListEntry ; i++ ) {
            userFontNames[ i ] = SafeMemPtrNew( 1 +
                                     StrLen( dbListEntry->name ) );
            StrCopy( userFontNames[ i ], dbListEntry->name );
            dbListEntry = ListNext( userFontDBList, dbListEntry );
        }
        numberOfUserFonts = i;
    }
}



/* Go from name to position in user font list, or NO_SUCH_USER_FONT */
Int16 GetUserFontNumber(
    Char* name    /* name of user font to search for */
    )
{
    Int16 i;
    if ( numberOfUserFonts == 0 )
        return NO_SUCH_USER_FONT;
    for ( i = 0 ; i < numberOfUserFonts ; i++ )
        if ( ! StrCompare( userFontNames[ i ] , name ) )
            return i;
    return NO_SUCH_USER_FONT;
}



/* Go from position in user font list to name */
Char* GetUserFontName(
    Int16 number    /* number of user font */
    )
{
    if ( numberOfUserFonts == 0 || number == NO_SUCH_USER_FONT )
        return "";
    else
        return userFontNames[ number ];
}



/* Load a particular user font prc */
/* Can take NO_SUCH_USER_FONT as an argument */
void LoadUserFont(
    Int16 number  /* number of user font to load */
    )
{
    DBEntryType*  dbListEntry;
    CloseUserFont();
    if ( numberOfUserFonts == 0 || number == NO_SUCH_USER_FONT ) {
        currentUserFontNumber = NO_SUCH_USER_FONT;
        RefreshCustomFonts();
        return;
    }
    dbListEntry = ListGet( userFontDBList, number + 1 );
    if ( dbListEntry != NULL ) {
        currentUserFontDBRef = DmOpenDatabase( dbListEntry->cardNo,
                                   dbListEntry->dbID, dmModeReadOnly );
        currentUserFontNumber = number;
    }
    else {
        currentUserFontNumber = NO_SUCH_USER_FONT;
    }
    return;
}



/* Unload the current user font prc, if any */
void CloseUserFont( void )
{
  if ( currentUserFontNumber != NO_SUCH_USER_FONT ) {
        DmCloseDatabase( currentUserFontDBRef );
        currentUserFontNumber = NO_SUCH_USER_FONT;
  }
}



/* Check the current user font number */
Int16 GetCurrentUserFontNumber( void )
{
    return currentUserFontNumber;
}



/* Clear internal lists of user font prcs and close any user font */
void DeinitializeUserFontDBs( void )
{
    DBEntryType*  dbListEntry;
    DBEntryType*  nextDBListEntry;
    Int16         i;

    CloseUserFont();

    if ( NULL == userFontDBList )
        return;

    dbListEntry = ListFirst( userFontDBList );
    while ( dbListEntry != NULL ) {
        nextDBListEntry = ListNext( userFontDBList, dbListEntry );
        SafeMemPtrFree( dbListEntry );
        dbListEntry = nextDBListEntry;
    }

    ListDelete( userFontDBList );
    userFontDBList = NULL;

    for ( i = 0 ; i < numberOfUserFonts ; i++ )
        SafeMemPtrFree( userFontNames[ i ] );

    SafeMemPtrFree( userFontNames );
    userFontNames = NULL;
}



/* Initialize drop down list of user fonts */
void InitializeUserFontList(
    UInt16 listID    /* ID number of list for user font names */
    )
{
    ListType*   list;

    list = GetObjectPtr( listID );

    if ( 0 < numberOfUserFonts ) {
        LstSetListChoices( list, userFontNames, numberOfUserFonts );
        LstSetHeight( list, numberOfUserFonts );
    }
    else {
        LstSetListChoices( list, emptyUserFontNameList, 1 );
        LstSetHeight( list, 1 );
    }
}



/* How many user font prcs are there? */
Int16 GetNumberOfUserFonts( void )
{
    return numberOfUserFonts;
}
