/*
 * $Id: resultform.c,v 1.37 2003/09/24 13:02:03 prussar Exp $
 *
 * Viewer - a part of Plucker, the free off-line HTML viewer for PalmOS
 * Copyright (c) 1998-2002, Mark Ian Lillywhite and Michael Nordstrom
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 *
 */

#include "const.h"
#include "control.h"
#include "debug.h"
#include "os.h"
#include "prefsdata.h"
#include "resourceids.h"
#include "search.h"
#include "util.h"
#include "silkscreen.h"
#include "font.h"

#include "resultform.h"


/***********************************************************************
 *
 *      Internal Constants
 *
 ***********************************************************************/
#define DEFAULT_X           2
#define DEFAULT_Y           28
#define DEFAULT_EXTENT_X    156
#define DEFAULT_EXTENT_Y    11
#define MAX_RESULT_SONY_SILK_MIN   16
#define MAX_RESULT_NORMAL          9


/***********************************************************************
 *
 *      Internal Types
 *
 ***********************************************************************/
static struct {
    RectangleType   coordinates;
    UInt16          record;
    UInt16          pos;
    UInt16          length;
} SearchResult[ MAX_RESULT_SONY_SILK_MIN ];


/***********************************************************************
 *
 *      Local functions
 *
 ***********************************************************************/
static void ClearSearchResult( void ) RESULTFORM_SECTION;
static void ShowCancel( void ) RESULTFORM_SECTION;
static void ShowCancelMore( void ) RESULTFORM_SECTION;
static void UpdateStatus( Boolean findMore ) RESULTFORM_SECTION;
static void ResultFormInit( void ) RESULTFORM_SECTION;
static Boolean SearchResultHandleEvent(
                Boolean startFromBeginning ) RESULTFORM_SECTION;


/***********************************************************************
 *
 *      Private variables
 *
 ***********************************************************************/
static FormType*        resultForm;
static UInt16           resultFormID;
static Int16            maxResult;
static RectangleType    rect = { { DEFAULT_X, DEFAULT_Y },
                                 { DEFAULT_EXTENT_X, DEFAULT_EXTENT_Y } };
static Int16            count;
static Int16            lineMatch;
static Boolean          newLine;
static FieldType*       fldPtr;
static Char             matches[ 25+MAX_PATTERN_LEN ];
static Char             noMatches[ 25+MAX_PATTERN_LEN ];
static Char             searching[ 21 ];
static Char             pattern[ MAX_PATTERN_LEN + 1 ];
static FontID           initialFontStyle;



/* Clear result data variables */
static void ClearSearchResult( void )
{
    MemSet( &SearchResult, sizeof( SearchResult ), 0 );
    count           = 0;
    newLine         = false;
    rect.topLeft.y  = DEFAULT_Y;
}



/* Write header info for search result, return true if end of record has
   been reached */
Boolean DrawResultHeader
    (
    Char* header    /* header string */
    )
{
    UInt16 len;
    UInt16 deltaX;
    UInt16 coordY;

    if ( newLine ) {
        newLine         = false;
        rect.topLeft.y += DEFAULT_EXTENT_Y;
        count++;
    }

    len     = FntCharsWidth( header, StrLen( header ) );
    deltaX  = ( DEFAULT_EXTENT_X - len ) / 2;
    coordY  = rect.topLeft.y + DEFAULT_EXTENT_Y / 2;

    WinEraseRectangle( &rect, 0 );

    WinDrawLine( DEFAULT_X, coordY, deltaX, coordY );
    WinDrawChars( header, StrLen( header ), DEFAULT_X + deltaX, coordY - 5 );
    WinDrawLine( DEFAULT_X + deltaX + len + 1, coordY, DEFAULT_EXTENT_X - 3,
        coordY );

    if ( maxResult - 1 <= count ) {
        count           = 0;
        newLine         = false;
        rect.topLeft.y  = DEFAULT_Y;
        return true;
    }

    return false;
}



/* Write search result, return true if end of record has been reached */
Boolean DrawResultString
    (
    UInt16  record, /* record ID */
    UInt16  pos,    /* position of search pattern in record */
    UInt16  length, /* length of pattern */
    Char*   result  /* result string */
    )
{
    if ( resultFormID == frmResultSonySilkMin )
        maxResult = MAX_RESULT_SONY_SILK_MIN;
    else
        maxResult = MAX_RESULT_NORMAL;

    count++;

    rect.topLeft.y += DEFAULT_EXTENT_Y;

    SearchResult[ count ].coordinates   = rect;
    SearchResult[ count ].record        = record;
    SearchResult[ count ].pos           = pos;
    SearchResult[ count ].length        = length;

    WinDrawChars( result, StrLen( result ), rect.topLeft.x, rect.topLeft.y );

    if ( maxResult - 1 <= count ) {
        count           = 0;
        newLine         = false;
        rect.topLeft.y  = DEFAULT_Y;
        return true;
    }
    newLine = true;

    return false;
}



/* Display Cancel button */
static void ShowCancel( void )
{
    FrmHideObject( resultForm, FrmGetObjectIndex( resultForm, frmResultStop ) );
    FrmShowObject( resultForm, FrmGetObjectIndex( resultForm,
                                frmResultCancel ) );
}



/* Display Cancel and Find More buttons */
static void ShowCancelMore( void )
{
    ShowCancel();
    FrmShowObject( resultForm, FrmGetObjectIndex( resultForm,
                                frmResultFindMore ) );
}



/* Display current status: "Matches for <pattern>" or "No matches for
   <pattern>" */
static void UpdateStatus
    (
    Boolean findMore    /* true if more records remains to be searched */
    )
{
    if ( ! findMore && count == 0 )
        FldSetTextPtr( fldPtr, noMatches );
    else
        FldSetTextPtr( fldPtr, matches );

    FldDrawField( fldPtr );
}



/* Initialize the result form */
static void ResultFormInit( void )
{
    Char        str[ 30 ];

    resultFormID = GetValidForm( frmResult );
    if ( resultFormID == frmResultSonySilkMin ) {
        maxResult = MAX_RESULT_SONY_SILK_MIN;
    }
    else
        maxResult = MAX_RESULT_NORMAL;
    resultForm = FrmGetFormPtr( resultFormID );
    
    SonySilkDisableResize(); /* FIXME!  This is a temporary work-around. */

    GetSearchString( pattern );

    SysCopyStringResource( searching, strResultSearching );

    SysCopyStringResource( str, strResultMatches );
    StrPrintF( matches, "%s \"%s\"", str, pattern );

    SysCopyStringResource( str, strResultNoMatches );
    StrPrintF( noMatches, "%s \"%s\"", str, pattern );

    fldPtr = GetObjectPtr( frmResultStatus );
    FldSetTextPtr( fldPtr, searching );

    FrmDrawForm( resultForm );

    ClearSearchResult();

    initialFontStyle = FntSetFont( stdFont );
}



/* Event handler for the result form */
static Boolean SearchResultHandleEvent
    (
    Boolean startFromBeginning  /* start from the first record in DB */
    )
{
    Boolean handled;
    Boolean done;
    Boolean findMore;

    handled     = false;
    done        = true;
    findMore    = false;

    for ( ;; ) {
        EventType event;

        do {
            EvtGetEvent( &event, evtWaitForever );

            if ( SysHandleEvent( &event ) )
                continue;

            handled = false;

            switch ( event.eType ) {
                case ctlSelectEvent:
                    if ( event.data.ctlEnter.controlID == frmResultStop ) {
                        ShowCancelMore();
                        UpdateStatus( false );
                        handled = true;
                    }
                    else
                        handled = false;
                    break;

                case penUpEvent:
                    EvtFlushPenQueue();
                    handled = false;
                    break;

                case keyDownEvent:
                    ShowCancelMore();
                    UpdateStatus( false );
                    handled = true;
                    break;

                case appStopEvent:
                    EvtAddEventToQueue( &event );
                    return false;

                default:
                    handled = false;
            }

            /* Check if the form can handle the event */
            if ( ! handled )
                FrmHandleEvent( resultForm, &event );
            else
                return true;

        } while ( EvtEventAvail() );

        if ( Prefs()->searchMode == SEARCH_DOWN_IN_DOC ) {
            SetSearchFromHere();
            startFromBeginning = false;
        }
        done = SearchDocument( pattern, startFromBeginning, &findMore, 
                   Prefs()->searchMode );

        startFromBeginning = false;

        if ( done ) {

            SonySilkEnableResize(); /* FIXME!  Temporary work-around. */

            if ( findMore )
                ShowCancelMore();
            else
                ShowCancel();
            UpdateStatus( findMore );

            return true;
        }
    }
}



/* Event handler for the result form */
Boolean ResultFormHandleEvent
    (
    EventType* event  /* pointer to an EventType structure */
    )
{
    RectangleType resultRect;

    Boolean handled;

    resultRect.topLeft.x = DEFAULT_X;
    resultRect.topLeft.y = DEFAULT_Y;
    resultRect.extent.x  = DEFAULT_EXTENT_X;
    resultRect.extent.y  = maxResult * DEFAULT_EXTENT_Y;

    handled = false;

    switch ( event->eType ) {
        case ctlSelectEvent:
            if ( event->data.ctlEnter.controlID == frmResultFindMore ) {
                WinEraseRectangle( &resultRect, 0 );

                FldSetTextPtr( fldPtr, searching );
                FldDrawField( fldPtr );

                FrmHideObject( resultForm, FrmGetObjectIndex( resultForm,
                                            frmResultCancel ) );
                FrmHideObject( resultForm, FrmGetObjectIndex( resultForm,
                                            frmResultFindMore ) );
                FrmShowObject( resultForm, FrmGetObjectIndex( resultForm,
                                            frmResultStop ) );

                ClearSearchResult();

                handled = SearchResultHandleEvent( false );
                break;
            }
            else if ( event->data.ctlEnter.controlID != frmResultCancel )
                break;

            FntSetFont( initialFontStyle );
            SonySilkEnableResize();
            FrmReturnToForm( GetMainFormId() );
            FrmUpdateForm( GetMainFormId(), frmRedrawUpdateCode );
            handled = true;
            break;

        case penDownEvent:
            if ( TopLeftY() < event->screenY && event->screenY <
                 resultRect.topLeft.y + resultRect.extent.y
               ) {
                Int16 i;
                
                lineMatch = NOT_FOUND;

                for ( i = 0; i < maxResult; i++ ) {
                    if ( RctPtInRectangle( event->screenX, event->screenY,
                             &SearchResult[ i ].coordinates )
                       ) {
                        WinInvertRectangle( &SearchResult[ i ].coordinates, 0 );
                        lineMatch = i;
                        break;
                    }
                    handled = true;
                }
            }
            break;

        case penUpEvent:
            if ( TopLeftY() < event->screenY && event->screenY <
                 resultRect.topLeft.y + resultRect.extent.y
               ) {
                if ( lineMatch != NOT_FOUND ) {
                    WinInvertRectangle( &SearchResult[ lineMatch ].coordinates,
                        0 );

                    if ( RctPtInRectangle( event->screenX, event->screenY,
                            &SearchResult[ lineMatch ].coordinates ) ) {
                        EventType match;

                        FrmReturnToForm( GetMainFormId() );

                        MemSet( &match, sizeof( EventType ), 0 );

                        match.eType                         = frmGotoEvent;
                        match.data.frmGoto.formID           = GetMainFormId();
                        match.data.frmGoto.recordNum        =
                                    SearchResult[ lineMatch ].record;
                        match.data.frmGoto.matchPos         =
                                    SearchResult[ lineMatch ].pos;
                        match.data.frmGoto.matchLen         =
                                    SearchResult[ lineMatch ].length;
                        match.data.frmGoto.matchFieldNum    = 0;
                        match.data.frmGoto.matchCustom      = 0;

                        EvtAddEventToQueue( &match );
                        SonySilkEnableResize(); /* FIXME!  Temporary work-around. */
                        
                    }
                    SndPlaySystemSound( sndClick );
                }
                handled = true;
            }
            break;

        case frmOpenEvent:
            ResultFormInit();
            handled = SearchResultHandleEvent( true );
            break;

        case frmCloseEvent:
            FntSetFont( initialFontStyle );
            SonySilkEnableResize();
            handled = false;
            break;

        default:
            handled = false;
            break;
    }

    return handled;
}

