/*
    pmacct (Promiscuous mode IP Accounting package)
    pmacct is Copyright (C) 2004 by Paolo Lucente
*/

/*
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
*/

#define __CFG_HANDLERS_C

/* includes */
#include "pmacct.h"
#include "pmacct-data.h"
#include "plugin_hooks.h"
#include "cfg_handlers.h"

#define OFF(x, y) ((unsigned long int)y - (unsigned long int)x)

void lower_string(char *string)
{
  int i = 0;
 
  while (string[i] != '\0') {
    string[i] = tolower(string[i]);
    i++;
  }
}

int set_all_int(unsigned int off, int value)
{
  struct plugins_list_entry *list = plugins_list;
  unsigned int **ptr;
  int changes = 0;

  while (list) {
    ptr = (unsigned int **) &list->cfg;

#ifdef OSF1
    if (off%sizeof(unsigned int **)) (u_char) ptr += off;
    else ptr += off/sizeof(unsigned int **);
#else
    ptr += off/sizeof(unsigned int **);
#endif
    
#ifdef IRIX
    *ptr = (unsigned int *) value;
#else
    (int) *ptr = value;
#endif

    changes++;
    list = list->next;
  }

  return changes;
}

int set_one_int(unsigned int off, char *name, int value)
{
  struct plugins_list_entry *list = plugins_list;
  unsigned int **ptr;
  int changes = 0;

  while (list) {
    if (!strcmp(name, list->name)) {
      ptr = (unsigned int **) &list->cfg;

#ifdef OSF1
      if (off%sizeof(unsigned int **)) (u_char) ptr += off;
      else ptr += off/sizeof(unsigned int **);
#else
      ptr += off/sizeof(unsigned int **);
#endif

#ifdef IRIX
      *ptr = (unsigned int *) value;
#else
      (int) *ptr = value;
#endif

      changes++;
      break;
    }
    list = list->next;
  }

  return changes;
}

int set_all_char_ptr(unsigned int off, char *value)
{
  struct plugins_list_entry *list = plugins_list;
  unsigned int **ptr;
  int changes = 0;

  while (list) {
    ptr = (unsigned int **) &list->cfg;

#ifdef OSF1
    if (off%sizeof(unsigned int **)) (u_char) ptr += off;
    else ptr += off/sizeof(unsigned int **);
#else
    ptr += off/sizeof(unsigned int **);
#endif

#ifdef IRIX
    *ptr = (unsigned int *) value;
#else
    (char *) *ptr = value;
#endif

    changes++;
    list = list->next;
  }

  return changes;
}

int set_one_char_ptr(unsigned int off, char *name, char *value)
{
  struct plugins_list_entry *list = plugins_list;
  unsigned int **ptr;
  int changes = 0;

  while (list) {
    if (!strcmp(name, list->name)) {
      ptr = (unsigned int **) &list->cfg;

#ifdef OSF1
      if (off%sizeof(unsigned int **)) (u_char) ptr += off;
      else ptr += off/sizeof(unsigned int **);
#else
      ptr += off/sizeof(unsigned int **);
#endif

#ifdef IRIX
      *ptr = (unsigned int *) value;
#else
      (char *) *ptr = value;
#endif

      changes++;
      break;
    }
    list = list->next;
  }

  return changes;
}

int parse_truefalse(char *value_ptr)
{
  int value, i = 0;

  lower_string(value_ptr);
  
  if (!strcmp("true", value_ptr)) value = TRUE;
  else if (!strcmp("false", value_ptr)) value = FALSE;
  else value = ERR;

  return value;
}

int cfg_key_debug(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  value = parse_truefalse(value_ptr); 
  if (value < 0) return ERR; 

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.debug), value);
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.debug), name, value);

  return changes;
}

int cfg_key_syslog(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.syslog), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.syslog), name, value_ptr);

  return changes;
}

int cfg_key_pidfile(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.pidfile), value_ptr);
  if (name) Log(LOG_WARNING, "WARN: plugin name not supported for key 'pidfile'. Globalized.\n");

  return changes;
}

int cfg_key_daemonize(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  value = parse_truefalse(value_ptr);
  if (value < 0) return ERR;

  changes = set_all_int(OFF(&list->cfg, &list->cfg.daemon), value);
  if (name) Log(LOG_WARNING, "WARN: plugin name not supported for key 'daemonize'. Globalized.\n"); 

  return changes;
}

int cfg_key_aggregate(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  char *count_token;
  int value = 0, changes = 0; /* XXX: 'value' to be changed in unsigned long int ? */

  trim_all_spaces(value_ptr);

  while (*value_ptr != '\0') {
    count_token = extract_token(&value_ptr, ',');
    if (!strcmp(count_token, "src_host")) value |= COUNT_SRC_HOST;
    else if (!strcmp(count_token, "dst_host")) value |= COUNT_DST_HOST;
    else if (!strcmp(count_token, "src_net")) value |= COUNT_SRC_NET;
    else if (!strcmp(count_token, "dst_net")) value |= COUNT_DST_NET;
    else if (!strcmp(count_token, "sum")) value |= COUNT_SUM_HOST;
    else if (!strcmp(count_token, "src_port")) value |= COUNT_SRC_PORT;
    else if (!strcmp(count_token, "dst_port")) value |= COUNT_DST_PORT;
    else if (!strcmp(count_token, "proto")) value |= COUNT_IP_PROTO;
    else if (!strcmp(count_token, "src_mac")) value |= COUNT_SRC_MAC;
    else if (!strcmp(count_token, "dst_mac")) value |= COUNT_DST_MAC;
    else Log(LOG_WARNING, "WARN: ignoring unknown aggregation method: %s.\n", count_token);
  }

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.what_to_count), value);
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.what_to_count), name, value);

  return changes;
}

int cfg_key_aggregate_filter(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) {
    Log(LOG_ERR, "ERROR: aggregation filter cannot be global. Not loaded.\n");
    changes++;
  }
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.a_filter), name, value_ptr);

  return changes;
}

int cfg_key_pcap_filter(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.clbuf), value_ptr);
  if (name) Log(LOG_WARNING, "WARN: plugin name not supported for key 'pcap_filter'. Globalized.\n");

  return changes;
}

int cfg_key_interface(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.dev), value_ptr);
  if (name) Log(LOG_WARNING, "WARN: plugin name not supported for key 'interface'. Globalized.\n");

  return changes;
}

int cfg_key_promisc(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  value = parse_truefalse(value_ptr);
  if (value < 0) return ERR;

  changes = set_all_int(OFF(&list->cfg, &list->cfg.promisc), value);
  if (name) Log(LOG_WARNING, "WARN: plugin name not supported for key 'promisc'. Globalized.\n");

  return changes;
}

int cfg_key_imt_path(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.imt_plugin_path), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.imt_plugin_path), name, value_ptr);

  return changes;
}

int cfg_key_imt_passwd(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.imt_plugin_passwd), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.imt_plugin_passwd), name, value_ptr);

  return changes;
}

int cfg_key_imt_buckets(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  value = atoi(value_ptr);
  if (value <= 0) {
    Log(LOG_ERR, "ERROR: invalid number of buckets.\n");
    exit(1);
  }

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.buckets), value);
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.buckets), name, value);

  return changes;
}

int cfg_key_imt_mem_pools_number(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;
  
  value = atoi(value_ptr);
  if (value < 0) {
    Log(LOG_ERR, "ERROR: invalid number of memory pools.\n");
    exit(1);
  }

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.num_memory_pools), value);
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.num_memory_pools), name, value);

  have_num_memory_pools = TRUE;
  return changes;
}

int cfg_key_imt_mem_pools_size(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.memory_pool_size), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.memory_pool_size), name, atoi(value_ptr));

  return changes;
}

int cfg_key_sql_db(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_db), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_db), name, value_ptr);

  return changes;
}

int cfg_key_sql_table(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_table), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_table), name, value_ptr);

  return changes;
}

int cfg_key_sql_data(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_data), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_data), name, value_ptr);

  return changes;
}

int cfg_key_sql_host(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_host), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_host), name, value_ptr);

  return changes;
}

int cfg_key_sql_recovery_backup_host(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_backup_host), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_backup_host), name, value_ptr);

  return changes;
}

int cfg_key_sql_user(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_user), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_user), name, value_ptr);

  return changes;
}

int cfg_key_sql_passwd(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_passwd), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_passwd), name, value_ptr);

  return changes;
}

int cfg_key_sql_refresh_time(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.sql_refresh_time), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.sql_refresh_time), name, atoi(value_ptr));

  return changes;
}

int cfg_key_sql_startup_delay(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.sql_startup_delay), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.sql_startup_delay), name, atoi(value_ptr));

  return changes;
}

int cfg_key_sql_optimize_clauses(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  value = parse_truefalse(value_ptr);
  if (value < 0) return ERR;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.sql_optimize_clauses), value);
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.sql_optimize_clauses), name, value);

  return changes;
}

int cfg_key_sql_history_roundoff(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_history_roundoff), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_history_roundoff), name, value_ptr);

  return changes;
}

int cfg_key_sql_recovery_logfile(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.sql_recovery_logfile), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.sql_recovery_logfile), name, value_ptr);

  return changes;
}

int cfg_key_sql_history(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0, sql_history, sql_history_howmany;

  parse_history_time(value_ptr, &sql_history, &sql_history_howmany);

  if (!name) {
    changes = set_all_int(OFF(&list->cfg, &list->cfg.sql_history), sql_history);
    changes += set_all_int(OFF(&list->cfg, &list->cfg.sql_history_howmany), sql_history_howmany);
  }
  else {
    changes = set_one_int(OFF(&list->cfg, &list->cfg.sql_history), name, sql_history);
    changes += set_one_int(OFF(&list->cfg, &list->cfg.sql_history_howmany), name, sql_history_howmany);
  }

  return changes;
}

int cfg_key_sql_cache_entries(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.sql_cache_entries), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.sql_cache_entries), name, atoi(value_ptr));

  return changes;
}

int cfg_key_plugin_pipe_size(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.pipe_size), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.pipe_size), name, atoi(value_ptr));

  return changes;
}

int cfg_key_plugin_buffer_size(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_int(OFF(&list->cfg, &list->cfg.buffer_size), atoi(value_ptr));
  else changes = set_one_int(OFF(&list->cfg, &list->cfg.buffer_size), name, atoi(value_ptr));

  return changes;
}

int cfg_key_networks_file(char *name, char *value_ptr)
{
  struct plugins_list_entry *list = plugins_list;
  int value, changes = 0;

  if (!name) changes = set_all_char_ptr(OFF(&list->cfg, &list->cfg.networks_file), value_ptr);
  else changes = set_one_char_ptr(OFF(&list->cfg, &list->cfg.networks_file), name, value_ptr);

  return changes;
}

void parse_history_time(char *value, int *mu, int *howmany)
{
  int k, j, len;

  len = strlen(value);
  for (j = 0; j < len; j++) {
    if (!isdigit(value[j])) {
      if (value[j] == 'm') *mu = COUNT_MINUTELY;
      else if (value[j] == 'h') *mu = COUNT_HOURLY;
      else if (value[j] == 'd') *mu = COUNT_DAILY;
      else {
        Log(LOG_WARNING, "WARN: Ignoring unknown time measuring unit: '%c'.\n", value[j]);
        *mu = 0;
        *howmany = 0;
        return;
      }
      if (*mu) {
        value[j] = '\0';
        break;
      }
    }
  }
  k = atoi(value);
  if (k > 0) *howmany = k;
  else {
    Log(LOG_WARNING, "WARN: ignoring invalid time value: %d\n", k);
    *mu = 0;
    *howmany = 0;
  }
}

