//
// CipherKey.h
//
// $Id: //poco/1.3/Crypto/include/Poco/Crypto/CipherKey.h#1 $
//
// Library: Crypto
// Package: CryptoCore
// Module:  CipherKey
//
// Definition of the CipherKey class.
//
// Copyright (c) 2007, Applied Informatics Software Engineering GmbH.
// and Contributors.
//
// Permission is hereby granted, free of charge, to any person or organization
// obtaining a copy of the software and accompanying documentation covered by
// this license (the "Software") to use, reproduce, display, distribute,
// execute, and transmit the Software, and to prepare derivative works of the
// Software, and to permit third-parties to whom the Software is furnished to
// do so, all subject to the following:
// 
// The copyright notices in the Software and this entire statement, including
// the above license grant, this restriction and the following disclaimer,
// must be included in all copies of the Software, in whole or in part, and
// all derivative works of the Software, unless such copies or derivative
// works are solely in the form of machine-executable object code generated by
// a source language processor.
// 
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE, TITLE AND NON-INFRINGEMENT. IN NO EVENT
// SHALL THE COPYRIGHT HOLDERS OR ANYONE DISTRIBUTING THE SOFTWARE BE LIABLE
// FOR ANY DAMAGES OR OTHER LIABILITY, WHETHER IN CONTRACT, TORT OR OTHERWISE,
// ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.
//


#ifndef Crypto_CipherKey_INCLUDED
#define Crypto_CipherKey_INCLUDED


#include "Poco/Crypto/Crypto.h"
#include "Poco/Crypto/CipherKeyImpl.h"


namespace Poco {
namespace Crypto {


class Crypto_API CipherKey
	/// class CipherKey stores the key information for decryption/encryption of data.
	/// To create a random key, using the following code:
	///
	///     CipherKey key("aes-256");
	///
	/// Note that you won't be able to decrypt data encrypted with a random key
	/// once the Cipher is destroyed unless you persist the generated key and IV.
	/// An example usage for random keys is to encrypt data saved in a temporary
	/// file.
	///
	/// To create a key using a human-readable password
	/// string, use the following code. We create a AES Cipher and 
	/// use a salt value to make the key more robust:
	///
	///     std::string password = "secret";
	///		CipherKey key("aes-256", password, salt);
	///
{
public:
	typedef CipherKeyImpl::Mode Mode;
	typedef CipherKeyImpl::ByteVec ByteVec;

	enum
	{
		DEFAULT_ITERATION_COUNT = 2000	/// Default iteration count to use with
										/// generateKey().  RSA security recommends
										/// an iteration count of at least 1000.
	};

	CipherKey(const std::string& name, 
		const std::string& passphrase, 
		const std::string& salt = "",
		int iterationCount = DEFAULT_ITERATION_COUNT);
		/// Creates a new CipherKeyImpl object

	CipherKey(const std::string& name, 
		const ByteVec& key, 
		const ByteVec& iv);
		/// Creates a new CipherKeyImpl object

	CipherKey(const std::string& name);
		/// Creates a new CipherKeyImpl object. Autoinitializes  key and iv

	~CipherKey();
		/// Destroys the CipherKeyImpl.

	const std::string& name() const;
		/// Returns the name of the Cipher.

	int keySize() const;
		/// Returns the key size of the Cipher.

	int blockSize() const;
		/// Returns the block size of the Cipher.

	int ivSize() const;
		/// Returns the IV size of the Cipher.

	Mode mode() const;
		/// Returns the Cipher's mode of operation.
	
	const ByteVec& getKey() const;
		/// Returns the key for the Cipher.

	void setKey(const ByteVec& key);
		/// Sets the key for the Cipher.

	const ByteVec& getIV() const;
		/// Returns the initialization vector (IV) for the Cipher.

	void setIV(const ByteVec& iv);
		/// Sets the initialization vector (IV) for the Cipher.

	CipherKeyImpl::Ptr impl();
		/// Returns the impl object

private:
	CipherKeyImpl::Ptr _pImpl;
};


inline const std::string& CipherKey::name() const
{
	return _pImpl->name();
}


inline int CipherKey::keySize() const
{
	return _pImpl->keySize();
}


inline int CipherKey::blockSize() const
{
	return _pImpl->blockSize();
}


inline int CipherKey::ivSize() const
{
	return _pImpl->ivSize();
}


inline CipherKey::Mode CipherKey::mode() const
{
	return _pImpl->mode();
}


inline const CipherKey::ByteVec& CipherKey::getKey() const
{
	return _pImpl->getKey();
}


inline void CipherKey::setKey(const CipherKey::ByteVec& key)
{
	_pImpl->setKey(key);
}


inline const CipherKey::ByteVec& CipherKey::getIV() const
{
	return _pImpl->getIV();
}


inline void CipherKey::setIV(const CipherKey::ByteVec& iv)
{
	_pImpl->setIV(iv);
}


inline CipherKeyImpl::Ptr CipherKey::impl()
{
	return _pImpl;
}


} } // namespace Poco::Crypto


#endif // Crypto_CipherKey_INCLUDED
