// ------------------------------------------------------------------------- //
// $Id: pointless_wrap.cpp,v 1.78 2004/01/06 21:18:36 weismann Exp $
// ------------------------------------------------------------------------- //
 
/* 
 * Copyright (c) 2002 
 *				see AUTHORS list
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */

// ------------------------------------------------------------------------- //

#if HAVE_CONFIG_H
# include <config.h>
#endif

#if HAVE_PYTHON2_2
# if defined WIN32 && (!defined __GNUC__)
#  ifdef _DEBUG
#   undef _DEBUG
#   include <Python.h>
#   define _DEBUG
#  else
#   include <Python.h>
#  endif
# else
#  include <python2.2/Python.h>
# endif
#elif HAVE_PYTHON2_3
#  include <python2.3/Python.h>
#endif

#ifndef HAVE_SDL
# include "x11win.h"
#else
# include "sdlwin.h"
#endif

#include "pointless_wrap.h"
#include "nodefactory.h"
#include "nodecontroller.h"
#include "common.h"

// ------------------------------------------------------------------------- //

class Clock g_clock;
class MainWindow* g_renderer = 0;

#ifdef __cplusplus
extern "C" {
#endif

/* ------------------------------------------------------------------------- 
 * Misc 
 * ------------------------------------------------------------------------- */

static PyObject *wrap_clock_time( PyObject *self, PyObject *args )
{
	return Py_BuildValue((char*)"d", g_clock.time());
}

/* ------------------------------------------------------------------------- 
 * Logging 
 * ------------------------------------------------------------------------- */

static PyObject *set_loglevel(PyObject *self, PyObject *args)
{
	int value = 0;
	if (!PyArg_ParseTuple(args, (char*)"i:set_loglevel", &value))
		return NULL;

	log_level = value;

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_log_abort( PyObject *self, PyObject *args )
{
	char *str = (char*)"";

	if (!PyArg_ParseTuple(args, (char*)"s:log_abort", &str))
		return NULL;

	log_abort(str);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_log_warning( PyObject *self, PyObject *args )
{
	char *str = (char*)"";

	if (!PyArg_ParseTuple(args, (char*)"s:log_warning", &str))
		return NULL;

	log_warning(str);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_log_info( PyObject *self, PyObject *args )
{
	char *str = (char*)"";

	if (!PyArg_ParseTuple(args, (char*)"s:log_info", &str))
		return NULL;

	log_info(str);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_log_debug( PyObject *self, PyObject *args )
{
	char *str = (char*)"";

	if (!PyArg_ParseTuple(args, (char*)"s:log_debug", &str))
		return NULL;

	log_debug(str);

	Py_INCREF(Py_None);
	return Py_None;
}

/* ------------------------------------------------------------------------- 
 * Renderer methods 
 * ------------------------------------------------------------------------- */

static PyObject *wrap_renderer_init( PyObject *self, PyObject* args )
{
	bool fullscreen;
	uint w, h;
	PyObject *placement;

	if (!PyArg_ParseTuple(args, (char*)"biiO:renderer_init", &fullscreen, 
				&w, &h, &placement))
		return NULL;

	log_debug("fullscreen is %d",fullscreen);
	WindowGeometry geometry(w, h);
	if (placement != Py_None) {
		int x, x_sign, y, y_sign;
		if (!PyArg_ParseTuple(placement, (char*)"iiii:renderer_init",
			&x, &x_sign, &y, &y_sign))
				return NULL;
		geometry.set_placement(x, x_sign, y, y_sign);
	}

#ifdef HAVE_SDL
	g_renderer = new SDLWindow();
#else
	g_renderer = new XWindow();
#endif
	g_renderer->create_window(geometry, fullscreen);
	g_renderer->init_gfx();
	g_renderer->reset_time();

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_get_resolution(PyObject *self, PyObject* args)
{
	std::string str = g_renderer->get_resolution();
	return Py_BuildValue((char*)"s", str.c_str());
}

static PyObject *wrap_renderer_get_fullscreen( PyObject *self, PyObject *args )
{
	return Py_BuildValue((char*)"i", g_renderer->get_fullscreen());
}

static PyObject *wrap_renderer_get_resolutions(PyObject *self, PyObject* args)
{
	std::string str = g_renderer->get_resolutions();
	return Py_BuildValue((char*)"s", str.c_str());
}

static PyObject *wrap_renderer_set_resolution(PyObject *self, PyObject* args)
{
	char *res = (char*)"";
	if (!PyArg_ParseTuple(args, (char*)"s:renderer_set_resolution", &res))
		return NULL;
	g_renderer->set_resolution(res);
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_toggle_fullscreen(PyObject *self, PyObject* args)
{
	g_renderer->toggle_fullscreen();
	return Py_BuildValue((char*)"");
}

static PyObject *wrap_renderer_shutdown(PyObject *self, PyObject* args)
{
	if (g_renderer) {
		delete g_renderer;
	}
	return Py_BuildValue((char*)"");
}

static PyObject *wrap_renderer_newroot(PyObject *self, PyObject* args)
{
	if (g_renderer) {
		g_renderer->new_root();
	}
	return Py_BuildValue((char*)"");
}

static PyObject *wrap_renderer_frame( PyObject *self, PyObject* args )
{
	g_renderer->frame();
	return Py_BuildValue((char*)"i", Node::_controllers_run);
}

static PyObject *wrap_renderer_get_root( PyObject *self, PyObject* args )
{
	return PyCObject_FromVoidPtrAndDesc((void*)g_renderer->get_root(), 
			(char*)"Group", NULL);
}

static PyObject *wrap_renderer_getevent( PyObject *self, PyObject *args )
{
	int block = 0;
	if (!PyArg_ParseTuple(args, (char*)"i:renderer_getevent", 
				&block))
		return NULL;

	const Event* ev;
	if ( (ev = g_renderer->getevent(block)) ) {
		switch (ev->get_type()) {
			case Event::Key:
				//std::cout << ev->key().mod << std::endl;
				return Py_BuildValue((char*)"i(iii)", SDL_KEYDOWN,
						ev->key().sym, ev->key().mod, ev->key().type);
			case Event::Resize:
				return Py_BuildValue((char*)"i(ii)", SDL_VIDEORESIZE,
						ev->resize().new_width, ev->resize().new_height);
			case Event::MouseButton:
				return Py_BuildValue((char*)"i(i)", SDL_MOUSEBUTTONDOWN,
						ev->mousebutton().button);
			default:
				log_warning("Unknown event type");
				break;
		}
	}
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_get_wireframe( PyObject *self, PyObject *args )
{
	return Py_BuildValue((char*)"i", g_renderer->get_wireframe());
}

static PyObject *wrap_renderer_set_wireframe( PyObject *self, PyObject *args )
{
	int value = 0;
	if (!PyArg_ParseTuple(args, (char*)"i:renderer_set_wireframe", 
				&value))
		return NULL;

	g_renderer->set_wireframe(value);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_set_debug_osd( PyObject *self, PyObject *args )
{
	int value = 0;
	if (!PyArg_ParseTuple(args, (char*)"i:renderer_set_debug_osd", 
				&value))
		return NULL;
	g_renderer->set_debug_osd(value);
	Py_INCREF(Py_None);
	return Py_None;
}

/*
static PyObject *wrap_renderer_viewer_set_step(PyObject *self, PyObject *args)
{
	int value = 0;
	if (!PyArg_ParseTuple(args, (char*)"i:renderer_viewer_set_step", &value))
		return NULL;

	g_renderer->_viewer.set_step(value);

	Py_INCREF(Py_None);
	return Py_None;
}
*/

static PyObject *wrap_renderer_print_at( PyObject *self, PyObject *args )
{
	uint x, y;
	char *str;

	if (!PyArg_ParseTuple(args, (char*)"iis:renderer_print_at", 
				&x, &y, &str))
		return NULL;

	g_renderer->print_at(x, y, str);

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_load_textures( PyObject *self, PyObject *args )
{
	g_renderer->load_textures();

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_renderer_get_height( PyObject *self, PyObject *args )
{
	return Py_BuildValue((char*)"i", g_renderer->get_height());
}

static PyObject *wrap_renderer_get_width( PyObject *self, PyObject *args )
{
	return Py_BuildValue((char*)"i", g_renderer->get_width());
}

static PyObject *wrap_renderer_init_gl(PyObject *self, PyObject *args)
{
	g_renderer->init_gl();
	return Py_BuildValue((char*)"");
}

static PyObject *wrap_renderer_make_screenshot(PyObject *self, PyObject *args)
{
	char *filename;
	float scale;
	if (!PyArg_ParseTuple(args, (char*)"sf:renderer_make_screenshot",
		&filename, &scale)) {
		return NULL;
	}

	int ok = g_renderer->make_screenshot(filename, scale);
	return Py_BuildValue((char*)"i", ok);
}

/* ------------------------------------------------------------------------- 
 * NodeFactory methods 
 * ------------------------------------------------------------------------- */

static PyObject *wrap_nodefactory_absrectangle(PyObject *self, PyObject *args)
{
	int ll_x, ll_y, ur_x, ur_y;
	float r, g, b, a;

	if (!PyArg_ParseTuple(args, (char*)"iiiiffff:nodefactory_absrectangle", 
				&ll_x, &ll_y, &ur_x, &ur_y, &r, &g, &b, &a))
		return NULL;

	Leaf* l = NodeFactory::instance()->rectangle(v2(ll_x, ll_y), 
			v2(ur_x, ur_y), rgba(r,g,b,a), rgba(r,g,b,a), rgba(r,g,b,a),
			rgba(r,g,b,a), 0, 0);

	if (!l) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) l, (char*)"AbsRectangle", NULL);
}

static PyObject *wrap_nodefactory_image( PyObject *self, PyObject* args )
{
	char *filename;
	int   center;
	int   gray_alpha;
	float r, g, b, a;
	float dropshadow;

	if (!PyArg_ParseTuple(args, (char*)"sii(ffff)f:nodefactory_image", 
				&filename, &center, &gray_alpha, &r, &g, &b, &a, &dropshadow))
		return NULL;

	Leaf* l = NodeFactory::instance()->image(filename, center, gray_alpha,
			rgba(r, g, b, a), dropshadow);

	if (!l) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) l, (char*)"Node", NULL);
}

static PyObject *wrap_nodefactory_backgroundimage( PyObject *self, PyObject* args )
{
	int width, height;
	char *filename;
	bool stretch;

	if (!PyArg_ParseTuple(args, (char*)"iibs:nodefactory_backgroundimage", 
				&width, &height, &stretch, &filename))
		return NULL;

	Leaf* l = NodeFactory::instance()->rectangle( 
			v2(0.0f, 0.0f), v2(width, height), 0, 0, 0, 0, stretch,  filename);

	if (!l) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) l, (char*)"Node", NULL);
}

static PyObject *wrap_nodefactory_rectangle( PyObject *self, PyObject* args )
{
	float width, height;
	float ll_r, ll_g, ll_b, ll_a;
	float lr_r, lr_g, lr_b, lr_a;
	float ul_r, ul_g, ul_b, ul_a;
	float ur_r, ur_g, ur_b, ur_a;

	if (!PyArg_ParseTuple(args, (char*)"ff(ffff)(ffff)(ffff)(ffff):nodefactory_rectangle", 
				&width, &height, &ll_r, &ll_g, &ll_b, &ll_a,
				&lr_r, &lr_g, &lr_b, &lr_a,
				&ul_r, &ul_g, &ul_b, &ul_a,
				&ur_r, &ur_g, &ur_b, &ur_a
				))
		return NULL;

	Leaf* l = NodeFactory::instance()->rectangle(v2(0.0f, 0.0f), 
			v2(width, height), rgba(ll_r, ll_g, ll_b, ll_a),
			rgba(lr_r, lr_g, lr_b, lr_a),
			rgba(ul_r, ul_g, ul_b, ul_a),
			rgba(ur_r, ur_g, ur_b, ur_a),  0, 0);

	if (!l) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) l, (char*)"Node", NULL);
}

static FreeTypeFont::handle fonthandle_from_cobject(PyObject *obj)
{
	if (!obj) return NULL;
	if (!PyCObject_Check(obj)) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Expected a pointer");
		return NULL;
	}
	return *(FreeTypeFont::handle*) PyCObject_AsVoidPtr(obj);
}

static void destr_font(void *obj, void *data)
{
	//std::cout << _("deleting\n");
	delete (FreeTypeFont::handle*)obj;
}

static PyObject *wrap_get_font(PyObject *self, PyObject *args)
{
	char *fontname;
	int size;
	if (!PyArg_ParseTuple(args, (char*)"si:get_font",
			&fontname, &size)) {
		return NULL;
	}

	FreeTypeFont::handle *font = new FreeTypeFont::handle(
			FreeTypeFont::get_font(fontname, size));
	if (!*font) {
		PyErr_SetString(PyExc_TypeError, "Failed to load font");
		return Py_BuildValue((char*)"");
	}

	return PyCObject_FromVoidPtrAndDesc((void*) font, (char*)"Font", destr_font);
}

static PyObject *wrap_get_font_metric( PyObject *self, PyObject* args )
{
	PyObject* obj;

	if (!PyArg_ParseTuple(args, (char*)"O:font_get_metric", &obj))
		return NULL;

	FreeTypeFont::handle font = fonthandle_from_cobject(obj);

	return Py_BuildValue((char*)"{s:f,s:f,s:f}",
			"baselineskip", font->get_baselineskip(),
			"underline_position", font->get_underline_position(),
			"underline_thickness", font->get_underline_thickness()
	);
}

static PyObject *wrap_nodefactory_letter( PyObject *self, PyObject* args )
{
	PyObject* obj;
	int   letter;
	float r, g, b, a;
	float   dropshadow;

	if (!PyArg_ParseTuple(args, (char*)"Oi(ffff)f:nodefactory_letter", 
				&obj, &letter, &r, &g, &b, &a, &dropshadow))
		return NULL;

	FreeTypeFont::handle font = fonthandle_from_cobject(obj);

	Letter* l = NodeFactory::instance()->letter(font, letter, 
			rgba(r, g, b, a), dropshadow);

	if (!l) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) l, (char*)"Letter", NULL);
}

static PyObject *wrap_nodefactory_group( PyObject *self, PyObject* args )
{
	char *name;

	if (!PyArg_ParseTuple(args, (char*)"s:nodefactory_group", &name))
		return NULL;

	Group* g = NodeFactory::instance()->group(name);

	if (!g) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) g, (char*)"Group", NULL);
}

/* ------------------------------------------------------------------------- 
 * Node methods 
 * ------------------------------------------------------------------------- */

static Node* node_from_cobject(PyObject *obj)
{
	if (!obj) return NULL;
	if (!PyCObject_Check(obj)) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Expected a pointer");
		return NULL;
	}
/* FIXME: This must me more sophisticated to handle inheritance
	if (strcmp((char*)PyCObject_GetDesc(obj), "Node")) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Type error: expected a Node pointer");
		return NULL;
	}
*/
	return (Node*) PyCObject_AsVoidPtr(obj);
}


static PyObject *wrap_node_name( PyObject *self, PyObject* args )
{
	PyObject *obj;

	if (!PyArg_ParseTuple(args, (char*)"O:node_name", &obj))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	return PyString_FromString(n->name().c_str());
}

static PyObject *wrap_node_full_name( PyObject *self, PyObject *args )
{
	PyObject *obj;
	if (!PyArg_ParseTuple(args, (char*)"O:node_full_name", &obj))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	return Py_BuildValue((char*)"s", n->full_name().c_str());
}

static PyObject *wrap_node_get_pos( PyObject *self, PyObject *args )
{
	PyObject *obj;
	if (!PyArg_ParseTuple(args, (char*)"O:node_get_pos", &obj))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	v3 pos = n->get_pos();
	return Py_BuildValue((char*)"fff", pos.x(), pos.y(), pos.z());
}

static PyObject *wrap_node_set_pos( PyObject *self, PyObject *args )
{
	PyObject *obj;
	float x, y, z;
	if (!PyArg_ParseTuple(args, (char*)"Offf:node_set_pos", &obj, 
				&x, &y, &z))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	n->set_pos(v3(x, y, z));

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_node_get_size( PyObject *self, PyObject *args )
{
	PyObject *obj;
	if (!PyArg_ParseTuple(args, (char*)"O:node_get_size", &obj))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	size3 size = n->get_size();
	return Py_BuildValue((char*)"ffffff", size.x.min, size.x.max, size.y.min, size.y.max, size.z.min, size.z.max);
}

static PyObject *wrap_node_set_visible( PyObject *self, PyObject *args )
{
	PyObject* obj;
	int i;
	if (!PyArg_ParseTuple(args, (char*)"Oi:node_set_visible", &obj, &i))
		return NULL;

	Node *n = node_from_cobject(obj);

	if (!n) return NULL;

	n->set_visible(i);
	
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_node_set_alpha(PyObject *self, PyObject *args)
{
	PyObject* obj;
	float a;
	if (!PyArg_ParseTuple(args, (char*)"Of:node_set_alpha", &obj, &a))
		return NULL;


	Node *n = node_from_cobject(obj);
	Leaf *l = (Leaf*)n;

	if (!n || !l) return NULL;

	l->set_alpha(a);
	
	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_node_replace( PyObject *self, PyObject *args )
{
	PyObject* old_obj;
	PyObject* new_obj;
	if (!PyArg_ParseTuple(args, (char*)"OO:node_replace", &old_obj, &new_obj))
		return NULL;

	Node *node_old = node_from_cobject(old_obj);
	Node *node_new = node_from_cobject(new_obj);

	if (!node_old || !node_new) return NULL;

	if (node_old->parent()) {
		Group* parent = node_old->parent();
		parent->remove(node_old);
		parent->add(node_new);
	}

	delete node_old;

	Py_INCREF(Py_None);
	return Py_None;
}

/* ------------------------------------------------------------------------- 
 * Group methods 
 * ------------------------------------------------------------------------- */

static Group* group_from_cobject(PyObject *obj)
{
	if (!obj) return NULL;
	if (!PyCObject_Check(obj)) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Expected a pointer");
		return NULL;
	}
	return (Group*) PyCObject_AsVoidPtr(obj);
}

static PyObject *wrap_group_add( PyObject *self, PyObject* args )
{
	PyObject *obj_self;
	PyObject *obj_child;

	if (!PyArg_ParseTuple(args, (char*)"OO:group_add", &obj_self, &obj_child))
		return NULL;
	
	Group* g = group_from_cobject(obj_self);
	Node*  n = node_from_cobject(obj_child);

	g->add(n); 
	Py_INCREF(Py_None);
	return Py_None;
}


/* ------------------------------------------------------------------------- 
 * EffectController methods 
 * ------------------------------------------------------------------------- */
	
/*
static EffectController* effect_controller_from_cobject(PyObject *obj)
{
	if (!obj) return NULL;
	if (!PyCObject_Check(obj)) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Expected a pointer");
		return NULL;
	}
	return (EffectController*) PyCObject_AsVoidPtr(obj);
}

static PyObject *wrap_effect_controller_start( PyObject *self, PyObject* args )
{
	PyObject *obj_self;

	if (!PyArg_ParseTuple(args, (char*)"O:effect_controller_start", 
				&obj_self))
		return NULL;

	EffectController* ec = effect_controller_from_cobject(obj_self);

	ec->start();

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *wrap_effect_controller_start_reverse( PyObject *self, 
		PyObject* args )
{
	PyObject *obj_self;

	if (!PyArg_ParseTuple(args, (char*)
				"O:effect_controller_start_reverse", &obj_self))
		return NULL;

	EffectController* ec = effect_controller_from_cobject(obj_self);

	ec->start_reverse();

	Py_INCREF(Py_None);
	return Py_None;
}

static PyObject *effect_controller_new( PyObject *self, PyObject* args)
{
	PyObject *obj_node;
	PyObject *controller_parameters;
	const char *controller_name = "None";

	if (!PyArg_ParseTuple(args, (char*)"OsO:effect_controller_new", 
				&obj_node, &controller_name, &controller_parameters))
		return NULL;

	Node* n = node_from_cobject(obj_node);

	if (!n) return NULL;

	EffectController* e = PythonEffectController::EffectControllerFactory(n, 
			controller_name, Parameters(controller_parameters));

	if (!e) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	return PyCObject_FromVoidPtrAndDesc((void*) e, 
			(char*)"EffectController", NULL);
}
*/
static PythonNodeController* python_controller_from_cobject(PyObject *obj)
{
	if (!obj) return NULL;
	if (!PyCObject_Check(obj)) {
		Py_DECREF(obj);
		PyErr_SetString(PyExc_TypeError, "Expected a pointer");
		return NULL;
	}
	return (PythonNodeController*) PyCObject_AsVoidPtr(obj);
}

static PyObject* python_controller_new(PyObject *self, PyObject* args)
{
	PyObject *obj_node;
	PyObject *callable;

	if (!PyArg_ParseTuple(args, (char*)"OO:python_controller_new", 
				&obj_node, &callable))
		return NULL;

	Node* n = node_from_cobject(obj_node);

	if (!n) return NULL;

	PythonNodeController* pnc = new PythonNodeController(n, callable);

	if (!pnc) {
		Py_INCREF(Py_None);
		return Py_None;
	}

	n->add_controller(pnc);

	return PyCObject_FromVoidPtrAndDesc((void*) pnc, 
			(char*)"PythonNodeController", NULL);
}

static PyObject* python_controller_set_active(PyObject *self, PyObject* args)
{
	PyObject *controller;
	PyObject *active;

	if (!PyArg_ParseTuple(args, (char*)"Oi:python_controller_set_active",
				&controller, &active))
		return NULL;

	PythonNodeController *c = python_controller_from_cobject(controller);

	if (!c) return NULL;

	c->set_active(active);

	Py_INCREF(Py_None);
	return Py_None;
}

/* ------------------------------------------------------------------------- 
 * Python method definition
 * ------------------------------------------------------------------------- */

static PyMethodDef libpointlessmethods[] = {

	/* Misc methods */
	{(char *)"clock_time",   wrap_clock_time, METH_VARARGS, (char *) ""},
	
	/* Logging methods */
	{(char *)"set_loglevel", set_loglevel, METH_VARARGS, (char *) ""},
	{(char *)"log_abort",    wrap_log_abort, METH_VARARGS, (char *) ""},
	{(char *)"log_warning",  wrap_log_warning, METH_VARARGS, (char *) ""},
	{(char *)"log_info",     wrap_log_info, METH_VARARGS, (char *) ""},
	{(char *)"log_debug",    wrap_log_debug, METH_VARARGS, (char *) ""},

	/* Font related methods */
	{(char *)"get_font",        wrap_get_font, METH_VARARGS, (char *) ""},
	{(char *)"get_font_metric", wrap_get_font_metric, METH_VARARGS, (char *) ""},

	/* Renderer methods */
	{(char *)"renderer_init",            wrap_renderer_init, METH_VARARGS, (char *) ""},
	{(char *)"renderer_shutdown",        wrap_renderer_shutdown, METH_VARARGS, (char *) ""},
	{(char *)"renderer_newroot",         wrap_renderer_newroot, METH_VARARGS, (char *) ""},
	{(char *)"renderer_frame",           wrap_renderer_frame, METH_VARARGS, 
		(char *) ""},
	{(char *)"renderer_get_root",        wrap_renderer_get_root, METH_VARARGS,
		(char *) ""},
	{(char *)"renderer_getevent",        wrap_renderer_getevent, METH_VARARGS, 
		(char *) ""},
	{(char *)"renderer_get_wireframe",   wrap_renderer_get_wireframe, 
		METH_VARARGS, (char *) ""},
	{(char *)"renderer_set_wireframe",   wrap_renderer_set_wireframe, 
		METH_VARARGS, (char *) ""},
	{(char *)"renderer_set_debug_osd",   wrap_renderer_set_debug_osd, 
			METH_VARARGS, (char *) ""},
	{(char *)"renderer_print_at",        wrap_renderer_print_at, METH_VARARGS, 
			(char *) ""},
	{(char *)"renderer_load_textures",   wrap_renderer_load_textures, 
		METH_VARARGS, (char *) ""},
	{(char *)"renderer_get_height",      wrap_renderer_get_height, METH_VARARGS, 
		(char *) ""},
	{(char *)"renderer_get_width",       wrap_renderer_get_width, METH_VARARGS, 
		(char *) ""},
	{(char *)"renderer_get_fullscreen",   wrap_renderer_get_fullscreen, 
		METH_VARARGS, (char *) ""},
    {(char *)"renderer_set_resolution",  wrap_renderer_set_resolution, 
		METH_VARARGS, (char *) ""}, 
    {(char *)"renderer_get_resolution",  wrap_renderer_get_resolution, 
		METH_VARARGS, (char *) ""}, 
	{(char *)"renderer_get_resolutions",  wrap_renderer_get_resolutions, 
		METH_VARARGS, (char *) ""},
	{(char *)"renderer_toggle_fullscreen", wrap_renderer_toggle_fullscreen, 
		METH_VARARGS, (char *) ""},
	{(char *)"renderer_init_gl",         wrap_renderer_init_gl, METH_VARARGS, 
		(char *) ""},
	{(char *)"renderer_make_screenshot", wrap_renderer_make_screenshot, METH_VARARGS, 
		(char *) ""},
/*	{(char *)"renderer_viewer_set_step", wrap_renderer_viewer_set_step, 
			METH_VARARGS, (char *) ""},	*/

	/* Node methods */
	{(char *)"node_name",      wrap_node_name, METH_VARARGS, (char *)""},
	{(char *)"node_full_name", wrap_node_full_name, METH_VARARGS, (char *)""},
	{(char *)"node_get_pos",   wrap_node_get_pos, METH_VARARGS, (char *)""},
	{(char *)"node_set_pos",   wrap_node_set_pos, METH_VARARGS, (char *)""},
	{(char *)"node_get_size",  wrap_node_get_size, METH_VARARGS, (char *)""},
	{(char *)"node_set_alpha", wrap_node_set_alpha, METH_VARARGS, (char *)""},
	{(char *)"node_set_visible", wrap_node_set_visible, METH_VARARGS, 
		(char *) ""},
	{(char *)"node_replace", wrap_node_replace, METH_VARARGS,
		(char *) ""},
	
	/* Leaf methods */
	
	/* Group methods */
	{(char *)"group_add",     wrap_group_add,     METH_VARARGS, (char *)""},
/*  {"Group_remove",       group_remove,       METH_VARARGS, (char *) ""},
 	{"Group_child",        group_child,        METH_VARARGS, (char *) ""}, */
	
	/* NodeFactory methods */
	{(char *)"nodefactory_image",           wrap_nodefactory_image, 
		METH_VARARGS, (char *) ""},
	{(char *)"nodefactory_backgroundimage",     wrap_nodefactory_backgroundimage, 
		METH_VARARGS, (char *) ""},
	{(char *)"nodefactory_letter",          wrap_nodefactory_letter,
		METH_VARARGS, (char *) ""},
	{(char *)"nodefactory_group",           wrap_nodefactory_group, 
		METH_VARARGS, (char *) ""},
	{(char *)"nodefactory_rectangle",       wrap_nodefactory_rectangle,  
		METH_VARARGS, (char *) ""},
	{(char *)"nodefactory_absrectangle",    wrap_nodefactory_absrectangle,  
		METH_VARARGS, (char *) ""},

	/* EffectController */
/*	{(char *)"effect_controller_start",	wrap_effect_controller_start, 
		METH_VARARGS, (char *) ""},
	{(char *)"effect_controller_start_reverse", 
		wrap_effect_controller_start_reverse, METH_VARARGS, (char *) ""},
	{(char *)"EffectController_new",	effect_controller_new, METH_VARARGS, 
		(char *) ""},*/
	{(char *)"PythonController_new",	python_controller_new, METH_VARARGS, 
		(char *) ""},
	{(char *)"PythonController_set_active",	python_controller_set_active, METH_VARARGS, 
		(char *) ""},
	{NULL, NULL, 0, NULL}
};

/* ------------------------------------------------------------------------- 
 * Constants
 * ------------------------------------------------------------------------- */

/* NOTE: These constants was shamelessly stolen from the pygame source
  (http://pygame.seul.org). Saved us a lot of typing! Thank you, guys! */


#define DEC_CONST(x)  PyModule_AddIntConstant(module, (char *)#x, SDL_##x)
#define DEC_CONSTK(x) PyModule_AddIntConstant(module, (char *)#x, SDL##x)
#define DEC_CONSTN(x) PyModule_AddIntConstant(module, (char *)#x, x)

static void declare_constants(PyObject* module)
{
	DEC_CONST(KEYDOWN);
	DEC_CONST(VIDEORESIZE);
	DEC_CONST(MOUSEBUTTONDOWN);

	DEC_CONST(BUTTON_LEFT);
	DEC_CONST(BUTTON_MIDDLE);
	DEC_CONST(BUTTON_RIGHT);
	DEC_CONST(BUTTON_WHEEL_UP);
	DEC_CONST(BUTTON_WHEEL_DOWN);

/*	DEC_CONST(NOEVENT);
	DEC_CONST(KEYDOWN);
	DEC_CONST(KEYUP);
	DEC_CONST(MOUSEMOTION);
	DEC_CONST(MOUSEBUTTONDOWN);
	DEC_CONST(MOUSEBUTTONUP);
	DEC_CONST(VIDEORESIZE);
	DEC_CONST(QUIT);
	DEC_CONST(SYSWMEVENT);
	DEC_CONST(USEREVENT);
	DEC_CONST(NUMEVENTS);
*/
	DEC_CONSTK(K_UNKNOWN);
	DEC_CONSTK(K_FIRST);
	DEC_CONSTK(K_BACKSPACE);
	DEC_CONSTK(K_TAB);
	DEC_CONSTK(K_CLEAR);
	DEC_CONSTK(K_RETURN);
	DEC_CONSTK(K_PAUSE);
	DEC_CONSTK(K_ESCAPE);
	DEC_CONSTK(K_SPACE);
	DEC_CONSTK(K_EXCLAIM);
	DEC_CONSTK(K_QUOTEDBL);
	DEC_CONSTK(K_HASH);
	DEC_CONSTK(K_DOLLAR);
	DEC_CONSTK(K_AMPERSAND);
	DEC_CONSTK(K_QUOTE);
	DEC_CONSTK(K_LEFTPAREN);
	DEC_CONSTK(K_RIGHTPAREN);
	DEC_CONSTK(K_ASTERISK);
	DEC_CONSTK(K_PLUS);
	DEC_CONSTK(K_COMMA);
	DEC_CONSTK(K_MINUS);
	DEC_CONSTK(K_PERIOD);
	DEC_CONSTK(K_SLASH);
	DEC_CONSTK(K_0);
	DEC_CONSTK(K_1);
	DEC_CONSTK(K_2);
	DEC_CONSTK(K_3);
	DEC_CONSTK(K_4);
	DEC_CONSTK(K_5);
	DEC_CONSTK(K_6);
	DEC_CONSTK(K_7);
	DEC_CONSTK(K_8);
	DEC_CONSTK(K_9);
	DEC_CONSTK(K_COLON);
	DEC_CONSTK(K_SEMICOLON);
	DEC_CONSTK(K_LESS);
	DEC_CONSTK(K_EQUALS);
	DEC_CONSTK(K_GREATER);
	DEC_CONSTK(K_QUESTION);
	DEC_CONSTK(K_AT);
	DEC_CONSTK(K_LEFTBRACKET);
	DEC_CONSTK(K_BACKSLASH);
	DEC_CONSTK(K_RIGHTBRACKET);
	DEC_CONSTK(K_CARET);
	DEC_CONSTK(K_UNDERSCORE);
	DEC_CONSTK(K_BACKQUOTE);
	DEC_CONSTK(K_a);
	DEC_CONSTK(K_b);
	DEC_CONSTK(K_c);
	DEC_CONSTK(K_d);
	DEC_CONSTK(K_e);
	DEC_CONSTK(K_f);
	DEC_CONSTK(K_g);
	DEC_CONSTK(K_h);
	DEC_CONSTK(K_i);
	DEC_CONSTK(K_j);
	DEC_CONSTK(K_k);
	DEC_CONSTK(K_l);
	DEC_CONSTK(K_m);
	DEC_CONSTK(K_n);
	DEC_CONSTK(K_o);
	DEC_CONSTK(K_p);
	DEC_CONSTK(K_q);
	DEC_CONSTK(K_r);
	DEC_CONSTK(K_s);
	DEC_CONSTK(K_t);
	DEC_CONSTK(K_u);
	DEC_CONSTK(K_v);
	DEC_CONSTK(K_w);
	DEC_CONSTK(K_x);
	DEC_CONSTK(K_y);
	DEC_CONSTK(K_z);
	DEC_CONSTK(K_DELETE);

	DEC_CONSTK(K_KP0);
	DEC_CONSTK(K_KP1);
	DEC_CONSTK(K_KP2);
	DEC_CONSTK(K_KP3);
	DEC_CONSTK(K_KP4);
	DEC_CONSTK(K_KP5);
	DEC_CONSTK(K_KP6);
	DEC_CONSTK(K_KP7);
	DEC_CONSTK(K_KP8);
	DEC_CONSTK(K_KP9);
	DEC_CONSTK(K_KP_PERIOD);
	DEC_CONSTK(K_KP_DIVIDE);
	DEC_CONSTK(K_KP_MULTIPLY);
	DEC_CONSTK(K_KP_MINUS);
	DEC_CONSTK(K_KP_PLUS);
	DEC_CONSTK(K_KP_ENTER);
	DEC_CONSTK(K_KP_EQUALS);
	DEC_CONSTK(K_UP);
	DEC_CONSTK(K_DOWN);
	DEC_CONSTK(K_RIGHT);
	DEC_CONSTK(K_LEFT);
	DEC_CONSTK(K_INSERT);
	DEC_CONSTK(K_HOME);
	DEC_CONSTK(K_END);
	DEC_CONSTK(K_PAGEUP);
	DEC_CONSTK(K_PAGEDOWN);
	DEC_CONSTK(K_F1);
	DEC_CONSTK(K_F2);
	DEC_CONSTK(K_F3);
	DEC_CONSTK(K_F4);
	DEC_CONSTK(K_F5);
	DEC_CONSTK(K_F6);
	DEC_CONSTK(K_F7);
	DEC_CONSTK(K_F8);
	DEC_CONSTK(K_F9);
	DEC_CONSTK(K_F10);
	DEC_CONSTK(K_F11);
	DEC_CONSTK(K_F12);
	DEC_CONSTK(K_F13);
	DEC_CONSTK(K_F14);
	DEC_CONSTK(K_F15);

	DEC_CONSTK(K_NUMLOCK);
	DEC_CONSTK(K_CAPSLOCK);
	DEC_CONSTK(K_SCROLLOCK);
	DEC_CONSTK(K_RSHIFT);
	DEC_CONSTK(K_LSHIFT);
	DEC_CONSTK(K_RCTRL);
	DEC_CONSTK(K_LCTRL);
	DEC_CONSTK(K_RALT);
	DEC_CONSTK(K_LALT);
	DEC_CONSTK(K_RMETA);
	DEC_CONSTK(K_LMETA);
	DEC_CONSTK(K_LSUPER);
	DEC_CONSTK(K_RSUPER);
	DEC_CONSTK(K_MODE);

	DEC_CONSTK(K_HELP);
	DEC_CONSTK(K_PRINT);
	DEC_CONSTK(K_SYSREQ);
	DEC_CONSTK(K_BREAK);
	DEC_CONSTK(K_MENU);
	DEC_CONSTK(K_POWER);
	DEC_CONSTK(K_EURO);
	DEC_CONSTK(K_LAST);

	DEC_CONSTN(KMOD_NONE);
	DEC_CONSTN(KMOD_LSHIFT);
	DEC_CONSTN(KMOD_RSHIFT);
	DEC_CONSTN(KMOD_LCTRL);
	DEC_CONSTN(KMOD_RCTRL);
	DEC_CONSTN(KMOD_LALT);
	DEC_CONSTN(KMOD_RALT);
	DEC_CONSTN(KMOD_LMETA);
	DEC_CONSTN(KMOD_RMETA);
	DEC_CONSTN(KMOD_NUM);
	DEC_CONSTN(KMOD_CAPS);
	DEC_CONSTN(KMOD_MODE);

	DEC_CONSTN(KMOD_CTRL);
	DEC_CONSTN(KMOD_SHIFT);
	DEC_CONSTN(KMOD_ALT);
	DEC_CONSTN(KMOD_META);
}

DL_EXPORT(void) initlibpointless(void)
{
	PyObject *m, *d;
	m = Py_InitModule((char *)"libpointless", libpointlessmethods);
	d = PyModule_GetDict( m );
	PyDict_SetItemString( d, (char *)"VERSION", PyString_FromString(VERSION) );
	PyDict_SetItemString( d, (char *)"SYSCONFDIR", 
			PyString_FromString(SYSCONFDIR) );
	PyDict_SetItemString( d, (char *)"PKGDATADIR", 
			PyString_FromString(PKGDATADIR) );
	PyDict_SetItemString( d, (char *)"LOCALEDIR", 
			PyString_FromString(LOCALEDIR) );
	declare_constants(m);
}

#ifdef __cplusplus
}	/* end extern C */
#endif

// ------------------------------------------------------------------------- //

