#
# Copyright (C) 2004 Mekensleep
#
# Mekensleep
# 24 rue vieille du temple
# 75004 Paris
#       licensing@mekensleep.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
#
# Authors:
#  Loic Dachary <loic@gnu.org>
#

from twisted.internet import reactor
from underware.client import UGAMEClientProtocol

from string import join
from time import sleep
from formatter import DumbWriter
from StringIO import StringIO

from underware import animated
from pokereval import PokerEval
from poker.pokergame import PokerGameClient, PokerPlayer
from poker.pokercards import PokerCards
from poker.pokerchips import PokerChips
from poker.pokerpackets import *
from poker import pokerinterface
from poker import pokerjabber
from poker.pokerinteractor import PokerInteractor
from poker.pokerchat import PokerChat

SEARCHING = 1
RECONNECTING = 2
SWITCHING = 3
SEATING = 4
IDLE = 5
INITIAL_LOGIN = 6
LIST_HANDS = 7
BUY_IN = 8
    
class PokerRenderer:
    """Poker client 3d rendering"""

    def __init__(self, factory):
        self.replayStepping = True
        self.displayedGameId = None
        self.replayGameId = None
        self.scheduledAction = {}
        self.futureAction = {}
        self.state = SEARCHING
        self.leaving = False
        self.delays = False
        self.top_menu = { 'show': False, 'message': 'No table' }
        self.interactors = {
            "check": PokerInteractor("check", self.interactorActionCheck, self.interactorDisplayNode, self.interactorSelectedCallback),
            "fold": PokerInteractor("fold", self.interactorActionFold, self.interactorDisplayNode, self.interactorSelectedCallback),
            "raise": PokerInteractor("raise", self.interactorActionRaise, self.interactorDisplayNull, self.interactorSelectedInterfaceCallback)}
        self.factory = factory
        self.protocol = None
        self.streamMode = True
        
    def setProtocol(self, protocol):
        self.protocol = protocol
        if protocol:
            self.protocol.play_money = -1
            protocol.registerHandler("current", None, self._handleConnection)
            protocol.registerHandler("not_current", None, self._handleConnection)

    def setDisplayedGameId(self, game_id):
        self.displayedGameId = game_id
        if self.protocol:
            self.protocol.setCurrentGameId(game_id)

    def logout(self):
        if ( self.state == IDLE or self.state == SEARCHING ) and self.protocol.user.isLogged():
            self.factory.killJabberClient()
            self.deleteGames()
            self.protocol.logout()
            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
            self.state = SEARCHING

    def quit(self):
        self.factory.quit()

    def confirmQuit(self):
        if self.protocol:
            self.sendPacket(PacketQuit())
        self.setDisplayedGameId(None)
        
    def autoBlind(self, auto):
        if auto:
            self.protocol.sendPacket(PacketPokerAutoBlindAnte(game_id = self.displayedGameId,
                                                              serial = self.protocol.getSerial()))
        else:
            self.protocol.sendPacket(PacketPokerNoautoBlindAnte(game_id = self.displayedGameId,
                                                                serial = self.protocol.getSerial()))
        
    def sitOut(self, yesno):
        if yesno:
            self.protocol.sendPacket(PacketPokerSitOut(game_id = self.displayedGameId,
                                                       serial = self.protocol.getSerial()))
        else:
            game = self.factory.getGame(self.displayedGameId)
            if game.isBroke(self.protocol.getSerial()):
                self.requestBuyIn(game)
            else:
                self.protocol.sendPacket(PacketPokerSit(serial = self.protocol.getSerial(),
                                                        game_id = game.id))
        
    def payAnte(self, game, amount):
        interface = self.factory.interface
        if not interface.callbacks.has_key(pokerinterface.INTERFACE_POST_BLIND):
            message = "Pay the ante (%d) ?" % amount
            interface.blindMessage(message, "no")
            interface.registerHandler(pokerinterface.INTERFACE_POST_BLIND, self.confirmPayAnte)
            
    def confirmPayAnte(self, response):
        if response == "yes":
            self.protocol.sendPacket(PacketPokerAnte(game_id = self.displayedGameId,
                                                     serial = self.protocol.getSerial()))
        else:
            self.protocol.sendPacket(PacketPokerSitOut(game_id = self.displayedGameId,
                                                       serial = self.protocol.getSerial()))
        self.factory.interface.blindHide()
            
    def payBlind(self, game, amount, dead, is_late):
        interface = self.factory.interface
        if not interface.callbacks.has_key(pokerinterface.INTERFACE_POST_BLIND):
            message = "Pay the "
            if dead > 0:
                message += "big blind (%d) + dead (%d)" % ( amount, dead )
            elif amount == game.blind_info["big"]:
                message += "big blind (%d)" % amount
            else:
                message += "small blind (%d)" % amount
            message += "?"
            wait_blind = is_late and "yes" or "no"
            interface.blindMessage(message, wait_blind)
            interface.registerHandler(pokerinterface.INTERFACE_POST_BLIND, self.confirmPayBlind)

    def confirmPayBlind(self, response):
        if response == "yes":
            self.protocol.sendPacket(PacketPokerBlind(game_id = self.displayedGameId,
                                             serial = self.protocol.getSerial()))
        elif response == "wait":
            self.protocol.sendPacket(PacketPokerWaitBigBlind(game_id = self.displayedGameId,
                                                    serial = self.protocol.getSerial()))
        else:
            self.protocol.sendPacket(PacketPokerSitOut(game_id = self.displayedGameId,
                                              serial = self.protocol.getSerial()))
        self.factory.interface.blindHide()
            
    def requestLogin(self):
        interface = self.factory.interface
        remember = self.factory.remember
        if remember:
            name = self.factory.name
            password = self.factory.password
        else:
            name = ""
            password = ""
        interface.requestLogin(name, password, remember)
        interface.registerHandler(pokerinterface.INTERFACE_LOGIN, self.interfaceCallbackLogin)

    def requestAuth(self):
        self.state = INITIAL_LOGIN
        self.requestLogin()

    def requestSlider(self, dummy):
        interface = self.factory.interface
        if self.displayedGameId:
            game = self.factory.getGame(self.displayedGameId)
            if game.isRunning():
                result = game.betLimits(self.protocol.getSerial())
                if result != 0:           
                    if self.interactors["raise"].enabled:
                        (min_bet, max_bet, to_call) = result
                        step = self.bet_step
                        assert(interface != None)
                        interface.requestSlider(to_call, min_bet, max_bet, step)
                        if self.factory.verbose > 1:
                            print to_call, " ", min_bet, " ", max_bet, " ", step
                        interface.registerHandler(pokerinterface.INTERFACE_SLIDER, self.interfaceCallbackSlider)

    def interfaceCallbackSlider(self, value):
        step = self.bet_step
        self.requestBet(int(value)/step)
        return
    
    def sendBetPacket(self, packet):
        value = packet.amount[0]
        self.requestBet(value)
        return

    def requestBet(self, value):
        if self.factory.verbose > 1:
            print "requestBet: ", value
        if self.displayedGameId:
            game = self.factory.getGame(self.displayedGameId)
            result = game.betLimits(self.protocol.getSerial())
            if result == 0:
                print "*CRITICAL* requestBet: no limits !"
                return
            (min_bet, max_bet, to_call) = result

            callValue = to_call/(self.bet_step)
            #print value, "  ",  callValue
            if (int(value) == int(0)):
                if self.factory.verbose > 1:
                    print "requestBet: canceled"
                return
            
            elif (int(value) == int(callValue)):
                if self.factory.verbose > 1:
                    print "requestBet: call"
                self.interactors["raise"].actionCallback = self.interactorActionCall
            else:                
                step = self.bet_step
                amounts = (int(value))
                if self.factory.verbose > 1:
                    print "requestBet: raise"
                self.interactors["raise"].actionCallback = lambda x: self.interactorActionRaise(x, amounts)
            self.interactors["raise"].setSelected(1)
            self.cancelAllInteractorButThisOne("raise")
            self.handleInteractors(game)
        
    def interfaceReady(self, interface):
        interface.registerHandler(pokerinterface.INTERFACE_SHOW_TABLE_LIST, self.askTableList)
        interface.registerHandler(pokerinterface.INTERFACE_SHOW_CASHIER, self.showUserInfo)
        interface.registerHandler(pokerinterface.INTERFACE_SHOW_OUTFITS, self.showOutfits)
        interface.registerHandler(pokerinterface.INTERFACE_CHAT_HISTORY, self.chatHistory)
        interface.registerHandler(pokerinterface.INTERFACE_CHAT_LINE, self.chatLine)
        if self.factory.remember:
            if self.protocol:
                if self.factory.verbose:
                    print "connection ready, ask for password"
                self.requestAuth()
            else:
                if self.factory.verbose:
                    print "connection not established, will ask password later"

    def chatFormatMessage(self, message):
        config = self.factory.chat_config
        #
        # This is crude but is only meant to cope with a server that
        # would insist on sending more chars than the client wants.
        #
        message = message[:config['max_chars']] 
        format = DumbWriter(StringIO(), config['line_length'])
        format.send_flowing_data(message)
        return format.file.getvalue()
        
    def chatHide(self):
        interface = self.factory.interface
        if interface:
            interface.chatHide()
            
    def chatHistory(self, yesno):
        if self.displayedGameId != None:
            game = self.factory.getGame(self.displayedGameId)        
            self.render(game, PacketPokerChatHistory(show = yesno))

    def chatLine(self, line):
        if self.displayedGameId != None:
            self.protocol.sendPacket(PacketPokerChat(game_id = self.displayedGameId,
                                            serial = self.protocol.getSerial(),
                                            message = line))
    
    def askTableList(self, yesno):
        if self.state == IDLE:
            self.state = SEARCHING
            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
        
    def interfaceCallbackLogin(self, ok_or_cancel, name, password, remember):
        if ok_or_cancel != "ok":
            self.protocol._handleConnection(PacketAuthCancel(), *self.protocol._expected_args, **self.protocol._expected_kwargs)
#            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
#            self.state = SEARCHING            
            return
        
        interface = self.factory.interface
        (ok, reason) = self.protocol.user.checkNameAndPassword(name, password)
        if ok:
            if self.state != INITIAL_LOGIN:
                self.protocol.sendPacket(PacketPokerPlayerInfo(serial = 0,
                                                               name = name,
                                                               outfit = self.factory.getOutfit()))
            self.protocol.sendPacket(PacketLogin(name = name,
                                        password = password))
            self.protocol.user.name = name
            self.protocol.user.password = password
            self.factory.saveAuthToFile(name, password, remember)
        else:
            self.showMessage(reason, self.requestLogin)

    def showMessage(self, message, callback):
        interface = self.factory.interface
        if interface:
            interface.messageBox(message)
            if callback:
                interface.registerHandler(pokerinterface.INTERFACE_MESSAGE_BOX, callback)
        if self.factory.verbose:
            print message

    def showUserInfo(self, yesno):
        interface = self.factory.interface
        if interface:
            interface.userInfo("Account Information",
                               "Login name: %s" % self.protocol.user.name,
                               "Bankroll: %d" % self.protocol.play_money)

    def handleSerial(self, packet):
        if self.factory.verbose:
            print "handleSerial: we now have serial %d" % packet.serial
        self.protocol.user.serial = packet.serial
        underware = self.factory.underware
        underware.PythonAccept(packet)
        self.factory.runJabberClient()
        self.top_menu['show'] = True
        self.updateTopMenu()

    def updateTableInfo(self, game):
        message = ""
        if game:
            message += "Table %s\n" % game.name
            message += "%s\n" % game.getVariantName()
            if game.isTournament() and game.level > 0:
                ( delay, unit ) = game.delayToLevelUp()
                if delay <= 0:
                    message += "level up next hand\n"
                else:
                    message += "next level in "
                    if unit == "hand":
                        message += "%d hands\n" % delay
                    elif unit == "second":
                        if delay > 60 * 60:
                            message += "%dH:" % ( delay / ( 60 * 60 ) )
                        if delay > 60:
                            message += "%dM\n" % ( ( delay / 60 ) % 60 )
                        elif delay == 60:
                            message += "one minute\n"
                        else:
                            message += "less than one minute\n"
                ( blind_info, ante_info ) = game.getLevelValues(game.level + 1)
                if game.blind_info:
                    message += "current blinds %d/%d\n" % ( game.blind_info["small"], game.blind_info["big"] )
                    message += "next blinds %d/%d\n" % ( blind_info["small"], blind_info["big"] )
                if game.ante_info:
                    message += "current ante %d\n" % game.ante_info["value"]
                    message += "next ante %d\n" % ante_info["value"]
            else:
                message += "%s\n" % game.getBettingStructureName()
        else:
            message += 'No table'
        self.top_menu['message'] = message
        self.updateTopMenu()
        
    def updateTopMenu(self):
        interface = self.factory.interface
        interface.topMenuParams(self.top_menu['message'])
        if self.top_menu['show']:
            interface.topMenuShow()
        else:
            interface.topMenuHide()
            
    def _handleConnection(self, protocol, packet):
        game = self.factory.packet2game(packet)

        if ( packet.type == PACKET_POKER_BEST_CARDS or
             packet.type == PACKET_POKER_PLAYER_NO_CARDS or
             packet.type == PACKET_POKER_CHIPS_PLAYER2BET or
             packet.type == PACKET_POKER_CHIPS_BET2POT or
             packet.type == PACKET_POKER_CHIPS_POT2PLAYER or
             packet.type == PACKET_POKER_CHIPS_POT_MERGE or
             packet.type == PACKET_POKER_CHIPS_POT_RESET or
             (packet.type == PACKET_POKER_DEAL_CARDS and self.streamMode) ):
            self.render(game, packet)

        elif packet.type == PACKET_POKER_STREAM_MODE:
            self.streamMode=True

        elif packet.type == PACKET_POKER_BATCH_MODE:
            self.streamMode=False
           

        elif packet.type == PACKET_POKER_BET_LIMIT:
            self.bet_step = packet.step
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_HAND_LIST:
            if self.state == LIST_HANDS:
                self.showHands(packet.hands)
                self.state = IDLE
            else:
                print "handleGame: unexpected state for HAND_LIST: %d" % self.state

        elif packet.type == PACKET_BOOTSTRAP:
            self.bootstrap()
            
        elif packet.type == PACKET_PROTOCOL_ERROR:
            self.showMessage(packet.message, lambda: self.factory.confirmQuit(True))
            self.factory.reconnect = False
            
        elif packet.type == PACKET_POKER_TABLE_LIST:
            if self.state == SEARCHING:
                self.showTables(packet.packets)
            elif self.state == SWITCHING:
                if not packet.packets:
                    self.state = SEARCHING
                    self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
                else:
                    self.connectTable(packet.packets)
            elif self.state == RECONNECTING:
                self.choseTable(packet.packets)
            else:
                print "handleGame: unexpected state for TABLE_LIST: %d" % self.state

        elif packet.type == PACKET_POKER_TABLE_DESTROY:
            self.leaving = False

            if self.replayGameId == packet.serial:
                self._lagmax = self.factory.delays.get("lag", 15)
                self.replayGameId = 0
                self.protocol.sendPacket(PacketPokerTableSelect(string = "my"))
                self.state = SWITCHING
            if self.displayedGameId == packet.serial:
                underware = self.factory.underware
                underware.PythonAccept(packet)
                self.setDisplayedGameId(None)
                self.updateTableInfo(None)
                self.chatHide()
            
        elif packet.type == PACKET_POKER_TABLE:
            self.leaving = False
            
            game = self.factory.getGame(packet.id)
            self.scheduledAction[game.id] = False
            self.futureAction[game.id] = {}
            if not game:
                self.showMessage("server refused our request", None)
            else:
                if game.name == "*REPLAY*":
                    self._lagmax = 0
                    self.replayGameId = game.id
                self.setDisplayedGameId(game.id)
                packet.seats_all = game.seats_all
                self.render(game, packet)
                self.updateTableInfo(game)
                self.factory.runBasicClient()
            self.state = IDLE

        elif packet.type == PACKET_AUTH_REQUEST:
            if self.factory.interface:
                self.requestLogin()
            else:
                self.protocol.sendPacket(PacketLogin(name = self.factory.name,
                                            password = self.factory.password))

        elif packet.type == PACKET_AUTH_EXPIRES:
            print "server timeout waiting for our login packet"
            self.factory.interface.hideLogin()
            self.showMessage("Server timed out waiting for login", None)
            self.state = IDLE

        elif packet.type == PACKET_AUTH_REFUSED:
            self.showMessage("Invalid login or passwd", lambda: self.askTableList(self.factory.interface))
            self.state = IDLE

        elif packet.type == PACKET_AUTH_CANCEL:
            if self.state == INITIAL_LOGIN:
                self.state = SEARCHING
                self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
            else:
                self.protocol.sendPacket(PacketAuthCancel())
                self.state = IDLE

        elif packet.type == PACKET_AUTH_OK:
            if self.factory.verbose:
                print "login accepted"

        elif packet.type == PACKET_SERIAL:
            self.handleSerial(packet)
            if self.state == INITIAL_LOGIN:
                self.protocol.sendPacket(PacketPokerPlayerInfo(serial = self.protocol.getSerial(),
                                                      name = self.protocol.getName(),
                                                      outfit = self.factory.getOutfit()))
                self.state = RECONNECTING
                self.deleteGames()
                self.protocol.sendPacket(PacketPokerTableSelect(string = "my"))

        if not game:
            return
            
        if packet.type == PACKET_POKER_START:
            self.render(game, packet)
            self.updateTableInfo(game)

        elif packet.type == PACKET_POKER_CANCELED:
            self.factory.interface.clearCallbacks(pokerinterface.INTERFACE_POST_BLIND)

        elif packet.type == PACKET_POKER_PLAYER_INFO:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_PLAYER_ARRIVE:
            self.render(game, packet)

            interface = self.factory.interface
            if packet.serial == self.protocol.getSerial():
                interface.sitActionsAuto("no")
                interface.sitActionsShow()
                if not interface.callbacks.has_key(pokerinterface.INTERFACE_AUTO_BLIND):
                    interface.registerHandler(pokerinterface.INTERFACE_AUTO_BLIND, self.autoBlind)
                    interface.registerHandler(pokerinterface.INTERFACE_SIT_OUT, self.sitOut)

        elif ( packet.type == PACKET_POKER_PLAYER_LEAVE or
               packet.type == PACKET_POKER_TABLE_MOVE ) :
            self.render(game, PacketPokerPlayerLeave(serial = packet.serial))
            if ( packet.serial == self.protocol.getSerial() and
                 packet.game_id == self.displayedGameId ):
                self.factory.interface.sitActionsHide()
                self.leaving = False
                self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
                self.state = SEARCHING
                self.protocol.sendPacket(PacketPokerGetUserInfo(serial = packet.serial))

        elif packet.type == PACKET_POKER_END_ROUND:
            self.clearFutureAction(game)
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_SELF_IN_POSITION:
            if not game.isBlindAnteRound():
                if not self.updateFutureAction(game):
                    interface = self.factory.interface
                    interface.inPositionShow()
                    self.updateInPosition(game)

        elif packet.type == PACKET_POKER_SELF_LOST_POSITION:
            self.updateFutureAction(game)
            interface = self.factory.interface
            interface.inPositionHide()

        elif packet.type == PACKET_POKER_HIGHEST_BET_INCREASE:
            self.updateFutureAction(game)
               
        elif packet.type == PACKET_POKER_POSITION:
            packet.serial = game.getSerialInPosition()
            self.render(game, packet)
            self.delay(game, "position")

        elif packet.type == PACKET_POKER_CHAT:
            if game.id == self.displayedGameId:
                interface = self.factory.interface
                interface.chatHistory(packet.message)
                # duplicate PacketPokerChat
                # in order to preseve integrity of original packet
                message = self.chatFormatMessage(packet.message)
                message = PokerChat.filterChatTrigger(message)
                chatPacket = PacketPokerChat(game_id = packet.game_id,
                                             serial = packet.serial,
                                             message = message)
                if chatPacket.message.strip() != "":
                    self.render(game, chatPacket)

        elif packet.type == PACKET_POKER_BLIND_REQUEST:
            if ( game.getSerialInPosition() == self.protocol.getSerial() ):
                self.payBlind(game, packet.amount, packet.dead, packet.is_late)
                
        elif packet.type == PACKET_POKER_ANTE_REQUEST:
            if ( game.getSerialInPosition() == self.protocol.getSerial() ):
                self.payAnte(game, packet.amount)
                
        elif packet.type == PACKET_POKER_SEAT:
            if packet.seat == 255:
                self.showMessage("This seat is busy", None)
                self.state = IDLE
            else:
                if not game.isTournament():
                    self.requestBuyIn(game)
            
        elif packet.type == PACKET_POKER_SEATS:
            self.state = IDLE
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_PLAYER_CARDS:
            if game.variant == "7stud":
                packet.visibles = "best"
            else:
                packet.visibles = "hole"
            self.render(game, packet)

        elif packet.type == PACKET_POKER_BOARD_CARDS:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_DEALER:
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_SIT_OUT:
            self.render(game, packet)
            if packet.serial == self.protocol.getSerial():
                self.factory.interface.sitActionsSitOut("yes", "sit out")
                self.leaving = False
                self.delays = False

        elif packet.type == PACKET_POKER_AUTO_FOLD:
            if packet.serial == self.protocol.getSerial():
                self.factory.interface.sitActionsSitOut("yes", "sit out")
                self.delays = False

        elif packet.type == PACKET_POKER_SIT:
            self.render(game, packet)
            if packet.serial == self.protocol.getSerial():
                self.delays = True
                self.factory.interface.sitActionsSitOut("no", "sit out next hand")
            
        elif packet.type == PACKET_POKER_TIMEOUT_WARNING:
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_TIMEOUT_NOTICE:
            self.render(game, packet)
            if packet.serial == self.protocol.getSerial():
                self.leaving = True
                if self.factory.interface:
                    self.factory.interface.blindHide()
            
        elif packet.type == PACKET_POKER_WAIT_FOR:
            if packet.serial == self.protocol.getSerial():
                self.factory.interface.sitActionsSitOut("yes", "wait for %s blind" % packet.reason)
            
        elif packet.type == PACKET_POKER_IN_GAME:
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_WIN:
            self.render(game, packet)
            self.delay(game, "showdown")
            
        elif packet.type == PACKET_POKER_PLAYER_CHIPS:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_FOLD:
            self.handleFold(game, packet)
            self.render(game, packet)
            
        elif packet.type == PACKET_POKER_CALL:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_RAISE:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_CHECK:
            self.render(game, packet)

        elif packet.type == PACKET_POKER_BLIND:
            interface = self.factory.interface
            if interface.callbacks.has_key(pokerinterface.INTERFACE_POST_BLIND):
                interface.blindHide()
                interface.clearCallbacks(pokerinterface.INTERFACE_POST_BLIND)

        elif packet.type == PACKET_POKER_ANTE:
            interface = self.factory.interface
            if interface.callbacks.has_key(pokerinterface.INTERFACE_POST_BLIND):
                interface.blindHide()
                interface.clearCallbacks(pokerinterface.INTERFACE_POST_BLIND)

        elif packet.type == PACKET_POKER_STATE:
            self.updateFutureAction(game)
            if packet.string != "end":
                self.delay(game, "round")

        self.handleInteractors(game)

    def requestBuyIn(self, game):
        player = game.getPlayer(self.protocol.getSerial())

        min_amount = max(0, game.buyIn() - player.money.toint())
        max_amount = game.maxBuyIn() - player.money.toint()

        if max_amount < 0:
            self.showMessage("You can't bring more money\nto the table", None)
            return

        if min_amount > self.protocol.play_money:
            self.showMessage("You don't have enough money to\nparticipate in the game", None)
            return

        if min_amount == self.protocol.play_money:
            self.buyIn(game, self.protocol.play_money)
            return 

        self.state = BUY_IN
        if player.isBuyInPayed():
            legend = "How much do you want to rebuy ?"
        else:
            legend = "Which amount do you want to bring at the table ?"
        
        interface = self.factory.interface

        if max_amount >= self.protocol.play_money:
            label = "All your bankroll"
        else:
            label = "Maximum buy in"
        interface.buyInParams(min_amount, min(max_amount, self.protocol.play_money), legend, label)
        interface.buyInShow()
        interface.registerHandler(pokerinterface.INTERFACE_BUY_IN, lambda value: self.buyIn(game, value))
        
    def buyIn(self, game, value):
        interface = self.factory.interface
        interface.buyInHide()
        interface.clearCallbacks(pokerinterface.INTERFACE_BUY_IN)
        self.protocol.sendPacket(PacketPokerBuyIn(serial = self.protocol.getSerial(),
                                         game_id = game.id,
                                         amount = int(float(value))))
        self.protocol.sendPacket(PacketPokerSit(serial = self.protocol.getSerial(),
                                       game_id = game.id))
        self.state = IDLE
        
    def delayPredicate(self, packet):
        if not self.delays:
            return False
        game = self.factory.packet2game(packet)
        return game and game.id == self.displayedGameId
        
    def delay(self, game, event):
        if game.id != self.displayedGameId:
            return

        if self.leaving:
            return
        
        if ( game.id == self.replayGameId and
             self.replayStepping ):
            self.protocol.hold(120, self.delayPredicate)
            return

        if event == "position":
            if not self.protocol._hold_packets:
                self.protocol.hold(self.factory.delays.get("position", 1), self.delayPredicate)
        elif event == "round":
            self.protocol.hold(self.factory.delays.get("round", 1), self.delayPredicate)
        elif event == "showdown":
            self.protocol.hold(self.factory.delays.get("showdown", 1), self.delayPredicate)
        else:
            print "*CRITICAL* delay unexpected event %s " % event

    def handleFold(self, game, packet):
        if packet.serial == self.protocol.getSerial():
            self.scheduledAction[game.id] = None

    def handleInPosition(self, game, packets, index, value):
        if index == 2:
            chips = PokerChips(game.chips_values, value)
            packets[index].amount = chips.chips
        self.protocol.sendPacket(packets[index])
        
    def updateInPosition(self, game):
        serial = self.protocol.getSerial()
        interface = self.factory.interface
        packets = [None, None, None]
        buttons = ['', '', '']
        interface.clearCallbacks(pokerinterface.INTERFACE_IN_POSITION)
        interface.registerHandler(pokerinterface.INTERFACE_IN_POSITION, lambda index, value: self.handleInPosition(game, packets, index, value))
        (min_raise, max_raise, to_call) = game.betLimits(serial)
        def fill(index, label, packet):
            packets[index] = packet
            buttons[index] = label
        fill(0, "Fold", PacketPokerFold(game_id = game.id,
                                        serial = serial))
        if game.canCheck(serial):
            fill(1, "Check", PacketPokerCheck(game_id = game.id,
                                              serial = serial))
        if game.canCall(serial):
            fill(1, "Call %d" % to_call, PacketPokerCall(game_id = game.id,
                                                         serial = serial))
        if game.canRaise(serial):
            if game.highestBetNotFold() > 0:
                name = "Raise"
            else:
                name = "Bet"
            if min_raise == max_raise:
                label = "%s %d" % ( name, min_raise )
            else:
                label = name + ' %d'
            fill(2, label, PacketPokerRaise(game_id = game.id,
                                            serial = serial))
            
        interface.inPositionParams(min_raise, max_raise, self.bet_step, to_call,
                                   buttons[0], buttons[1], buttons[2])
        
        
    def handleFutureAction(self, game, index):
        if not self.futureAction.has_key(game.id):
            return
        
        info = self.futureAction[game.id]

        if not info.has_key("names"):
            return

        if index < 0:
            info["selected"] = None
            
        if index >= 0 and index < len(info["names"]):
            info["selected"] = info["names"][index]
            if self.factory.verbose > 1:
                print "handleFutureAction: %s now selected" % info["selected"]

    def clearFutureAction(self, game):
        self.futureAction[game.id] = {}
        
    def updateFutureAction(self, game):
        interface = self.factory.interface
        serial = self.protocol.getSerial()
        info = self.futureAction[game.id]
        packetSent = False
        if game.isInGame(serial) and not game.isBlindAnteRound():
            if not info:
                interface.futureActionShow()
            interface.clearCallbacks(pokerinterface.INTERFACE_FUTURE_ACTION)
            interface.registerHandler(pokerinterface.INTERFACE_FUTURE_ACTION, lambda index: self.handleFutureAction(game, index))

            player = game.getPlayer(serial)
            bet_limits = game.betLimits(serial)
            (min_raise, max_raise, to_call) = bet_limits
            bet = player.bet.toint()
            money = player.money.toint()
            highest_bet = game.highestBetNotFold()

            current_selected = info.get("selected", None)
                
            if not info.has_key("bet_limits") or bet_limits != info["bet_limits"]:
                    
                if current_selected not in info.get("any", []):
                    current_selected = None

                info["any"] = []
                info["bet_limits"] = bet_limits
                names = []
                name2packet = {}
                def fill(name, packet, any):
                    names.append(name)
                    name2packet[name] = packet
                    if any:
                        info["any"].append(name)
                packet_fold = PacketPokerFold(game_id = game.id,
                                              serial = serial)
                packet_check = PacketPokerCheck(game_id = game.id,
                                                serial = serial)
                packet_call = PacketPokerCall(game_id = game.id,
                                              serial = serial)
                chips = PokerChips(game.chips_values, 0)
                packet_raise = PacketPokerRaise(game_id = game.id,
                                                serial = serial,
                                                amount = chips.chips)
                fill('Fold', lambda: packet_fold, False)
                if game.canCheck(serial):
                    def checkFold():
                        if(game.canCheck(serial)):
                            return packet_check
                        else:
                            return packet_fold
                    fill('Check', lambda: packet_check, False)
                    fill('Check/Fold', checkFold, False)
                if to_call < money:
                    if game.canCall(serial):
                        fill('Call %d' % to_call, lambda: packet_call, False)
                    fill('Call any', lambda: packet_call, True)
                    if min_raise < money:
                        if game.canRaise(serial):
                            fill('Raise %d' % min_raise, lambda: packet_raise, False)
                            if min_raise < max_raise:
                                fill('Raise any', lambda: packet_raise, True)
                    else:
                        if game.canRaise(serial):
                            fill('Raise all-in %d' % money, lambda: packet_raise, True)
                else:
                    if game.canCall(serial):
                        fill('Call all-in %d' % money, lambda: packet_call, True)

                info["names"] = names
                info["name2packet"] = name2packet

                if current_selected not in names:
                    info["selected"] = None
                    current_selected = None
                
            if serial == game.getSerialInPosition():
                if current_selected:
                    build_packet = info["name2packet"][current_selected]
                    self.protocol.sendPacket(build_packet())
                    packetSent = True
                self.futureAction[game.id] = {}
                interface.futureActionHide()
            else:
                interface.futureActionParams(info["names"], current_selected)
        else:
            interface.futureActionHide()
        return packetSent

    def handleInteractors(self, game):
        if game.isInGame(self.protocol.getSerial()) and not game.isBlindAnteRound():
            player = game.getPlayer(self.protocol.getSerial())
            my_bet = player.bet.toint()
            self.interactors["check"].setEnabled(game.betsNull() or game.highestBetNotFold() <= my_bet)
            self.interactors["fold"].setEnabled(1)
            self.interactors["raise"].setEnabled(not game.betsNull() and game.highestBetNotFold() > my_bet or not game.round_cap_left == 0)
#            self.interactors["callnraise"].setEnabled((not game.betsNull() and game.highestBetNotFold() > my_bet) or not game.round_cap_left == 0)
            
            for interactor in ( "check", "fold", "raise"):
                i = self.interactors[interactor]
                i.setInPosition(self.protocol.getSerial() == game.getSerialInPosition())
                i.update()
                i.display()
        else:
            #update interactor inposition state
            for interactor in ( "check", "fold", "raise"):
                i = self.interactors[interactor]
                i.disable()
                i.update()
                defaultNodeName = i.getDefault()
                clickedNodeName = i.getClicked()
                self.render(game, PacketPokerDisplayNode(name = interactor, state = "default", node = defaultNodeName))
                self.render(game, PacketPokerDisplayNode(name = interactor, state = "clicked", node = clickedNodeName))

    def disableAllInteractorButThisOne(self, name):
        for interactor in (self.interactors.keys()):
            if interactor != name:
                self.interactors[interactor].disable()
        return
    def cancelAllInteractorButThisOne(self, name):
        for interactor in (self.interactors.keys()):
            if interactor != name:
                self.interactors[interactor].cancel()
        return
    
    def interactorDisplayNode(self, interactor):        
        game = self.factory.getGame(self.displayedGameId)        
        defaultNodeName = interactor.getDefault()
        clickedNodeName = interactor.getClicked()
        self.render(game, PacketPokerDisplayNode(name = interactor.name, state = "default", node = defaultNodeName))
        self.render(game, PacketPokerDisplayNode(name = interactor.name, state = "clicked", node = clickedNodeName))
        return

    def interactorDisplayNull(self, interactor):
        return
#    def interactorDisplayInterface(interactor):
#        return

    def interactorActionFold(self, interactor):
        self.disableAllInteractorButThisOne(interactor.name)
        game = self.factory.getGame(self.displayedGameId)
        action = PacketPokerFold(game_id = game.id,
                                 serial = self.protocol.getSerial())
        self.protocol.sendPacket(action)        
        return

    def interactorActionCheck(self, interactor):
        self.disableAllInteractorButThisOne(interactor.name)
        game = self.factory.getGame(self.displayedGameId)
        action = PacketPokerCheck(game_id = game.id,
                                  serial = self.protocol.getSerial())
        self.protocol.sendPacket(action)
        return

    def interactorActionCall(self, interactor):
        self.disableAllInteractorButThisOne(interactor.name)
        game = self.factory.getGame(self.displayedGameId)
        action = PacketPokerCall(game_id = game.id,
                                 serial = self.protocol.getSerial())
        self.protocol.sendPacket(action)
        return

    def interactorActionRaise(self, interactor, value):
        self.disableAllInteractorButThisOne(interactor.name)
        game = self.factory.getGame(self.displayedGameId)
#        self.handleInteractors(game)
        amountInChips = PokerChips(game.chips_values,
                                   value * self.bet_step)
        action = PacketPokerRaise(game_id = game.id,
                                  serial = self.protocol.getSerial(),
                                  amount = amountInChips.chips)        
        self.protocol.sendPacket(action)
        return
        
    def interactorSelectedCallback(self, interactor):
        interactor.setSelected(1)
        self.cancelAllInteractorButThisOne(interactor.name)
        return
    def interactorSelectedInterfaceCallback(self, interactor):
        dummy = 0
        self.requestSlider(dummy)
        return
    
    def interactorSelected(self, packet):
        if self.displayedGameId:
            game = self.factory.getGame(self.displayedGameId)
            name = packet.name
            event = packet.event
            if event == "selected":
                self.interactors[name].select()
                self.handleInteractors(game)
            
    def render(self, game, packet):
        if self.displayedGameId == game.id:
            underware = self.factory.underware
            underware.PythonAccept(packet)
        
    def scheduleAction(self, packet):
        game = self.factory.packet2game(packet)
        if game.isRunning():
            action = False
            if packet.type == PACKET_POKER_RAISE:
                amount = PokerChips(game.chips_values,
                                    packet.amount[0] * self.bet_step)
                action = PacketPokerRaise(game_id = game.id,
                                          serial = self.protocol.getSerial(),
                                          amount = amount.chips)
            elif packet.action == "raise":
                amount = PokerChips(game.chips_values)
                action = PacketPokerRaise(game_id = game.id,
                                          serial = self.protocol.getSerial(),
                                          amount = amount.chips)
            elif packet.action == "fold":
                action = PacketPokerFold(game_id = game.id,
                                         serial = self.protocol.getSerial())
            elif packet.action == "call":
                action = PacketPokerCall(game_id = game.id,
                                         serial = self.protocol.getSerial())
            elif packet.action == "check":
                action = PacketPokerCheck(game_id = game.id,
                                          serial = self.protocol.getSerial())

            if self.protocol.getSerial() != game.getSerialInPosition():
                if type(self.scheduledAction[game.id]) == type(action):
                    self.scheduledAction[game.id] = None
                else:
                    self.scheduledAction[game.id] = action
            else:                
                self.protocol.sendPacket(action)

    def wantToLeave(self):
        self.protocol.hold(0, None)
        self.leaving = True
        
        if ( self.displayedGameId and
             self.displayedGameId == self.replayGameId ):
            self.protocol.sendPacket(PacketPokerTableDestroy(game_id = self.replayGameId))
            return

        if self.displayedGameId:
            interface = self.factory.interface
            if interface.callbacks.has_key(pokerinterface.INTERFACE_POST_BLIND):
                interface.blindHide()
                interface.clearCallbacks(pokerinterface.INTERFACE_POST_BLIND)
                
            game = self.factory.getGame(self.displayedGameId)
            if (game.isSeated(self.protocol.getSerial())):
                packet = PacketPokerPlayerLeave(game_id = self.displayedGameId,
                                                serial = self.protocol.getSerial())
                self.leaveSeat(packet)
                return

        if self.state == IDLE:
            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
            self.state = SEARCHING
        else:
            if self.factory.verbose:
                print "wantToLeave: unexpected state %d" % self.state
                    
    def showKeyBinding(self):
        interface = self.factory.interface
        if not interface:
            print "No interface"
            return False        
        bindings = ( ("ESC", "Quit"),
                     ("F1", "Show key binding"),
                     ("F3", "Select character outfits"),
                     ("F6", "Logout"),
                     ("F7", "Replay"),
                     ("Shift", "Instant show tooltip"),
                     ("Tab", "Show/Hide tooltip"), #do not remove ,
                     )
        interface.showKeyBinding(bindings)
        return True

    def showOutfits(self, yesno):
        interface = self.factory.interface
        if not interface:
            print "Cannot select outfit, no interface"
            return False

        if not self.state == IDLE:
            print "Cannot select outfit while in state %d" % self.state
            return False
        
        if self.displayedGameId:
            game = self.factory.getGame(self.displayedGameId)
            if game.isSeated(self.protocol.getSerial()):
                self.showMessage("You must leave the table to select an outfit", None)
                return False

        interface.showOutfits(self.factory.getOutfits())
        interface.registerHandler(pokerinterface.INTERFACE_OUTFITS, self.selectOutfit)
        return True

    def selectOutfit(self, outfit):
        interface = self.factory.interface
        if self.state == IDLE and outfit != None and type != None:
            self.factory.setOutfit(outfit)
            self.protocol.sendPacket(PacketPokerPlayerInfo(serial = self.protocol.getSerial(),
                                                           name = self.protocol.user.name,
                                                           outfit = self.factory.getOutfit()
                                                           ))
        elif outfit != None or type != None:
            print "selectTables: ignored because not in IDLE state"

    def selectTables(self, table):
        interface = self.factory.interface
        self.top_menu['table_list'] = 'no'
        self.updateTopMenu()
        if self.state == SEARCHING:
            self.connectTable([table])
        else:
            print "selectTables: ignored because not in SEARCHING state"
    
    def showTables(self, tables):
        interface = self.factory.interface
        if interface:
            for table in tables:
                table.variant = self.factory.translateFile2Name(table.variant)
                table.betting_structure = self.factory.translateFile2Name(table.betting_structure)
            interface.showTables(tables)
            interface.registerHandler(pokerinterface.INTERFACE_TABLES, self.selectTables)
            interface.registerHandler(pokerinterface.INTERFACE_TABLES_REFRESH, self.refreshTables)

    def refreshTables(self, query):
        self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
        
    def handReplay(self, hand):
        self.protocol.sendPacket(PacketPokerHandReplay(serial = hand))

    def replayStep(self):
        self.protocol.hold(0, self.delayPredicate)
    
    def selectHands(self, hand):
        interface = self.factory.interface
        if self.state == IDLE and hand != None:
            self.handReplay(hand)
        elif hand != None:
            print "selectHand: ignored because not in IDLE state"
    
    def showHands(self, hands):
        interface = self.factory.interface
        if interface:
            if hands:
                interface.showHands(hands)
                interface.registerHandler(pokerinterface.INTERFACE_HANDS, self.selectHands)
            else:
                self.showMessage("Your hand history is empty", None)

    def queryHands(self):
        self.state = LIST_HANDS
        self.protocol.sendPacket(PacketPokerHandSelect(string = ""))
        
    def queryAllHands(self):
        self.state = LIST_HANDS
        self.protocol.sendPacket(PacketPokerHandSelectAll(string = "hands.name is not null"))
        
    def choseTable(self, tables):
        if self.state != RECONNECTING:
            print "choseTable: unexpected state %d" % self.state
            return
        
        if not tables:
            #
            # There are no tables in which we are playing, widen
            # the search.
            #
            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
            self.state = SEARCHING
        else:
            for table in tables:
                self.protocol.sendPacket(PacketPokerTableJoin(game_id = table.id,
                                                     serial = self.protocol.getSerial()))
            self.state = IDLE
        
    def connectTable(self, tables):
        #
        # First available table 
        #
        found = tables[0]

        if found and self.displayedGameId != found.id:
            game = self.factory.getGame(self.displayedGameId)
            if game and not game.isSeated(self.protocol.getSerial()):
                #
                # Forget about tables where we do not sit
                #
                self.protocol.sendPacket(PacketPokerTableQuit(game_id = game.id,
                                                              serial = self.protocol.getSerial()))
                self.deleteGame(game.id)
                self.setDisplayedGameId(None)
                
            game = False
            if self.factory.games.has_key(found.id):
                game = self.factory.games[found.id]

            if not game:
                #
                # Join the table we've not joined yet
                #
                self.protocol.sendPacket(PacketPokerTableJoin(game_id = found.id,
                                                     serial = self.protocol.getSerial()))
            elif game.id == self.displayedGameId:
                #
                # Game is already displayed, do nothing
                #
                self.state = IDLE
                print "connectTable: unexpected game.id == self.displayedGameId (%d)" % game.id
                pass
            else:
                #
                # Restore the display of a table for which we already
                # know everything.
                #
                self.setDisplayedGameId(game.id)
                reactor.callLater(0, self.protocol.resendPackets, self.displayedGameId)
                self.leaving = False
                self.protocol.hold(0, None)
                self.state = IDLE
        else:
            self.state = IDLE

    def deleteGames(self):
        if self.displayedGameId:
            underware = self.factory.underware
            underware.PythonAccept(PacketPokerTableDestroy(serial = self.displayedGameId))
        self.setDisplayedGameId(None)
        for game_id in self.factory.games.keys():
            self.deleteGame(game_id)
        self.protocol.deleteGames()
        
    def deleteGame(self, game_id):
        self.protocol.deleteGame(game_id)
        if self.scheduledAction.has_key(game_id): del self.scheduledAction[game_id]
        if self.futureAction.has_key(game_id): del self.futureAction[game_id]

    def sendPacketSitOut(self, packet):
        self.sendPacket(packet)
        self.factory.interface.sitActionsSitOut("yes", "sit out next hand")
        
    def sendPacketSit(self, packet):
        self.sendPacket(packet)
        self.factory.interface.sitActionsSitOut("no", "sit out")
        
    def sendPacket(self, packet):
        print "render sendPacket %s" % packet
        return self.protocol.sendPacket(packet)
        
    def getSeat(self, packet):
        if self.state == IDLE:
            self.state = SEATING
            self.protocol.sendPacket(packet)

    def leaveSeat(self, packet):
        if self.state == IDLE:
            self.protocol.sendPacket(packet)
        
    def bootstrap(self):
        if self.factory.remember:
            if self.factory.interface:
                if self.factory.verbose:
                    print "interface ready, ask for password"
                self.requestAuth()
            else:
                if self.factory.verbose:
                    print "interface not ready, will ask password later"
        else:
            self.state = SEARCHING
            self.protocol.sendPacket(PacketPokerTableSelect(string = "all"))
        self.factory.underware.SetClient(self)

    def jabberMessageReceive(self, chat, author, message):
        if not self.displayedGameId:
            return

        game = self.factory.getGameByNameNoCase(chat)
        if not game:
            print "jabberMessageReceive: %s is not a known game" % chat
            return

        if not game.id == self.displayedGameId:
            if self.factory.verbose > 3:
                print "jabberMessageReceive: %s is not displayed" % chat
            return

        serial = game.getSerialByNameNoCase(author)
        if serial == 0:
            if self.factory.verbose > 3:
                print "jabberMessageReceive: %s is not seated at the table" % author
            return

        
        if self.factory.verbose > 3:
            print "jabberMessageReceive: %d/%d/%s" % ( game.id, serial, message )
        packet = PacketPokerChat(game_id = game.id,
                                 serial = serial,
                                 message = self.chatFormatMessage(message))
        underware = self.factory.underware
        underware.PythonAccept(packet)

    def reload(self):
        self.factory.reload()
