/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifndef maf_animate2d_h 
#define maf_animate2d_h

#include <vector>
#include <map>

#include <osg/ref_ptr>
#include <osg/Referenced>
#include <osg/Vec2>
#include <osg/Vec4>

#include <maf/mafexport.h>

namespace osg {
  class MatrixTransform;
  class Group;
  class Geode;
};

class MAF_EXPORT MAFApplication2DAnimateAction : public osg::Referenced {
public:
  virtual ~MAFApplication2DAnimateAction() {}

  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen) = 0;

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window) = 0;

  virtual void Unmap(osg::Group* group) = 0;

  virtual void Map(osg::Group* group) = 0;

  virtual void Update(bool mouseOver, double delta) = 0;
};

typedef std::vector<osg::ref_ptr<MAFApplication2DAnimateAction> > MAFApplication2DAnimateActionList;

class MAF_EXPORT MAFApplication2DAnimate : public MAFApplication2DAnimateActionList {
public:
  virtual ~MAFApplication2DAnimate() {}

  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window);

  virtual void Unmap(osg::Group* group);

  virtual void Map(osg::Group* group);

  virtual void Update(bool mouseOver, double delta);
};

typedef std::map<std::string, MAFApplication2DAnimate*> MAFName2Animate;

class MAF_EXPORT MAFApplication2DSlide : public MAFApplication2DAnimateAction {
public:
  enum {
    AXIS_X = 0x00,
    AXIS_Y = 0x01,
    AXIS_MASK = 0x01,
    DIRECTION_NEGATIVE = 0x00,
    DIRECTION_POSITIVE = 0x02,
    DIRECTION_MASK = 0x02
  };
  MAFApplication2DSlide(int directionFlags);
  virtual ~MAFApplication2DSlide();

  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window);

  virtual void Unmap(osg::Group* group) {}

  virtual void Map(osg::Group* group) {}

  virtual void Update(bool mouseOver, double delta);

  int GetDirectionFlags(void) { return mDirectionFlags; }
  void SetDirectionFlags(int directionFlags) { mDirectionFlags = directionFlags; }

  bool GetMouseTrigger(void) { return mMouseTrigger; }
  void SetMouseTrigger(bool mouseTrigger) { mMouseTrigger = mouseTrigger; }

  bool GetVisible(void) { return mVisible; }
  void SetVisible(bool visible);

  bool GetSwitching(void) { return mSwitching; }
  void SetSwitching(bool switching) { mSwitching = switching; }
  
  float GetDelay(void) { return mDelay; }
  void SetDelay(float delay) { mDelay = delay; }
  
protected:
  void Slide(const osg::Vec2& offset);

  int mDirectionFlags;
  osg::Vec2 mDisplacement;
  osg::Vec2 mOrigin;
  bool mMouseTrigger;
  bool mVisible;
  bool mSwitching;
  float mSwitchingTime;
  float mDelay;
  bool mDestroyped;
  osg::ref_ptr<osg::MatrixTransform> mWindow;
};

class MAF_EXPORT MAFApplication2DSlideInOut : public MAFApplication2DSlide {
public:
  MAFApplication2DSlideInOut(int directionFlags);
  virtual ~MAFApplication2DSlideInOut() {}

  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window);
  
  virtual void Unmap(osg::Group* group);

  virtual void Map(osg::Group* group);

};

class MAF_EXPORT MAFApplication2DAlpha : public MAFApplication2DAnimateAction {
public:
  MAFApplication2DAlpha(float alpha) : mAlpha(alpha) {}
  virtual ~MAFApplication2DAlpha() {}
  
  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window) {}

  virtual void Unmap(osg::Group* group) {}

  virtual void Map(osg::Group* group) {}

  virtual void Update(bool mouseOver, double delta) {}

protected:
  float mAlpha;
};

class MAF_EXPORT MAFApplication2DDecorate : public MAFApplication2DAnimateAction {
public:
  MAFApplication2DDecorate();
  virtual ~MAFApplication2DDecorate();
  
  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);

  virtual void Destroy(osg::Group* group, osg::MatrixTransform* window);

  virtual void Unmap(osg::Group* group);

  virtual void Map(osg::Group* group);

  virtual void Update(bool mouseOver, double delta) {}

protected:
  osg::ref_ptr<osg::Geode> mDecoration;
};

class MAF_EXPORT MAFApplication2DDecorateSquare : public MAFApplication2DDecorate {
public:
  MAFApplication2DDecorateSquare() :
    mLeft(10.f),
    mBottom(10.f),
    mRight(10.f),
    mTop(10.f),
    mBorder(0.f),
    mBackgroundColor(.5f, .5f, .5f, .5f),
    mBorderColor(1.f, 1.f, 1.f, 1.f)
    {}

  ~MAFApplication2DDecorateSquare() {}

  void SetGeometry(float left, float bottom, float right, float top) {
    mLeft = left;
    mBottom = bottom;
    mRight = right;
    mTop = top;
  }
  void SetBackgroundColor(const osg::Vec4& color) { mBackgroundColor = color; }
  void SetBorderColor(const osg::Vec4& color) { mBorderColor = color; }
  void SetBorder(float border) { mBorder = border; }

  virtual void Configure(osg::Group* group, osg::MatrixTransform* window, const osg::Vec2& origin, const osg::Vec2& geometry, const osg::Vec2& screen);
  
protected:
  float mLeft;
  float mBottom;
  float mRight;
  float mTop;
  float mBorder;
  osg::Vec4 mBackgroundColor;
  osg::Vec4 mBorderColor;
};

#endif // maf_animate2d_h
