/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Johan Euphrosine <johan@mekensleep.com>
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifndef _mafbezier_h_
#define _mafbezier_h_

#ifndef MAF_USE_VS_PCH
#include <vector>
#endif

namespace math
{
  struct lerp
  {
    template<typename U>
    static inline void doit(U &out, const U& from, const U &to, float t)
    {
      out = from + ((to - from)*t);
    }
  };
  template<typename U>
  struct length
  {
    static inline float doit(const U &v)
    {
      return v.length();
    }
  };
  template<>
  struct length<float>
  {
    static inline float doit(const float &v)
    {
      return v;
    }
  };
};

template<typename T>
struct MAFBezier
{
  struct Node
  {
    T mIn;
    T mTIn;
    T mTOut;
    T mOut;
    float mLength;
    Node() : mLength(0) {
    }
    Node(const T& in, const T& tin, const T& tout, const T& out) : mLength(0) {
      Init(in, tin, tout, out);
    }
    void Get(T &out, float t) const
    {
      const T &a = mIn;
      const T &b = mTIn;
      const T &c = mTOut;
      const T &d = mOut;
      T ab;
      T bc;
      T cd;
      T abc;
      T bcd;
      math::lerp::doit(ab, a, b, t);
      math::lerp::doit(bc, b, c, t);
      math::lerp::doit(cd, c, d, t);
      math::lerp::doit(abc, ab, bc, t);
      math::lerp::doit(bcd, bc, cd, t);
      math::lerp::doit(out, abc, bcd, t);
    }
    void Init(const T& in, const T& tin, const T& tout, const T& out)
    {
      mIn = in;
      mTIn = tin;
      mTOut = tout;
      mOut = out;
      mLength = math::length<T>::doit((tin - in) + (tout - tin) + (out -  tout));
      //mLength = math::length<T>::doit(in - out);
      //length = (in - out).length();
    }
  };

  MAFBezier() : mLength(0) {}

  void Insert(const T& in, const T& tin, const T& tout, const T& out)
  {
    int size = mNodes.size();
    mNodes.resize(size + 1);
    Node &node = mNodes[size];
    node.Init(in, tin, tout, out);
    mLength += node.mLength;
  }
  void Insert(const Node &node)
  {
    mNodes.push_back(node);
    mLength += node.mLength;
  }

  void Get(T &result, float t) const
  {
    const Node &node = GetNode(t);
    node.Get(result, t);
  }

  const Node &GetNode(float &t) const
  { 
    if (t >= 1.0f)
      return mNodes.back();
    if (t <= 0.0f)
      return mNodes.front();

    typedef typename std::vector<Node>::const_iterator It;
    It begin = mNodes.begin(), end = mNodes.end();

    float nodeToFind = t * mLength, nodeIn = 0.0f;
    
    // foreach node
    while(begin != end)
      {
	float nodeDelta = (*begin).mLength;
	float nodeOut = nodeIn + nodeDelta;
	if (nodeOut > nodeToFind)
	  {
	    // node found
	    // renormalize t on node space
	    t = (nodeToFind - nodeIn)/nodeDelta;
	    break;
	  }
	nodeIn = nodeOut;
	begin++;
      }
    return (*begin);
  }

  void Init() {
    mLength=0;
    mNodes.clear();
  }
  float mLength;
  std::vector<Node> mNodes;
};

#endif
