/**
   Copyright (C) 2004 Cedric Pinson <cpinson@freesheep.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

   ****************************************************************************
   * @file   prf_node.h
   *
   * @brief   Profile system
   *
   *****************************************************************************
   *
   * @author  Cedric Pinson, Based from article on Gems Game programming 3
   *	    Original authors GregHjelstrom and Byon Garrabrant
   *
   * @date    Created 2002/11
   *
   * @version $Id: profile_node.h,v 1.5 2004/09/30 16:28:34 izidor79 Exp $
   *
   ****************************************************************************/

#ifndef MAF_PROFILE_NODE_H
#define MAF_PROFILE_NODE_H

#ifndef MAF_USE_VS_PCH
#include <vector>
#include <ostream>
#endif

#ifdef USE_PROFILE

/** 
 *  This class is used as node in a array tree
 */
class ProfileNode : public std::vector<ProfileNode*>
{
  
 public:

  typedef std::vector<ProfileNode*> ContainerNode;

 private:

  /// start time
  double mStartTime;

  /// end time
  double mEndTime;

  /// time of function
  double mDuringTime;

  /// total time of function
  double mTotalTime;

  /// number of call
  int mNbCalls;

  /// level of the current node, we can see if a function is recursif with it
  int mLevel;

  /// specific data to display informations
  int mMaxSubSizeName;

  /// Frame counter
  unsigned int mCount;


  /// Total time reference
  static double mGlobalTime;

  /// Reference frame counter
  static unsigned int mReferenceCount;

  /// depth of profile
  static int mFilter;


  /// Name of node
  const char* mName;

  /// Childs
  //    ContainerNode mNodes;

  /// Parent node
  ProfileNode* mParent;


 public:

 
  /// Default constructor
  ProfileNode(const char* _name) {
    mName=_name; 
    Reset();
    mTotalTime=mStartTime=mDuringTime=0;
    mCount=0;
    mNbCalls=0;
    mParent=0;
    mLevel=0;
    mMaxSubSizeName=8;
    reserve(10);
  }



  /**
   *  Default destructor
   */
  ~ProfileNode() {
    while (!empty()) {
      delete back();
      pop_back();
    } 
  }

  
  inline double GetTotalTime() { return mTotalTime;}
  inline double GetGlobalTime() { return mGlobalTime;}
  inline int GetNbCalls() { return mNbCalls;}
  inline int GetSubSizeName() { return mMaxSubSizeName;}
  inline int GetLevel() { return mLevel;}

  /** 
   *  Set parent to this node
   *
   *  @param ProfileNode* parent to affect
   */
  void SetParent(ProfileNode* _parent) { mParent=_parent;}



  /** 
   *  Get Parent of this node
   *
   *  @return ProfileNode* parent
   */
  ProfileNode* GetParent() { return mParent;}


  /** 
   *  Add a child node of this node
   *
   *  @param ProfileNode* not to add
   */
  void AddNode(ProfileNode* _node) {
    int strl=strlen(_node->GetName());
    if (mMaxSubSizeName<strl)
      mMaxSubSizeName=strl;
    _node->SetParent(this); 
    push_back(_node);
    _node->SetLevel(GetLevel()+1);
  }




  /** 
   *  Return the name of node
   *
   *  @param const char* name of node
   */
  const char* GetName() const { return mName;}


  /** 
   *  Check if a node with the given name exist
   *
   *  @param const char* name of node to check
   *  @return ProfileNode* node if found else 0
   */
  ProfileNode* CheckSubName(const char* _name) { 
    ContainerNode::iterator it=begin();
    for (;it!=end();it++) 
      if ( (*it)->GetName() == _name)
	return *it;
    return 0;
  }



  /**
   *  Get the level of this node
   *
   *  @return int the level of node
   */
  int GetLevel() const { return mLevel;}



  /**
   *  Set level of this node
   *
   *  @param int level to affect
   */
  void SetLevel(int _level) { mLevel=_level;}



  /// Incremente number of call
  void AddCalls() { mNbCalls+=1;}


  /// Reset
  void Reset() {
    mTotalTime=mStartTime=mDuringTime=mGlobalTime=0;
    mCount=mReferenceCount=0;
    mNbCalls=0;
  }



  /// Start profile
  void StartProfile() { 

    if (mFilter && mFilter<mLevel)
      return;

    if (mReferenceCount!=mCount) {
      mCount=mReferenceCount;

      mTotalTime=0;
      mNbCalls=0;
    }

    mStartTime=GetRealTime(); 
  }



  /// Stop profile
  void StopProfile() { 
    if (mFilter && mFilter<mLevel)
      return ;

    AddCalls(); 
    mDuringTime=GetRealTime(); 
    mDuringTime-=mStartTime; 
    mTotalTime+=mDuringTime;

    if (mLevel==1)
      mGlobalTime+=mDuringTime;
  }

    
  /** 
   *  Set filter. It affect all node
   *
   *  @param int filter depth
   */
  void SetFilter(int _depth) { mFilter=_depth;}



  /** 
   *  New frame restart profiling for next frame
   *
   *  @param double global time to use
   */
  void SetGlobalTime(double _globalTime) {
    mGlobalTime=_globalTime;
    mReferenceCount++;
  }



  void DisplayHeader(std::ostream& _out) {
    char header[128];
    sprintf(header," %-*s - %*s - %*s - %*s - %*s - %*s\n",
	    10, GetName(),
	    10,"Parent %%",
	    10,"Global %%",
	    10,"Time s",
	    13,"Time/Call s",
	    9,"nbCalls");
    _out << header;
  }

  void DisplayItemAsParent(std::ostream& _out) {
    double ptime=GetTotalTime();
    double totalTime=ptime;
    double globalTime=GetGlobalTime();
    int nbCalls=GetNbCalls();
    ProfileNode* parent=GetParent();
    if (parent && parent->GetTotalTime()>0.)
      ptime=parent->GetTotalTime();

    static char te[256] ;
    static char l[256];
    
    sprintf(te," %-*s - %9.2f - %9.2f - %10f - %13f - %9d <\n",
	    GetSubSizeName(),GetName(),
	    totalTime*100.0/ptime,totalTime*100.0/globalTime,
	    totalTime,totalTime/nbCalls,
	    nbCalls);

    _out << te;
    int t=strlen(te);
    l[0]=' ';
    for (int i=1;i<t-3;i++)
      l[i]='-';
    l[t-3]='\n';
    l[t-2]=0;
    
    _out << l;
  }


  void DisplayChildItem(std::ostream& _out,const char *addBack) {
    double ptime=GetTotalTime();
    double totalTime=ptime;
    double globalTime=GetGlobalTime();
    int nbCalls=GetNbCalls();
    ProfileNode* parent=GetParent();
    if (parent && parent->GetTotalTime()>0.)
      ptime=parent->GetTotalTime();
  
    if (!addBack)
      addBack=" ";

    char ch[128];
    sprintf(ch," %-*s - %9.2f - %9.2f - %10f - %13f - %9d (%2d) %s\n",
	    parent->GetSubSizeName(),GetName(),
	    totalTime*100.0/ptime,totalTime*100.0/globalTime,
	    totalTime,totalTime/nbCalls,
	    nbCalls,size(),addBack);
    _out << ch;
  }




  void DisplayUnlogedTime(std::ostream& _out,double subtime) {
    char ch[128];
    double globalTime=GetGlobalTime();
    double totalTime=GetTotalTime();
    sprintf(ch," %-*s - %9.2f - %9.2f - %10f\n",
	    GetSubSizeName(),"Unloged",
	    subtime*100.0/totalTime,subtime*100.0/globalTime,
	    subtime);
    _out << ch;
  }



  void DisplayNode(std::ostream& _out,const char* addBack,ProfileNode* i) {

    DisplayHeader(_out);

    if (GetLevel()>0)
      DisplayItemAsParent(_out);
  
    double subTime=0;

    ContainerNode::iterator n=begin();
    for (;n!=end();n++) {
      (*n)->DisplayChildItem(_out, (*n==i?addBack:0) ) ;
      subTime+=(*n)->GetTotalTime();
    }

    if (subTime>0.) {
      subTime=GetTotalTime()-subTime;
      if (subTime>0.)
	DisplayUnlogedTime(_out,subTime);
    }
  }



  void DisplayFlatItem(std::ostream& _out) {
    double ptime=GetTotalTime();
    double totalTime=ptime;
    double globalTime=GetGlobalTime();
    int nbCalls=GetNbCalls();
    ProfileNode* parent=GetParent();

    if (parent && parent->GetTotalTime()>0.)
      ptime=parent->GetTotalTime();

    char str[256];
    char littleString[]=" ";
    sprintf(str,"%*s %-*s prt %6.2f glb %6.2f - time %f avg %f - ncall %-7d\n",
	    (GetLevel()-1)*5,littleString,
	    parent->GetSubSizeName(),GetName(),
	    totalTime*100.0/ptime,totalTime*100.0/globalTime,
	    totalTime,totalTime/nbCalls,
	    nbCalls);
    _out << str;
  }


  void DisplayFlatUnlogedTime(std::ostream& _out,double subtime) {
    double ptime=GetTotalTime();
    double totalTime=ptime;
    double globalTime=GetGlobalTime();

    char littleString[]=" ";
    char str[128];
    sprintf(str,"%*s %-*s prt %6.2f glb %6.2f - time %f\n",
	    GetLevel()*5,littleString,
	    GetSubSizeName(),"Unloged",
	    subtime*100.0/totalTime,subtime*100.0/globalTime,
	    subtime);
    _out << str;
  }


  void DisplayFlatStats(std::ostream& _out) {

    if (GetLevel()>0)
      DisplayFlatItem(_out);

    double subTime=0;

    ContainerNode::iterator n=begin();
    for (;n!=end();n++) {
      (*n)->DisplayFlatStats(_out);
      subTime+=(*n)->GetTotalTime();
    }

    if (subTime>0.) {
      subTime=GetTotalTime()-subTime;
      if (subTime>0.)
	DisplayFlatUnlogedTime(_out,subTime);
    }
  }

};
#endif

#endif 
