/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "../../config_win32.h"
#endif

#include <iostream>
#include <iomanip>
#include <map>

#ifdef _DEBUG // for Windows python23_d.lib is not in distribution... ugly but works
 #undef _DEBUG
 #include <Python.h>
 #define _DEBUG
#else
 #include <Python.h>
#endif


#include <libxml/tree.h>

#define UNDERWAREAPI_ANIMATED_INTERNAL
#include <pyunderware.h>

#include <cal3d/scheduler.h>

#include <maf/mafexport.h>
#include <maf/maferror.h>
#include <maf/application.h>
#include <maf/MultipleAnimationPathCallback.h>
#include <ugame/animated.h>

class Animated {
public:
  Animated() : mCal3d(0), mOsg(0) {}

  osg::ref_ptr<UGAMEAnimatedController> mCal3d;
  osg::ref_ptr<osg::Node> mOsg;
};

class PythonStopCallback : public CalAnimationAlt::StopCallback
{
public:
  PythonStopCallback(PyObject* callback) :
    mCallback(callback) {
    Py_INCREF(callback);
  }

  virtual void process(CalModel* model, CalAnimationAlt* animation) {
    PyObject* result = PyObject_CallObject(mCallback, 0);
    if(result == 0)
      throw new MAFError(UNDERWARE_MAF_ERROR_CAL3D, "PythonStopCallback::process: callback failed");
    Py_DECREF(result);
    Py_DECREF(mCallback);
    mCallback = 0;
    animation->setStopCallback(0);
    delete this;
  }

private:
  PyObject* mCallback;
};

//
//
// Animated class
//
//

static void
Animated_dealloc(Animated_Object *self)
{
  //  std::cerr << "Animated_dealloc" << std::endl;
  if(self->_Animated)
    delete self->_Animated;
  self->ob_type->tp_free((PyObject *)self);
}

static PyObject *
Animated_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  Animated_Object *self;

  assert(type != NULL && type->tp_alloc != NULL);

  self = (Animated_Object*)type->tp_alloc(type, 0);

  if(self != NULL) {
    self->_Animated = new Animated();
  }

  return (PyObject*)self;
}

static char Animated_run_doc[] =
"Run animation \n"
"\n";

static PyObject *
Animated_run(Animated_Object *self, PyObject *args)
{
  int channel;
  char* name;
  float length;
  float weight;
  float fade_in;
  float fade_out;
  float delay;
  if(!PyArg_ParseTuple(args, "isfffff:run", &channel, &name, &length, &weight, &fade_in, &fade_out, &delay))
    return NULL;

  if(self->_Animated->mOsg.valid()) {
    osg::MultipleAnimationPathCallback* animated = dynamic_cast<osg::MultipleAnimationPathCallback*>(self->_Animated->mOsg->getUpdateCallback());
    animated->setCurrentAnimationPath(name);
    if(animated->m_currentAnimationPath == 0) {
      PyErr_Format(PyExc_RuntimeError, "%s was not found", name);
      return 0;
    }
    if(length < 0.f)
      animated->m_currentAnimationPath->setLoopMode(osg::AnimationPath::LOOP);
    else
      animated->m_currentAnimationPath->setLoopMode(osg::AnimationPath::NO_LOOPING);

    animated->reset();
    animated->setPause(false);
    return Py_BuildValue("s", name);

  } else if(self->_Animated->mCal3d.valid()) {
    UGAMEAnimatedController* animated = self->_Animated->mCal3d.get();
    int coreId = animated->GetCoreAnimationId(name);
    CalScheduler::WeightFunction* weight_function = 0;
    if(fade_in > 0.0001f || fade_out > 0.0001f)
      weight_function = new CalScheduler::FadeInOut(fade_in, fade_out);
    CalAnimationAlt* animation = animated->GetScheduler()->run((CalScheduler::Channel)channel,
							       coreId,
							       length,
							       weight,
							       weight_function,
							       delay);
    if(animation == 0) {
      PyErr_Format(PyExc_RuntimeError, "run %s failed", name);
      return 0;
    }

    return Py_BuildValue("i", animation->getAnimationId());

  } else {
    PyErr_Format(PyExc_RuntimeError, "no animated object set");
    return 0;
  }
}

static char Animated_stop_doc[] =
"Stop animation \n"
"\n";

static PyObject *
Animated_stop(Animated_Object *self, PyObject *args)
{
  char* name;
  float fade_out;
  float delay;
  if(!PyArg_ParseTuple(args, "sff:stop", &name, &fade_out, &delay))
    return NULL;

  if(self->_Animated->mOsg.valid()) {
    osg::MultipleAnimationPathCallback* animated = dynamic_cast<osg::MultipleAnimationPathCallback*>(self->_Animated->mOsg->getUpdateCallback());
    animated->reset();
    animated->setPause(true);
    return Py_BuildValue("s", name);

  } else if(self->_Animated->mCal3d.valid()) {
    UGAMEAnimatedController* animated = self->_Animated->mCal3d.get();
    int anyId;
    if(isdigit(name[0])) {
      anyId = atoi(name);
    } else if(!strcmp(name, "-1")) { 
      anyId = -1;
    } else {
      anyId = animated->GetCoreAnimationId(name);
    }

    CalScheduler::WeightFunction* weight_function = 0;
    if(fade_out > 0.0001f) {
      weight_function = new CalScheduler::FadeOut(fade_out);
    }
      
    CalAnimationAlt* animation = animated->GetScheduler()->stop(anyId,
								weight_function,
								delay);
    int resId=-1;
    if(animation && anyId!=-1) {
      resId=animation->getAnimationId();
    }
    return Py_BuildValue("i", resId);

  } else {
    PyErr_Format(PyExc_RuntimeError, "no animated object set");
    return 0;
  }
}

static char Animated_setStopCallback_doc[] =
"SetStopCallback animation \n"
"\n";

static PyObject *
Animated_setStopCallback(Animated_Object *self, PyObject *args)
{
  char* name;
  PyObject* callback;
  if(!PyArg_ParseTuple(args, "sO:setStopCallback", &name, &callback))
    return NULL;

  if(!PyCallable_Check(callback)) {
    PyErr_Format(PyExc_TypeError, "second argument must be a callable");
    return 0;
  }    

  if(self->_Animated->mOsg.valid()) {
    PyErr_Format(PyExc_RuntimeError, "cannot set a stop callback on an AnimationPath");
    return 0;

  } else if(self->_Animated->mCal3d.valid()) {
    UGAMEAnimatedController* animated = self->_Animated->mCal3d.get();
    int anyId;
    if(isdigit(name[0]))
      anyId = atoi(name);
    else
      anyId = animated->GetCoreAnimationId(name);

    CalAnimationAlt* animation = animated->GetScheduler()->getAnimation(anyId);
    animation->setStopCallback(new PythonStopCallback(callback));

    Py_INCREF(Py_None);
    return Py_None;

  } else {
    PyErr_Format(PyExc_RuntimeError, "no animated object set");
    return 0;
  }
}

static char Animated_getDuration_doc[] =
"Get the total duration of the animation \n"
"\n";

static PyObject *
Animated_getDuration(Animated_Object *self, PyObject *args)
{
  char* name;
  if(!PyArg_ParseTuple(args, "s:getDuration", &name))
    return NULL;

  float duration = 0.f;

  if(self->_Animated->mOsg.valid()) {
    osg::MultipleAnimationPathCallback* animated = dynamic_cast<osg::MultipleAnimationPathCallback*>(self->_Animated->mOsg->getUpdateCallback());
    duration = animated->getAnimationDuration(name);

  } else if(self->_Animated->mCal3d.valid()) {
    UGAMEAnimatedController* animated = self->_Animated->mCal3d.get();
    duration = animated->GetDuration(name);

  } else {
    PyErr_Format(PyExc_RuntimeError, "no animated object set");
    return 0;
  }

  return Py_BuildValue("f", duration);
}



static char Animated_isAnimationActive_doc[] =
"Return true if animation is playing else false\n"
"\n";

static PyObject *
Animated_isAnimationActive(Animated_Object *self, PyObject *args)
{
  char* name;
  int playing=0;
  if(!PyArg_ParseTuple(args, "s:isAnimationActive", &name))
    return NULL;

  if(self->_Animated->mOsg.valid()) {
    osg::MultipleAnimationPathCallback* animated = dynamic_cast<osg::MultipleAnimationPathCallback*>(self->_Animated->mOsg->getUpdateCallback());
    if(animated) playing=(int)animated->isPlaying();

  } else if(self->_Animated->mCal3d.valid()) {
    UGAMEAnimatedController* animated = self->_Animated->mCal3d.get();
    int coreId = animated->GetCoreAnimationId(name);
    playing=animated->GetScheduler()->isAnimationActive(coreId);
  } else {
    PyErr_Format(PyExc_RuntimeError, "no animated object set");
    return 0;
  }

  return Py_BuildValue("i", playing);
}



static char Animated_create_doc[] =
"Constructor for the animation \n"
"\n";

static PyObject *
Animated_create(Animated_Object *self, PyObject *args)
{
  PyObject* application_object = 0;
  char* name;
  if(!PyArg_ParseTuple(args, "Os:create", &application_object, &name))
    return NULL;

  if(!MAFApplication_Check(application_object)) {
    PyErr_Format(PyExc_TypeError, "first argument must be an underware object");
    return 0;
  }

  MAFApplication_Object* application = (MAFApplication_Object*)application_object;

  osg::Referenced* animated = application->_MAFApplication->SearchAnimated(name);
  if(!animated) {
    PyErr_Format(PyExc_LookupError, "no animated matching %s in the application", name);
    return 0;
  }

  self->_Animated->mCal3d = 0;
  self->_Animated->mOsg = 0;

  if(dynamic_cast<UGAMEAnimatedController*>(animated)) {
    self->_Animated->mCal3d = dynamic_cast<UGAMEAnimatedController*>(animated);
  } else {
    osg::Node* node = dynamic_cast<osg::Node*>(animated);
    osg::NodeCallback* callback = node->getUpdateCallback();
    if(callback == 0) {
      PyErr_Format(PyExc_LookupError, "osg::Node %s has no update callback", name);
      return 0;
    }
    if(dynamic_cast<osg::MultipleAnimationPathCallback*>(callback) == 0) {
      PyErr_Format(PyExc_LookupError, "osg::Node %s has no osg::MultipleAnimationPathCallback callback", name);
      return 0;
    }
    self->_Animated->mOsg = node;
  }
  
  Py_INCREF(Py_None);
  return Py_None;
}

static PyMethodDef Animated_methods[] = {
  {"create",	(PyCFunction)Animated_create, METH_VARARGS, Animated_create_doc},
  {"run",	(PyCFunction)Animated_run, METH_VARARGS, Animated_run_doc},
  {"stop",	(PyCFunction)Animated_stop, METH_VARARGS, Animated_stop_doc},
  {"setStopCallback",	(PyCFunction)Animated_setStopCallback, METH_VARARGS, Animated_setStopCallback_doc},
  {"getDuration",	(PyCFunction)Animated_getDuration, METH_VARARGS, Animated_getDuration_doc},
  {"isAnimationActive",	(PyCFunction)Animated_isAnimationActive, METH_VARARGS, Animated_isAnimationActive_doc},
  {NULL,	NULL}		/* sentinel */
};

static char Animated_doc[] =
"Animated object\n"
"\n";

static PyTypeObject animated_Type = {
	PyObject_HEAD_INIT(&PyType_Type)
	0,
	"CAnimated",                           	/* tp_name */	  
	sizeof(Animated_Object),	     	/* tp_basicsize */
	0,				     	/* tp_itemsize */ 
	(destructor)Animated_dealloc,		/* tp_dealloc */
	0,					/* tp_print */
	0,			 		/* tp_getattr (obsoleted by tp_getattro) */
	0,			 		/* tp_setattr (obsoleted by tp_setattro) */
	0,					/* tp_compare */
	0,					/* tp_repr */
	0,					/* tp_as_number */
	0,					/* tp_as_sequence */
	0,					/* tp_as_mapping */
	0,					/* tp_hash */
	0,					/* tp_call */
	0,					/* tp_str */
	0,					/* tp_getattro */
	0,					/* tp_setattro */
	0,					/* tp_as_buffer */
	Py_TPFLAGS_DEFAULT | Py_TPFLAGS_BASETYPE, /* tp_flags */
	Animated_doc,				/* tp_doc */
	0,					/* tp_traverse */
	0,					/* tp_clear */
	0,					/* tp_richcompare */
	0,					/* tp_weaklistoffset */
	0,					/* tp_iter */
	0,					/* tp_iternext */
	Animated_methods,			/* tp_methods */
	0,					/* tp_members */
	0,					/* tp_getset */
	0,					/* tp_base */
	0,					/* tp_dict */
	0,					/* tp_descr_get */
	0,					/* tp_descr_set */
	0,					/* tp_dictoffset */
	0,					/* tp_init */
	0,					/* tp_alloc */
	Animated_new,				/* tp_new */
	0,					/* tp_free */
	0,					/* tp_is_gc */
	0,					/* tp_bases */
	0,					/* tp_mro */
	0,					/* tp_cache */
	0,					/* tp_subclasses */
	0					/* tp_weaklist */
};

//
//
// Module
//
//

static PyMethodDef animated_methods[] = {
  {NULL, NULL, 0, NULL}
};

PyMODINIT_FUNC
initc_animated(void)
{
  int index;
  PyObject *module;
  PyObject *dictionary;
  PyObject *apiobj;
  static void* c_api[UNDERWAREAPI_ANIMATED_NUMSLOTS];

  //
  // Declare methods, if any
  //
  if((module = Py_InitModule("c_animated", animated_methods)) == NULL) {
    /* error */
    return;
  }

  if(PyType_Ready(&animated_Type) < 0) return;
  
  dictionary = PyModule_GetDict(module);
  if(PyDict_SetItemString(dictionary, "CAnimated", (PyObject*)&animated_Type) < 0)
    return;

  index = 0;

  c_api[index++] = &animated_Type;

  apiobj = PyCObject_FromVoidPtr(c_api, NULL);
  PyDict_SetItemString(dictionary, UNDERWAREAPI_LOCAL_ENTRY, apiobj);
  Py_DECREF(apiobj);

  import_underware_base();
  import_mafapplication();
}
