/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *  Henry Precheur <henry@precheur.org>
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/maferror.h>
#include <maf/hit.h>
#include <maf/window.h>
#include <maf/application.h>
#include <maf/scene.h>
#include <maf/vision.h>
#include <maf/profile.h>
#include <maf/application2d.h>

#include <osg/Group>
#include <osg/LightSource>
#include <osg/MatrixTransform>
#include <osg/Projection>
#include <osg/Fog>
#include <osg/Viewport>
#include <osg/LineSegment>
#include <osg/State>
#include <osgUtil/IntersectVisitor>
#include <osgUtil/CullVisitor>

#include <osgCal/SubMeshSoftware>

#include <AL/alc.h>
#include <openalpp/alpp.h>

#include <maf/maferror.h>
#include <maf/window.h>
#include <maf/application.h>
#include <maf/scene.h>
#include <maf/vision.h>

#include <osgAL/SoundNode>
#include <osgAL/SoundRoot>
#include <osgAL/SoundManager>
#include <osgAL/SoundState>

#include <maf/application2d.h>

#include <osgUtil/TriStripVisitor>

#include <maf/timer.h>

#endif

// Model

MAFSceneModel::~MAFSceneModel() {
  g_debug("MAFSceneModel::~MAFSceneModel");

  mPickCache.clear();
  mPath2VisionController.clear();

  RecursiveClearUserData(mGroup.get());

  g_debug("MAFSceneModel::~MAFSceneModel: HUDGroup");
  mGroup->removeChild(mHUDProjection.get());
  mHUDProjection = 0;
  {
    osg::NodeVisitor* leakCheck = RecursiveLeakCollect(mHUDGroup.get());
    g_assert(mHUDGroup->referenceCount() == 1);
    mHUDGroup = 0;
    RecursiveLeakCheck(leakCheck);
  }
  mCamera = 0;
  g_assert(mScene->referenceCount() == 1);
  mScene = 0;
  g_debug("MAFSceneModel::~MAFSceneModel: Group");
  {
    osg::NodeVisitor* leakCheck = RecursiveLeakCollect(mGroup.get());
    g_assert(mGroup->referenceCount() == 1);
    mGroup = 0;
    RecursiveLeakCheck(leakCheck);
  }
  mLastPicked = 0;
  if(mAudio)
    osgAL::SoundManager::instance()->shutdown();
  mLastHit = 0;
}

void MAFSceneModel::Init(void)
{
  mGroup = new osg::Group;

  mScene = new osgUtil::SceneView();

  mScene->setDefaults();

  mScene->setLightingMode(osgUtil::SceneView::NO_SCENEVIEW_LIGHT);
  //  mScene->getGlobalStateSet()->setAssociatedModes(mScene->getLight(),osg::StateAttribute::OFF);
  mScene->getGlobalStateSet()->setMode(GL_ALPHA_TEST, osg::StateAttribute::OFF);
  mScene->getGlobalStateSet()->setMode(GL_CULL_FACE, osg::StateAttribute::ON);
  //  mScene->getGlobalStateSet()->setMode(GL_LIGHTING,osg::StateAttribute::OFF);
  //  mScene->getGlobalStateSet()->setAttributeToInherit(osg::StateAttribute::TEXENV);

  mScene->setCullMask(MAF_VISIBLE_MASK);

  mScene->setSceneData(mGroup.get());
  mScene->setComputeNearFarMode(osgUtil::CullVisitor::DO_NOT_COMPUTE_NEAR_FAR);
  HUDCreate();

  try {
    if(mAudio) {
      osgAL::SoundManager::instance()->init(MAF_AUDIO_NB_CHANNELS);
      osgAL::SoundManager::instance()->getEnvironment()->setDistanceModel(openalpp::InverseDistance);
    }
  } catch (...) {
    mAudio=false;
  }

}

void MAFSceneModel::HUDCreate(void)
{
  mHUDGroup = new osg::Group();

  // turn lighting off for the text and disable depth test to ensure its always ontop.
  osg::StateSet* stateset = mHUDGroup->getOrCreateStateSet();
  stateset->setMode(GL_LIGHTING,osg::StateAttribute::OFF);

  // disable depth test, and make sure that the hud is drawn after everything 
  // else so that it always appears ontop.
  stateset->setMode(GL_DEPTH_TEST,osg::StateAttribute::OFF);
  stateset->setRenderBinDetails(11,"RenderBin");

  // create the hud.
  osg::MatrixTransform* modelview_abs = new osg::MatrixTransform;
  modelview_abs->setReferenceFrame(osg::Transform::ABSOLUTE_RF);
  modelview_abs->setMatrix(osg::Matrix::identity());
  modelview_abs->addChild(mHUDGroup.get());

  mHUDProjection = new osg::Projection;
  mHUDProjection->addChild(modelview_abs);

  mGroup->addChild(mHUDProjection.get());
}


// View



/*virtual*/ void MAFSceneView::Init(void)
{
  MAFView::Init();

  if (NULL != GetModel())
    {
      GetModel()->mScene->setFrameStamp(new osg::FrameStamp());
      mStartTime=GetRealTime();
    }
}

void MAFSceneView::Update( MAFWindow* pWin )
{
  // Update Time
  if (NULL != GetModel())
    {
      const_cast<osg::FrameStamp*>(GetModel()->mScene->getFrameStamp())->setReferenceTime(GetRealTime()-mStartTime);
      const_cast<osg::FrameStamp*>(GetModel()->mScene->getFrameStamp())->setFrameNumber(mFrameNumber);
      mFrameNumber++;
    }

  osgUtil::SceneView* scene = GetModel()->mScene.get();

  scene->setViewport( 0, 0, pWin->GetWidth(), pWin->GetHeight() );

  GetModel()->mHUDProjection->setMatrix(osg::Matrix::ortho2D(0,pWin->GetWidth(),0,pWin->GetHeight()));

  if(GetModel()->mCamera.valid()) {
    MAFCameraModel* camera = GetModel()->mCamera->GetModel();
    osg::BoundingSphere bs = scene->getSceneData()->getBound();
    scene->setProjectionMatrixAsPerspective(camera->GetFov(),
					    (((float)pWin->GetWidth())/pWin->GetHeight()),
					    1.0f, // Be carefull when setting zNear: some cards/drivers (nVidia for instance) do not like when zNear is below 1
					    bs.radius()*2.0f);
    scene->setViewMatrixAsLookAt(camera->GetPosition(),
				 camera->GetTarget(),
				 camera->GetUp());
  } else {
    scene->setProjectionMatrixAsOrtho(0.f, pWin->GetWidth(),
				      0.f, pWin->GetHeight(),
				      1.f, 5000.f);
    scene->setViewMatrixAsLookAt(osg::Vec3(0.f, 0.f, 0.f),
				 osg::Vec3(0.f, 0.f, 0.f),
				 osg::Vec3(0.f, 1.f, 0.f));
  }

  if(GetModel()->GetAudio())
    osgAL::SoundManager::instance()->setListenerMatrix(scene->getViewMatrix());

  {
    PROFILE_SAMPLE("update");
    scene->update();
  }
  {
    PROFILE_SAMPLE("cull");
    scene->cull();
  }
  {
    PROFILE_SAMPLE("draw");
    scene->draw();
  }

  if(GetModel()->GetAudio()) {
    PROFILE_SAMPLE("sound");
    osgAL::SoundManager::instance()->update();
  }

  {
    PROFILE_SAMPLE("swapbuffer");
    pWin->SwapBuffers();
  }


#if 0
  if(GetModel()->GetAudio()) {
    g_debug("Sound active source %d/%d available %d",
	    osgAL::SoundManager::instance()->getNumActiveSources(), 
	    osgAL::SoundManager::instance()->getNumSources(),
	    osgAL::SoundManager::instance()->getNumAvailableSources()
	    );
  }
#endif
}

//////////////////////////////////////////////////////////////////////////////
//
// Picking intersection visitor.
//

class PickIntersectVisitor : public osgUtil::IntersectVisitor
{
public:
  PickIntersectVisitor()
  { 
  }
  virtual ~PickIntersectVisitor() {}

  HitList& getIntersections(osg::Node *scene, const osg::Vec3& nr, const osg::Vec3& fr)
  { 
    // option for non-sceneView users: you need to get the screen perp line and call getIntersections
    // if you are using Projection nodes you should also call setxy to define the xp,yp positions for use with
    // the ray transformed by Projection
    _lineSegment = new osg::LineSegment;
    _lineSegment->set(nr,fr); // make a line segment

    addLineSegment(_lineSegment.get());

    scene->accept(*this);
    return getHitList(_lineSegment.get());
  }
private:
  osg::ref_ptr<osg::LineSegment> _lineSegment;
  friend class osgUtil::IntersectVisitor;
};

// PickVisitor traverses whole scene and checks below all Projection nodes
class PickVisitor : public osg::NodeVisitor
{
public:
  PickVisitor()
    : _traverseScene(false)
  { 
    xp=yp=0;
    setTraversalMode(osg::NodeVisitor::TRAVERSE_ALL_CHILDREN);
  }
  ~PickVisitor() {}

  void setTraversalMask(osg::Node::NodeMask traversalMask)
  {
    NodeVisitor::setTraversalMask(traversalMask);
    _piv.setTraversalMask(traversalMask);   
  }

  // Aug 2003 added to pass the nodemaskOverride to the PickIntersectVisitor
  //   may be used make the visitor override the nodemask to visit invisible actions
  inline void setNodeMaskOverride(osg::Node::NodeMask mask) {
    _piv.setNodeMaskOverride(mask);
    _nodeMaskOverride = mask; }


  virtual void apply(osg::Projection& pr)
  { // stack the intersect rays, transform to new projection, traverse
    // Assumes that the Projection is an absolute projection
    if(pr.getName() == "MAFCursor")
      return;

    osg::Matrixd mt;
    mt.invert(pr.getMatrix());
    osg::Vec3 npt=osg::Vec3(xp,yp,-1.0f) * mt, farpt=osg::Vec3(xp,yp,1.0f) * mt;

    // traversing the nodes children, using the projection direction
    for (unsigned int i=0; i<pr.getNumChildren(); i++) 
      {
	osg::Node *nodech=pr.getChild(i)->asGroup()->getChild(0);
	osgUtil::IntersectVisitor::HitList &hli=_piv.getIntersections(nodech,npt, farpt);
	for(osgUtil::IntersectVisitor::HitList::iterator hitr=hli.begin();
	    hitr!=hli.end();
	    ++hitr)
	  { // add the projection hits to the scene hits.
	    // This is why _lineSegment is retained as a member of PickIntersectVisitor
	    _PIVsegHitList.push_back(*hitr);
	  }
	traverse(*nodech);
      }
  }

  osgUtil::IntersectVisitor::HitList& getHits(osg::Node *node, const osg::Vec3& near_point, const osg::Vec3& far_point)
  {
    // High level get intersection with sceneview using a ray from x,y on the screen
    // sets xp,yp as pixels scaled to a mapping of (-1,1, -1,1); needed for Projection from x,y pixels

    traverse(*node); // check for projection nodes
    // HUD has precendence over scene
    if(_PIVsegHitList.empty() && _traverseScene)
      _PIVsegHitList=_piv.getIntersections(node,near_point,far_point);

    return _PIVsegHitList;
  }

  osgUtil::IntersectVisitor::HitList& getHits(osg::Node *node, const osg::Matrixd &projm, const float x, const float y)
  {
    // utility for non=sceneview viewers
    // x,y are values returned by 
    osg::Matrixd inverseMVPW;
    inverseMVPW.invert(projm);
    osg::Vec3 near_point = osg::Vec3(x,y,-1.0f)*inverseMVPW;
    osg::Vec3 far_point = osg::Vec3(x,y,1.0f)*inverseMVPW;
    setxy(x,y);    
    getHits(node,near_point,far_point);
    return _PIVsegHitList;
  }

  inline void setxy(float xpt, float ypt) { xp=xpt; yp=ypt; }
  inline bool hits() const { return _PIVsegHitList.size()>0;}

  void	traverseScene(bool t) { _traverseScene = t; }

private:

  PickIntersectVisitor _piv;
  float xp, yp; // start point in viewport fraction coordiantes
  osgUtil::IntersectVisitor::HitList       _PIVsegHitList;
  bool	_traverseScene;
};

MAFVisionController* MAFSceneView::Pick(osgUtil::SceneView* scene, bool traverse_all, Uint16 x, Uint16 y)
{
  osg::Vec3 point_far, point_near;
  osg::Viewport* viewport = scene->getViewport();
  bool r=scene->projectWindowXYIntoObject(x, viewport->height()-y, point_near, point_far);
  if (point_near.isNaN() || point_far.isNaN() || !r)
    return 0;

  PickVisitor picker;
  picker.traverseScene(traverse_all);
  picker.setNodeMaskOverride(0);
  picker.setTraversalMask(MAF_COLLISION_MASK);
  picker.setxy((float(x) / viewport->width()) * 2 - 1, (float(viewport->height() - y) / viewport->height()) * 2 - 1);
  osgUtil::IntersectVisitor::HitList& hits = picker.getHits(scene->getSceneData(), point_near, point_far);

  MAFVisionController* controller = 0;
  if(hits.size() > 0 ) {

    float ratio = 1.f;
    unsigned int hit_index = 0;
    unsigned int closest_hit_index = 0;
    for(unsigned int i = 0; i < hits.size(); i++) {
      if (hits[i]._ratio < ratio) {

	// check if the ray intersect in frontface of ray direction
	osg::Vec3 dirOfRay=hits[i]._localLineSegment->end()-hits[i]._localLineSegment->start();
	if (dirOfRay*hits[i]._intersectNormal>=0)
	  continue;

	
	for(osg::NodePath::iterator j = hits[i]._nodePath.begin();j != hits[i]._nodePath.end();j++) {
	  osg::Node* node = *j;
	  
	  controller = dynamic_cast<MAFVisionController*>(node->getUserData());
	  ratio = hits[i]._ratio;
	  hit_index = i;

	  if(controller)
	    break;
	}

	closest_hit_index = i;
      }
    }

    {
      std::string sub_path;
      std::string path;
      for(osg::NodePath::iterator i = hits[closest_hit_index]._nodePath.begin();
	  i != hits[closest_hit_index]._nodePath.end();
	  i++) {
	path += "/" + (*i)->getName();
      }
      osg::Drawable* drawable = hits[closest_hit_index]._drawable.get();
      if(dynamic_cast<osgCal::SubMeshSoftware*>(drawable)) {
	osgCal::SubMeshSoftware* submesh = dynamic_cast<osgCal::SubMeshSoftware*>(drawable);
	path += "/" + submesh->getName();
      }

      MAFVisionController* found = 0;

      MAFSceneModel* model = GetModel();
      //g_debug("MAFSceneView::Pick: %s", path.c_str());
      for(MAFSceneModel::PickCache::iterator i = model->mPickCache.begin();
	  i != model->mPickCache.end();
	  i++) {
	MAFSceneModel::PickPair& pair = *i;
	if(pair.first == path) {
	  found = pair.second.second.get();
	  sub_path = pair.second.first;
	  if(i != model->mPickCache.begin()) {
	    //
	    // Move cache hit to the front of the cache
	    //
	    MAFSceneModel::PickPair tmp = pair;
	    model->mPickCache.erase(i);
	    model->mPickCache.push_front(tmp);
	    // g_debug("MAFSceneView::Pick: cache hit");
	    break;
	  }
	}
      }

      if(!found) {
	for(MAFSceneModel::Path2VisionController::iterator i = model->mPath2VisionController.begin();
	    i != model->mPath2VisionController.end();
	    i++) {
	  if(path.find(i->first) != std::string::npos) {
	    found = i->second.get();
	    sub_path = i->first;
	    model->mPickCache.push_front(MAFSceneModel::
					 PickPair(path,
						  MAFSceneModel::
						  ControllerPair(i->first,
								 found)));
	    if(model->mPickCache.size() >= model->mPickCacheSize)
	      model->mPickCache.pop_back();
	    // g_debug("MAFSceneView::Pick: matches %s", i->first.c_str());
	    break;
	  }
	}
      }

      MAFVisionController*	last_picked = GetModel()->mLastPicked;
      if(found)
	{
	  if (found != last_picked && last_picked)
	    last_picked->HandleNotHit();
	  MAFHit hit(hits[closest_hit_index], x, y);
	  hit.mPath = sub_path;
	  found->HandleHit(hit);
	  GetModel()->mLastPicked = found;
	}
      else
	if (last_picked)
	  last_picked->HandleNotHit(), GetModel()->mLastPicked = 0;
    }

    MAFVisionController*	last_hit = GetModel()->mLastHit;
    if (last_hit && last_hit != controller)
      {
	last_hit->HandleNotHit();
	GetModel()->mLastHit = 0;
      }
    
    if(controller) {
      MAFHit hit(hits[hit_index], x, y);
      controller->HandleHit(hit);
      GetModel()->mLastHit = controller;
    }
  }
  
  return controller;
}

// Controller

bool MAFSceneController::Update(MAFApplication* application)
{
  return true;
}

void MAFSceneController::DoIntersection(MAFApplication* application, int x, int y)
{
  SDL_Event*	event = application->GetLastEventIgnoreLocking();

  bool	traverse_all = event && event->type == SDL_MOUSEMOTION;
  traverse_all=true;
  MAFVisionController* hit = GetView()->Pick(GetModel()->mScene.get(),traverse_all, x, y);

  if (hit)
    application->SetFocus(hit);
  else if (traverse_all)
    application->SetFocus(0);
}

void MAFSceneController::Init(void)
{
  if(GetModel() == 0)
    SetModel(new MAFSceneModel);
  if(GetView() == 0)
    SetView(new MAFSceneView);
  MAFController::Init();
}

void MAFSceneController::Insert( MAFVisionController* controller )
{
  MAFSceneModel* model = GetModel();
  model->mGroup->addChild(controller->GetModel()->GetNode());
}

void MAFSceneController::Remove( MAFVisionController* controller )
{
  MAFSceneModel* model = GetModel();
  model->mGroup->removeChild(controller->GetModel()->GetNode());
}

void MAFSceneController::HUDInsert(MAFVisionController* controller)
{
  MAFSceneModel* model = GetModel();
  osg::Node* node = controller->GetModel()->GetNode();
  model->mHUDGroup->addChild(node);
}

void MAFSceneController::HUDRemove(MAFVisionController* controller)
{
  MAFSceneModel* model = GetModel();
  model->mHUDGroup->removeChild(controller->GetModel()->GetNode());
}

void MAFSceneController::RegisterPickCallback(const std::string& path, MAFVisionController* controller)
{
  MAFSceneModel* model = GetModel();
  model->mPath2VisionController[path] = controller;
  model->mPickCache.clear();
}

ALuint MAFSceneController::AllocateBuffer() {
  MAFSceneModel* model = GetModel();
  return model->mBuffers[model->mBufferIndex++];
}

ALuint MAFSceneController::AllocateSource() {
  MAFSceneModel* model = GetModel();
  return model->mSources[model->mSourceIndex++];
}
