/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *  Vincent Caron <zerodeux@gnu.org>
 *
 */
/*
 *
 * Copyright (C) Nicolas Roussel
 *
 * See the file LICENSE for information on usage and redistribution of
 * this file, and for a DISCLAIMER OF ALL WARRANTIES.
 * Note: LICENSE is LGPL
 *
 */

#include "mafStdAfx.h"

#ifndef MAF_USE_VS_PCH

#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif
#ifdef WIN32
#include "config_win32.h"
#endif

#include <maf/wnc_conversion.h>

#endif

static inline void	ARGB2RGB(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst)
{
  src++ ;
  for (unsigned int i= w * h; i; i--)
    {
      memcpy(dst, src, 3) ;
      src += 4 ;
      dst += 3 ;
    }
}

static inline void	ABGR2RGB(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst)
{
  for (unsigned int i=w*h; i; i--)
    {
      *dst++ = src[3] ;
      *dst++ = src[2] ;
      *dst++ = src[1] ;
      src += 4 ;
    }
}

static inline void	RGBA2RGB(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst)
{
  for (unsigned int i=w*h; i; i--)
    {
      memcpy(dst, src, 3) ;
      src += 4 ;
      dst += 3 ;
    }
}


static inline void	RGB2L(unsigned int w, unsigned int h,
			      unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; i--) {
    double r=src[0], g=src[1], b=src[2] ;
    *dst++ = (unsigned char)(0.11*b + 0.59*g + 0.3*r) ;
    src += 3 ;
  }
}

static inline void	RGB2ARGB(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst,
				 unsigned char alpha) {
  for (unsigned int i=w*h; i; i--) {
    *dst++ = alpha ;
    memcpy(dst, src, 3) ;
    dst += 3 ;
    src += 3 ;
  }
}

static inline void	RGB2RGBA(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst,
				 unsigned char alpha) {
  for (unsigned int i=w*h; i; i--) {
    memcpy(dst, src, 3) ;
    src += 3 ;
    dst += 3 ;
    *dst++ = alpha ;
  }
}

static inline void	RGB2ABGR(unsigned int w, unsigned int h,
				 unsigned char *src, unsigned char *dst,
				 unsigned char alpha) {
  for (unsigned int i=w*h; i; i--) {
    *dst++ = alpha ;  // A
    *dst++ = src[2] ; // B
    *dst++ = src[1] ; // G
    *dst++ = src[0] ; // R
    src+=3 ;
  }
}

// ABGR2RGBA and RGBA2ABGR
static inline void	ABGR_RGBA(unsigned int w, unsigned int h,
				  unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; i--) {
    *dst++ = src[3] ;
    *dst++ = src[2] ;
    *dst++ = src[1] ;
    *dst++ = src[0] ;
    src += 4 ;
  }
}

static inline void	RGBA2ARGB(unsigned int w, unsigned int h,
				  unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; i--) {
    *dst++ = src[3] ;
    memcpy(dst, src, 3) ;
    dst += 3 ;
    src += 4 ;
  }
}

static inline void	ARGB2RGBA(unsigned int w, unsigned int h,
				  unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; i--) {
    dst[3] = *src ;
    memcpy(dst, src+1, 3) ;
    dst += 4 ;
    src += 4 ;
  }
}

static inline void	ARGB2L(unsigned int w, unsigned int h,
			       unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; i--) {
    double r=src[1], g=src[2], b=src[3] ;
    *dst++ = (unsigned char)(0.11*b + 0.59*g + 0.3*r) ;
    src += 4 ;
  }
}

static inline void	L2RGB(unsigned int w, unsigned int h,
			      unsigned char *src, unsigned char *dst) {
  for (unsigned int i=w*h; i; --i) {
    memset(dst, *src, 3) ;
    src++ ;
    dst += 3 ;
  }
}

void	RGB2YpCbCr420(WncImage *img) {
  unsigned int src_w=img->getWidth(), src_h=img->getHeight() ;
  unsigned int dst_w=src_w&0xFFF0, dst_h=src_h&0xFFF0 ;
  unsigned int dst_nbpixels = dst_w*dst_h ;
  unsigned int dst_size = (int)(dst_nbpixels*1.5) ;

  unsigned char *src = (unsigned char *)img->getData() ;
  unsigned char *dst = (unsigned char *)WncImage::AllocMem(dst_size) ;

  unsigned char *cbplane = (unsigned char *)WncImage::AllocMem(dst_size) ;
  unsigned char *crplane = (unsigned char *)WncImage::AllocMem(dst_size) ;

  unsigned char *ypplane = dst ;
  unsigned char *cbptr = cbplane ;
  unsigned char *crptr = crplane ;

  for (unsigned int y=0; y<dst_h; ++y) {
    unsigned char *ptr = src + (y*src_w*3) ;
    for (unsigned int x=0; x<dst_w; ++x) {
      unsigned char Rp=*ptr++ ;
      unsigned char Gp=*ptr++ ;
      unsigned char Bp=*ptr++ ;	   
      *ypplane++ = (unsigned char)(16 + (65.738*Rp + 129.057*Gp +25.064*Bp)/256.0) ;
      *cbptr++ = (unsigned char)(128 + (-37.945*Rp - 74.494*Gp + 112.439*Bp)/256.0) ;
      *crptr++ = (unsigned char)(128 + (112.439*Rp - 94.154*Gp - 18.285*Bp)/256.0) ;
    }
  }

  cbptr = dst + dst_nbpixels ;
  crptr = dst + (int)(dst_nbpixels*1.25) ;

  for (unsigned int y=0; y<dst_h;) {
    for (unsigned int x=0; x<dst_w;) {
      int cb1 = cbplane[y*dst_w+x] ;
      int cb2 = cbplane[y*dst_w+x+1] ;
      int cb3 = cbplane[(y+1)*dst_w+x] ;
      int cb4 = cbplane[(y+1)*dst_w+x+1] ;
      *cbptr++ = (unsigned char)((cb1+cb2+cb3+cb4)/4) ;

      int cr1 = crplane[y*dst_w+x] ;
      int cr2 = crplane[y*dst_w+x+1] ;
      int cr3 = crplane[(y+1)*dst_w+x] ;
      int cr4 = crplane[(y+1)*dst_w+x+1] ;
      *crptr++ = (unsigned char)((cr1+cr2+cr3+cr4)/4) ;

      x += 2 ;
    }
    y += 2 ;
  }

  WncImage::FreeMem(&cbplane) ;
  WncImage::FreeMem(&crplane) ;

  img->setEncoding(WncImage::YpCbCr420) ;
  img->setData(dst, dst_size, WncImage::FREEMEM) ;
  img->setDims(dst_w, dst_h) ;
}

// ------------------------------------------------------------------

/*
 * Turn a YUV4:2:0 block into an RGB block (this code comes from
 * ov511.c) 
 *
 * Color space conversion coefficients taken from the excellent
 * http://www.inforamp.net/~poynton/ColorFAQ.html
 * In his terminology, this is a CCIR 601.1 YCbCr -> RGB.
 * Y values are given for all 4 pixels, but the U (Pb)
 * and V (Pr) are assumed constant over the 2x2 block.
 *
 * To avoid floating point arithmetic, the color conversion
 * coefficients are scaled into 16.16 fixed-point integers.
 * They were determined as follows:
 *
 *	double brightness = 1.0;  (0->black; 1->full scale) 
 *	double saturation = 1.0;  (0->greyscale; 1->full color)
 *	double fixScale = brightness * 256 * 256;
 *	int rvScale = (int)(1.402 * saturation * fixScale);
 *	int guScale = (int)(-0.344136 * saturation * fixScale);
 *	int gvScale = (int)(-0.714136 * saturation * fixScale);
 *	int buScale = (int)(1.772 * saturation * fixScale);
 * int yScale = (int)(fixScale);
 */

/* LIMIT: convert a 16.16 fixed-point value to a byte, with clipping. */
#define LIMIT(x) ((x)>0xffffff?0xff: ((x)<=0xffff?0:((x)>>16)))

void	YpCbCr4202RGB(WncImage *img)
{
  const unsigned int w=img->getWidth(), h=img->getHeight() ;
  const int numpix = w*h ;

  unsigned char *pIn0 = img->getData() ;
  unsigned int size = numpix*3 ;
  unsigned char *buffer = (unsigned char *)WncImage::AllocMem(size) ;

  unsigned char *pY = pIn0 ;
  unsigned char *pU = pY + numpix ;
  unsigned char *pV = pU + numpix / 4 ;
  unsigned char *pOut = buffer ;
  const int skip = 3 * w ;

  const int rvScale = 91881 ;
  const int guScale = -22553 ;
  const int gvScale = -46801 ;
  const int buScale = 116129 ;
  const int yScale  = 65536 ;

  for (unsigned int j = 0; j <= h - 2; j += 2) {
    for (unsigned int i = 0; i <= w - 2; i += 2) {
      int yTL=(*pY)*yScale, yTR=(*(pY + 1))*yScale ;
      int yBL=(*(pY + w))*yScale, yBR=(*(pY + w + 1))*yScale ;

      int u = (*pU++) - 128 ;
      int v = (*pV++) - 128 ;
      int g = guScale * u + gvScale * v;
      int r = buScale * u;
      int b = rvScale * v;

      // Write out top two pixels
      pOut[0] = LIMIT(b+yTL); pOut[1] = LIMIT(g+yTL); pOut[2] = LIMIT(r+yTL) ;
      pOut[3] = LIMIT(b+yTR); pOut[4] = LIMIT(g+yTR); pOut[5] = LIMIT(r+yTR) ;

      // Skip down to next line to write out bottom two pixels
      pOut[skip+0] = LIMIT(b+yBL); pOut[skip+1] = LIMIT(g+yBL); pOut[skip+2] = LIMIT(r+yBL) ;
      pOut[skip+3] = LIMIT(b+yBR); pOut[skip+4] = LIMIT(g+yBR); pOut[skip+5] = LIMIT(r+yBR) ;

      pY += 2;
      pOut += 2 * 3 ; // 3 bytes
    }
    pY += w ;
    pOut += w * 3 ; // 3 bytes
  }

  img->setEncoding(WncImage::RGB) ;
  img->setData(buffer, size, WncImage::FREEMEM) ;
}

// ------------------------------------------------------------------

bool	convertImage(WncImage *src,
		     WncImage *dst, WncImage::Encoding dst_encoding,
		     int quality, unsigned char alpha) {
  WncImage::Encoding src_encoding = src->getEncoding() ;

  if (src_encoding==dst_encoding || dst_encoding==WncImage::PREFERRED) {
    *dst = *src ;
    return true ;
  }

  unsigned int width = src->getWidth() ;
  unsigned int height = src->getHeight() ;
    
  bool ok = true ;

  switch (src_encoding) {

  case WncImage::RGB:
    switch (dst_encoding) {
    case WncImage::ARGB:
      dst->prepareFor(width, height, dst_encoding) ;
      RGB2ARGB(width, height, src->getData(), dst->getData(),alpha) ;
      break ;
    case WncImage::RGBA:
      dst->prepareFor(width, height, dst_encoding) ;
      RGB2RGBA(width, height, src->getData(), dst->getData(),alpha) ;
      break ;
    case WncImage::ABGR:
      dst->prepareFor(width, height, dst_encoding) ;
      RGB2ABGR(width, height, src->getData(), dst->getData(),alpha) ;
      break ;
    case WncImage::L:
      dst->prepareFor(width, height, dst_encoding) ;
      RGB2L(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::YpCbCr420:
      dst->linkDataFrom(*src) ;
      RGB2YpCbCr420(dst) ;
      break ;
    default: ok = false ;
    }
    break ;

  case WncImage::ABGR:
    switch (dst_encoding) {
    case WncImage::RGB:
      dst->prepareFor(width, height, dst_encoding) ;
      ABGR2RGB(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::RGBA:
      dst->prepareFor(width, height, dst_encoding) ;
      ABGR_RGBA(width, height, src->getData(), dst->getData()) ;
      break ;
    default: ok = false ;
    }
    break ;

  case WncImage::ARGB:
    switch (dst_encoding) {
    case WncImage::L:
      dst->prepareFor(width, height, dst_encoding) ;
      ARGB2L(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::RGB:
      dst->prepareFor(width, height, dst_encoding) ;
      ARGB2RGB(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::RGBA:
      dst->prepareFor(width, height, dst_encoding) ;
      ARGB2RGBA(width, height, src->getData(), dst->getData()) ;
      break ;
    default: ok = false ;
    }
    break ;

  case WncImage::RGBA:
    switch (dst_encoding) {
    case WncImage::RGB:
      dst->prepareFor(width, height, dst_encoding) ;
      RGBA2RGB(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::ABGR:
      dst->prepareFor(width, height, dst_encoding) ;
      ABGR_RGBA(width, height, src->getData(), dst->getData()) ;
      break ;
    case WncImage::ARGB:
      dst->prepareFor(width, height, dst_encoding) ;
      RGBA2ARGB(width, height, src->getData(), dst->getData()) ;
      break ;
    default: ok = false ;
    }
    break ;

  case WncImage::L:
    switch (dst_encoding) {
    case WncImage::RGB:
      dst->prepareFor(width, height, dst_encoding) ;
      L2RGB(width, height, src->getData(), dst->getData()) ;
      break ;
    default: ok = false ;
    }
    break ;
	 
  default: ok = false ;
  
 case WncImage::YpCbCr420:
   switch (dst_encoding) {
   case WncImage::RGB:
     dst->linkDataFrom(*src) ;
     YpCbCr4202RGB(dst) ;
     break ;
   case WncImage::RGBA: {
     WncImage tmp ;
     tmp.linkDataFrom(*src) ;
     YpCbCr4202RGB(&tmp) ;
     dst->prepareFor(width, height, dst_encoding) ;
     RGB2RGBA(tmp.getWidth(), tmp.getHeight(), tmp.getData(), dst->getData(), alpha) ;
   } break ;
   case WncImage::L: {
     WncImage tmp ;
     tmp.linkDataFrom(*src) ;
     YpCbCr4202RGB(&tmp) ;
     dst->prepareFor(width, height, dst_encoding) ;
     RGB2L(tmp.getWidth(), tmp.getHeight(), tmp.getData(), dst->getData()) ;
   } break ;
#ifdef HAVE_JPEG
   case WncImage::JPEG: {
     WncImage tmp ;
     tmp.linkDataFrom(*src) ;
     YpCbCr4202RGB(&tmp) ;
     ok = JpegEncode(&tmp, dst, quality) ;
   } break ;
#endif
   default: ok = false ;
   }
   break ;
  }

  if (ok)
    return true ;
  else
   return false ;
}

  // ------------------------------------------------------------------

bool	convertImage(WncImage *img,
		     WncImage::Encoding dst_encoding,
		     int quality, unsigned char alpha) {
  if (img->getEncoding()==dst_encoding) {
    return true ;
  }

  WncImage dst ;
  bool res = convertImage(img, &dst, dst_encoding, quality, alpha) ;
  if (res) img->stealDataFrom(dst) ;
  return res ;
}

