/* -*- c++ -*-
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 *  Loic Dachary <loic@gnu.org>
 *
 */

//
// This is an OpenSceneGraph IO plugin, see the betslider namespace
// documentation for a usage description. In order to work as
// expected, this code must be available using one of these three
// methods:
//
// 1) It is dynamicaly or statically linked to the executable.  
//
// 2) It is part of a dynamically loadable library named
// libosgdb_betslider.so (or something similar depending on the
// operating system) and located in the plugin directory of
// OpenSceneGraph (usually /usr/lib/osgPlugins on GNU/Linux). It will
// be loaded when osgDB::ReadNodeFile or osgDB::WriteNodeFile save a
// file using the OSG format and run into an osg::Node who claims to
// belong to the betslider library (as returned by the libraryName()
// method).
//
// 3) It is part of dynamically loadable library named libbetslider.so
// (or something similar depending on the operating system) and
// located in the directories searched by the operating system
// functions (LD_LIBRARY_PATH on GNU/Linux for instance). It will be
// loaded when osgDB::WriteNodeFile tries to save an osg::Node who
// claims to belong to the betslider library (as returned by the
// libraryName() method) or when osgDB::ReadNodFile is provided with
// a filename whose extension is .betslider.
//
// In all cases this plugin relies on static constructors and
// destructors to hook themselves to the osgDB::Registry instance (all
// objects whose name starts with static_). When the library is loaded
// the constructors are called and register pointers to the code in 
// the registry. When the library is unloaded, the destructors are called
// and deregister the pointers.
//

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif // HAVE_CONFIG_H

#include "ugameStdAfx.h"

#ifndef DATADIR
#define DATADIR "."
#endif // DATADIR

#include <ugame/ugameexport.h>
#ifndef WIN32
#include <ugame/BetSlider>
#endif // WIN32

#include <osgDB/FileNameUtils>
#include <osgDB/FileUtils>
#include <osgDB/Registry>
#include <osgDB/Input>
#include <osgDB/Output>
#include <osgDB/ReadFile>

//
// OSG interface to read/write from/to a file.
//
class ReaderWriterbetslider : public osgDB::ReaderWriter {
public:
  virtual const char* className() { return "betslider object reader"; }

  virtual bool acceptsExtension(const std::string& extension) const {
    return osgDB::equalCaseInsensitive(extension, "betslider");
  }

  virtual ReadResult readNode(const std::string& fileName, const Options* options = NULL) const {
    std::string ext = osgDB::getLowerCaseFileExtension(fileName);
    if (!acceptsExtension(ext)) return ReadResult::FILE_NOT_HANDLED;

    osgDB::FilePathList* filePathList = 0;
    if(options) {
      filePathList = const_cast<osgDB::FilePathList*>(&(options->getDatabasePathList()));
      filePathList->push_back(DATADIR);
    }

    std::string path = osgDB::findDataFile(fileName);
    if(path.empty()) return ReadResult::FILE_NOT_FOUND;

    osg::ref_ptr<betslider::BetSlider> slider = new betslider::BetSlider;
    if(!slider->unserialize(path, options))
      return ReadResult("failed to load " + path);
    else
      return ReadResult(slider.get());

    if(options && filePathList)
      filePathList->pop_back();
  }

  virtual WriteResult writeNode(const osg::Node& node, const std::string& fileName, const Options* options = NULL) const {
    std::string ext = osgDB::getLowerCaseFileExtension(fileName);
    if (!acceptsExtension(ext)) return WriteResult::FILE_NOT_HANDLED;

    if(!static_cast<const betslider::BetSlider&>(node).serialize(fileName))
      return WriteResult("failed to save " + node.getName() + " in file " + fileName);
    else
      return WriteResult(WriteResult::FILE_SAVED);
  }
};

static osgDB::RegisterReaderWriterProxy<ReaderWriterbetslider> static_readerWriter_betslider_Proxy;

//
// OSG interface for inclusion within a .osg file
//
static bool readLocalData(osg::Object &obj, osgDB::Input &fr)
{
  return true;
}

static bool writeLocalData(const osg::Object &obj, osgDB::Output &fw)
{
  return true;
}

static osgDB::RegisterDotOsgWrapperProxy static_BetSlider(
    new betslider::BetSlider,
    "BetSlider",
    "Group",
    readLocalData,
    writeLocalData
);
