/*
 *
 * Copyright (C) 2004 Mekensleep
 *
 *	Mekensleep
 *	24 rue vieille du temple
 *	75004 Paris
 *       licensing@mekensleep.com
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Authors:
 * Henry Precheur	<henry@precheur.org>
 *
 */

#include "ugameStdAfx.h"

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif /* HAVE_CONFIG_H */
#ifdef WIN32
#include "config_win32.h"
#endif

#include <osg/Array>
#include <osg/Geode>
#include <osg/Geometry>
#include <osg/BlendFunc> // FIXME may be unused
#include <osg/AlphaFunc> // FIXME may be unused
#include <osg/PolygonMode>
#include <osg/PolygonOffset>

#include "maf/maferror.h"
#include "ugame/text.h"

static const float	__tt_size = 8.0f;

static const osg::Vec3	vertices[16] =
  {
    osg::Vec3(-1 * __tt_size,	-1 * __tt_size,	0), // ----
    osg::Vec3(0 * __tt_size,	-1 * __tt_size,	0), // TOP
    osg::Vec3(0 * __tt_size,	-1 * __tt_size,	0), // TOP
    osg::Vec3(1 * __tt_size,	-1 * __tt_size,	0), // ----
    
    osg::Vec3(-1 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(0 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(0 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(1 * __tt_size,	0 * __tt_size,	0),
    
    osg::Vec3(-1 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(0 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(0 * __tt_size,	0 * __tt_size,	0),
    osg::Vec3(1 * __tt_size,	0 * __tt_size,	0),
    
    osg::Vec3(-1 * __tt_size,	1 * __tt_size,	0), // ----
    osg::Vec3(0 * __tt_size,	1 * __tt_size,	0), // BOTTOM
    osg::Vec3(0 * __tt_size,	1 * __tt_size,	0), // BOTTOM
    osg::Vec3(1 * __tt_size,	1 * __tt_size,	0), // ----
  };

static const osg::Vec3*	vertices_begin = vertices;
static const osg::Vec3*	vertices_end = vertices + 16;

UGAMEBasicText::UGAMEBasicText(const std::string& str, osgText::Font* font)
{
  osgText::Text*	text = new osgText::Text;
  text->setFont(font);
  text->setCharacterSize(16); // FIXME
  text->setPosition(osg::Vec3(0, 0, 0)); // FIXME uninitialized in
					 // contructor
  osg::StateSet*	state = text->getOrCreateStateSet();
  state->setRenderingHint(osg::StateSet::TRANSPARENT_BIN);
  state->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
  getOrCreateStateSet()->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
  mText = text;
  setStringUTF8(str);
  addDrawable(mText.get());
}

osgText::Text*	UGAMEBasicText::getText()
{
  return mText.get();
}

const osgText::Text*	UGAMEBasicText::getText() const
{
  return mText.get();
}

void	UGAMEBasicText::setStringUTF8(const std::string& str)
{
  mText->setText(osgText::String(str, osgText::String::ENCODING_UTF8));
}

std::string	UGAMEBasicText::getStringUTF8() const
{
  return mText->getText().createUTF8EncodedString();
}

UGAMEFramedText::UGAMEFramedText(const std::string& str, osg::Image* img, osgText::Font* font)
  : UGAMEBasicText(str, font)
{
  g_assert(img);
  mText->setAlignment(osgText::Text::LEFT_BOTTOM);

  osg::Texture2D*	tex = new osg::Texture2D(img);
  // FIXME to avoid mipmaping problem
  tex->setFilter(osg::Texture::MIN_FILTER, osg::Texture::NEAREST);
  tex->setFilter(osg::Texture::MAG_FILTER, osg::Texture::NEAREST);

  mGeom = new osg::Geometry;
  mGeom->setUseVertexBufferObjects(false);
  mGeom->setUseDisplayList(false);

  static const osg::Vec2	tex_coord[16] =
    {
      osg::Vec2(0,	0),
      osg::Vec2(0.25,	0),
      osg::Vec2(0.75,	0),
      osg::Vec2(1.0,	0),
      
      osg::Vec2(0,	0.25),
      osg::Vec2(0.25,	0.25),
      osg::Vec2(0.75,	0.25),
      osg::Vec2(1.0,	0.25),

      osg::Vec2(0,	0.75),
      osg::Vec2(0.25,	0.75),
      osg::Vec2(0.75,	0.75),
      osg::Vec2(1.0,	0.75),

      osg::Vec2(0,	1.0),
      osg::Vec2(0.25,	1.0),
      osg::Vec2(0.75,	1.0),
      osg::Vec2(1.0,	1.0)
    };

  mVertice = new osg::Vec3Array(vertices_begin, vertices_end);
  osg::Vec2Array*	tex_coord_array = new osg::Vec2Array(tex_coord, tex_coord + 16);

  mGeom->setVertexArray(mVertice.get());
  mGeom->setTexCoordArray(0, tex_coord_array);

  osg::StateSet*	stateset = mGeom->getOrCreateStateSet();
  stateset->setMode(GL_LIGHTING, osg::StateAttribute::OFF);
//   stateset->setAttributeAndModes(new osg::AlphaFunc(osg::AlphaFunc::GEQUAL, 0.5), osg::StateAttribute::ON);
  stateset->setAttributeAndModes(new osg::BlendFunc(osg::BlendFunc::SRC_ALPHA,
						    osg::BlendFunc::ONE_MINUS_SRC_ALPHA),
				 osg::StateAttribute::ON);
  stateset->setAttributeAndModes(new osg::PolygonOffset(-1.0, 4.0), osg::StateAttribute::ON);

  stateset->setTextureAttributeAndModes(0, tex, osg::StateAttribute::ON);

  mGeom->setStateSet(stateset);

  static unsigned int	tri_strip_left[8] =
    {
      0, 1, 4, 5, 8, 9, 12, 13
    };

  static unsigned int	tri_strip_middle[8] =
    {
      1, 2, 5, 6, 9, 10, 13, 14
    };

  static unsigned int	tri_strip_right[8] =
    {
      2, 3, 6, 7, 10, 11, 14, 15
    };

   mGeom->addPrimitiveSet
    (new osg::DrawElementsUInt(osg::PrimitiveSet::TRIANGLE_STRIP,
			       8, tri_strip_left));
  mGeom->addPrimitiveSet
    (new osg::DrawElementsUInt(osg::PrimitiveSet::TRIANGLE_STRIP,
			       8, tri_strip_middle));
  mGeom->addPrimitiveSet
    (new osg::DrawElementsUInt(osg::PrimitiveSet::TRIANGLE_STRIP,
			       8, tri_strip_right));

  addDrawable(mGeom.get());
  setStringUTF8(str); // FIXME
}

static osg::Vec3Array*	compute_x_y_len(osg::Vec3Array*	array, float x_len, float y_len)
{
  static const osg::Vec2	scale[16] =
    {
      osg::Vec2(0, 0), // ----
      osg::Vec2(0, 0), // TOP
      osg::Vec2(1, 0), // TOP
      osg::Vec2(1, 0), // ----

      osg::Vec2(0, 0), // ----
      osg::Vec2(0, 0), // MIDDLE TOP
      osg::Vec2(1, 0), // MIDDLE TOP
      osg::Vec2(1, 0), // ----

      osg::Vec2(0, 1), // ----
      osg::Vec2(0, 1), // MIDDLE BOTTOM
      osg::Vec2(1, 1), // MIDDLE BOTTOM
      osg::Vec2(1, 1), // ----

      osg::Vec2(0, 1), // ----
      osg::Vec2(0, 1), // BOTTOM
      osg::Vec2(1, 1), // BOTTOM
      osg::Vec2(1, 1) // ----
    };

  for (unsigned u = 0; u < 16; u++)
    {
      (*array)[u].x() = vertices[u].x() + scale[u].x() * x_len;
      (*array)[u].y() = vertices[u].y() + scale[u].y() * y_len;
    }
  return array;
}

void	UGAMEFramedText::setStringUTF8(const std::string& str)
{
  UGAMEBasicText::setStringUTF8(str);
  const osg::BoundingBox&	bbox = mText->getBound();

  float	x_len = bbox.xMax() - bbox.xMin();
  float	y_len = bbox.yMax() - bbox.yMin();

  compute_x_y_len(mVertice.get(), x_len, y_len);
}

osg::Geometry*	UGAMEFramedText::getGeometry()
{
  return mGeom.get();
}

const osg::Geometry*	UGAMEFramedText::getGeometry() const
{
  return mGeom.get();
}


/*
 * used by UGAMEPlaceTextInHud
 */
static osg::Vec3	align2position(enum osgText::Text::AlignmentType align)
{
  osg::Vec3	v;

  switch (align)
    {
    case osgText::Text::LEFT_BOTTOM:
    case osgText::Text::LEFT_BASE_LINE:
      v = osg::Vec3f(0.0f, 0.0f, 0.0f);
      break;
    case osgText::Text::LEFT_CENTER:
      v = osg::Vec3f(0.0f, 0.5f, 0.0f);
      break;
    case osgText::Text::LEFT_TOP:
      v = osg::Vec3f(0.0f, 1.0f, 0.0f);
      break;
    case osgText::Text::CENTER_BOTTOM:
    case osgText::Text::CENTER_BASE_LINE:
      v = osg::Vec3f(0.5f, 0.0f, 0.0f);
      break;
    case osgText::Text::CENTER_CENTER:
      v = osg::Vec3f(0.5f, 0.5f, 0.0f); 
      break;
   case osgText::Text::CENTER_TOP:
      v = osg::Vec3f(0.5f, 1.0f, 0.0f);
      break;
    case osgText::Text::RIGHT_BOTTOM:
    case osgText::Text::RIGHT_BASE_LINE:
      v = osg::Vec3f(1.0f, 0.0f, 0.0f);
      break;
    case osgText::Text::RIGHT_CENTER:
      v = osg::Vec3f(1.0f, 0.5f, 0.0f);
      break;
    case osgText::Text::RIGHT_TOP:
      v = osg::Vec3f(1.0f, 1.0f, 0.0f);
      break;
    }
  return v;
}

void	UGAMEPlaceTextInHud(osgText::Text* text, enum osgText::Text::AlignmentType align,
			  unsigned screen_width, unsigned screen_height)
{
  text->setAlignment(align);
  osg::Vec3	v = align2position(align);
  g_assert(v == osg::Vec3(0, 0, 0));
  v.x() *= screen_width;
  v.y() *= screen_height;
  text->setPosition(v);
}
