/************************************************************************
 *
 * newrole
 *
 * SYNOPSIS:
 *
 * This program allows a user to change their SELinux RBAC role and/or
 * SELinux TE type (domain) in a manner similar to the way the traditional
 * UNIX su program allows a user to change their identity.
 *
 * USAGE:
 *
 * newrole [ -r role ] [ -t type ] [ -l level ] [ -- args ]
 *
 * BUILD OPTIONS:
 *
 * option USE_PAM:
 *
 * Set the USE_PAM constant if you want to authenticate users via PAM.
 * If USE_PAM is not set, users will be authenticated via direct
 * access to the shadow password file.
 *
 * If you decide to use PAM must be told how to handle newrole.  A
 * good rule-of-thumb might be to tell PAM to handle newrole in the
 * same way it handles su, except that you should remove the pam_rootok.so
 * entry so that even root must re-authenticate to change roles. 
 *
 * If you choose not to use PAM, make sure you have a shadow passwd file
 * in /etc/shadow.  You can use a symlink if your shadow passwd file
 * lives in another directory.  Example:
 *   su
 *   cd /etc
 *   ln -s /etc/auth/shadow shadow
 *
 * If you decide not to use PAM, you will also have to make newrole
 * setuid root, so that it can read the shadow passwd file.
 * 
 *
 * option CANTSPELLGDB:
 *
 * If you set CANTSPELLGDB you will turn on some debugging printfs.
 *
 *
 * Authors:  Tim Fraser , 
 *           Anthony Colatrella <amcolat@epoch.ncsc.mil>
 * Various bug fixes by Stephen Smalley <sds@epoch.ncsc.mil>
 *
 *************************************************************************/

#include <stdio.h>
#include <stdlib.h>               /* for malloc(), realloc(), free() */
#include <pwd.h>                  /* for getpwuid() */
#include <sys/types.h>            /* to make getuid() and getpwuid() happy */
#include <sys/wait.h>		  /* for wait() */
#include <getopt.h>               /* for getopt_long() form of getopt() */
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <selinux/selinux.h>      /* for is_selinux_enabled() */
#include <selinux/flask.h>        /* for SECCLASS_CHR_FILE */
#include <selinux/context.h>      /* for context-mangling functions */
#include <selinux/get_default_type.h>
#include <selinux/get_context_list.h> /* for SELINUX_DEFAULTUSER */
#include <signal.h>
#ifdef USE_NLS
#include <locale.h>			    /* for setlocale() */
#include <libintl.h>			    /* for gettext() */
#define _(msgid) gettext (msgid)
#else
#define _(msgid) (msgid)
#endif
#ifndef PACKAGE
#define PACKAGE "policycoreutils"   /* the name of this package lang translation */
#endif

/* USAGE_STRING describes the command-line args of this program. */
#define USAGE_STRING "USAGE: newrole [ -r role ] [ -t type ] [ -l level ] [ -- args ]"

#define DEFAULT_CONTEXT_SIZE 255  /* first guess at context size */

char *xstrdup(const char *s)
{
  char *s2;

  s2 = strdup(s);
  if (!s2) {
	  fprintf(stderr, _("Out of memory!\n"));
	  exit(1);
  }
  return s2;
}

static char *
build_new_range(char *newlevel, const char *range)
{
  char *newrangep = NULL, *tmpptr;
  size_t len;

  /* a missing or empty string */
  if (!range || !strlen(range) || !newlevel || !strlen(newlevel))
    return NULL;

  /* if the newlevel is actually a range - just use that */
  if (strchr(newlevel, '-')) {
      newrangep = strdup(newlevel);
      return newrangep;
  }

  /* look for MLS range */
  tmpptr = strchr(range, '-');

  if (tmpptr) {
    /* we are inserting into a ranged MLS context */
    len = strlen(newlevel) + 1 + strlen(tmpptr + 1) + 1;
    newrangep = (char *)malloc(len);
    if (!newrangep)
      return NULL;
    snprintf(newrangep, len, "%s-%s", newlevel, tmpptr + 1);
  } else {
    /* we are inserting into a currently non-ranged MLS context */
    if (!strcmp(newlevel, range)) {
      newrangep = strdup(range);
    } else {
      len = strlen(newlevel) + 1 + strlen(range) + 1;
      newrangep = (char *)malloc(len);
      if (!newrangep)
        return NULL;
      snprintf(newrangep, len, "%s-%s", newlevel, range);
    }
  }

  return newrangep;
}

#ifdef USE_PAM

/************************************************************************
 *
 * All PAM code goes in this section.
 *
 ************************************************************************/

#include <unistd.h>               /* for getuid(), exit(), getopt() */

#include <security/pam_appl.h>    /* for PAM functions */
#include <security/pam_misc.h>    /* for misc_conv PAM utility function */

#define SERVICE_NAME "newrole"    /* the name of this program for PAM */

int authenticate_via_pam( const struct passwd * );

/* authenticate_via_pam()
 *
 * in:     pw - struct containing data from our user's line in 
 *                         the passwd file.
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           1     PAM thinks that the user authenticated themselves properly
 *           0     otherwise
 *
 * This function uses PAM to authenticate the user running this
 * program.  This is the only function in this program that makes PAM
 * calls.
 *
 */

int authenticate_via_pam( const struct passwd *pw ) {

  int result = 0;    /* our result, set to 0 (not authenticated) by default */
  pam_handle_t *pam_handle;      /* opaque handle used by all PAM functions */

  /* This is a jump table of functions for PAM to use when it wants to *
   * communicate with the user.  We'll be using misc_conv(), which is  *
   * provided for us via pam_misc.h.                                   */
  struct pam_conv pam_conversation = {
    misc_conv,
    NULL
  };

  /* Make `p_pam_handle' a valid PAM handle so we can use it when *
   * calling PAM functions.                                       */
  if( PAM_SUCCESS != pam_start( SERVICE_NAME,
				pw->pw_name,
				&pam_conversation,
				&pam_handle ) ) {
    fprintf( stderr, _("failed to initialize PAM\n") );
    exit( -1 );
  }

  /* Ask PAM to authenticate the user running this program */
  if( PAM_SUCCESS == pam_authenticate(pam_handle,0) ) {
    result = 1;  /* user authenticated OK! */
  }

  /* Ask PAM to verify acct_mgmt */
  if( PAM_SUCCESS != pam_acct_mgmt(pam_handle,0) ) {
    result = 0;  /* user authenticated OK! */
  }

  /* We're done with PAM.  Free `pam_handle'. */
  pam_end( pam_handle, PAM_SUCCESS );
 
  return( result );

} /* authenticate_via_pam() */

#else /* else !USE_PAM */


/************************************************************************
 *
 * All shadow passwd code goes in this section.
 *
 ************************************************************************/


#include <unistd.h>                         /* for getuid(), exit(), crypt() */
#include <shadow.h>                         /* for shadow passwd functions */
#include <string.h>                         /* for strlen(), memset() */

#define PASSWORD_PROMPT _("Password:")         /* prompt for getpass() */

int authenticate_via_shadow_passwd( const struct passwd * );

/* authenticate_via_shadow_passwd()
 *
 * in:     pw - struct containing data from our user's line in 
 *                         the passwd file.
 * out:    nothing
 * return: value   condition
 *         -----   ---------
 *           1     user authenticated themselves properly according to the
 *                 shadow passwd file.
 *           0     otherwise
 *
 * This function uses the shadow passwd file to thenticate the user running
 * this program.
 *
 */

int authenticate_via_shadow_passwd( const struct passwd *pw ) {

  struct spwd *p_shadow_line; /* struct derived from shadow passwd file line */
  char *unencrypted_password_s;        /* unencrypted password input by user */
  char *encrypted_password_s; /* user's password input after being crypt()ed */

  /* Make `p_shadow_line' point to the data from the current user's *
   * line in the shadow passwd file.                                */
  setspent();            /* Begin access to the shadow passwd file. */
  p_shadow_line = getspnam( pw->pw_name );
  endspent();            /* End access to the shadow passwd file. */
  if( !( p_shadow_line ) ) {
    fprintf( stderr, _("Cannot find your entry in the shadow passwd file.\n"));
    exit( -1 );
  }

  /* Ask user to input unencrypted password */
  if( ! ( unencrypted_password_s = getpass( PASSWORD_PROMPT ) ) ) {
    fprintf( stderr, _("getpass cannot open /dev/tty\n"));
    exit( -1 );
  }

  /* Use crypt() to encrypt user's input password.  Clear the *
   * unencrypted password as soon as we're done, so it is not * 
   * visible to memory snoopers.                              */
  encrypted_password_s = crypt( unencrypted_password_s,
				p_shadow_line->sp_pwdp );
  memset( unencrypted_password_s, 0, strlen( unencrypted_password_s ) );

  /* Return 1 (authenticated) iff the encrypted version of the user's *
   * input password matches the encrypted password stored in the      *
   * shadow password file.                                            */
  return( !strcmp( encrypted_password_s, p_shadow_line->sp_pwdp ) );

} /* authenticate_via_shadow_passwd() */

#endif /* if/else USE_PAM */

/************************************************************************
 *
 * All code used for both PAM and shadow passwd goes in this section.
 *
 ************************************************************************/

int main( int argc, char *argv[] ) {

  security_context_t new_context=NULL;          /* our target security context */
  security_context_t old_context=NULL;	   /* our original securiy context */
  security_context_t tty_context=NULL;	   /* The current context of tty file */
  security_context_t new_tty_context=NULL;	   /* The new context of tty file */
  security_context_t chk_tty_context= NULL;

  context_t context;		 	   /* manipulatable form of new_context */


  const char *se_username;  /* SELinux user identity */
  struct passwd *pw;                 /* struct derived from passwd file line */
  struct passwd pw_copy;

  int clflag;                        /* holds codes for command line flags */
  int flag_index;                    /* flag index in argv[] */
  struct option long_options[] = {   /* long option flags for getopt() */
    { "role", 1, 0, 'r' },
    { "type", 1, 0, 't' },
    { "level", 1, 0, 'l' },
    { NULL, 0, 0, 0 }
  };
  char *role_s = NULL;               /* role spec'd by user in argv[] */
  char *type_s = NULL;               /* type spec'd by user in argv[] */
  char *level_s = NULL;              /* level spec'd by user in argv[] */
  char *ttyn   = NULL;		     /* tty path */
  pid_t childPid=0;			     
  int fd=0;

  /* Terminate on SIGHUP. */
  signal(SIGHUP, SIG_DFL);

#ifdef USE_NLS
  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE, LOCALEDIR);
  textdomain (PACKAGE);
#endif

  /*
   *
   * Step 1:  Handle command-line arguments.
   *
   */


  if( !is_selinux_enabled() ) {
    fprintf( stderr, 
	     _("Sorry, newrole may be used only on a SELinux kernel.\n") );
    exit(-1);
  }

  while (1) {
    clflag=getopt_long(argc,argv,"r:t:l:",long_options,&flag_index);
    if (clflag == -1)
	break;
    
    switch( clflag ) {
    case 'r':
      /* If role_s is already set, the user spec'd multiple roles - bad. */
      if( role_s ) {
	fprintf( stderr, _("Error: multiple roles specified\n"));
	exit( -1 );
      }
      role_s = optarg;  /* save the role string spec'd by user */
      break;

    case 't':
      /* If type_s is already set, the user spec'd multiple types - bad. */
      if( type_s ) {
	fprintf( stderr, _("Error: multiple types specified\n"));
	exit( -1 );
      }
      type_s = optarg;  /* save the type string spec'd by user */
      break;

    case 'l':
      if(!is_selinux_mls_enabled() ) {
        fprintf(stderr, _("Sorry, -l may be used with SELinux MLS support.\n"));
        exit(-1);
      }
      /* If level_s is already set, the user spec'd multiple levels - bad. */
      if(level_s) {
	fprintf( stderr, _("Error: multiple levels specified\n"));
	exit( -1 );
      }
      level_s = optarg;  /* save the level string spec'd by user */
      break;

    default:
      fprintf(stderr, "%s\n",USAGE_STRING);
      exit(-1);
    } /* switch( clflag ) */
  } /* while command-line flags remain for newrole */

  
  /* Verify that the combination of command-line arguments we were *
   * given is a viable one.                                        */
  if( !(role_s || type_s || level_s) ) {
    fprintf(stderr, "%s\n",USAGE_STRING);
    exit(-1);
  }

  /* Fill in a default type if one hasn't been specified */
  if( role_s && !type_s ) {
    if( get_default_type(role_s,&type_s) )
      {
	fprintf(stderr,_("Couldn't get default type.\n"));
	exit(-1);
      }
#ifdef CANTSPELLGDB
  printf( "Your type will be %s.\n", type_s );
#endif  
  }

  /*
   *
   * Step 2:  Authenticate the user.
   *
   */

  /*
   * Get the context of the caller, and extract
   * the username from the context.  Don't rely on the Linux
   * uid information - it isn't trustworthy.
   */

  /* Put the caller's context into `old_context'. */
  if( 0!=(getprevcon(&old_context)) ) {
    fprintf(stderr,_("failed to get old_context.\n"));
    exit(-1);
  }

#ifdef CANTSPELLGDB
  printf( "Your old context was %s\n", old_context );
#endif

  /* 
   * Create a context structure so that we extract and modify 
   * components easily. 
   */
  context=context_new(old_context);
  if(context == 0) {
    fprintf(stderr,_("failed to get new context.\n"));
    exit(-1);
  }

  freecon(old_context);
  /* Make `pw' point to a structure containing the data              *
   * from our user's line in the passwd file.  If the current user's
   * SELinux user identity is the default (SELINUX_DEFAULTUSER), then
   * we authenticate using the user's UID.  Otherwise we use the SELinux
   * user identity.
   */
  se_username = context_user_get(context);
  if (!strcmp (se_username, SELINUX_DEFAULTUSER))
    pw = getpwuid(getuid());
  else
    pw=getpwnam(se_username);
  if( !pw ) {
    fprintf(stderr,_("cannot find your entry in the passwd file.\n"));
    exit(-1);
  }
  pw_copy = *pw;
  pw = &pw_copy;
  pw->pw_name = xstrdup(pw->pw_name);
  pw->pw_dir = xstrdup(pw->pw_dir);
  pw->pw_shell = xstrdup(pw->pw_shell);

  printf(_("Authenticating %s.\n"),pw->pw_name);

  /* Authenticate the user running this program. */
#ifdef USE_PAM
  if( !authenticate_via_pam(pw) ) 
#else /* !USE_PAM */
  if( !authenticate_via_shadow_passwd(pw) ) 
#endif /* if/else USE_PAM */
    {
      fprintf(stderr,_("newrole: incorrect password for %s\n"), pw->pw_name);
      return(-1);
    }
  /* If we reach here, then we have authenticated the user. */
#ifdef CANTSPELLGDB
  printf( "You are authenticated!\n" );
#endif  

  /*
   *
   * Step 3:  Construct a new context based on our old context and the
   *          arguments specified on the command line.
   *
   */

  /* The first step in constructing a new context for the new shell we  *
   * plan to exec is to take our old context in `context' as a   *
   * starting point, and modify it according to the options the user *
   * specified on the command line.                                  */

  /* If the user specified a new role on the command line (if `role_s'   *
   * is set), then replace the old role in `context' with this new role. */
  if( role_s ) {
    if( context_role_set(context,role_s)) {
      fprintf(stderr,_("failed to set new role %s\n"),role_s);
      exit(-1);
    }
#ifdef CANTSPELLGDB
    printf("Your new role is %s\n",context_role_get(context));
#endif
  } /* if user specified new role */

  /* If the user specified a new type on the command line (if `type_s'   *
   * is set), then replace the old type in `context' with this new type. */
  if( type_s ) {
    if( context_type_set(context,type_s)) {
      fprintf(stderr,_("failed to set new type %s\n"),type_s);
      exit(-1);
    }
#ifdef CANTSPELLGDB
    printf("Your new type is %s\n",context_type_get(context));
#endif
  } /* if user specified new type */

  /* If the user specified a new level on the command line (if `level_s'   *
   * is set), then replace the old level in `context' with this new level. */
  if(level_s) {
    char *range_s = build_new_range(level_s, context_range_get(context));
    if (!range_s) {
      fprintf(stderr, _("failed to build new range with level %s\n"), level_s);
      exit(-1);
    }
    if(context_range_set(context, range_s)) {
      fprintf(stderr, _("failed to set new range %s\n"), range_s);
      free(range_s);
      exit(-1);
    }
    free(range_s);
#ifdef CANTSPELLGDB
    printf("Your new range is %s\n", context_range_get(context));
#endif
  } /* if user specified new level */

  /* The second step in creating the new context is to convert our modified *
   * `context' structure back to a context string and then to a Context.    */

  if( !(new_context=context_str(context))) {
    fprintf(stderr,_("failed to convert new context to string\n") );
    exit(-1);
  }


#ifdef CANTSPELLGDB
  printf("Your new context is %s\n",new_context);
#endif

  if (security_check_context(new_context) < 0) {
    fprintf(stderr, _("%s is not a valid context\n"), new_context);
    exit(-1);
  }

  /*
   *
   * Step 4:  Handle relabeling of the tty.
   *
   */

  /* Fetch TTY information */
  ttyn=ttyname(0);
  if (!ttyn || *ttyn == '\0') {
	fprintf(stderr, _("Warning!  Could not retrieve tty information.\n"));
	exit(-1);
  }

  fd = open(ttyn, O_RDWR);
  if (fd < 0) {
	fprintf(stderr, _("Warning!  Could not open %s.\n"), ttyn);
	exit(-1);
  }

  tty_context = NULL;
  if (fgetfilecon(fd, &tty_context) < 0)
	fprintf(stderr, _("Warning!  Could not get current context for %s, not relabeling.\n"), ttyn);
    
#ifdef CANTSPELLGDB
  if (tty_context)
    printf("Your tty %s was labeled with context %s\n", ttyn, tty_context);
#endif

  new_tty_context = NULL;
  if (tty_context && (security_compute_relabel(new_context,tty_context,SECCLASS_CHR_FILE,&new_tty_context) < 0))
       fprintf(stderr, _("Warning!  Could not get new context for %s, not relabeling.\n"), ttyn);

#ifdef CANTSPELLGDB
  if (new_tty_context)
    printf("Relabeling tty %s to context %s\n", ttyn, new_tty_context);
#endif

  if (new_tty_context) {
    if (fsetfilecon(fd,new_tty_context) < 0) {
      fprintf(stderr, _("Warning!  Could not set new context for %s\n"), ttyn);
      freecon(new_tty_context);
      new_tty_context = NULL;
    }
  }

  /* Fork, allowing parent to clean up after shell has executed */
  childPid=fork();
  if( childPid<0 ) {
    int errsv=errno;
    fprintf(stderr,_("newrole: failure forking: %s"),strerror(errsv));
    exit(-1);
  } else if (childPid) {
    /* PARENT */
    wait(NULL);

    if (!new_tty_context)
      exit(0);

    /* Verify that the tty still has the context set by newrole. */
    if (fgetfilecon(fd,&chk_tty_context) < 0) {
      fprintf(stderr, "Could not fgetfilecon %s.\n", ttyn);
      exit (-1);
    }

    if (strcmp(chk_tty_context, new_tty_context)) {
      fprintf(stderr,_("%s changed labels.\n"), ttyn);
      exit(-1);
    }

    freecon(new_tty_context);

#ifdef CANTSPELLGDB
    printf("Restoring tty %s back to context %s\n", ttyn, tty_context);
#endif

    fsetfilecon(fd,tty_context);
    freecon(tty_context);

    /* Done! */
    exit(0);
  }

  /* CHILD */

  close(fd);

  /* Close and reopen descriptors 0 through 2 */
  if( close(0) || close(1) || close(2) )
    {
      fprintf(stderr,_("Could not close descriptors.\n"));
      exit(-1);
    }
  fd = open(ttyn,O_RDWR);
  if (fd != 0) {
      exit(-1);
  }
  fd = open(ttyn,O_RDWR);  
  if (fd != 1) {
      exit(-1);
  }
  fd = open(ttyn,O_RDWR);  
  if (fd != 2) {
      exit(-1);
  }

  /*
   *
   * Step 5:  Execute a new shell with the new context in `new_context'. 
   *
   */

  if (optind < 1) optind = 1;
  argv[optind-1] = pw->pw_shell;
#ifdef CANTSPELLGDB
  {
	  int i;
	  printf("Executing ");
	  for (i = optind-1; i < argc; i++)
		  printf("%s ", argv[i]);
	  printf("with context %s\n", new_context);
  }
#endif
  if (setexeccon(new_context) < 0) {
       fprintf(stderr, _("Could not set exec context to %s.\n"), new_context);
       exit(-1);
  }
  execv(argv[optind-1],argv+optind-1);

  /* If we reach here, then we failed to exec the new shell. */
  perror(_("failed to exec shell\n"));
  return(-1);
} /* main() */

