/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-main.h"
#include "polyxedit-editctxt.h"
#include "polyxmass-polchemdefctxt.h"
#include "polyxedit-ui-seqed-wnd.h"

/****************************************************************************
		 This file is the entry point of the

			   polyxedit module

		   in the polyxmass software suite.
****************************************************************************/


PxmEditCtxt *
polyxedit_main_prepare_new_context_new_file (void)
{
  GtkWidget *window = NULL;

  gchar *help = NULL;
  
  PxmPolchemdefSpec *ps = NULL;

  PxmPlminfo *plminfo = NULL;

  PxmEditCtxt *editctxt = NULL;
  

  /* The caller asks that a new polymer sequence editing context be
     launched. We comply with the command.

     Since we do not know the filename of the new polymer sequence,
     and thus we do not know of what polymer chemistry definition it
     should be, we'll have to ask the user to provide the polymer
     chemistry definition that should be used to prepare the new
     context and the filename of the new polymer sequence.
  */
  
  window = 
    polyxmass_filechoosers_show_wnd (GTK_WIDGET (polyxmass_main_wnd),
				     PXM_FILE_CHOOSE_POLCHEMDEF
				     | PXM_FILE_CHOOSE_POLSEQDATA_INIT);
  g_assert (window != NULL);
  
  /* Now we should have all the data required to successfully start
     a new polymer sequence file.
     
     Screen the data that are expected to be set as datum objects
     in the polyxmass_main_wnd by the
     polyxmass_filechoosers_show_wnd ().
  */
  
  /* We should have received a PxmPolchemdefSpec * set as a datum
     to the polyxmass_main_wnd window. Note that the pointer that
     we get is a reference to an object allocated elsewhere. Do
     not free it.
  */
  ps = g_object_get_data (G_OBJECT (polyxmass_main_wnd), 
			  "polchemdefspec");
  
  if (ps == NULL)
    {
      /* All we can do is return because we could not get the
	 infos required to correctly create a polyxedit context.
      */
      g_critical (_("%s@%d: failed to get polymer chemistry definition"
		    " details for new polymer sequence.\n"),
		  __FILE__, __LINE__);
      
      return NULL;
    }
  
  /* At this point we have the type of the polymer chemistry
     definition in ps->type.
  */
  /* Also check immediately that we have the new polymer sequence
     filename.
  */
  help = g_object_get_data (G_OBJECT (polyxmass_main_wnd),
			    "new_polseq_filename");
  
  if (help == NULL)
    {
      g_critical (_("%s@%d: failed to get the new polymer sequence"
		    " filename.\n"),
		  __FILE__, __LINE__);
      
      return NULL;
    }
  
  /* Ok, these two elements are enough for us to start a new
     polymer sequence, so allocate the PxmPlminfo instance that
     we'll fill with data...
  */
  plminfo = pxmchem_plminfo_new ();
  
  /* The type of the required polymer chemistry definition.
   */
  pxmchem_polymer_plminfo_set_type (plminfo, ps->type);
  
  /* The name of the polymer sequence file:
   */
  pxmchem_polymer_plminfo_set_file (plminfo, help);
  
  /* The name of the polymer sequence:
   */
  help = g_object_get_data (G_OBJECT (polyxmass_main_wnd),
			    "polseq_name_entry_data");
  
  if (help == NULL)
    {
      pxmchem_polymer_plminfo_set_name (plminfo, _("Name not set"));
    }
  else
    {
      pxmchem_polymer_plminfo_set_name (plminfo, help);
    }
  
  /* The code of the polymer sequence:
   */
  help = g_object_get_data (G_OBJECT (polyxmass_main_wnd),
			    "polseq_code_entry_data");
  if (help == NULL)
    {
      pxmchem_polymer_plminfo_set_code (plminfo, _("Code not set"));
    }
  else
    {
      pxmchem_polymer_plminfo_set_code (plminfo, help);
    }
      
  /* Since we know what's the file of this new polymer chemistry
     definition file, all we do is send the contents of PxmPlminfo in
     this file on disk. That goes along with the writing to the disk
     file of the DTD...

     Once this is done we'll ask that this sequence be read.
  */
  if (0 >= pxmchem_plminfo_write_xml_file (plminfo, 
					   plminfo->file, NULL))
    {
      g_critical (_("%s@%d: failed to save pseudo-sequence to file: '%s'\n"),
		  __FILE__, __LINE__, plminfo->file);
	  
      pxmchem_plminfo_free (plminfo);
	  
      return NULL;
    }
      
  /* At this point the empty sequence should be on disk in the file
     requested by the user.

     We now shamelessly call the context-preparing function that works
     using a polymer sequence filename for which enough data are
     present to actually create a full-featured editctxt...

     So we use the plminfo->file member as the name of the file that
     we just got writtent to the disk. Once the context is prepared,
     we free the 'plminfo' object.

  */
  editctxt = polyxedit_main_prepare_new_context_open_file (plminfo->file);
  
  pxmchem_plminfo_free (plminfo);

  return editctxt;
}



PxmEditCtxt *
polyxedit_main_prepare_new_context_open_file (gchar *filename)
{
  gchar *type = NULL;

  gint count = 0;

  PxmPolchemdefCtxt *polchemdefctxt = NULL;

  gboolean new_polchemdefctxt = FALSE;

  PxmEditCtxt *editctxt = NULL;

  

  /* The user asks that a new polymer sequence editing context be
     launched. We comply with the command. 

     The 'filename' parameter cannot be NULL.
  */
  
  g_assert (filename != NULL);
  

  editctxt = polyxedit_editctxt_new ();
  

  if (FALSE == g_file_test (filename, G_FILE_TEST_EXISTS))
    {
      g_critical (_("%s@%d: file asked for opening not found: '%s'\n"),
		  __FILE__, __LINE__, 
		  filename);
	  
      polyxedit_editctxt_free (editctxt);

      return NULL;
    }
      
  /* At this point we know that the polymer sequence file is there.
     Try to open it.
  */
     
  /* Each polymer sequence file (that's an xml-format file) should
     contain an xml element that indicates what is the polymer
     chemistry definition that should be used for it.

     Try to get that definition name. For example, if we were
     loading a protein sequence, the type would almost certainly
     be something like "protein" or "peptide".

     Later, we will have to make the connection between that type
     and the file on disk that describes it and the directory
     where all the data related to that polymer chemistry
     definition are located.

     Note that 'defintion_type' is allocated.
  */
  type = pxmchem_polymer_get_type_from_xml_file (filename);
      
  if (type == NULL)
    {
      g_critical (_("%s@%d: failed to get polymer type "
		    "from sequence file\n"),
		  __FILE__, __LINE__);
      
      polyxedit_editctxt_free (editctxt);

      return NULL;
    }
	  
  /* We now know what's the polymer chemistry definition of the
     polymer sequence that we are trying to read from file. 

     Polymer chemistry definitions are made available to the GNU
     polyxmass program in the form of contexts of type
     PxmPolchemdefCtxt.

     Each time a PxmPolchemdefCtxt is allocated it is made
     available in the global array of such instances
     (polyxmass_polchemdefctxtGPA).

     We ask if the PxmPolchemdefCtxt type we are looking for is
     already available in this array.

     Attention, the polchemdefctxt is not allocated for us. Do not
     free it.
  */
  polchemdefctxt = polyxmass_polchemdefctxt_get_ptr_by_type 
    (type, polyxmass_polchemdefctxtGPA);

  if (polchemdefctxt == NULL)
    {
      /* Apparently we are trying to load a polymer sequence of a
	 polymer chemistry definition that's never been
	 encountered yet. So we have to go through all the steps
	 required to load the polymer chemistry definition from
	 file...
      */
	  
      /* Set a gboolean variable to let us know later that we had to 
	 create the PxmPolchemdefCtxt ex nihilo.
      */
      new_polchemdefctxt = TRUE;

      /* Allocate a PxmPolchemdefCtxt instance so that we'll be
	 able to feed it with data read from a polymer chemistry
	 definition context.
      */
      polchemdefctxt = polyxmass_polchemdefctxt_new ();

      /* Now that we have allocated a new polchemdefctxt, we can
	 initialize its member data by indicating what the polymer
	 chemistry definition we are willing to put in: "protein" or
	 "dna" or whatever...
      */
      if (FALSE == polyxmass_polchemdefctxt_init_with_type (polchemdefctxt,
							    type,
							    TRUE))
	{
	  g_critical (_("%s@%d: failed to initialize the polchemdefctxt "
			"of type: '%s'\n"),
		      __FILE__, __LINE__, type);

	  polyxedit_editctxt_free (editctxt);
	  
	  polyxmass_polchemdefctxt_free (polchemdefctxt);
	  
	  g_free (type);
	  
	  return NULL;
	}

      /* At this point, everything indicates that the polymer
	 chemistry definition is fully rendered in memory. We can add
	 it to the global array of available polymer chemistry
	 definitions so that we don't need to remake all that work
	 later if another polymer sequence of the same polymer
	 chemistry definition type is loaded.
      */
      g_ptr_array_add (polyxmass_polchemdefctxtGPA, polchemdefctxt);
    }
  /* End of 
     if (polchemdefctxt == NULL)
  */

  /* At this point, we are sure that we have a proper polymer
     chemistry definition (either found in the available ones or
     created out of nothing above) for our polymer sequence, so
     let 'editctxt' immediately know to what polchemdefctxt it
     should turn itself when polymer chemistry data are necessary.
  */
  editctxt->polchemdefctxt = polchemdefctxt;

  /* Since we do not allocate, each time we use it, a
     polchemdefctxt instance, we reference it once to let the
     program know it is used once more.
  */
  polyxmass_polchemdefctxt_ref (polchemdefctxt);

  /* Starting from now it will not be possible to free the
     polchemdefctxt directly, we'll have to free the editctxt
     itself, which will take care of properly unrefing the
     polchemdefctxt...
  */

  /* Allocate immediately the calcopt member, otherwise it will
     be NULL when we'll need it.
  */
  editctxt->calcopt = pxmchem_calcopt_new ();
  pxmchem_calcopt_set_default (editctxt->calcopt);
      
  /* Allocate immediately the ionizerule member, otherwise it will be
     NULL when we'll need it. Also the allocation is performed by 
     duplicating the one that sits in the polchemdef object, so that
     we start fresh with default sensible values. 
	 
     Later the user will be able to modify this structure using a nice
     window (in the calc_mass_options stufff...).
  */
  g_assert (editctxt->polchemdefctxt->polchemdef != NULL);
  g_assert (editctxt->polchemdefctxt->polchemdef->ionizerule != NULL);
  editctxt->ionizerule = pxmchem_ionizerule_dup (editctxt->polchemdefctxt->
						 polchemdef->ionizerule);
      
  /* Now yes, time has come to load the 'filename' from disk and
     render it into the editctx->polymer instance!
  */
  editctxt->polymer =  
    pxmchem_polymer_render_xml_file (filename,
				     editctxt->polchemdefctxt->
				     polchemdef->monomerGPA,
				     editctxt->polchemdefctxt->
				     polchemdef->codelen,
				     NULL);
      
  if (editctxt->polymer == NULL)
    {
      g_critical (_("%s@%d: failed to render polymer "
		    "sequence from file: '%s'\n"),
		  __FILE__, __LINE__, filename);
	  
      polyxedit_editctxt_free (editctxt);

      return NULL;
    }

  /* Because we know we could correctly render the polymer
     sequence in 'filename' to the editctx->polymer, let's the
     editctx->polymer be self-conscious of the file whence it
     came. That's a sanity check, important indeed, because when
     the user will "save" the sequence we really want to only
     write the sequence to the initial true file!
  */
  g_assert (editctxt->polymer->plminfo != NULL);
  g_assert (editctxt->polymer->plminfo->file != NULL);
  g_assert (0 == strcmp (filename, editctxt->polymer->plminfo->file));
      
  /* Since we now have all we need to make the graphical rendering
     of the raw data in the polymer sequence, we should prepare
     the rendering framework. 

     This framework is based on the PxmSeqEditorCtxt structure
     which is the polymer sequence editing context. 

     The seqeditorctxt structure holds a number of critical widget
     pointers that are needed to perform the graphical rendering
     of the data in the polymer sequence editor.

     Bridging the raw data material with the graphical logics is
     performed in the editctxt structure, that makes the link
     between the polymer PxmPolymer instance and the seqeditorctxt
     PxmSeqEditorCtxt instance.
  */
  editctxt->seqeditorctxt = polyxedit_seqeditorctxt_new ();
      
  /* We have to allocate a string of a correct length for two
     variables: eval_code and elab_code, which must be of a length
     large enough to accomodate a monomer code:
  */
  polyxedit_seqeditorctxt_new_eval_code (editctxt->seqeditorctxt,
					 editctxt->polchemdefctxt->
					 polchemdef->codelen);
      
  polyxedit_seqeditorctxt_new_elab_code (editctxt->seqeditorctxt,
					 editctxt->polchemdefctxt->
					 polchemdef->codelen);
  
  /* Now that all the crucial instances have been created an
     configured, we can start making the graphical rendering of
     the polymer sequence.
  */
  editctxt->seqeditorctxt->sequence_editor_wnd = 
    polyxedit_seqed_wnd_setup (editctxt);

  if (editctxt->seqeditorctxt->sequence_editor_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to set up the polymer "
		    "sequence editor.\n"),
		  __FILE__, __LINE__);
	  
      polyxedit_editctxt_free (editctxt);
	  
      return NULL;
    }

  /* At this point we have a fully constructed polymer sequence
     editor, in which we should however render the polymer
     sequence!
  */
  count = polyxedit_seqed_wnd_render_polseq (editctxt, 
					     /* rerender? */ FALSE);
      
  if (count == -1)
    {
      g_critical (_("%s@%d: failed to render polymer sequence: '%s'\n"),
		  __FILE__, __LINE__, editctxt->polymer->plminfo->name);
	  
	  
      polyxedit_editctxt_free (editctxt);
	  
      return NULL;
    }

  /* Store the pointer of the editctxt in the array of all the
     allocated PxmEditCtxt instances , so that we can ubiquitously
     get to it later.
  */
  g_ptr_array_add (polyxedit_editctxtGPA, editctxt);
      
  /*
    debug_printf (("adding editctxt %p - %s\n",
    editctxt, editctxt->polymer->plminfo->name));
  */

  return editctxt;
}


