/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-completions.h"
#include "polyxedit-rendering.h"
#include "polyxedit-ui-seqed-kbd.h"

enum
  {
    COMP_COLUMN_MNM_PIXBUF,
    COMP_COLUMN_MNM_CODE,
    COMP_COLUMN_MNM_NAME,
    COMP_COLUMN_MNM_COL_COUNT
  };


GtkWidget *
polyxedit_completions_mnm_code_show_list (PxmEditCtxt *editctxt,
					  GPtrArray *GPA)
{
  GtkWidget *window = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  /* Note that each polymer sequence editor window may have its own
   * completions' list window, so we may not actually create one in
   * the call below, but just get a pointer to a window that existed
   * and was visible already.
   */
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);


  /* The window may be exising already!
   */
  if (seqeditorctxt->monomer_completions_wnd != NULL)
    {
      polyxedit_completions_wnd_empty_list_completions (editctxt);
      polyxedit_completions_wnd_update_list_completions (editctxt, GPA);
      
      return seqeditorctxt->monomer_completions_wnd;
    }
  
  /* Apparently the current editing context has no completions window 
   * opened, so we have to go through all the creation process.
   */
  window = polyxedit_completions_wnd_setup (editctxt, GPA);
  
  g_assert (window != NULL);
  
  /* Now that we have the window in which we want to list the
   * available completions, do it. However, remember that a window may
   * be pre-existing, and if it is so, it may contain already
   * previously needed completions. So, remove any existing
   * completions from the list_store.
   */
  seqeditorctxt->monomer_completions_wnd = window;
  
  /* Note that GPA is emptied after this function returns, so it is not
     possible to use the GPA pointer.
  */

  return seqeditorctxt->monomer_completions_wnd;
}



GtkWidget *
polyxedit_completions_wnd_setup (PxmEditCtxt *editctxt, GPtrArray *GPA)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;

  GtkWidget *completions_vbox = NULL;
  GtkWidget *scroll_wnd = NULL;
  GtkWidget *treeview = NULL;
  GtkTreeModel *model = NULL;
  GtkCellRenderer *renderer = NULL;
  GtkTreeViewColumn* column = NULL;
  
  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  gui_file = 
    g_strdup_printf ("%s/polyxedit-completions.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "completions_list_wnd", 
		       PACKAGE);
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "completions_list_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the completions window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);


  /* Set the polymer chemistry definition type to its correspondent
     GtkEntry.
   */
  widget = glade_xml_get_widget (xml, "polchemdef_type_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polchemdef_type_entry", widget);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polchemdefctxt->polchemdef->type);
  

 /* Get from the glade definition file the vertical box into which
   * we are going to pack a number of widgets:
   */

  completions_vbox = glade_xml_get_widget (xml, "completions_vbox");
  g_assert (completions_vbox != NULL);


  /* Create the scrolled window that we'll pack into widget.
   */
  scroll_wnd = gtk_scrolled_window_new (NULL, NULL);

  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW (scroll_wnd),
				       GTK_SHADOW_ETCHED_IN);

  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scroll_wnd),
				  GTK_POLICY_AUTOMATIC,
				  GTK_POLICY_AUTOMATIC);

  gtk_box_pack_start (GTK_BOX (completions_vbox), scroll_wnd, 
		      TRUE, TRUE, 0);  
  
  g_object_set_data (G_OBJECT (window), 
		     "scroll_wnd", scroll_wnd);
  


  /* Create the treeview model.
   */
  model = polyxedit_completions_wnd_create_treeview_model (editctxt,
							   GPA);
  
  /* Set to the model a datum with a pointer to the window!
   */
  g_object_set_data (G_OBJECT (model), "window", window);

  /* And now set the window a datum with a pointer to the mode,
   * so that later the model is accessible.
   */
  g_object_set_data (G_OBJECT (window), "TREEVIEW_MODEL", model);
    

  /* Create the treeview proper.
   */
  treeview = gtk_tree_view_new_with_model (model);

  g_object_unref (G_OBJECT (model));

  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (treeview), TRUE);

  /* Set to the window a datum with a pointer to the treeview, so that
   * is accessible later.
   */
  g_object_set_data (G_OBJECT (window), "TREEVIEW", treeview);


  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);

  /* Reacts when a double-click happens on an item.
   */
  g_signal_connect (G_OBJECT (treeview),
		    "row-activated",
		    G_CALLBACK 
		    (polyxedit_completions_treeview_row_activated_event), 
		    window);
  
  /* We want to be able to trap the ENTER key.
   */
  gtk_widget_add_events (GTK_WIDGET (treeview), GDK_KEY_PRESS_MASK);
  
  g_signal_connect (G_OBJECT (treeview),
		    "key-release-event",
		    G_CALLBACK 
		    (polyxedit_completions_treeview_key_release_event),
		    window);
  
  /* Pixbuf column.
   */
  renderer = gtk_cell_renderer_pixbuf_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Monicon"), renderer,
     "pixbuf", COMP_COLUMN_MNM_PIXBUF,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COMP_COLUMN_MNM_PIXBUF);


  /* Code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column",
		      (gint *) COMP_COLUMN_MNM_CODE);

  column = 
    gtk_tree_view_column_new_with_attributes (_("Code"),
					      
					      renderer,
					      
					      "text", 
					      COMP_COLUMN_MNM_CODE,
					      
					      NULL);

  gtk_tree_view_column_set_sort_column_id (column, 
					   COMP_COLUMN_MNM_CODE);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);


  /* Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COMP_COLUMN_MNM_NAME);


  renderer = gtk_cell_renderer_text_new ();

  g_object_set (G_OBJECT (renderer), "xalign", 0.5, NULL);

  g_object_set_data (G_OBJECT (renderer), "column",
		      (gint *) COMP_COLUMN_MNM_CODE);

  column = 
    gtk_tree_view_column_new_with_attributes (_("Name"),
					      
					      renderer,
					      
					      "text", 
					      COMP_COLUMN_MNM_NAME,
					      
					      NULL);

  gtk_tree_view_column_set_sort_column_id (column, 
					   COMP_COLUMN_MNM_NAME);

  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  gtk_tree_view_column_set_clickable (GTK_TREE_VIEW_COLUMN (column), TRUE);
  gtk_tree_view_column_set_resizable (GTK_TREE_VIEW_COLUMN (column), TRUE);







  /* Add the treeview to the scroll_wnd.
   */
  gtk_container_add (GTK_CONTAINER (scroll_wnd), treeview);
  
  gtk_widget_show_all (window);
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));


  /* Set the name of the polymer sequence to the title bar of the
   * window.
   */
  help = g_strdup_printf (_("Completions: %s"), 
			  editctxt->polymer->plminfo->name);
  
  gtk_window_set_title (GTK_WINDOW (window), help);
  
  g_free (help);


  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_completions_wnd_delete_event), 
		    editctxt);
  
  return window;
}



gint
polyxedit_completions_wnd_empty_list_completions (PxmEditCtxt *editctxt)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gint count = 0;

  gboolean valid = FALSE;
  
  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeView *treeview = NULL;
  GtkTreePath *path = NULL;
  
  GdkPixbuf *pixbuf = NULL;
    
  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* We get a pointer to a polymer sequence context that contains a
   * polymer editing context that itself contains a pointer to a
   * window that may already be listing a number of completions
   * items. We first have to remove them all.
   */
  if (seqeditorctxt->monomer_completions_wnd == NULL)
    return count;
  
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (seqeditorctxt->monomer_completions_wnd), 
				       "TREEVIEW");

  model = gtk_tree_view_get_model (treeview);

  path = gtk_tree_path_new_first ();  

  if (TRUE == gtk_tree_model_get_iter_first (model, &treeiter))
    {
      /* The tree is not empty. We should first empty it.
       */
      
      /* Get the first iter in the list.
       */
      valid = gtk_tree_model_get_iter_first (model, &treeiter);

      while (valid == TRUE)
	{
	  /* Walk through the list, reading each row.
	   */
	  /* Make sure to terminate calls to gtk_tree_model_get()
	   * with a '-1' value.
	   */
	  gtk_tree_model_get (model, &treeiter, 
			      COMP_COLUMN_MNM_PIXBUF, &pixbuf,
			      -1);
	  
	  /* Unref the pixbuf that was gotten from previous call.
	   */
	  g_object_unref (G_OBJECT (pixbuf));
	  
	  /* Do not do anything with the monomer's name and code, because
	   * they are pointed to in the model, and were not duplicated !
	   * This may need a fix later, with a duplication step when the
	   * items are appended to the list_store.
	   */

	  count++;
	  
	  valid = gtk_tree_model_iter_next (model, &treeiter);
	}
      
      /* Now that we have made the necessary with all the items, we 
       * can remove them from the list in a single step:
       */
      gtk_list_store_clear (GTK_LIST_STORE (model));
    }

  return count;
}


gint
polyxedit_completions_wnd_update_list_completions (PxmEditCtxt *editctxt,
						   GPtrArray *GPA)
{
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gint count = 0;
  gint iter = 0;

  GtkTreeModel *model = NULL;
  GtkTreeIter treeiter ;
  GtkTreeView *treeview = NULL;
  GtkTreePath *path = NULL;
  
  GdkPixbuf *pixbuf = NULL;
    
  PxmMonomer *monomer;

  
  g_assert (editctxt != NULL);
  g_assert (GPA != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  
  /* We get a pointer to a polymer sequence context that contains a
   * polymer editing context that itself contains a pointer to a
   * window that may already be listing a number of completions
   * items. We first have to remove them all and next fill them again
   * using the data in GPA.
   */
  if (seqeditorctxt->monomer_completions_wnd == NULL)
    return count;
  
  treeview = 
    (GtkTreeView *) g_object_get_data (G_OBJECT (seqeditorctxt->monomer_completions_wnd), 
				       "TREEVIEW");

  model = gtk_tree_view_get_model (treeview);

  path = gtk_tree_path_new_first ();  

  polyxedit_completions_wnd_empty_list_completions (editctxt);
    
  /* Now that we have emptied the store, we can append the items that 
   * are in the GPA.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (GTK_LIST_STORE (model), &treeiter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);

	  /* We now have to get a pointer to the monomer's monicon's
	   * pixbuf.
	   */
	  pixbuf = 
	    polyxedit_rendering_pixbuf_get_from_data_list (monomer->code,
							   NULL,
							   editctxt);
	  
	  if (pixbuf != NULL)
	    {
	      /* We should refcount this instance.
	       */
	      g_object_ref (G_OBJECT (pixbuf));
	    }
	  
	  if (pixbuf == NULL)
	    {
	      pixbuf = 
		polyxedit_rendering_pixbuf_render_no_modif (monomer->code,
							    seqeditorctxt->
							    monicon_size,
							    editctxt);
	      if (pixbuf != NULL)
		{
		  /* Let the polymer chemistry definition context know
		   * that we have rendered the pixbuf for the monomer
		   * 'code'.  Note that we just created a pixbuf from
		   * nothing and it was created by a reference count
		   * of 1, so we do not need to do anything.
		   */
		  polyxedit_rendering_pixbuf_set_to_data_list (pixbuf, 
							       monomer->code,
							       NULL, 
							       editctxt);
		}
	      else
		{
		  g_error (_("%s@%d: failed to render pixbuf for code: '%s'\n"),
			 __FILE__, __LINE__, monomer->code);
		}
	    }
	  
	  gtk_list_store_set (GTK_LIST_STORE (model), &treeiter,
			      
			      COMP_COLUMN_MNM_PIXBUF,
			      pixbuf,
			      
			      COMP_COLUMN_MNM_CODE, 
			      monomer->code,
			      
			      COMP_COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      -1);

	  count++;
	}
    }
    
  return count;
}


GtkTreeModel *
polyxedit_completions_wnd_create_treeview_model (PxmEditCtxt *editctxt,
						 GPtrArray *GPA)
{
  gint iter = 0;
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  
  GtkListStore *model;
  GtkTreeIter treeiter;

  GdkPixbuf *pixbuf = NULL;
    
  PxmMonomer *monomer;
  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

    
  /* Create the list store.
   */
  model = gtk_list_store_new (COMP_COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			      GDK_TYPE_PIXBUF, /* monicon's pixbuf */
			      G_TYPE_STRING, /* monomer code */
			      G_TYPE_STRING /* monomer name */);
  
  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &treeiter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);
	  
	  /* We now have to get a pointer to the monomer's monicon's
	   * pixbuf. The function below refcounts the pixbuf if it is
	   * found existing already, so that it is not necessary to do
	   * it here.
	   */
	  pixbuf = polyxedit_rendering_pixbuf_render_no_modif (monomer->code,
							       seqeditorctxt->
							       monicon_size,
							       editctxt);
	  	  
	  gtk_list_store_set (model, &treeiter,
			      
			      COMP_COLUMN_MNM_PIXBUF,
			      pixbuf,
			      
			      COMP_COLUMN_MNM_CODE, 
			      monomer->code,
			      
			      COMP_COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      -1);
	}
    }
  
  
  return GTK_TREE_MODEL (model);
}


/* Reacts when a double-click happens on an item.
 */
void
polyxedit_completions_treeview_row_activated_event (GtkTreeView *treeview,
						    GtkTreePath *path,
						    GtkTreeViewColumn *col,
						    gpointer data)
{
  GtkWindow *window = data;
  
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
  GtkTreeSelection* selection = NULL; 
  
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gchar* err = NULL;
  gchar* code = NULL;
  
  
  g_assert (window != NULL);
  g_assert (treeview != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  

  /* The user has selected an item, which means that she wants the
     corresponding monomer to be inserted in the polymer sequence editor.
     Thus we need to get the name of the monomer item that's selected.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  selection = gtk_tree_view_get_selection (treeview);

  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&iter))
    return; /*no selection, just return. Should never be the case. */
  

  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the iter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &iter, 
		      COMP_COLUMN_MNM_CODE, &code,
		      -1);

  g_assert (code != NULL);
  
  /* The evaluation of the code is the process by which a given
   * monomer code is changed into a real monomer in the
   * array of monomers in the polymer sequence.
   */
  if (FALSE == polyxedit_seqed_kbd_evaluate_code (code,
						  seqeditorctxt->last_point_1_idx,
						  editctxt))
    {
      g_free (code);

      err = g_strdup_printf (_("Failed completion monomer code: '%s'"),
			     code);
      
      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			  err);
      
      g_free (err);

      return;
    }

  g_free (code);

  /* If we are here, that means that the evaluation of the eval_code
     has been successful, so we have to do the following:
    
     1. emtpy the elab_code and the eval_code, so that we later can use
     them fresh.
     
     
     2. set to 0 the count of typed characters, since we have finished
     dealing witht he previously elaborating code.

     3. set to empty the text entry that hold the elaborating code.
  */
  memset (seqeditorctxt->eval_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  memset (seqeditorctxt->elab_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);

  seqeditorctxt->kb_typed_chars = 0;
  
  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
  return ;
}




/* Reacts when a key is released in the completions window.
 */
void
polyxedit_completions_treeview_key_release_event (GtkWidget *widget,
						  GdkEventKey *event,
						  gpointer data)
{
  GtkWindow *window = data;

  GtkTreeView *treeview = GTK_TREE_VIEW (widget);
  GtkTreeSelection* selection = NULL; 
  GtkTreeIter iter;
  GtkTreeModel *model = NULL;
 
  PxmEditCtxt *editctxt = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  gchar* err = NULL;
  gchar* code = NULL;
  
  
  GdkEventKey *gdk_event_key = (GdkEventKey *) event;
  
  if (GDK_Return != gdk_event_key->keyval)
    return;
  
  g_assert (window != NULL);
  g_assert (treeview != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);

  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);
  

  /* The user has selected an item, which means that she wants the
     corresponding monomer to be inserted in the polymer sequence editor.
     Thus we need to get the name of the monomer item that's selected.
  */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);

  selection = gtk_tree_view_get_selection (treeview);

  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&iter))
    return; /*no selection, just return. Should never be the case. */
  

  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the iter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &iter, 
		      COMP_COLUMN_MNM_CODE, &code,
		      -1);

  g_assert (code != NULL);
  
  /* The evaluation of the code is the process by which a given
   * monomer code is changed into a real monomer in the
   * array of monomers in the polymer sequence.
   */
  if (FALSE == polyxedit_seqed_kbd_evaluate_code (code,
						  seqeditorctxt->last_point_1_idx,
						  editctxt))
    {
      g_free (code);

      err = g_strdup_printf (_("Failed completion monomer code: '%s'"),
			     code);
      
      gtk_entry_set_text (GTK_ENTRY (seqeditorctxt->error_code_entry),
			  err);
      
      g_free (err);

      return;
    }

  g_free (code);

  /* If we are here, that means that the evaluation of the eval_code
     has been successful, so we have to do the following:
    
    1. emtpy the elab_code and the eval_code, so that we later can use
       them fresh.
   
   
    2. set to 0 the count of typed characters, since we have finished
    dealing witht he previously elaborating code.

    3. set to empty the text entry that hold the elaborating code.
  */
  memset (seqeditorctxt->eval_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  memset (seqeditorctxt->elab_code, '\x0', 
	  polchemdefctxt->polchemdef->codelen + 1);
  
  seqeditorctxt->kb_typed_chars = 0;
  
  polyxedit_seqed_kbd_echo_elab_code (editctxt);
	  
  return ;
}


gint 
polyxedit_completions_wnd_delete_event (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  PxmEditCtxt *editctxt = data;
  PxmSeqEditorCtxt *seqeditorctxt = NULL;

  
  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  

  /* We must empty the list of completions and set to NULL the 
   * member of the seqeditorctxt structure so that when completions are
   * required later a new window may be created ex nihilo.
   */
  polyxedit_completions_wnd_empty_list_completions (editctxt);
  
  seqeditorctxt->monomer_completions_wnd = NULL;
    
  return FALSE;
}
