/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-monomer-modif.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxedit-monomer-modif.h"
#include "polyxedit-ui-masses-display-wnd.h"


GtkWidget *
polyxedit_ui_monomer_modif_setup_wnd (PxmEditCtxt *editctxt)
{
  GtkWidget *window = NULL;
  GtkWidget *widget = NULL;
  GtkTreeView *treeview = NULL;
  GtkTreeSelection* selection = NULL; 
  
  PxmSeqEditorCtxt *seqeditorctxt = NULL;
  PxmMonomer *monomer = NULL;
  
  GladeXML *xml = NULL;

  gint idx = -1;
  gint start = -1;
  gint end = -1;
  

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (editctxt != NULL);
  
  seqeditorctxt = editctxt->seqeditorctxt;
  g_assert (seqeditorctxt != NULL);
  


  gui_file = 
    g_strdup_printf ("%s/polyxedit-monomer-modif.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "monomer_modif_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "monomer_modif_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the monomer modification window\n"),
	     __FILE__, __LINE__);

      g_object_unref (G_OBJECT (xml));

      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "messages_entry",
		     widget);


  /* Set the polymer name to the relative entry.
   */
  widget = glade_xml_get_widget (xml, "polymer_sequence_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polymer_sequence_name_entry", 
		     widget);
  
  if (editctxt->polymer->plminfo->name != NULL)
    gtk_entry_set_text (GTK_ENTRY (widget), 
			editctxt->polymer->plminfo->name);
  else
    gtk_entry_set_text (GTK_ENTRY (widget), _("Not set"));
  

  widget = glade_xml_get_widget (xml, "polymer_sequence_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), "polymer_sequence_id_number_entry", 
		     widget);
  
  /* Set the polymer id number (the pointer to editctxt
     to the relative entry.
   */
  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);
  
  /* Now deal systematically with the widgets that we will use later.
   */
  widget = glade_xml_get_widget (xml, 
				 "selected_monomer_code_entry");
  g_object_set_data (G_OBJECT (window), 
		     "selected_monomer_code_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "selected_monomer_position_entry");
  g_object_set_data (G_OBJECT (window), 
		     "selected_monomer_position_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "available_modifications_treeview");
  g_object_set_data (G_OBJECT (window), 
		     "available_modifications_treeview",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_selected_monomer_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_selected_monomer_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_same_code_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_same_code_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_from_list_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_from_list_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_all_monomers_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_all_monomers_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_spec_pos_radiobutton");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_spec_pos_radiobutton",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_spec_pos_entry");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_spec_pos_entry",
		     widget);

  widget = glade_xml_get_widget (xml, 
				 "target_monomers_treeview");
  g_object_set_data (G_OBJECT (window), 
		     "target_monomers_treeview",
		     widget);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (widget));
  
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

  /* The textview where errors are to be displayed.
   */
  widget = glade_xml_get_widget (xml, 
				 "monomer_modification_errors_textview");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_errors_textview",
		     widget);
  
  /* The buttons are connected through signals to related functions.
   */
  widget = 
    glade_xml_get_widget (xml, "monomer_modification_modify_button");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_modify_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_ui_monomer_modif_modify_button), 
		    window);

  widget = 
    glade_xml_get_widget (xml, "monomer_modification_un_modify_button");
  g_object_set_data (G_OBJECT (window), 
		     "monomer_modification_un_modify_button",
		     widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK (polyxedit_ui_monomer_modif_un_modify_button),
		    window);


  gtk_widget_show_all (GTK_WIDGET (window));
  
  /* We have finished setting up the window, and so also using
   * the xml data, unref them
   */
  g_object_unref (G_OBJECT (xml));

  /* And now that we know all the window is set up, we can fill the
     data in it: the list of available modifications, the list of the
     monomers in the polymer chemistry definition context, the
     position of the possibly selected monomer...
  */

  /* First the index of the currently selected monomer. Function returns
     a pointer to a monomer if only one is selected, and the index
     of the selected monomer into idx.
   */
  idx = -1;
  monomer = polyxedit_seqed_wnd_get_selected_monomer (editctxt, &idx);
  
  if (monomer != NULL)
    {
      /* If a single monomer is selected, then its idx cannot be -1.
       */
      g_assert (idx != -1);
      
      /* We can print the relevant info in their entries.
       */
      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_code_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), monomer->code);
      
      /* Print the position of the selected monomer, not its index.
       */
      help = g_strdup_printf ("%d", idx + 1);

      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_position_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget), help);

      g_free (help);
    }
  else
    {
      /* We can print the relevant explanations the entry.
       */
      widget = g_object_get_data (G_OBJECT (window), 
				  "selected_monomer_code_entry");
      g_assert (widget != NULL);
      
      gtk_entry_set_text (GTK_ENTRY (widget),
			  _("Not a single monomer selected"));

      /* Get the indices of the selection if there is one.
       */
      if (FALSE != 
	  polyxedit_seqed_wnd_get_selection_indices (editctxt,
						     &start, &end))
	{
	  /* Remember that due to the handling of the polymer sequence
	     displayed in the sequence editor, the end value that is 
	     returned is indeed corresponding to the real index of the last
	     selected monomer + 1, so care should be taken not to outbound
	     the polymer sequence array.
	  */
	  
	  /* We show the (start + 1) value because we want to show
	     a position and not an index.
	  */
	  help = 
	    g_strdup_printf ("[%d..%d]", start + 1, end /* NOT + 1 */);
	  
	  /* Print the position of the selection, not its index.
	   */
	  widget = g_object_get_data (G_OBJECT (window), 
				      "selected_monomer_position_entry");
	  g_assert (widget != NULL);
	  
	  gtk_entry_set_text (GTK_ENTRY (widget), help);
	  
	  g_free (help);
	}
      else
	{
	  /* Print that there is no selection.
	   */
	  widget = g_object_get_data (G_OBJECT (window), 
				      "selected_monomer_position_entry");
	  g_assert (widget != NULL);
	  
	  gtk_entry_set_text (GTK_ENTRY (widget), _("No selection"));
	  
	}

      /* Note that if there is no single monomer selected, it makes no
	 sense to have the radio button "target_selected_monomer_radiobutton"
	 selected. We set it to the other one, where the user selects the
	 positions with regex-like strings.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "target_monomers_spec_pos_radiobutton");
      g_assert (widget != NULL);
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  /* Second fill the treeview of modifications avaialable to the
     polymer chemistry definition context of that specific polymer
     sequence.
  */
  treeview = 
    GTK_TREE_VIEW (g_object_get_data (G_OBJECT (window), 
				      "available_modifications_treeview"));

  if (FALSE == 
      polyxedit_ui_monomer_modif_setup_modifs_treeview (treeview,
							editctxt->
							polchemdefctxt->
							polchemdef))
    {
      g_critical (_("%s@%d: failed to fill the modifications list\n"),
	     __FILE__, __LINE__);
    }
  
  /* Third fill the treeview of all the monomer codes present in the
     polymer chemistry definition context of that specific polymer
     sequence.
  */
  treeview = 
    GTK_TREE_VIEW (g_object_get_data (G_OBJECT (window), 
				      "target_monomers_treeview"));

  if (FALSE == 
      polyxedit_ui_monomer_modif_setup_monomcodes_treeview (treeview,
							    editctxt->
							    polchemdefctxt->
							    polchemdef))
    {
      g_critical (_("%s@%d: failed to fill the monomer codes list\n"),
	     __FILE__, __LINE__);
    }
  


  /* The signal of the window itself.
   */
  /* Signal / callback connections.
   */
  g_signal_connect 
    (G_OBJECT (window),
     "delete_event",
     G_CALLBACK (polyxedit_ui_monomer_modif_wnd_delete_event), 
     editctxt);

  g_signal_connect 
    (G_OBJECT (window),
     "destroy_event",
     G_CALLBACK (polyxedit_ui_monomer_modif_wnd_destroy_event), 
     editctxt);

  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one monomer modification window opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("monomer_modif_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_ui_monomer_modif_wnd_really_close);
  
  g_free (help);
 

  return window;
}


gboolean
polyxedit_ui_monomer_modif_setup_modifs_treeview (GtkTreeView *treeview,
						  PxmPolchemdef *polchemdef)
{
  GPtrArray *GPA = NULL;

  GtkCellRenderer *renderer = NULL;
  GtkListStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;

  PxmModif *modif;


  g_assert (polchemdef != NULL);
  GPA = polchemdef->modifGPA;
  
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MDF_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING,
			      G_TYPE_STRING,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  modif = g_ptr_array_index (GPA, iter);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MDF_NAME, 
			      modif->name,
			      
			      COLUMN_MDF_ACTFORM, 
			      modif->actform,
			      
			      COLUMN_MDF_EDITABLE, FALSE,
			      -1);
	}
    }
  
  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (model));  

  g_object_unref (G_OBJECT (model));

  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_SINGLE);
  

  /* Modif Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  /*
    GtkTreeViewColumn *column = NULL;
    
    column = 
    gtk_tree_view_column_new_with_attributes (_("Name"), 
    renderer,
    "text", COLUMN_MDF_NAME,
    NULL); 
    
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  */
  
  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MDF_NAME,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_NAME);


  /* Modif Actform column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Actform"), renderer,
     "text", COLUMN_MDF_ACTFORM,
     "editable", COLUMN_MDF_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MDF_ACTFORM);

  return TRUE;
}


gboolean
polyxedit_ui_monomer_modif_setup_monomcodes_treeview (GtkTreeView *treeview,
						      PxmPolchemdef *polchemdef)
{
  GPtrArray *GPA = NULL;

  GtkCellRenderer *renderer = NULL;
  GtkListStore *model;
  GtkTreeIter tree_iter;

  gint iter = 0;

  PxmMonomer *monomer;



  g_assert (polchemdef != NULL);
  GPA = polchemdef->monomerGPA;
  
  /* Create the list store.
   */
  model = gtk_list_store_new (COLUMN_MNM_COL_COUNT, /*Numb. of columns*/
			      G_TYPE_STRING, /* monomer name */
			      G_TYPE_STRING, /* monomer code */
			      G_TYPE_STRING /* monomer formula */,
			      G_TYPE_BOOLEAN /* editability */);

  /* Add the items if GPA is non-null and there are items in it.
   */
  if (GPA != NULL)
    {
      for (iter = 0 ; iter < GPA->len ; iter++)
	{
	  /* Acquire an iterator.
	   */
	  gtk_list_store_append (model, &tree_iter);
	  
	  monomer = g_ptr_array_index (GPA, iter);
	  g_assert (monomer != NULL);
	  
	  gtk_list_store_set (model, &tree_iter,
			      
			      COLUMN_MNM_NAME, 
			      monomer->name,
			      
			      COLUMN_MNM_CODE, 
			      monomer->code,
			      
 			      COLUMN_MNM_FORMULA, 
			      monomer->formula,

			      COLUMN_MNM_EDITABLE, FALSE,
			      -1);
	}
    }

  gtk_tree_view_set_model (treeview, GTK_TREE_MODEL (model));  

  g_object_unref (G_OBJECT (model));

  gtk_tree_selection_set_mode (gtk_tree_view_get_selection 
			       (GTK_TREE_VIEW (treeview)),
			       GTK_SELECTION_MULTIPLE);
  

  /* Monomer Name column.
   */
  renderer = gtk_cell_renderer_text_new ();

  /*
    GtkTreeViewColumn *column = NULL;
    
    column = 
    gtk_tree_view_column_new_with_attributes (_("Name"), 
    renderer,
    "text", COLUMN_MNM_NAME,
    NULL); 
    
    gtk_tree_view_append_column (GTK_TREE_VIEW (treeview), column);
  */
  
  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Name"), renderer,
     "text", COLUMN_MNM_NAME,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_NAME);


  /* Monomer Code column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Code"), renderer,
     "text", COLUMN_MNM_CODE,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_CODE);


  /* Monomer Formula column.
   */
  renderer = gtk_cell_renderer_text_new ();

  gtk_tree_view_insert_column_with_attributes 
    (GTK_TREE_VIEW (treeview),
     -1, _("Formula"), renderer,
     "text", COLUMN_MNM_FORMULA,
     "editable", COLUMN_MNM_EDITABLE,
     NULL);
  
  g_object_set_data (G_OBJECT (renderer), "column", 
		     (gint *) COLUMN_MNM_FORMULA);
  
  return TRUE;
}


void
polyxedit_ui_monomer_modif_modify_button (GtkWidget *button,
					  gpointer data)
{
  gint pos_count = -1;
  
  gint full_success = 0;
  gint render_failure = 0;
  gint full_failure = 0;
  gint chem_failure = 0;
  
  GtkWidget *window = data;

  GtkTreeView * treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection* selection = NULL; 
  GtkTreeModel *model = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter;
  GtkTextBuffer *buffer = NULL;

  GArray *GA = NULL;

  PxmEditCtxt *editctxt = NULL;

  gchar *modif_name = NULL;
  gchar *errors = NULL;
  gchar *help = NULL;
  


  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), 
				      "available_modifications_treeview");
  
  /* Ensure right away that there is one modification selected.
   */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  selection = gtk_tree_view_get_selection (treeview);
  
  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&treeiter))
    {
      polyxmass_timeoutmsg_message_set 
	((GtkWindow *) window,
	 _("Select a modification to apply"),
	 POLYXMASS_NORM_MSG_TIMEOUT);
    
      return; /*no selection, just return. */
    }
  

  /* At this point treeiter points to the selected item in the treeview.
     This means that we must get the name out of the item.
   */
  /* Make sure you terminate calls to gtk_tree_model_get()
     with a '-1' value. We can do this call because we have the treeiter
     pointing to the selected node!
   */
  gtk_tree_model_get (model, &treeiter, 
		      COLUMN_MDF_NAME, &modif_name,
		      -1);

  g_assert (modif_name != NULL); /* Free it later.*/
  
  /* We now have the name of the modif that is selected in the
     treeview. We can go on and allocate the GArray that will hold
     all the positions where the modification has to occur. These
     positions will be analyzed in another function.
   */
  GA = g_array_new (TRUE, TRUE, (guint) sizeof (gint));

  pos_count = polyxedit_ui_monomer_modif_prepare_indices_array (window,
								GA);

  if (-1 == pos_count)
    return;

  /*
    debug_printf (("number of positions to modify: %d\n", pos_count));
  */
  
  /* At this stage we have an array of integers that represent all the
    indices at which the monomers of the polymer sequence should be
    modified using the modification that we got previously in the
    modif_name variable (see above). Time has come to perform the
    modification proper.
   */
  polyxedit_monomer_modif_modify_monomer_GA (GA,
					     modif_name,
					     editctxt, 
					     &errors,
					     &full_success,
					     &full_failure,
					     &chem_failure,
					     &render_failure);

  g_free (modif_name);
  
  /* If at least one chemical modification was performed, irrespective
     of the graphical rendering, the polymer sequence is modified.
  */
  if (render_failure != 0 || chem_failure != 0 || full_success != 0)
    polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
  
  help = g_strdup_printf (_("Full success = %d; Full failure = %d; "
			    "Chem failure = %d; Render failure = %d"),
			  full_success, full_failure, 
			  chem_failure, render_failure);
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					 help, 
					 POLYXMASS_LONG_MSG_TIMEOUT);
  
  g_free (help);
  

  /* If we got a string back in errors of length > 0, display its
     contents in the errors textview.
  */
  if (errors != NULL)
    {
      textview = g_object_get_data (G_OBJECT (window),
				    "monomer_modification_errors_textview");
      g_assert (textview != NULL);
      
      buffer = gtk_text_view_get_buffer (textview);
      
      /* Get start of buffer.
       */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter, 0);
      
      gtk_text_buffer_insert (buffer, &text_iter, errors, -1);
      
      g_free (errors);
    }
  
  /* Finally free the whole indices' GArray.
   */
  g_array_free (GA, TRUE);

  
  /* The selection polygon (if there was any around monomer(s) which
     have been modified) may have been partically "overwritten".
  */
  polyxedit_seqed_wnd_update_sel_polygon (editctxt->seqeditorctxt);
    

  /* The sequence and the selection.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return;
}

void
polyxedit_ui_monomer_modif_un_modify_button (GtkWidget *button,
					     gpointer data)
{
  gint pos_count = -1;
  
  gint full_success = 0;
  gint render_failure = 0;
  gint full_failure = 0;
  gint chem_failure = 0;
  
  GtkWidget *window = data;

  GtkTreeView * treeview = NULL;
  GtkTreeIter treeiter;
  GtkTreeSelection* selection = NULL; 
  GtkTreeModel *model = NULL;

  GtkTextView *textview = NULL;
  GtkTextIter text_iter;
  GtkTextBuffer *buffer = NULL;

  GArray *GA = NULL;

  PxmEditCtxt *editctxt = NULL;

  gchar *modif_name = NULL;
  gchar *errors = NULL;
  gchar *help = NULL;
  


  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  
  treeview = g_object_get_data (G_OBJECT (window), 
				      "available_modifications_treeview");
  
  /* Check if there is one modification selected. Indeed, the
     processing here is a bit different than for the modification:
     there can be no modification selected. If such is the case, that
     means that the un-modification processing will be performed
     whatever the current modification of the target monomers as
     determined later by the call to
     polyxedit_ui_monomer_modif_prepare_indices_array ().  Instead, if
     a modification is selected, then the un-modification will only
     effectively be performed on the monomers from the indices GArray
     that are modified with this modification precisely.
   */
  model = gtk_tree_view_get_model (treeview);
  g_assert (model != NULL);
  
  selection = gtk_tree_view_get_selection (treeview);
  
  if (FALSE == gtk_tree_selection_get_selected (selection,
						NULL,
						&treeiter))
    {
      /* No modification is selected. The un-modification will be
	 performed on absolutely all the monomers at the indices
	 obtained by calling
	 polyxedit_ui_monomer_modif_prepare_indices_array ().
      */
    }
  else
    {
      /* At this point treeiter points to the selected item in the treeview.
	 This means that we must get the name out of the item.
      */
      /* Make sure you terminate calls to gtk_tree_model_get()
	 with a '-1' value. We can do this call because we have the treeiter
	 pointing to the selected node!
      */
      gtk_tree_model_get (model, &treeiter, 
			  COLUMN_MDF_NAME, &modif_name,
			  -1);
      
      g_assert (modif_name != NULL); /* Free it later.*/
    }
  
  /* Now modif_name is either NULL or points to an allocated string
     containing the modification name.
  */

  /* We now have the name of the modif that is selected in the
     treeview. We can go on and allocate the GArray that will hold
     all the positions where the modification has to occur. These
     positions will be analyzed in another function.
   */
  GA = g_array_new (TRUE, TRUE, (guint) sizeof (gint));

  pos_count = polyxedit_ui_monomer_modif_prepare_indices_array (window,
								GA);

  if (-1 == pos_count)
    return;
  debug_printf (("number of positions to un-modify: %d\n", pos_count));
  
  
  /* At this stage we have an array of integers that represent all the
    indices at which the monomers of the polymer sequence should be
    modified using the modification that we got previously in the
    modif_name variable (see above). Time has come to perform the
    modification proper.
   */
  polyxedit_monomer_modif_un_modify_monomer_GA (GA,
						modif_name,
						editctxt, 
						&errors,
						&full_success,
						&full_failure,
						&chem_failure,
						&render_failure);
  
  if (modif_name != NULL)
    g_free (modif_name);
  /* If at least one chemical modification was performed, irrespective
     of the graphical rendering, the polymer sequence is modified.
  */
  if (render_failure != 0 || chem_failure != 0 || full_success != 0)
    polyxedit_seqed_wnd_set_polymer_modified (editctxt, TRUE);
  
  help = g_strdup_printf (_("Full success = %d; Full failure = %d; "
			    "Chem failure = %d; Render failure = %d"),
			  full_success, full_failure, 
			  chem_failure, render_failure);
  
  polyxmass_timeoutmsg_message_set  ((GtkWindow *) window,
					  help, 
					  POLYXMASS_NORM_MSG_TIMEOUT);
  
  g_free (help);
  

  /* If we got a string back in errors of length > 0, display its
     contents in the errors textview.
  */
  if (errors != NULL)
    {
      textview = g_object_get_data (G_OBJECT (window),
				    "monomer_modification_errors_textview");
      g_assert (textview != NULL);
      
      buffer = gtk_text_view_get_buffer (textview);
      
      /* Get start of buffer.
       */
      gtk_text_buffer_get_iter_at_offset (buffer, &text_iter, 0);
      
      gtk_text_buffer_insert (buffer, &text_iter, errors, -1);
      
      g_free (errors);
    }
  
  /* Finally free the whole indices' GArray.
   */
  g_array_free (GA, TRUE);

  
  /* The selection polygon (if there was any around monomer(s) which
     have been modified) may have been partically "overwritten".
  */
  polyxedit_seqed_wnd_update_sel_polygon (editctxt->seqeditorctxt);
    

  /* The sequence and the selection.
   */
  polyxedit_ui_masses_display_wnd_update_sequence_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  polyxedit_ui_masses_display_wnd_update_selection_masses
    ((PxmEditCtxt *) polyxedit_last_editctxt);
  
  return;
}


gint
polyxedit_ui_monomer_modif_prepare_indices_array (GtkWidget *window,
						  GArray *GA)
{
  gint iter = 0;
  gint max = 0;
  gint target = -1;
  
  gboolean valid = FALSE;
  gboolean selected = FALSE;

  gchar *code = NULL;
  gchar *text = NULL;
  
  GtkWidget *entry = NULL;

  GtkTreeView * treeview = NULL;
  GtkTreeSelection* selection = NULL; 
  GtkTreeIter treeiter;
  GtkTreeModel *model = NULL;

  GPtrArray *monomerGPA = NULL;

  PxmMonomer *monomer_iter = NULL;
  PxmMonomer *monomer_sel = NULL;

  PxmEditCtxt *editctxt = NULL;



  g_assert (window != NULL);
  g_assert (GA != NULL);

  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  monomerGPA = editctxt->polymer->monomerGPA;
  g_assert (monomerGPA != NULL);
  
  
  /* Get the target of the modification operation.
   */
  target = polyxedit_ui_monomer_modif_get_target (window);

  if (target == MODIF_TARGET_ALL_MONOMERS)
    {
      /* Each single monomer in the sequence must be modified! Thus
	 put in the array of all the monomer indices for which
	 modification should occur all the indices of the polymer! That
	 is so easy!
      */
      for (iter = 0; iter < monomerGPA->len; iter++)
	GA = g_array_append_val (GA, iter);
    }

  else if (target == MODIF_TARGET_SEL_MONOMER
	   || target == MODIF_TARGET_ALL_MONOMERS_CODE)
    {
      /* Modification should be applied on the selected monomer or on
	 all the monomers of the sequence that have the same code as
	 the selected one. Which means for both cases only one monomer
	 should currently be selected in the sequence editor. Reuse
	 the iter variable to get the index of the putatively selected
	 monomer: that's recycling!
       */
      iter = -1;
      monomer_sel = polyxedit_seqed_wnd_get_selected_monomer (editctxt,
							      &iter);

      if (monomer_sel == NULL)
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
						 _("Select a single monomer"),
						 POLYXMASS_NORM_MSG_TIMEOUT);
	  return -1;
	}
      
      /* A selected monomer may not have an index = -1. Good sanity
	 check.
       */
      g_assert (iter != -1);
      
      /* We can handle the modification right away.
       */
      if (target == MODIF_TARGET_SEL_MONOMER)
	{
	  /* Only the monomer that is currently selected should be
	     modified. And we just finished getting its index, so that
	     we can put it in the array of the indices (actually one
	     only in our present case) of the monomers to be modified.
	   */
	  GA = g_array_append_val (GA, iter);
	}
      
      /* If it is not SEL_MONOMER, then it is ALL_MONOMERS_CODE!
       */
      else 
	{
	  /* All monomers that have the same code as the one which is
	     currently selected should be modified. We have the code
	     right away because we have the monomer!
	   */

	  /* Iterate in the polymer sequence and for each monomer
	     check if it has the same code as the one we have at
	     hand. If so, just add the index to the array of positions
	     that should be modified.
	  */
	  for (iter = 0; iter < monomerGPA->len; iter++)
	    {
	      monomer_iter = g_ptr_array_index (monomerGPA, iter);
	      
	      if (0 == strcmp (monomer_sel->code, monomer_iter->code))
		GA = g_array_append_val (GA, iter);
	    }
	} 
    }
  /* end of
     else if (target == MODIF_TARGET_SEL_MONOMER
     || target == MODIF_TARGET_ALL_MONOMERS_CODE)
  */

  else if (target == MODIF_TARGET_ALL_MONOMERS_LIST)
    {
      /* The code(s) of the monomers for which the modification 
       * is to be applied is(are) currently selected in the 
       * codes' treeview. We'll have to check that the list actually 
       * contains selected items:
       */
      
      /* Get the treeview widget in which the user may select no/one/more
	 monomer to be modified.
      */
      treeview = g_object_get_data (G_OBJECT (window), 
					    "target_monomers_treeview");
      
      model = gtk_tree_view_get_model (treeview);
      g_assert (model != NULL);
      
      /* Remember that we have a multiple-selection treeview here,
	 conversely to what was for the modification treeview. This is
	 why we cannot ask for the selection as we did in the modif
	 treeview code above.
      */
      selection = gtk_tree_view_get_selection (treeview);
      g_assert (selection != NULL);
      
      /* Get the first treeiter in the list.
       */
      valid = gtk_tree_model_get_iter_first (model, &treeiter);
      while (valid)
	{
	  /* Walk through the treeveiw list of monomers , reading each
	     row, only if actually selected.
	   */
	  selected = gtk_tree_selection_iter_is_selected (selection, 
							  &treeiter);
      
	  if (selected != TRUE)
	    {
	      /* Go the the next treeview item.
	       */
	      valid = gtk_tree_model_iter_next (model, &treeiter);
	      
	      continue;
	    }
	  	  
	  /* The row is selected, thus we can ask that the contents of
	     the CODE column be allocated in 'code'. Make sure to
	     terminate the call to gtk_tree_model_get() * with a '-1'
	     value.
	  */
	  gtk_tree_model_get (model, &treeiter, 
			      COLUMN_MNM_CODE, &code,
			      -1);
	  
	  /* We have a monomer code allocated in code, that we should
	     compare with the monomer code of each monomer iterated in
	     the polymer's monomerGPA array. If there is a match
	     that means that the iter value should be stored in the
	     array of all the positions where the modification should
	     occur.
	  */
	  for (iter = 0; iter < monomerGPA->len; iter++)
	    {
	      monomer_iter = g_ptr_array_index (monomerGPA, iter);
	      
	      if (0 == strcmp (monomer_iter->code, code))
		GA = g_array_append_val (GA, iter);
	      
	    }

	  /* Finished doing the comparison work with the currently 
	     iterated list's monomer code 'code'. We have to free it
	     because the gtk_tree_model_get () call above allocated it.
	  */
	  g_free (code);
	  
	  /* And now go the the next treeview item.
	   */
	  valid = gtk_tree_model_iter_next (model, &treeiter);
	}
    }
  /* end of
     else if (target == MODIF_TARGET_ALL_MONOMERS_LIST)
  */
  
  else if (target == MODIF_TARGET_SPEC_POS_MONOMERS)
    {
      /* The user should have written some positions in the entry.
       */
      entry = g_object_get_data (G_OBJECT (window), 
				 "target_monomers_spec_pos_entry");
      g_assert (entry != NULL);
      
      text = gtk_editable_get_chars (GTK_EDITABLE (entry), 0, -1);
      g_assert (text != NULL);
      
      /* We should make sure that there is no single
	 space/tab/whatever character in the string that we got.
       */
      text = libpolyxmass_globals_unspacify_string (&text);

      if (strlen (text) <= 0)
	{
	  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					     _("Enter valid position(s)"),
					     POLYXMASS_NORM_MSG_TIMEOUT);

	  g_free (text);
	  
	  return -1;
	}
      
      max = editctxt->polymer->monomerGPA->len - 1;
      
      if (-1 == libpolyxmass_globals_fill_int_array_match_exp (text, GA, max))
	g_critical (_("%s@%d: failed to process string: '%s'\n"),
	       __FILE__, __LINE__, text);
      
      g_free (text);
    }
  /* end of 
     else if (target == MODIF_TARGET_SPEC_POS_MONOMERS)
  */

  return GA->len;
}


gint
polyxedit_ui_monomer_modif_get_target (GtkWidget *window)
{
  GSList *group = NULL;
  GtkWidget *radio = NULL;
  GSList *iter = NULL;
  
  
  g_assert (window != NULL);
  
  radio = g_object_get_data (G_OBJECT (window), 
			     "target_selected_monomer_radiobutton");
  g_assert (radio != NULL);
  
  group = gtk_radio_button_get_group (GTK_RADIO_BUTTON (radio));
  
  iter = group;
  
  
  while (iter != NULL)
    {
      radio = iter->data;
      
      if (TRUE == 
	  gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (radio)))
	{
	  if (0 == strcmp ("target_selected_monomer_radiobutton", 
		      gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_SEL_MONOMER;
	  
	  else if  (0 == strcmp ("target_monomers_same_code_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS_CODE;

	  else if  (0 == strcmp ("target_monomers_from_list_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS_LIST;

	  else if  (0 == strcmp ("target_all_monomers_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_ALL_MONOMERS;

	  else if  (0 == strcmp ("target_monomers_spec_pos_radiobutton", 
			    gtk_widget_get_name (GTK_WIDGET (radio))))
	    return MODIF_TARGET_SPEC_POS_MONOMERS;

	  else
	    g_warning (_("%s@%d: failed to get the radiobutton widget\n"),
		   __FILE__, __LINE__);
	}

      iter = g_slist_next (iter);
    }
  return -1;
}










/* WINDOW LIFE-CYCLE FUNCTIONS.
 */
void
polyxedit_ui_monomer_modif_wnd_really_close (GtkWidget *window)
{
  gint count = 0;

  
  g_assert (window != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
  
  gtk_widget_destroy (window);
}


gboolean
polyxedit_ui_monomer_modif_wnd_delete_event (GtkWidget *window, 
					     GdkEventAny *event, 
					     gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  help = g_strdup_printf ("monomer_modif_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called, thus
     destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT
		     (editctxt->seqeditorctxt->sequence_editor_wnd), 
		     help, NULL);
  g_free (help);
    
  return TRUE;
}

  
gboolean
polyxedit_ui_monomer_modif_wnd_destroy_event (GtkWidget *window, 
					      GdkEventAny *event, 
					      gpointer data)
{
  return FALSE;
}
