/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 59 Temple Place - Suite 330,
   Boston, MA 02111-1307, USA.
*/
#include "polyxedit-ui-self-read.h"
#include "polyxedit-ui-seqed-wnd.h"
#include "polyxmass-self-read.h"


GtkWidget *
polyxedit_self_read_wnd_setup (PxmEditCtxt *editctxt)
{
  GtkWidget *widget = NULL;
  GtkWidget *window = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar *help = NULL;


  g_assert (editctxt != NULL);



  gui_file = 
    g_strdup_printf ("%s/polyxedit-self-read.glade", userspec->gladedir);

  xml = glade_xml_new (gui_file, "self_read_wnd", 
		       PACKAGE);

  g_free (gui_file);
  
  if (xml == NULL)
    {
      g_error (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      g_free (gui_file);
      
      return NULL;
    }
  
  window = glade_xml_get_widget (xml, "self_read_wnd");
  
  if (window == NULL)
    {
      g_critical (_("%s@%d: failed to create the self-read config window\n"),
	     __FILE__, __LINE__);
      
      g_object_unref (G_OBJECT (xml));
      
      return NULL;
    }

  /* Immediately set to the window a pointer to the editctxt:
   */
  g_object_set_data (G_OBJECT (window), "editctxt", editctxt);


  /* We may need to displayed timed-out messages in here:
   */
  widget = glade_xml_get_widget (xml, "messages_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "messages_entry", widget);
  

  widget = glade_xml_get_widget (xml, "polymer_sequence_name_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polymer_sequence_name_entry", widget);
  
  gtk_entry_set_text (GTK_ENTRY (widget), 
		      editctxt->polymer->plminfo->name);

  widget = glade_xml_get_widget (xml, "polymer_sequence_id_number_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "polymer_sequence_id_number_entry", widget);

  help = g_strdup_printf ("%p", editctxt);
  gtk_entry_set_text (GTK_ENTRY (widget), help);
  g_free (help);

  
  widget = glade_xml_get_widget (xml, "update_sequence_region_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "update_sequence_region_button", widget);

  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_self_read_wnd_sequence_region_button), 
		    window);

  widget = glade_xml_get_widget (xml, "sel_start_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "sel_start_pos_entry", widget);
  
  widget = glade_xml_get_widget (xml, "sel_end_pos_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "sel_end_pos_entry", widget);
  
  widget = glade_xml_get_widget (xml, "whole_sequence_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "whole_sequence_checkbutton", widget);

  /* At this point we should be able to set the values to the text
     entries above, if some portion of the protein is already selected...
  */
  polyxedit_self_read_wnd_sequence_region_button (NULL,
						   window);
  
  widget = glade_xml_get_widget (xml, "monomer_names_reading_radiobutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "monomer_names_reading_radiobutton", widget);
  
  widget = glade_xml_get_widget (xml, "monomer_codes_reading_radiobutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "monomer_codes_reading_radiobutton", widget);
  
  widget = glade_xml_get_widget (xml, "monomer_modif_reading_checkbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "monomer_modif_reading_checkbutton", widget);
  
  widget = glade_xml_get_widget (xml, "start_self_read_delay_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "start_self_read_delay_spinbutton", widget);
  
  widget = glade_xml_get_widget (xml, "inter_monomer_delay_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "inter_monomer_delay_spinbutton", widget);
  
  widget = glade_xml_get_widget (xml, "extra_delay_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "extra_delay_spinbutton", widget);
  
  widget = glade_xml_get_widget (xml, "every_other_monomer_spinbutton");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "every_other_monomer_spinbutton", widget);
  
  widget = glade_xml_get_widget (xml, "select_output_file_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "select_output_file_button", widget);
  
  g_signal_connect (G_OBJECT (widget),
		    "clicked",
		    G_CALLBACK 
		    (polyxedit_self_read_wnd_select_output_file_button), 
		    window);

  widget = glade_xml_get_widget (xml, "output_filename_entry");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window), 
		     "output_filename_entry", widget);
  
  widget = glade_xml_get_widget (xml, "make_self_read_playlist_button");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (window),
		     "make_self_read_playlist_button", widget);
  
  g_signal_connect 
    (G_OBJECT (widget),
     "clicked",
     G_CALLBACK (polyxedit_self_read_wnd_make_self_read_playlist_button), 
     window);
  
  
  gtk_widget_show_all (GTK_WIDGET (window));


  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "delete_event",
		    G_CALLBACK (polyxedit_self_read_wnd_delete_event), 
		    editctxt);
  
  /* Signal / callback connections.
   */
  g_signal_connect (G_OBJECT (window),
		    "destroy_event",
		    G_CALLBACK (polyxedit_self_read_wnd_destroy_event), 
		    editctxt);



  /* We have finished setting up the window, and so also using the xml
     data, unref them.
   */
  g_object_unref (G_OBJECT (xml));
  
  /* Set this window pointer as a full datum to the polymer sequence
     editor window, so that when it is closed this window is closed
     also. 

     There might be more than one window of the present kind opened
     for a given polymer seqence editing window, and we do not want
     that the second window destroys the datum name of the first
     window, so we create an uambiguous datum name each time.
  */
  help = g_strdup_printf ("self_read_wnd-%p", window);
  
  g_object_set_data_full 
    (G_OBJECT (editctxt->seqeditorctxt->sequence_editor_wnd),
     help, GTK_WIDGET (window), 
     (GDestroyNotify) polyxedit_self_read_wnd_really_close);
  
  g_free (help);
 
  return window;
}



gboolean
polyxedit_self_read_wnd_get_sequence_interval (GtkWidget *window,
						gint *start_idx,
						gint *end_idx)
{
  GtkWidget *widget = NULL;

  gchar *help = NULL;
  gint len = 0;

  gint start = 0;
  gint end = 0;
  
  PxmEditCtxt *editctxt = NULL;

  


  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  g_assert (start_idx != NULL && end_idx != NULL);
  

  g_assert (editctxt->polymer != NULL);
  g_assert (editctxt->polymer->monomerGPA != NULL);
  len = editctxt->polymer->monomerGPA->len;

  if (len == 0)
    {
      return -1;
    }
  

  /* We must check if the user wants that the calculation be made for
     the whole sequence or for the selection region of the polymer sequence
     only, or for the interval that is defined with the start_pos and 
     end_pos entry widgets.
  */
  widget = 
    g_object_get_data (G_OBJECT (window), "whole_sequence_checkbutton");
  g_assert (widget != NULL);

  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    {
      start = 0;
      end = len;
    }
  else
    {
      /* We should check what the start_pos and end_pos entries say:
       */
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  start = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == libpolyxmass_globals_strtoi (help, &start, 10))
	    start = 0;
	  else
	    {
	      /* Remember that the user "counts" monomers starting at 1,
		 and we need to count starting at 0, because we use indexes
		 and not positions. So decrement by one the value entered
		 by the user.
	      */
	      start--;
	    }
	}
      
      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      
      help = gtk_editable_get_chars (GTK_EDITABLE (widget), 0, -1);
      g_assert (help != NULL);
      
      if (strlen (help) <= 0)
	{
	  end = 0;
	}
      else
	{
	  /* There is a string in the entry. Convert it to gint.
	   */
	  if (FALSE == libpolyxmass_globals_strtoi (help, &end, 10))
	    end = 0;
	  else
	    {
	      /* Remember that for the *end_idx, the calculations need
		 to have it incremented by one with respect to the real
		 last monomer index. So, contrary to start_idx, we do not
		 change its value.
		 
		 end;
	      */
	    }
	}

      /* Now make some final checks:
       */

      if (start > len)
	start = len;

      if (start < 0)
	start = 0;
      
      if (end > len)
	end = len;

      if (end < 0)
	end = 0;
    }

  *start_idx = start;
  *end_idx = end;
  
  return TRUE;
}


void
polyxedit_self_read_wnd_sequence_region_button (GtkWidget *button,
						 gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;
  
  PxmEditCtxt *editctxt = NULL;
  
  gchar *help = NULL;
  
  gint start = 0;
  gint end = 0;
  
  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);
  
  /* Prepare the variables from the sequence editor window, so that
     we can put them immediately in the start/end positions's entry
     widgets.
  */
  if (FALSE != polyxedit_seqed_wnd_get_selection_indices (editctxt,
							  &start, &end))
    {
      /* Remember that due to the handling of the polymer sequence
	 displayed in the sequence editor, the end value that is 
	 returned is indeed corresponding to the real index of the last
	 selected monomer + 1, so care should be taken not to outbound
	 the polymer sequence array.
      */
      
      /* We show the (start + 1) value because we want to show
	 a position and not an index, while the end needs not be
	 incremented because it is by itself already incremented.
      */      
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      help = g_strdup_printf ("%d", start + 1);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      widget =  g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);
      help = g_strdup_printf ("%d", end);
      gtk_entry_set_text (GTK_ENTRY (widget), help);
      g_free (help);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);

      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), FALSE);
    }
  else
    {
      widget = g_object_get_data (G_OBJECT (window), "sel_start_pos_entry");
      g_assert (widget != NULL);

      widget = g_object_get_data (G_OBJECT (window), "sel_end_pos_entry");
      g_assert (widget != NULL);

      /* If there is a sequence selected, by default we disable the
	 "whole sequence" checkbutton. Otherwise we enable it.
      */
      widget = g_object_get_data (G_OBJECT (window), 
				  "whole_sequence_checkbutton");
      g_assert (widget != NULL);
      
      gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (widget), TRUE);
    }
  
  return;
}


void
polyxedit_self_read_wnd_select_output_file_button (GtkWidget *button,
						    gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *dialog = NULL;
  GtkWidget *entry = NULL;
  
  gint result = 0;
  
  gchar *filename;

  g_assert (window != NULL);
  
  dialog = gtk_file_chooser_dialog_new (_("Select Output File"),
					GTK_WINDOW (window),
					GTK_FILE_CHOOSER_ACTION_SAVE,
					GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
					GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
					NULL);
  
  result = gtk_dialog_run (GTK_DIALOG (dialog));
  
  if (result == GTK_RESPONSE_ACCEPT)
    {
      filename = gtk_file_chooser_get_filename (GTK_FILE_CHOOSER (dialog));
     
      /* We have the filename, so that we can set it to the widget in
	 the window.*/
      entry = g_object_get_data (G_OBJECT (window), 
				 "output_filename_entry");
      g_assert (entry != NULL);

      gtk_entry_set_text (GTK_ENTRY (entry), filename);
      
      g_free (filename);
    }
  else if (result == GTK_RESPONSE_CANCEL)
    ;
  
  gtk_widget_destroy (dialog);
  
  return ;
}




void
polyxedit_self_read_wnd_make_self_read_playlist_button (GtkWidget *button,
							  gpointer data)
{
  GtkWidget *window = data;
  GtkWidget *widget = NULL;

  PxmEditCtxt *editctxt = NULL;

  gboolean name_or_code;
  gboolean modif;
  gboolean result = FALSE;
  
  gint len = 0;
  gint start_idx = 0;
  gint end_idx = 0;
  
  const gchar *filename = NULL;
  
  gint start_delay = 0;
  gint inter_monomer_delay = 0;
  gint  extra_delay = 0;
  gint every_other_monomer = 0;
  



  g_assert (window != NULL);
  
  editctxt = g_object_get_data (G_OBJECT (window), "editctxt");
  g_assert (editctxt != NULL);

  g_assert (editctxt->polymer != NULL);
  g_assert (editctxt->polymer->monomerGPA != NULL);
  len = editctxt->polymer->monomerGPA->len;

  if (len == 0)
    return;
  
  /* We have a number of things to know first: 

     First OFF : initialize the array of monomerspec instances so that
     they have all the names of the ogg files for the monomer codes
     and names... Same for modifs.

     Is the whole sequence or a region to be self-spoken out ?
     Should the monomer names' ogg files be played or their codes' ?
  */

  result = polyxedit_self_read_wnd_get_sequence_interval (window,
							   &start_idx,
							   &end_idx);
  if (result == FALSE)
    {
      g_critical (_("%s@%d: failed to get sequence interval\n"),
	     __FILE__, __LINE__);
      
      return;
    }
  
  /* OK, by that time we should know what sequence portion should
     read itself out.
  */

  widget = g_object_get_data (G_OBJECT (window), 
			      "monomer_codes_reading_radiobutton");
  g_assert (widget != NULL);
  
  if (TRUE == gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget)))
    name_or_code = CODE_SOUND;
  else
    name_or_code = NAME_SOUND;

  /* Should the modifications in the monomers read themselves out ?
   */
  widget = g_object_get_data (G_OBJECT (window), 
			      "monomer_modif_reading_checkbutton");
  g_assert (widget != NULL);
  
  modif = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (widget));
  
  
  /* We now know if the ogg files to be used for the self-read out
     are the ones for the monomer codes or for the monomer names.
  */
  widget = g_object_get_data (G_OBJECT (window), 
			      "start_self_read_delay_spinbutton");
  g_assert (widget != NULL);
  start_delay = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));


  widget = g_object_get_data (G_OBJECT (window), 
			      "inter_monomer_delay_spinbutton");
  g_assert (widget != NULL);
  inter_monomer_delay = 
    (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));


  widget = g_object_get_data (G_OBJECT (window), 
			      "extra_delay_spinbutton");
  g_assert (widget != NULL);
  extra_delay = (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));


  widget = g_object_get_data (G_OBJECT (window), 
			      "every_other_monomer_spinbutton");
  g_assert (widget != NULL);
  every_other_monomer = 
    (gint) gtk_spin_button_get_value (GTK_SPIN_BUTTON (widget));

  /* Before asking that the sound file be created, we should check if
     the user has asked that a file be created specifically. When the
     user asks that a filename be used, that filename is presented in
     its corresponding text entry widget; so read it from there.
  */
  widget = g_object_get_data (G_OBJECT (window), 
			      "output_filename_entry");
  g_assert (widget != NULL);
  
  filename = (gchar *) gtk_entry_get_text (GTK_ENTRY (widget));
  
  if (NULL == filename || strlen (filename) <= 0)
    {
      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("Define an output file first."),
					POLYXMASS_MEDI_MSG_TIMEOUT);
      return;
    }
  
  /* At this point we have a filename so that it becomes possible to
     send ogg sound data to the FILE after having opened it.
  */
  if (-1 == 
      polyxedit_self_read_wnd_create_polseq_playlist (editctxt,
						       (gchar *) filename,
						       name_or_code,
						       modif,
						       start_idx,
						       end_idx,
						       start_delay,
						       inter_monomer_delay,
						       extra_delay,
						       every_other_monomer))
    {
      g_critical (_("%s@%d: failed to create sound list file for sequence region.\n"),
	     __FILE__, __LINE__);

      polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
					_("The operation failed"), 
					POLYXMASS_LONG_MSG_TIMEOUT);
      return;
    }
  
  polyxmass_timeoutmsg_message_set ((GtkWindow *) window,
				    _("The operation succeeded"), 
				    POLYXMASS_NORM_MSG_TIMEOUT);
    
  return;
}





gint
polyxedit_self_read_wnd_create_polseq_playlist (PxmEditCtxt *editctxt,
						 gchar *filename,
						 gint name_or_code,
						 gboolean modif,
						 gint start_idx,
						 gint end_idx,
						 gint start_delay,
						 gint inter_monomer_delay,
						 gint extra_delay,
						 gint every_other_monomer)
{
  gint iter = 0;
  gint jter = 0;
  gint count = 0;
  gint success_count = 0;

  gint result = 0;
    
  PxmMonomerSpec *mnmspec = NULL;
  PxmModifSpec *mdfspec = NULL;

  PxmMonomer *monomer = NULL;
  PxmPolchemdefCtxt *polchemdefctxt = NULL;
  PxmPolymer *polymer = NULL;

  GPtrArray *mnmspecGPA = NULL;
  GPtrArray *mdfspecGPA = NULL;
  
  gchar *silence_sound = NULL;
  gchar *sound = NULL;
  gchar *help = NULL;
      
  FILE *file = NULL;
  
  

  /* We get the name of a file, that we have to create in order to
     send the different filenames to it according to the sequence
     portion contained between start_idx and end_idx in the polymer
     sequence contained in 'editctxt'. That will create the playlist.
  */
  g_assert (filename != NULL);
  
  g_assert (editctxt != NULL);
  polymer = editctxt->polymer;
  g_assert (polymer != NULL);
  polchemdefctxt = editctxt->polchemdefctxt;
  g_assert (polchemdefctxt != NULL);

  mnmspecGPA = polchemdefctxt->mnmspecGPA;
  g_assert (mnmspecGPA != NULL);
  mdfspecGPA = polchemdefctxt->mdfspecGPA;
  g_assert (mdfspecGPA != NULL);
  
  
  file = fopen (filename, "w");
  
  if (NULL == file)
    {
      g_critical (_("%s@%d: failed to open sound list file: '%s'\n"),
	     __FILE__, __LINE__, filename);
      
      return -1;
    }
  
  /* Great, at this point we know we can start creating the playlist
     file with each file name corresponding to the sequence monomers.

     Iterate in the polymer sequence array of monomers
     (editctxt->polymer->monomerGPA array), between indices start_idx
     and end_idx, and for each iterated monomer:

     Determine the filename of the monomer being iterated, and 
     send it to the playlist.
  */
  if (start_delay > 0 || inter_monomer_delay > 0 || 
      (extra_delay > 0 && every_other_monomer > 0))
    {
      help = 
	polyxmass_self_read_get_silent_sound_file_name (polchemdefctxt);
      
      if (help == NULL)
	{
	  g_critical (_("%s@%d: failed to get silence sound file name.\n"),
		 __FILE__, __LINE__);
	  
	  return -1;
	}
      silence_sound = g_strdup_printf ("%s\n", help);
      
      g_free (help);
    }
  
  if (start_delay > 0)
    {
      for (jter = 0 ; jter < start_delay ; jter++)
	{
	  result = fputs (silence_sound, file);

	  if (result == EOF || result < 0)
	    {
	      g_critical (_("%s@%d: failed to write string to the sound list file: '%s'\n"),
		     __FILE__, __LINE__, filename);
	      
	      fclose (file);

	      g_free (silence_sound);
	  	      
	      return -1;
	    }
	}
    }
  
  for (iter = start_idx ; iter < end_idx; iter++)
    {
      /* First get the monomer that were are iterating upon in the 
	 polymer sequence.
      */
      monomer = g_ptr_array_index (polymer->monomerGPA, iter);
      g_assert (monomer != NULL);
      
      mnmspec = libpolyxmass_monomerspec_get_ptr_by_code (mnmspecGPA,
							  monomer->code);
      g_assert (mnmspec != NULL);
      g_assert (mnmspec->name != NULL);
      
      if (NAME_SOUND == name_or_code)
	{
	  if (NULL == mnmspec->name_sound)
	    {
	      g_critical (_("%s@%d: failed to create sound list file: '%s'. No sound"
		       " file name for monomer name: '%s'\n"),
		     __FILE__, __LINE__, filename, monomer->name);
	      
	      fclose (file);
	      
	      g_free (silence_sound);

	      return -1;
	    }
	  
	  sound = g_strdup_printf ("%s\n", mnmspec->name_sound);
	  
	  result = fputs (sound, file);
	  
	  g_free (sound);
	  
	  if (result == EOF || result < 0)
	    {
	      g_critical (_("%s@%d: failed to write string to the sound list file: '%s'\n"),
		     __FILE__, __LINE__, filename);
	      
	      fclose (file);
	      
	      g_free (silence_sound);

	      return -1;
	    }
	}
      /* End of 
	 if (NAME_SOUND == name_or_code)
      */
      else if (CODE_SOUND == name_or_code)
	{
	  if (NULL == mnmspec->code_sound)
	    {
	      g_critical (_("%s@%d: failed to create the sound list file: '%s'. No sound"
		       " file name for monomer code: '%s'\n"),
		     __FILE__, __LINE__, filename, monomer->code);
	      
	      fclose (file);
	      
	      g_free (silence_sound);

	      return -1;
	    }
	  
	  sound = g_strdup_printf ("%s\n", mnmspec->code_sound);
	  
	  result = fputs (sound, file);
	  
	  g_free (sound);
	  
	  if (result == EOF || result < 0)
	    {
	      g_critical (_("%s@%d: failed to write string to the sound list file: '%s'\n"),
		     __FILE__, __LINE__, filename);
	      
	      fclose (file);
	      
	      g_free (silence_sound);

	      return -1;
	    }
	}
      else
	g_assert_not_reached ();
      
      /* At this point we know we could send the sound file name for
	 the currently iterated monomer.
      */
      success_count++;
      count++;

      /* Now we should send the name of the modification that could be
	 sitting in the monomer, if the 'modif' param were TRUE.
      */
      if (TRUE == modif)
	{
	  help = pxmchem_monomer_get_modif_name (monomer);
	  
	  if (NULL != help)
	    {
	      mdfspec = libpolyxmass_modifspec_get_ptr_by_name (mdfspecGPA,
								help);
	      g_free (help);
	      
	      g_assert (mdfspec != NULL);
	      g_assert (mdfspec->name != NULL);
	      
	      if (NULL != mdfspec->name_sound)
		{
		  sound = g_strdup_printf ("%s\n", mdfspec->name_sound);
		  
		  result = fputs (sound, file);
		  
		  g_free (sound);
		  
		  if (result == EOF || result < 0)
		    {
		      g_critical (_("%s@%d: failed to write string to "
			       "the sound list file: '%s'\n"),
			     __FILE__, __LINE__, filename);
		      
		      fclose (file);
		      
		      g_free (silence_sound);
		      
		      return -1;
		    }
		}
	      /* End of 
		 if (NULL != mdfspec->name_sound)
	      */
	    }
	  /* End of
	     if (NULL != help)
	  */
	}
      /* End of
	 if (TRUE == modif)
      */
      

      /* Check if the user asked that between each monomer a delay be 
	 inserted between monomer sounds.
      */
      for (jter = 0 ; jter < inter_monomer_delay ; jter++)
	{
	  result = fputs (silence_sound, file);

	  if (result == EOF || result < 0)
	    {
	      g_critical (_("%s@%d: failed to write string to the sound list file: '%s'\n"),
		     __FILE__, __LINE__, filename);
	      
	      fclose (file);
	      
	      g_free (silence_sound);

	      return -1;
	    }
	}
      
      /* Also check if we have read 'every_other_monomer' monomers, in which
	 case, we might have to inser an extra delay.
      */
      if (count == every_other_monomer)
	{
	  for (jter = 0 ; jter < extra_delay ; jter++)
	    {
	      result = fputs (silence_sound, file);
	      
	      if (result == EOF || result < 0)
		{
		  g_critical (_("%s@%d: failed to write string to the sound list file: '%s'\n"),
			 __FILE__, __LINE__, filename);
		  
		  fclose (file);
		  
		  g_free (silence_sound);

		  return -1;
		}
	    }
	  count = 0;
	}
    }
  /* End of 
     for (iter = start_idx ; iter < end_idx; iter++)
  */
  
  /* At this point, the sequence should be in the file. We can close
     that file now.
  */
  if (EOF == fclose (file))
    {
      g_critical (_("%s@%d: failed to close [fclose()] the FILE* : '%s'\n"), 
	     __FILE__, __LINE__, (gchar *) g_strerror);
    }
  
  /*
    debug_printf (("Count of sound writes: %d", success_count));
  */

#if 0
  help = g_strdup_printf ("xmms %s&", filename);
  
  system (help);
  
  g_free (help);
#endif

  return success_count;
}


/* WINDOW LIFE-CYCLE FUNCTIONS.
 */

void
polyxedit_self_read_wnd_really_close (GtkWidget *window)
{
  gint count = 0;

  
  g_assert (window != NULL);
  

  /* 
     Prior to closing the window, we want to make sure that no
     pending timed-out messages are there...
  */
  count = polyxmass_timeoutmsg_messages_remove ((GtkWindow *) window);
   
  gtk_widget_destroy (window);
}


gint
polyxedit_self_read_wnd_delete_event  (GtkWidget *window, 
					GdkEventAny *event, 
					gpointer data)
{
  PxmEditCtxt *editctxt = data;
  
  gchar *help = NULL;
  

  g_assert (window != NULL);
  g_assert (editctxt != NULL);
  

  help = g_strdup_printf ("self_read_wnd-%p", window);


  /* We set the datum to NULL, then the GDestroyNotify callback that
     was specified when the datum was set first (that is
     xxx_wnd_really_close () function above) will be called,
     thus destroying *this window, which is exactly what we want !
  */
  g_object_set_data (G_OBJECT
		     (editctxt->seqeditorctxt->sequence_editor_wnd), 
		     help, NULL);
  g_free (help);
    
  return TRUE;
}


gint
polyxedit_self_read_wnd_destroy_event  (GtkWidget *window, 
					 GdkEventAny *event, 
					 gpointer data)
{
  return FALSE;
}




