/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/
#include "polyxcalc-globals.h"

#include "polyxcalc-ui-calculator.h"
#include "polyxcalc-ui-chempad.h"
#include "polyxcalc-ui-recorder.h"


/* CHEMPAD KEY/GEO HELPER FUNCTIONS
 */
PxmChempadkey *
polyxcalc_chempadkey_new (void)
{
  PxmChempadkey *cpk = NULL;
  
  cpk = g_malloc0 (sizeof (PxmChempadkey));
  
  cpk->label = g_strdup ("");
  cpk->actform = g_strdup ("");
  cpk->tooltip = g_strdup ("");

  return cpk;
}


PxmChempadgeo *
polyxcalc_chempadgeo_new (void)
{
  PxmChempadgeo *cpg = NULL;
  
  cpg = g_malloc0 (sizeof (PxmChempadgeo));
  
  cpg->rows = 0;
  cpg->cols=0;
  
  return cpg;
}


gboolean
polyxcalc_chempadkey_set_label (PxmChempadkey *cpk, gchar *label)
{
  g_assert (cpk != NULL);
  g_assert (label != NULL);
  
  g_assert (cpk->label != NULL);
  
  g_free (cpk->label);
  
  cpk->label = g_strdup (label);
  
  return TRUE;
}


gboolean
polyxcalc_chempadkey_set_actform (PxmChempadkey *cpk, gchar *actform)
{
  g_assert (cpk != NULL);
  g_assert (actform != NULL);
  
  g_assert (cpk->actform != NULL);
  
  g_free (cpk->actform);
  
  cpk->actform = g_strdup (actform);
  
  return TRUE;
}


gboolean
polyxcalc_chempadkey_set_tooltip (PxmChempadkey *cpk, gchar *tooltip)
{
  g_assert (cpk != NULL);
  g_assert (tooltip != NULL);
  
  g_assert (cpk->tooltip != NULL);
  
  g_free (cpk->tooltip);
  
  cpk->tooltip = g_strdup (tooltip);
  
  return TRUE;
}


gboolean
polyxcalc_chempadgeo_set_rows (PxmChempadgeo *cpg, gint rows)
{
  g_assert (cpg != NULL);

  cpg->rows = rows;
  
  return TRUE;
}


gboolean
polyxcalc_chempadgeo_set_cols (PxmChempadgeo *cpg, gint cols)
{
  g_assert (cpg != NULL);

  cpg->cols = cols;
  
  return TRUE;
}


gboolean
polyxcalc_chempadkey_free (PxmChempadkey *cpk)
{
  g_assert (cpk != NULL);

  if (cpk->label != NULL)
    g_free (cpk->label);
  
  if (cpk->actform != NULL)
    g_free (cpk->actform);
  
  if (cpk->tooltip != NULL)
    g_free (cpk->tooltip);

  g_free (cpk);
  
  return TRUE;
}

gboolean
polyxcalc_chempadgeo_free (PxmChempadgeo *cpg)
{
  g_assert (cpg != NULL);

  g_free (cpg);
  
  return TRUE;
}

gint
polyxcalc_chempadkey_GPA_free (GPtrArray *GPA)
{
  PxmChempadkey *cpk;
  
  gint count = 0;
  
  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      cpk = g_ptr_array_remove_index (GPA, 0);
      g_assert (cpk != NULL);
      
      polyxcalc_chempadkey_free (cpk);
      
      count++;
    }
  
  g_ptr_array_free (GPA, TRUE);
  
  return count;
}





GtkWidget *
polyxcalc_ui_chempad_wnd_setup_window (PxmCalcCtxt *calcctxt)
{
  GtkWidget *widget = NULL;

  GladeXML *xml = NULL;

  gchar *gui_file = NULL;
  gchar * chempad_file = NULL;
  gchar * dir = NULL;

  gboolean result = FALSE;
  gboolean try_global = FALSE;
  gboolean try_private = FALSE;
  
  gint parent_width = 0;


  g_assert (calcctxt != NULL);
  

  /* We are creating the chempad window ex nihilo.  Go through all
     the steps:
  */
  gui_file = 
    g_strdup_printf ("%s/polyxcalc.glade", userspec->gladedir);
  
  g_assert (gui_file != NULL);
  
  xml = glade_xml_new (gui_file, "polyxcalc_chempad_wnd", 
		       PACKAGE);
  g_free (gui_file);

  if (xml == NULL)
    {
      g_critical (_("%s@%d: failed to load the interface\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  calcctxt->chempad_wnd = glade_xml_get_widget (xml, 
						"polyxcalc_chempad_wnd");

  if (calcctxt->chempad_wnd == NULL)
    {
      g_critical (_("%s@%d: failed to create the chempad window\n"),
	     __FILE__, __LINE__);

      return NULL;
    }

  widget = glade_xml_get_widget (xml, "polyxcalc_chempad_vbox");
  g_assert (widget != NULL);
  g_object_set_data (G_OBJECT (calcctxt->chempad_wnd),
		     "polyxcalc_chempad_vbox", widget);
      
  /* We don't need the GladeXML object any more, so unref it
   * to save some memory 
   */
  g_object_unref (G_OBJECT (xml));
      
  
  /* Now that we have the calcctxt->chempad_wnd skeleton, we should
     fill it with the buttons. But to know what chempad keys we have
     to create, we need to know what polymer chemistry definition is currently
     loaded (if any). When setting up the calculator window, we did
     load the polymer chemistry definition, if any. And thus we did
     set the directory of that polymer chemistry definition data.
  */
  if (calcctxt->polchemdefctxt != NULL)
    {
      chempad_file = 
	g_strdup_printf ("%s/chempad.conf", 
			 calcctxt->polchemdefctxt->polchemdef->dir);
      
      result = g_file_test (chempad_file, G_FILE_TEST_EXISTS);
      
      if (TRUE == result)
	{
	  /* Read the file and construct the chempad layout.
	   */
	  result = polyxcalc_ui_chempad_wnd_fill_layout (calcctxt, 
							 chempad_file);
	  
	  if (result == FALSE)
	    try_private = TRUE;
	}
      else
	{
	  /* First try the private configuration file. If it goes wrong
	     the try the global configuration file.
	  */
	  try_private = TRUE;
	}

      g_free (chempad_file);
    }
  else
    {
      /* The calculator is loaded without requiring any given polymer
	 chemistry definition. So we just just load the private
	 configuration file, and if that does not work the global
	 configuration file.
      */
      try_private = TRUE;
    }
  
  
  if (try_private == TRUE)
    {
      /* Two cases can bring us here: 1. no polchemdef was found in
	 the calling window. 2. the polymer chemistry definition was
	 there, but no chempad.conf file was found in its data
	 directory.
       
	 Thus we still have the chance to find a chempad.conf
	 configuration file in the user's config directory.
      */
      dir = libpolyxmass_config_get_user_polyxmass_config_dir ();
	
      chempad_file = g_strdup_printf ("%s/chempad.conf", dir);

      g_free (dir);
      
      result = g_file_test (chempad_file, G_FILE_TEST_EXISTS);
      
      if (FALSE == result)
	{
	  g_message (_("%s@%d: no private chempad configuration file found.\n"),
		 __FILE__, __LINE__);
	  
	  try_global = TRUE;
	}
      else
	{
	  /* Read the file and construct the chempad layout.
	   */
	  result = 
	    polyxcalc_ui_chempad_wnd_fill_layout (calcctxt, chempad_file);

	  if (result == FALSE)
	    try_global = TRUE;
	}
      
      g_free (chempad_file);
    }
  
  if (try_global == TRUE)
    {
      /* We are here as a last resort...
       */
      dir = libpolyxmass_config_get_system_polyxmass_config_dir ();
      
      chempad_file = g_strdup_printf ("%s/chempad.conf", dir);

      g_free (dir);
      
      result = g_file_test (chempad_file, G_FILE_TEST_EXISTS);
      
      if (FALSE == result)
	{
	  g_critical (_("%s@%d: the general chempad configuration file does"
		   " not exist.\n"),
		 __FILE__, __LINE__);
	}
      else
	{
	  /* Read the file and construct the chempad layout.
	   */
	  polyxcalc_ui_chempad_wnd_fill_layout (calcctxt, chempad_file);
	}
      
      g_free (chempad_file);
    }
  
  g_signal_connect (G_OBJECT (calcctxt->chempad_wnd),
		    "delete_event",
		    G_CALLBACK 
		    (polyxcalc_ui_chempad_wnd_delete_event),
		    calcctxt);
  
  g_signal_connect (G_OBJECT (calcctxt->chempad_wnd),
		    "destroy_event",
		    G_CALLBACK 
		    (polyxcalc_ui_chempad_wnd_destroy_event),
		    calcctxt);


  /* We should have a reasonable size for the chempad window, that is
     reasonably sized with respect to the calculator window.
   */
  parent_width = GTK_WIDGET (calcctxt->calculator_wnd)->allocation.width;
  
  gtk_widget_set_size_request (GTK_WIDGET (calcctxt->chempad_wnd),
                               parent_width /2,
			       parent_width / 3);

        
  gtk_widget_show_all (GTK_WIDGET (calcctxt->chempad_wnd));

  return calcctxt->chempad_wnd;
}


/* The togglebutton that emits this signal is in the program's main
 * window.
 */
gboolean
polyxcalc_ui_chempad_wnd_fill_layout (PxmCalcCtxt *calcctxt, gchar *file)
{
  GtkWidget *vbox = NULL;
  GtkWidget *table = NULL;
  GtkWidget *button = NULL;
  GtkWidget *tooltips = NULL;

  gint count = 0;
  
  gint iter = 0;
  gint iter_row = 0;
  gint iter_col = 0;
  gfloat ratio = 0;
  
  gchar *help = NULL;
  
  PxmChempadkey *cpk = NULL;
  PxmChempadgeo *cpg = NULL;
    
  GPtrArray *cpkGPA = NULL;
  

  g_assert (calcctxt != NULL);
  g_assert (file != NULL);

  /* Allocate the array into which all the keys (buttons) descriptions
     are going to be stored.
  */
  cpkGPA = g_ptr_array_new ();
  
  /* Allocate the geometry object where 'rows' and 'columns' data are
     going to be stored upon parsing of the chempad.conf file.
  */
  cpg = polyxcalc_chempadgeo_new ();
  
  /* And now that we have all the envelopes, we can stuff them
     with all the data parsed from the file.
  */
  if (FALSE == polyxcalc_chempad_parse_file (file, cpkGPA, cpg, 
					     calcctxt->atomGPA))
    {
      g_critical (_("%s@%d: failed to parse the chempad file: '%s'\n"),
	     __FILE__, __LINE__, file);

      polyxcalc_chempadkey_GPA_free (cpkGPA);
      
      polyxcalc_chempadgeo_free (cpg);
      
      return FALSE;
    }
  
  /* Let's see how many buttons are going to be stuffed in the 
     chemical pad. If 0, then we just return after freeing allocated 
     data.
  */
  count = cpkGPA->len;
  
  if (count == -1)
    {
      polyxcalc_chempadkey_GPA_free (cpkGPA);

      polyxcalc_chempadgeo_free (cpg);
  
      return TRUE;
    }
  
  /* At this point we know that we have at least one button to 
   * put in the widget window passed as parameter.
   */
  
  /* The buttons are packed into the vertical box that is in the
   * widget passed as parameter (the window in which the chempad is to
   * be set up.
   */
  vbox = g_object_get_data (G_OBJECT (calcctxt->chempad_wnd), 
			    "polyxcalc_chempad_vbox");
  g_assert (vbox != NULL);
  
  /* And now we are ready to synthesize the buttons and to 
   * place them correctly in the vertical box.
   */

  /* We should make sure that correct values were parsed out of the
     file for the cols and rows members of the cpg struct.
  */
  if (cpg->rows <= 0 && cpg->cols <= 0)
    {
      cpg->cols = 3;
      
      /* We should calculate the number of rows, given the fact that we
	 have arbitrarily decided that we would lay out the chempad in
	 three columns.
      */
      ratio = count/cpg->cols;

      if ((gint) ratio < ratio)
	ratio = ((gint) ratio) + 1;
      
      if (ratio <= 0)
	ratio = 1;
      
      cpg->rows = ratio;
    }
      
  else if (cpg->rows <= 0)
    {
      /* We should calculate the number of rows, knowing that the user
	 has asked that the chempad be laid out using cpg->cols.
      */
      ratio = count/cpg->cols;
      /*
	debug_printf (("ratio is %f\n", ratio));
      */

      if ((gint) ratio < ratio)
	ratio = ((gint) ratio)  + 1;
      
      if (ratio <= 0)
	ratio = 1;
      
      /*
	debug_printf (("now ratio is %f\n", ratio));
      */
      cpg->rows = ratio;
    }

  else if (cpg->cols <= 0)
    {
      /* We should calculate the number of cols, knowing that the user
	 has asked that the chempad be laid out using cpg->rows.
      */
      ratio = (gfloat) count/(gfloat) cpg->rows;
      /*
	debug_printf (("ratio is %f\n", ratio));
      */

      if ((gint) ratio < ratio)
	ratio = ((gint) ratio) + 1;
      
      if (ratio <= 0)
	ratio = 1;
      
      /*
	debug_printf (("now ratio is %f\n", ratio));
      */
      cpg->cols = ratio;
    }

  /* Now we can either make the table if we do not find it (which
   * means that we are constructing the window ex nihilo) or first
   * destroy it if we find it already existing (which means we are
   * updating the layout of the chempad window).
   */
  table = g_object_get_data (G_OBJECT (calcctxt->chempad_wnd), "table");
  
  if (table != NULL)
    gtk_widget_destroy (GTK_WIDGET (table));
  
  /* And now we can just start the construction anew.
   */
  table = gtk_table_new (cpg->rows, cpg->cols, FALSE);

  /* Add the table to the vertical box container.
   */
  gtk_container_add (GTK_CONTAINER (vbox), table);

  /* Set a pointer to the table in the window itself so that later we can
   * access this table and destroy it if we are asked to re-setup a new
   * chempad (typically if the user asks that another polymer definition
   * file be opened for calculations).
   */
  g_object_set_data (G_OBJECT (calcctxt->chempad_wnd), "table", table);
  
  /* Create the tooltips.
   */
  tooltips = g_object_get_data (G_OBJECT (calcctxt->chempad_wnd), "tooltips");
  
  if (tooltips == NULL)
    {
      tooltips = (GtkWidget *) gtk_tooltips_new ();
      g_assert (tooltips != NULL);
      g_object_set_data (G_OBJECT (calcctxt->chempad_wnd), 
			 "tooltips", tooltips);
    }
  
  /* Now we know how many iterations we'll have to deal with.
   */
  for (iter = 0 ; iter < count ; iter++)
    {
      cpk = g_ptr_array_index (cpkGPA, iter);
      g_assert (cpk != NULL);
      
      button = gtk_button_new_with_label (cpk->label);
      g_assert (button != NULL);
      

      /*
	gtk_table_attach (GTK_TABLE (table), button,
	iter_col, iter_col + 1,
	iter_row, iter_row + 1, 
	GTK_FILL|GTK_SHRINK, GTK_FILL|GTK_SHRINK,
	0, 0);
	
      */
      gtk_table_attach_defaults (GTK_TABLE (table), button,
				 iter_col, iter_col + 1,
				 iter_row, iter_row + 1);
      iter_col++;
      
      if (iter_col >= cpg->cols)
	{
	  iter_col = 0;
	  iter_row++;
	}

      /* Each button has its own tooltip.
       */
      gtk_tooltips_set_tip (GTK_TOOLTIPS (tooltips), button, 
			    cpk->tooltip, NULL);
      
      /* Each button should have a datum set to it with the 
       * actionformula of the chempadkey. Also true for the label.
       */
      help = g_strdup (cpk->actform);
      g_assert (help != NULL);
      
      g_object_set_data_full (G_OBJECT (button),
			      "actform", help, (GDestroyNotify) g_free);
      
      help = g_strdup (cpk->label);
      g_assert (help != NULL);
      
      g_object_set_data_full (G_OBJECT (button),
			      "label", help, (GDestroyNotify) g_free);
      
      gtk_widget_show (button);
      

      g_signal_connect (G_OBJECT (button),
			"clicked",
			GTK_SIGNAL_FUNC 
			(polyxcalc_ui_chempad_wnd_chempadkey_button),
			calcctxt);
    }


  gtk_widget_show_all (GTK_WIDGET (calcctxt->chempad_wnd));
  
  polyxcalc_chempadkey_GPA_free (cpkGPA);

  polyxcalc_chempadgeo_free (cpg);

  return TRUE;
}


gboolean
polyxcalc_chempad_parse_file (gchar *file, 
			      GPtrArray *GPA,
			      PxmChempadgeo *cpg,
			      GPtrArray *atom_refGPA)
{
  gint count = 0;

  gchar *line = NULL;
  gchar *read = NULL;
   
  FILE *stream = NULL;
   
  PxmChempadkey *cpk = NULL;
   

  /* We get a chempad.conf file and we want to extract data from it:
     ---------------------------------------------------------------

     -- a number of chempad keys (the description of which is going
     be stored in the GPA array passed as parameter;
     -- the columns gint value that indicates the geometry of the chempad
     (to be filled in the cpg object);
     -- the row gint value that indicates the geometry of the chempad
     (to be filled in the cpg object);
  */

  g_assert (file != NULL);
  g_assert (GPA != NULL);
  g_assert (cpg != NULL);
  g_assert (atom_refGPA != NULL);
  
  
  if (FALSE == g_file_test (file, G_FILE_TEST_EXISTS))
    {
      g_critical (_("%s@%d: polymer chemistry definition file does"
	       " not exist: '%s'\n"),
	     __FILE__, __LINE__, file);
      
      return FALSE;
      
    }
  
  stream = fopen (file, "r");
  
  if (NULL == stream)
    {
      g_critical (_("%s@%d: failed to open the chempad configuration"
	       " file: '%s'\n"),
	     __FILE__, __LINE__, file);

      return FALSE;
    }

  /* We now have a configuration file that we must parse to get all
   * the specifications...
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));
  g_assert (line != NULL);
  read = line;

  /* This is a typical chempad.conf file:
   *
   * chempad_rows$3
   * chempad_columns$3
   * 
   * chempadkey=protonate%+H1%adds a proton
   * chempadkey=hydrate%+H2O1%adds a water molecule
   * chempadkey=0H-ylate%+O1H1%adds an hydroxyl group
   * chempadkey=acetylate%-H1+C2H3O1%adds an acetyl group
   */

  /* There are two different kinds of line in the file: 
   * 1. the ones of the form 'chempadkey=protonate%+H1%adds a proton'
   * 2. the ones related to the geometry of the buttons: 
   *       - chempad_rows$5
   *       - chempad_columns$5
   */

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, stream);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Check if the line is a comment:
	   */
	  if (read [0] == '#')
	    continue;
	  
	  /* Check if the line is a geometry line:
	   */
	  if (NULL != strstr (read, "chempad_rows$")
	      ||
	      NULL != strstr (read, "chempad_columns$"))
	    {
	      polyxcalc_chempad_parse_geometry (read, cpg);
	      
	      /*
		debug_printf (("geometry line is %s\n", read));
	      */

	      continue;
	    }
	  
	  /* Check if the line is a chempadkey line. If so allocate a
	     chempadkey object so that we can transmit its pointer to
	     the parsing function.
	   */
	  else if (NULL != strstr (read, "chempadkey="))
	    {
	      cpk = polyxcalc_chempadkey_new ();

	      if (FALSE != polyxcalc_chempad_parse_chempadkey (read, cpk,
							       atom_refGPA))
		{
		  g_ptr_array_add (GPA, cpk);
		  count++;
		}
	      else
		{
		  /* We are not going to use that cpk object, thus we
		     can free it right away.
		  */
		  polyxcalc_chempadkey_free (cpk);
		}
	      	      
	      continue;
	    }
	}
    }

#if 0
  for (iter = 0 ; iter < p_GPA->len ; iter++)
    {
      cpk = g_ptr_array_index (p_GPA, iter);
      
      printf ("Chempadkey at index %d: '%s/%s/%s'\n",
	      iter,
	      cpk->name, 
	      cpk->actform,
	      cpk->tooltip);
    }
#endif
  
  g_free (line);
  fclose (stream);

  return TRUE;
}


gboolean
polyxcalc_chempad_parse_geometry (gchar *line, PxmChempadgeo *cpg)
{
  gint iter = 0;
  gint val = -1;
  
  gchar * found = NULL;

  GString *gs = NULL;
  

  g_assert (line != NULL);
  g_assert (cpg != NULL);
  
  /* The format of the line is either 
     chempad_rows$5
     or 
     chempad_columns$5
   */

  found = strchr (line, '$');
  /* We have to be certain we get a line with a '$' otherwise that
     would mean that the calling function was bugged.
  */
  g_assert (found != NULL);

  /* We want to skip the '$'.
   */
  iter ++;
  
  gs = g_string_new ("");
  
  while (found [iter] != '\x0' && found [iter] != '\n')
    {
      gs = g_string_append_c (gs, found [iter]);

      iter++;
    }
  
  /* At this point, gs should contain a stringal representation of 
     a numerical value.
   */
  if (strlen (gs->str) <= 0)
    {
        g_critical (_("%s@%d: failed to parse line: '%s'\n"),
	 __FILE__, __LINE__, line);

	g_string_free (gs, TRUE);
	
      return FALSE;
    }
  
  /* Convert string to &val.
   */
  if (FALSE == libpolyxmass_globals_strtoi (gs->str, &val, 10))
    {
      g_critical (_("%s@%d: failed to convert from '%s' to gint\n"),
	     __FILE__, __LINE__, gs->str);
      
      g_string_free (gs, TRUE);
      
      return FALSE;
    }

  /* Now that the value is in val, we should manage to know if it is
     for the cols or for the rows variable so that we can set the
     right value in the cpg struct passed as param!
  */
  if (NULL != strstr (line, "chempad_rows$"))
    polyxcalc_chempadgeo_set_rows (cpg, val);
  else if (NULL != strstr (line, "chempad_columns$"))
    polyxcalc_chempadgeo_set_cols (cpg, val);
  else
    {
      g_critical (_("%s@%d: failed to parse line: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      g_string_free (gs, TRUE);
      
      return FALSE;
    }
  
  return TRUE;
}


gboolean
polyxcalc_chempad_parse_chempadkey (gchar *line, PxmChempadkey *cpk,
				    GPtrArray *atom_refGPA)
{
  gint iter = 0;
  
  gboolean label_done = FALSE;
  gboolean actform_done = FALSE;
  gboolean tooltip_done = FALSE;
  
  gchar *found = NULL;

  GString *gs = NULL;

  g_assert (line != NULL && strlen (line) > 0);
  
  g_assert (cpk != NULL);

  g_assert (atom_refGPA != NULL);
  
  /* The general shape of a line is :
   * chempadkey=0H-ylate%+O1H1%"adds an hydroxyl group"
   */
  found = strchr (line, '=');
  g_assert (found != NULL);

  g_assert (cpk != NULL);
  
  /* We want to skip the '='.
   */
  iter ++;
  
  gs = g_string_new ("");
  
  while (found [iter] != '\x0' && found [iter] != '\n')
    {
      if (found [iter] == '%')
	{
	  if (FALSE == label_done)
	    {
	      /* gs now contains the label of the chempadkey.
	       */
	      if (strlen (gs->str) <= 0)
		{
		  g_critical (_("%s@%d: failed to parse line: '%s'\n"),
			 __FILE__, __LINE__, line);
		  
		  g_string_free (gs, TRUE);
		  
		  return FALSE;
		}
	      
	      polyxcalc_chempadkey_set_label (cpk, gs->str);

	      g_string_free (gs, TRUE);
	      gs = g_string_new ("");
	      
	      label_done = TRUE;
	      iter++;
	    }
	  else if (FALSE == actform_done)
	    {
	      /* gs now contains the actionformula of the chempadkey.
	       */
	      if (strlen (gs->str) <= 0)
		{
		  g_critical (_("%s@%d: failed to parse line: '%s'\n"),
			 __FILE__, __LINE__, line);
		  
		  g_string_free (gs, TRUE);
		  
		  return FALSE;
		}

	      /* Check that the formula is OK, anyway.
	       */
	      if (FALSE == pxmchem_actform_check (gs->str, atom_refGPA))
		{
		  g_critical (_("%s@%d: failed to parse line: '%s'\n"),
			 __FILE__, __LINE__, line);
		  
		  g_string_free (gs, TRUE);
		  
		  return FALSE;
		}

	      polyxcalc_chempadkey_set_actform (cpk, gs->str);
	      
	      g_string_free (gs, TRUE);
	      gs = g_string_new ("");

	      actform_done = TRUE;
	      iter++;
	    }
	}
      /* end of if (found [iter] == '%')
       */
      else
	{
	  gs = g_string_append_c (gs, found [iter]);
	  iter++;
	}
    }
  
  /* If we are here, that means that we got to the end of the line,
   * which means that we still have the tooltip in gs.
   */
  if (FALSE == tooltip_done)
    {
      /* gs now contains the tooltip of the chempadkey.
       */
      if (strlen (gs->str) <= 0)
	{
	  g_critical (_("%s@%d: failed to parse line: '%s'\n"),
		 __FILE__, __LINE__, line);
	  
	  g_string_free (gs, TRUE);
	  
	  return FALSE;
	}

      polyxcalc_chempadkey_set_tooltip (cpk, gs->str);

      g_string_free (gs, TRUE);
      
      tooltip_done = TRUE;
    }

  /* Finally the three elements should have been correctly set:
   */
  if (FALSE == label_done 
      || FALSE == actform_done 
      || FALSE == tooltip_done)
    {
      g_critical (_("%s@%d: failed to parse line: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      return FALSE;
    }

  return TRUE;
}



gint
polyxcalc_ui_chempad_wnd_chempadkey_button (GtkWidget * button, 
					    gpointer data)
{
  PxmCalcCtxt *calcctxt = data;
  
  GtkWidget *entry = NULL;

  GString *gs = NULL;
  
  gchar *actform = NULL;

  gint error = 1;
  
  gchar *help = 0;



  g_assert (calcctxt != NULL);
    

  libpolyxmass_masspair_reset (calcctxt->masspair_init);
  libpolyxmass_masspair_reset (calcctxt->masspair_temp);
  
  /* We are going to add the result of the mass calculation for the
   * actform that is set to the chempad key button that was clicked to
   * the result masses. No matter what is in the initial masses
   * entries.
   */

  /* MONO
   */
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The mono result mass contains something.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->mono)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);
	  
	  return 0;
	}
      
      /* Since we are going to use the result avg mass, we want to
       * store it so that we can later put it to the init avg mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_mono (calcctxt->masspair_init, 
				      calcctxt->masspair_temp->mono); 
    }

  /* AVG
   */
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  
  help = (gchar *) gtk_entry_get_text (GTK_ENTRY (entry));
  
  if (help != NULL && strlen (help) > 0)
    {
      /* The avg result mass contains something.
       */
      if (FALSE == 
	  libpolyxmass_globals_strtod ((gchar *) help,
				       &(calcctxt->masspair_temp->avg)))
	{
	  g_critical (_("%s@%d: failed to convert from '%s' to gdouble\n"),
		 __FILE__, __LINE__, help);

	  return 0;
	}
      
      /* Since we are going to use the result avg mass, we want to
       * store it so that we can later put it to the init avg mass,
       * therefore allowing the user to sort of undo the last
       * calculation, if necessary.  
       */
      libpolyxmass_masspair_set_avg (calcctxt->masspair_init, 
				     calcctxt->masspair_temp->avg); 
    }


  /* ACTFORM CALCULATION PROPER.
   */

  /* Allocate the GString in which we'll construct the string
   * describing the operation for the recorder.
   */
  gs = g_string_new ("");
  
  /* Get the actform from the button that was clicked in the chempad.
   */
  actform = g_object_get_data (G_OBJECT (button), "actform");
  g_assert (actform != NULL);

  if (strlen (actform) <= 0)
    {
      g_string_free (gs, TRUE);

      return 1;
    }
  
  if (FALSE == pxmchem_actform_account_mass (actform, calcctxt->atomGPA,
					     1, calcctxt->masspair_temp))
    {
      g_critical (_("%s@%d: failed to account for actform chempad key: '%s'\n"),
	     __FILE__, __LINE__, actform);
      
      g_string_append_printf (gs, 
			      _("Failed to account for actform chempad key:"
				" '%s'\n"), actform);
      
      error = 0;
    }
  else
    g_string_append_printf (gs, 
			    _("Account for actform chempad key: '%s' ==>"
			      " mono: %f ; avg: %f\n"),
			    actform, calcctxt->masspair_temp->mono, 
			    calcctxt->masspair_temp->avg);

  /* Put the operation on queue to the recorder text.
   */
  polyxcalc_ui_recorder_wnd_append_record (calcctxt,
				    gs->str);
  
  g_string_free (gs, TRUE);
  

  /* DISPLAY OF THE RESULTS.
   */

  /* We can now set to the result mass entries the values obtained.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->mono);
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "result_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_temp->avg);
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "result_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  /* Set the old values to the initial mass entries.
   */
  help = g_strdup_printf ("%f", calcctxt->masspair_init->mono);
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "initial_mono_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);
  
  help = g_strdup_printf ("%f", calcctxt->masspair_init->avg);
  entry = g_object_get_data (G_OBJECT (calcctxt->calculator_wnd), 
			     "initial_avg_entry");
  g_assert (entry != NULL);
  gtk_entry_set_text (GTK_ENTRY (entry), help);
  g_free (help);

  return 1;
}


/* WINDOW LIFE CYCLE-RELATED FUNCTIONS
 */
gboolean
polyxcalc_ui_chempad_wnd_delete_event (GtkWidget *widget,
				    GdkEvent *event,
				    gpointer data)
{
  PxmCalcCtxt *calcctxt = data;


  g_assert (calcctxt != NULL);
  g_assert (calcctxt->chempad_wnd != NULL);

  gtk_widget_hide (GTK_WIDGET (calcctxt->chempad_wnd));

  return TRUE;
}


gboolean
polyxcalc_ui_chempad_wnd_destroy_event (GtkWidget *widget,
					GdkEvent *event,
					gpointer data)
{
  return FALSE;
}


