/* GNU polyxmass - the massist's program.
   -------------------------------------- 
   Copyright (C) 2000,2001,2002,2003,2004 Filippo Rusconi

   http://www.polyxmass.org

   This file is part of the "GNU polyxmass" project.
   
   The "GNU polyxmass" project is an official GNU project package (see
   www.gnu.org) released ---in its entirety--- under the GNU General
   Public License and was started at the Centre National de la
   Recherche Scientifique (FRANCE), that granted me the formal
   authorization to publish it under this Free Software License.

   This software is free software; you can redistribute it and/or
   modify it under the terms of the GNU  General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.
   
   This software is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU  General Public
   License along with this software; if not, write to the
   Free Software Foundation, Inc., 51 Franklin St, Fifth Floor,
   Boston, MA 02110-1301, USA.
*/

#include "libpolyxmass-modifspec.h"



/* NEW'ING FUNCTIONS, DUPLICATING FUNCTIONS ...
 */
PxmModifSpec *
libpolyxmass_modifspec_new (void)
{
  PxmModifSpec *ms = NULL;
  
  ms = g_malloc0 (sizeof (PxmModifSpec));

 return ms;
}


gboolean
libpolyxmass_modifspec_set_name (PxmModifSpec *ms, gchar *name)
{
  g_assert (ms != NULL && name != NULL);

  if (ms->name != NULL)  
    g_free (ms->name);
  
  ms->name = g_strdup (name);
  
  return TRUE;
}


gboolean
libpolyxmass_modifspec_set_action (PxmModifSpec *ms, gchar *action)
{
  g_assert (ms != NULL && action != NULL);

  if (ms->action != NULL)  
    g_free (ms->action);
  
  ms->action = g_strdup (action);
  
  return TRUE;
}


gboolean
libpolyxmass_modifspec_set_raster (PxmModifSpec *ms, gchar *file)
{
  g_assert (ms != NULL && file != NULL);

  if (ms->raster != NULL)  
    g_free (ms->raster);
  
  ms->raster = g_strdup (file);
  
  return TRUE;
}  


gboolean
libpolyxmass_modifspec_set_vector (PxmModifSpec *ms, gchar *file)
{
  g_assert (ms != NULL && file != NULL);

  if (ms->vector != NULL)  
    g_free (ms->vector);
  
  ms->vector = g_strdup (file);
  
  return TRUE;
}  


gboolean
libpolyxmass_modifspec_set_name_sound (PxmModifSpec *ms, gchar *name)
{
  g_assert (ms != NULL && name != NULL);
  
  if (ms->name_sound != NULL)  
    g_free (ms->name_sound);
  
  ms->name_sound = g_strdup (name);
  
  return TRUE;
}




/*  LOCATING FUNCTIONS
 */
PxmModifSpec *
libpolyxmass_modifspec_get_ptr_by_name (GPtrArray *GPA, gchar *name)
{
  return libpolyxmass_modifspec_get_ptr_top_by_name (GPA, name);
}


PxmModifSpec *
libpolyxmass_modifspec_get_ptr_top_by_name (GPtrArray *GPA, gchar *name)
{
  gint iter = 0;
  PxmModifSpec *modifspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);

  for (iter = 0; iter < GPA->len; iter ++)
    {
      modifspec = g_ptr_array_index (GPA, iter);
      g_assert (modifspec != NULL);

      if (0 == strcmp (modifspec->name, name))
	return modifspec;
    }
  
  return NULL;
}


PxmModifSpec *
libpolyxmass_modifspec_get_ptr_bottom_by_name (GPtrArray *GPA, 
						   gchar *name)
{
  gint iter = 0;
  PxmModifSpec *modifspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);

  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  modifspec = g_ptr_array_index (GPA, iter);
	  g_assert (modifspec != NULL);

	  if (0 == strcmp (modifspec->name, name))
	    return modifspec;
	}
    }
    
  return NULL;
}


gint
libpolyxmass_modifspec_get_index_by_name (GPtrArray *GPA, gchar *name)
{
  return libpolyxmass_modifspec_get_index_top_by_name (GPA, name);
}


gint
libpolyxmass_modifspec_get_index_top_by_name (GPtrArray *GPA, gchar *name)
{
  gint iter = 0;
  PxmModifSpec *modifspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);

  for (iter = 0; iter < GPA->len; iter ++)
    {
      modifspec = g_ptr_array_index (GPA, iter);
      g_assert (modifspec != NULL);

      if (0 == strcmp (modifspec->name, name))
	return iter;
    }
  
  return -1;
}

gint
libpolyxmass_modifspec_get_index_bottom_by_name (GPtrArray *GPA, 
						   gchar *name)
{
  gint iter = 0;
  PxmModifSpec *modifspec = NULL;
  

  g_assert (GPA != NULL);
  g_assert (name != NULL);

  if (GPA->len > 0)
    {
      for (iter = GPA->len -1 ; iter >= 0; iter--)
	{
	  modifspec = g_ptr_array_index (GPA, iter);
	  
	  if (0 == strcmp (modifspec->name, name))
	    return iter;
	}
    }
    
  return -1;
}



/* UTILITY FUNCTIONS
 */

PxmModifSpec *
libpolyxmass_modifspec_init_from_monicons_dic_line (gchar *line, 
						    gchar *dir) 
{
  gint iter = 0;

  GString * gs = NULL;

  gchar *help = NULL;

  gboolean modif_done = FALSE;
  gboolean action_done = FALSE;

  gboolean vector_done = FALSE;
  gboolean raster_done = FALSE;

  PxmModifSpec *ms = NULL;


  g_assert (line != NULL);
  g_assert (dir != NULL);


  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We have a line of the kind
     "modif;Phosphorylation%T%phospho.svg|phospho.png" (note the
     newline character may be found if the line in question is not the
     last of the file!), we just need to deconstruct this into pieces.
   
     Note that one of the two graphics files may miss. Which is that
     we may have something like:
   
     "modif;Amidation%O%asparagine.svg|asparagine.xpm"
     or
     "modif;Phosphorylation%T%|phospho.xpm"
     or
     "modif;Phosphorylation%T%phospho.svg|"
  */


  /* We should place the pointer to the first letter of the code, that
     is on the character right of the ';' of "monomer;".
  */
  g_assert (g_str_has_prefix (line, "modif;"));

  help = strchr (line, ';');
  line = ++help;
  

  /* Allocate the modifspec instance that we'll fill with data 
   * parsed out of the currently analyzed line.
   */
  ms = libpolyxmass_modifspec_new ();

  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '%')
	{
	  /* There are two '%' signs in a line. When we reach the
	     first, that means that we have just finished parsing the
	     modif name, which is modif_done is FALSE. When we reach
	     the second, that means that we already have finished
	     parsing and validating the modification name, which is
	     the name_one is TRUE. Just work out these two
	     possiblities, depending on the value of modif_done.
	   */
	  if (modif_done == FALSE)
	    {
	      /* gs should contain the modif name just terminated 
	       * parsing.
	       */
	      libpolyxmass_modifspec_set_name (ms, gs->str);

	      gs = g_string_truncate (gs, 0);

	      modif_done = TRUE;
	    }
	  else
	    {
	      /* gs should contain the action type string: "T" for 
	       * transparent, and "O" for opaque.
	       */
	      if (0 != strcmp (gs->str, "T") 
		  && 0 != strcmp (gs->str, "O"))
		{
		  g_string_free (gs, TRUE);
		 
		  libpolyxmass_modifspec_free (ms);
		  
		  return NULL;
		}

	      libpolyxmass_modifspec_set_action (ms, gs->str);
	  
	      gs = g_string_truncate (gs, 0);

	      action_done = TRUE;
	    }
	}
      
      else if (line [iter] == '|')
	{
	  /* We should have something in gs, namely the vector
	     filename. However, the string gs may be empty, as
	     explained above.
	   */
	  if (strlen (gs->str) > 0)
	    {
	      /* Make an absolute filename.
	       */
	      help = g_strdup_printf ("%s/%s", dir, gs->str);
	      
	      /* 
		 Make sure that the file actually exists. Also be
		 certain that the help string is an absolute filename.
	      */
	      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
	      g_assert (TRUE == g_path_is_absolute (help));
	      
	      libpolyxmass_modifspec_set_vector (ms, help);

	      g_free (help);
	      
	      gs = g_string_truncate (gs, 0);
	      
	      vector_done = TRUE;
	    }
	}

      else
	{
	  /* we are iterating in some character, so we should simply
	   * put this into the gs string.
	   */
	  gs = g_string_append_c (gs, line[iter]);
	}
      
      iter++;		    
    }

  /* And now we are at the end of the line, so the raster file name
   * should have completed now, in gs as above. However, the gs string
   * may be empty, because the raster filename may be empty, if the
   * vector filename is not. So we have to check this now. Also, do
   * not forget to check that the code was done already.
   */
  if (strlen (gs->str) > 0)
    {
      /* Make an absolute filename.
       */
      help = g_strdup_printf ("%s/%s", dir, gs->str);
      
      /* 
	 Make sure that the file actually exists. Also be
	 certain that the help string is an absolute filename.
      */
      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
      g_assert (TRUE == g_path_is_absolute (help));
      
      libpolyxmass_modifspec_set_raster (ms, help);
      
      g_free (help);
      
      raster_done = TRUE;
    }

  else 
    {
      if (vector_done == FALSE)
	{
	  g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
		 _("%s@%d: both vector/raster file names cannot be emtpy\n"),
		 __FILE__, __LINE__);

	  libpolyxmass_modifspec_free (ms);
	  
	  g_string_free (gs, TRUE);
	  
	  return NULL;
	}
    }
  
  /* 
     And now we are at the end of the line, so the file name should
     have completed now, but check that the other elements were done
     already.
   */
  if (modif_done == FALSE 
      || action_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: line is not well formed: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      libpolyxmass_modifspec_free (ms);
      
      g_string_free (gs, TRUE);
      
      return NULL;
    }
  
  g_string_free (gs, TRUE);
  
  return ms;
}


gint
libpolyxmass_modifspec_parse_monicons_dic_file (gchar *file,
						GPtrArray *GPA)
{
  gint count = 0;
  
  gchar *line = NULL; 
  gchar *read = NULL;
  gchar * dir = NULL;

  FILE *cfgp = NULL;

  PxmModifSpec *ms = NULL;


  g_assert (GPA != NULL);
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return -1;
  
  /* Note that we NEED that the file name passed as parameter be an
     absolute file name because we are going to use the dirname to
     construct the PxmPolchemdefSpec data.
   */
  if (FALSE == g_path_is_absolute (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file name is not absolute: '%s'\n"),
	     __FILE__, __LINE__, file);

      return -1;
    }

  /* Get the dir name from the abolute file name.
   */
  dir = g_path_get_dirname (file);

  cfgp = fopen (file, "r");
  
  if (NULL == cfgp)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file not found: '%s'\n"),
	     __FILE__, __LINE__, file);

      g_free (dir);
      
      return -1;
    }

  /* The lines that we are interested in always start with the prefix
     "modif;". So it is easy to find them...
  */
  
  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  if (FALSE == g_str_has_prefix (line, "modif;"))
	    continue;
	  
	  ms = 
	    libpolyxmass_modifspec_init_from_monicons_dic_line (read, dir);
	  
	  if (NULL == ms)
	    {
	      g_free (line);
	      
	      fclose (cfgp);
	      
	      g_free (dir);
	      
	      return -1;
	    }
	  else
	    {
	      count++;
	      
	      g_ptr_array_add (GPA, ms);
	      
	      continue;
	    }
	}
      /* End of
	 if (read != NULL && strlen (read) > 1)
      */
    }
  /* End of   
     while (read != NULL)
  */

  /* Now that we do not need this string anymore, since the file is
   * finished parsing, we can free the memory for reuse of the pointer
   */
  g_free (line);
  
  fclose (cfgp);

  g_free (dir);
  
  return count;
}



PxmModifSpec *
libpolyxmass_modifspec_init_from_sounds_dic_line (gchar *line, 
						  gchar *dir) 
{
  gint iter = 0;

  GString * gs = NULL;

  gchar *help = NULL;
  
  gboolean name_done = FALSE;
  
  PxmModifSpec *ms = NULL;


  g_assert (line != NULL);
  g_assert (dir != NULL);
  
  gs = g_string_new ("");
  g_assert (gs != NULL);

  /* We have a line of the kind "modif;Phosphorylation=phospho.ogg"
     (note the newline character may be found if the line in question
     is not the last of the file!), we just need to deconstruct this
     into pieces.
   */

  /* The lines that we have to parse here are in this format:
    
     modif;Phosphorylation=phospho.ogg
     
     Note that the filename may not miss, since then, the most easy
     thing would be to not list the item at all.

     Note that this file contains a line like the following

     silence-sound$silence.ogg
     
     which indicates which file should be used to insert silent spaces
     in the ogg files to be constructed (delays between monomer
     sounds, for example).
     
     That line is parsed by another function, so we do not care about it
     here.
  */

  /* We should place the pointer to the first letter of the name, that
     is on the character right of the ';' of "monomer;".
  */
  g_assert (g_str_has_prefix (line, "modif;"));
  
  help = strchr (line, ';');
  line = ++help;

  /* Allocate the modifspec instance that we'll fill with data 
   * parsed out of the currently analyzed line.
   */
  ms = libpolyxmass_modifspec_new ();

  while (line [iter] != '\x0' && line [iter] != '\n')
    {
      if (line [iter] == '=')
	{
	  /* We should have something in gs, namely the modification
	     name.
	   */
	  if (strlen (gs->str) <= 0)
	    {
	      g_string_free (gs, TRUE);
	      
	      libpolyxmass_modifspec_free (ms);
	      
	      return NULL;
	    }

	  libpolyxmass_modifspec_set_name (ms, gs->str);
	  
	  g_string_free (gs, TRUE);
	  gs = g_string_new ("");
	  
	  name_done = TRUE;
	}
      else
	{
	  /* We are iterating in some character, so we should simply
	   * put this into the gs string.
	   */
	  gs = g_string_append_c (gs, line[iter]);
	}

      iter++;		    
    }

  /* And now we are at the end of the line, so the name_sound filename
     should have completed now, in gs as above. Do not forget to check
     that the name was done already.
   */
  if (name_done == FALSE)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: line is not well formed: '%s'\n"),
	     __FILE__, __LINE__, line);
      
      libpolyxmass_modifspec_free (ms);
      
      g_string_free (gs, TRUE);
      
      return NULL;
    }
  
  /* And now we can go on with the last line chunk: the name of the
     sound file.
  */
  if (strlen (gs->str) > 0)
    {
      /* Make an absolute filename.
       */
      help = g_strdup_printf ("%s/%s", dir, gs->str);

      /*
      debug_printf (("the sound file is %s\n", help));
      */

      /* 
	 Make sure that the file actually exists. Also be
	 certain that the help string is an absolute filename.
      */
      g_assert (TRUE == g_file_test (help, G_FILE_TEST_EXISTS));
      g_assert (TRUE == g_path_is_absolute (help));
	      
      libpolyxmass_modifspec_set_name_sound (ms, help);
      
      g_free (help);
    }
  else 
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: modification sound file name cannot be emtpy\n"),
	     __FILE__, __LINE__);
      
      libpolyxmass_modifspec_free (ms);
      
      g_string_free (gs, TRUE);
      
      return NULL;
    }
  
  g_string_free (gs, TRUE);
  
  return ms;
}


gint
libpolyxmass_modifspec_parse_sounds_dic_file (gchar *file,
					      GPtrArray *GPA)
{
  gint count = 0;
  
  gchar *line = NULL; 
  gchar *read = NULL;
  gchar * dir = NULL;

  FILE *cfgp = NULL;

  PxmModifSpec *ms = NULL;
  PxmModifSpec *ms_local = NULL;

  g_assert (GPA != NULL);
  g_assert (file != NULL);
  
  if (strlen (file) <= 0)
    return -1;
  
  /* Note that we NEED that the file name passed as parameter be 
   * an absolute file name because we are going to use the dirname
   * to construct the PxmPolchemdefSpec data.
   */
  if (FALSE == g_path_is_absolute (file))
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file name is not absolute: '%s'\n"),
	     __FILE__, __LINE__, file);

      return -1;
    }

  /* Get the dir name from the abolute file name.
   */
  dir = g_path_get_dirname (file);

  cfgp = fopen (file, "r");
  
  if (NULL == cfgp)
    {
      g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
	     _("%s@%d: file not found: '%s'\n"),
	     __FILE__, __LINE__, file);

      g_free (dir);
      
      return -1;
    }

  /* The lines that we have to parse here are in this format:

     modif;Phosphorylation=phospho.ogg
   */

  /* The lines that we are interested in always start with the prefix
     "modif;". So it is easy to find them...
  */

  /* Allocate the memory into which each line of the file will be
   * store for recursive parsing.
   */
  line = g_malloc0 (sizeof (gchar) * (MAX_LINE_LENGTH + 1));

  read = line;

  while (read != NULL)
    {
      read = fgets (read, MAX_LINE_LENGTH, cfgp);

      if (read != NULL && strlen (read) > 1)
	{
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  /* Apparently something interesting should be 
	   * in 'read' for us to parse.
	   */
	  if (FALSE == g_str_has_prefix (line, "modif;"))
	    continue;

	  ms = libpolyxmass_modifspec_init_from_sounds_dic_line (read, dir);
	  
	  if (NULL == ms)
	    {
	      g_free (line);
		  
	      fclose (cfgp);

	      g_free (dir);
		  
	      return -1;
	    }
	  else
	    {
	      /* At this point, we add this new modif spec instance in
		 the array of the modif spec instances, only if one
		 such item does not exist already. If one such item
		 exists already, we have to update only the items we
		 are interested in: sound data exclusively, as other
		 modif raster/vector data were setup already in
		 another call and we do not want to overwrite these.
	      */
	      ms_local = libpolyxmass_modifspec_get_ptr_by_name (GPA, 
								 ms->name);
	      if (ms_local == NULL)
		{
		  /* No item in the array has the same name, so we
		     just add the new one and that's done.
		  */
		  count++;
		      
		  g_ptr_array_add (GPA, ms);
		      
		  continue;
		}
	      else
		{
		  /* Apparently an item by the same name already
		     exists.  We only do update the data that we are
		     interested in here:
		  */
		  if (ms->name_sound != NULL)
		    libpolyxmass_modifspec_set_name_sound (ms_local, 
							   ms->name_sound);
		  
		  /* At this time we can free the ms that was returned
		     from the parsing code, as, this time, we do not
		     need it.
		  */
		  count++;
		  
		  continue;
		}
	    }
	}
      /* End of 
      if (NULL != strchr (read, '=') && NULL != strchr (read, '|'))
      */
    }
  /* End of   
     while (read != NULL)
  */

  /* Now that we do not need this string anymore, since the file is
     finished parsing, we can free the memory for reuse of the pointer
  */
  g_free (line);
  
  fclose (cfgp);
  
  g_free (dir);
  
  return count;
}



/* FREE'ING FUNCTIONS
 */
gint
libpolyxmass_modifspec_free (PxmModifSpec *ms)
{
  g_assert (ms != NULL);
  
  if (ms->name != NULL)
    g_free (ms->name);
  
  if (ms->action != NULL)
    g_free (ms->action);

  if (ms->raster != NULL)
    g_free (ms->raster);
  
  if (ms->vector != NULL)
    g_free (ms->vector);
  
  if (ms->name_sound != NULL)
    g_free (ms->name_sound);
 
  g_free (ms);
  
  return 1;
}




/* GPtrArray-RELATED FUNCTIONS
 */
gint
libpolyxmass_modifspec_GPA_empty (GPtrArray *GPA)
{
  gint count = 0;
  
  PxmModifSpec *ms = NULL;
  

  g_assert (GPA != NULL);
  
  while (GPA->len > 0)
    {
      ms = g_ptr_array_remove_index (GPA, 0);
      g_assert (ms != NULL);
      libpolyxmass_modifspec_free (ms);
      count++;
    }
  
  return count;
}


gint
libpolyxmass_modifspec_GPA_free (GPtrArray *GPA)
{
  gint count = 0;
  

  g_assert (GPA != NULL);

  count = libpolyxmass_modifspec_GPA_empty (GPA);
    
  g_ptr_array_free (GPA, TRUE);

  return count;
}


