#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <arpa/inet.h>
#include <errno.h>

#include <pork_util.h>
#include <pork_list.h>
#include <pork_queue.h>
#include <pork_inet.h>
#include <pork_acct.h>
#include <pork_opt.h>
#include <pork_irc.h>

extern struct sockaddr_storage local_addr;
extern in_port_t local_port;

static int irc_send_server(int sock, const char *cmd) {
	return (sockprintf(sock, "%s\r\n", cmd));
}

int irc_flush_outq(irc_session_t *session) {
	char *cmd;
	int ret = 0;

	while ((cmd = queue_get(session->outq)) != NULL) {
		ret++;
		irc_send_server(session->sock, cmd);
		free(cmd);
	}

	return (ret);
}

int irc_connect(const char *server, char *port, int *sock) {
	struct sockaddr_storage ss;
	struct sockaddr_storage *lss = NULL;
	struct sockaddr_storage local;
	in_port_t port_num;

	if (server == NULL || port == NULL)
		return (-1);

	memset(&ss, 0, sizeof(ss));
	memset(&local, 0, sizeof(local));

	if (get_port(port, &port_num) != 0)
		return (-1);

	if (get_addr(server, &ss) != 0)
		return (-1);

	if (!opt_is_set(USE_ADDR)) {
		char *irchost = getenv("IRCHOST");

		if (irchost != NULL) {
			if (get_addr(irchost, &local) != 0)
				return (-1);
			lss = &local;
		}
	} else {
		memcpy(&local, &local_addr, sizeof(local));
		if (opt_is_set(USE_PORT))
			sin_set_port(&local, local_port);
		lss = &local;
	}

	return (nb_connect(&ss, lss, port_num, sock));
}

int irc_send(irc_session_t *session, char *command) {
	if (session->sock < 0) {
		char *cmd;

		cmd = xstrdup(command);
		if (queue_add(session->outq, cmd) != 0) {
			free(cmd);
			return (-1);
		}

		return (0);
	}

	return (irc_send_server(session->sock, command));
}

int irc_set_chan_mode(irc_session_t *session, char *chan, char *mode_str) {
	char buf[2048];
	int ret;

	if (mode_str != NULL)
		ret = snprintf(buf, sizeof(buf), "MODE %s :%s", chan, mode_str);
	else
		ret = snprintf(buf, sizeof(buf), "MODE %s", chan);

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_set_user_mode(irc_session_t *session, char *mode_str) {
	char buf[2048];
	struct pork_acct *acct = session->data;
	int ret;

	if (mode_str != NULL) {
		ret = snprintf(buf, sizeof(buf),
				"MODE %s :%s", acct->username, mode_str);
	} else
		ret = snprintf(buf, sizeof(buf), "MODE %s", acct->username);

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_pong(irc_session_t *session, char *dest) {
	char buf[2048];
	int ret;

	ret = snprintf(buf, sizeof(buf), "PONG %s", dest);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_set_away(irc_session_t *session, char *msg) {
	char buf[2048];
	int ret;

	if (msg != NULL)
		ret = snprintf(buf, sizeof(buf), "AWAY :%s", msg);
	else
		ret = snprintf(buf, sizeof(buf), "AWAY");

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_raw(irc_session_t *session, char *command) {
	return (irc_send(session, command));
}

int irc_send_join(irc_session_t *session, char *channel, char *key) {
	char buf[2048];
	int ret;

	if (key != NULL)
		ret = snprintf(buf, sizeof(buf), "JOIN %s %s", channel, key);
	else
		ret = snprintf(buf, sizeof(buf), "JOIN %s", channel);

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_login(irc_session_t *session) {
	char buf[2048];
	struct pork_acct *acct = session->data;
	int ret;

	ret = snprintf(buf, sizeof(buf), "NICK %s", acct->username);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	ret = irc_send(session, buf);
	if (ret == -1)
		return (-1);

	ret = snprintf(buf, sizeof(buf), "USER %s ojnk ojnk :%s",
			acct->username, acct->profile);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_kick(irc_session_t *session, char *chan, char *user, char *msg) {
	char buf[2048];
	int ret;

	ret = snprintf(buf, sizeof(buf), "KICK %s %s :%s",
			chan, user, (msg == NULL ? "" : msg));
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_privmsg(irc_session_t *session, char *dest, char *msg) {
	char buf[2048];
	int ret;

	ret = snprintf(buf, sizeof(buf), "PRIVMSG %s :%s", dest, msg);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_names(irc_session_t *session, char *chan) {
	char buf[2048];
	int ret;

	if (chan != NULL)
		ret = snprintf(buf, sizeof(buf), "NAMES :%s", chan);
	else
		ret = snprintf(buf, sizeof(buf), "NAMES");

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_who(irc_session_t *session, char *dest) {
	char buf[2048];
	int ret;

	if (dest != NULL)
		ret = snprintf(buf, sizeof(buf), "WHO :%s", dest);
	else
		ret = snprintf(buf, sizeof(buf), "WHO");

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_whois(irc_session_t *session, char *dest) {
	char buf[2048];
	int ret;
	struct pork_acct *acct = session->data;

	if (dest == NULL)
		dest = acct->username;

	ret = snprintf(buf, sizeof(buf), "WHOIS %s", dest);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_whowas(irc_session_t *session, char *dest) {
	char buf[2048];
	int ret;
	struct pork_acct *acct = session->data;

	if (dest == NULL)
		dest = acct->username;

	ret = snprintf(buf, sizeof(buf), "WHOWAS %s", dest);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_nick(irc_session_t *session, char *nick) {
	char buf[2048];
	int ret;

	if (nick == NULL)
		return (-1);

	ret = snprintf(buf, sizeof(buf), "NICK :%s", nick);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_part(irc_session_t *session, char *chan) {
	char buf[2048];
	int ret;

	if (chan == NULL)
		return (-1);

	ret = snprintf(buf, sizeof(buf), "PART %s", chan);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_quit(irc_session_t *session, char *reason) {
	char buf[2048];
	int ret;

	if (reason == NULL)
		reason = "Leaving";

	ret = snprintf(buf, sizeof(buf), "QUIT :%s", reason);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	pork_io_del(session);
	return (irc_send(session, buf));
}

int irc_send_notice(irc_session_t *session, char *dest, char *msg) {
	char buf[2048];
	int ret;

	ret = snprintf(buf, sizeof(buf), "NOTICE %s :%s", dest, msg);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_kick(irc_session_t *session, char *chan, char *nick, char *msg) {
	char buf[2048];
	int ret;

	ret = snprintf(buf, sizeof(buf), "KICK %s %s :%s", chan, nick, msg);
	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}

int irc_send_topic(irc_session_t *session, char *chan, char *topic) {
	char buf[2048];
	int ret;

	if (topic != NULL)
		ret = snprintf(buf, sizeof(buf), "TOPIC %s :%s", chan, topic);
	else
		ret = snprintf(buf, sizeof(buf), "TOPIC %s", chan);

	if (ret < 0 || (size_t) ret >= sizeof(buf))
		return (-1);

	return (irc_send(session, buf));
}
