/*
** pork_acct.c - account management.
** Copyright (C) 2002-2003 Ryan McCabe <ryan@numb.org>
**
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License, version 2,
** as published by the Free Software Foundation.
*/

#include <config.h>

#include <unistd.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/time.h>
#include <sys/types.h>

#include <pork.h>
#include <pork_missing.h>
#include <pork_util.h>
#include <pork_list.h>
#include <pork_imwindow.h>
#include <pork_buddy.h>
#include <pork_buddy_list.h>
#include <pork_proto.h>
#include <pork_acct.h>
#include <pork_screen.h>
#include <pork_screen_cmd.h>
#include <pork_msg.h>
#include <pork_chat.h>

static u_int32_t acct_last_refnum;

static int pork_acct_find_helper(void *l, void *r) {
	u_int32_t refnum = POINTER_TO_UINT(l);
	struct pork_acct *acct = r;

	return (refnum - acct->refnum);
}

/*
** Free everything that needs freeing.
*/

static void pork_acct_free(struct pork_acct *acct) {
	if (acct->proto->write_config != NULL)
		acct->proto->write_config(acct);

	if (acct->proto->free != NULL)
		acct->proto->free(acct);

	if (acct->blist != NULL) {
		blist_destroy(acct->blist);
		free(acct->blist);
	}

	buddy_destroy(acct);

	if (acct->passwd != NULL) {
		memset(acct->passwd, 0, strlen(acct->passwd));
		free(acct->passwd);
	}

	hash_destroy(&acct->autoreply);

	free(acct->away_msg);
	free(acct->username);
	free(acct->profile);
	free(acct);
}

inline dlist_t *pork_acct_find(u_int32_t refnum) {
	dlist_t *ret;

	ret = dlist_find(screen.acct_list,
			UINT_TO_POINTER(refnum), pork_acct_find_helper);

	return (ret);
}

struct pork_acct *pork_acct_find_name(const char *name, int protocol) {
	dlist_t *cur = screen.acct_list;

	while (cur != NULL) {
		struct pork_acct *acct = cur->data;

		if (acct->proto->protocol == protocol &&
			!strcasecmp(name, acct->username))
		{
			return (cur->data);
		}

		cur = cur->next;
	}

	return (NULL);
}

inline struct pork_acct *pork_acct_get_data(u_int32_t refnum) {
	dlist_t *node;

	node = pork_acct_find(refnum);
	if (node == NULL || node->data == NULL)
		return (NULL);

	return (node->data);
}

int pork_acct_add(struct pork_acct *acct) {
	screen.acct_list = dlist_add_head(screen.acct_list, acct);
	return (0);
}

int pork_acct_del_refnum(u_int32_t refnum, char *reason) {
	dlist_t *node = pork_acct_find(refnum);

	if (node == NULL)
		return (-1);

	pork_acct_del(node, reason);
	return (0);
}

void pork_acct_del(dlist_t *node, char *reason) {
	struct pork_acct *acct = node->data;
	dlist_t *cur;

	pork_signoff(acct, reason);
	chat_leave_all(acct);

	screen.acct_list = dlist_remove(screen.acct_list, node);

	cur = screen.window_list;
	if (cur != NULL) {
		do {
			struct imwindow *win = cur->data;

			if (win->owner == acct) {
				if (win->blist_visible)
					imwindow_blist_hide(win);

				win->owner->ref_count--;
				win->owner = screen.null_acct;

				if (win->owner != NULL)
					win->owner->ref_count++;
			}

			cur = cur->next;
		} while (cur != screen.window_list);
	}

	/* This must always be the case. */
	if (acct->ref_count == 0)
		pork_acct_free(acct);
}

void pork_acct_print_list(void) {
	dlist_t *cur = screen.acct_list;

	screen_status_msg("USERNAME\t\tPROTOCOL\tREFNUM");
	for (cur = screen.acct_list ; cur != NULL ; cur = cur->next) {
		struct pork_acct *acct = cur->data;

		if (acct->proto->protocol < 0)
			continue;

		screen_status_msg("%s\t\t\t%s\t\t\t%u",
			acct->username, acct->proto->name, acct->refnum);
	}
}

void pork_acct_del_all(char *reason) {
	dlist_t *cur = screen.acct_list;

	while (cur != NULL) {
		dlist_t *next = cur->next;

		pork_acct_del(cur, reason);
		cur = next;
	}
}

void pork_acct_update_blist_color(void) {
	dlist_t *cur = screen.acct_list;

	while (cur != NULL) {
		struct pork_acct *acct = cur->data;

		if (acct->blist != NULL)
			blist_draw(acct->blist);

		cur = cur->next;
	}
}

void pork_acct_update_blist_format(void) {
	dlist_t *cur = screen.acct_list;

	while (cur != NULL) {
		struct pork_acct *acct = cur->data;

		if (acct->blist != NULL)
			blist_changed_format(acct->blist);

		cur = cur->next;
	}
}

inline void pork_acct_update(void) {
	dlist_t *cur = screen.acct_list;

	for (cur = screen.acct_list ; cur != NULL ; cur = cur->next) {
		struct pork_acct *acct = cur->data;
		time_t time_now = time(NULL);

		if (acct->proto->update != NULL) {
			if (acct->proto->update(acct) != 0)
				continue;
		}

		if (acct->proto->set_idle_time != NULL && acct->report_idle &&
			!acct->marked_idle && opt_get_bool(OPT_REPORT_IDLE))
		{
			time_t time_diff = time_now - acct->last_input;
			int idle_after = opt_get_int(OPT_IDLE_AFTER);

			if (idle_after > 0 && time_diff >= 60 * idle_after) {
				screen_status_msg_ts("Marking you idle after %d minutes of inactivity",
					(int) time_diff / 60);
				acct->proto->set_idle_time(acct, time_diff);
			}
		}
	}
}

/*
** Initialize an account for the specified user.
*/

struct pork_acct *pork_acct_init(const char *user, int protocol) {
	struct pork_acct *acct;

	acct = xcalloc(1, sizeof(*acct));
	acct->username = xstrdup(user);
	acct->events = &screen.events;

	acct->proto = proto_get(protocol);

	if (protocol < 0)
		return (acct);

	if (buddy_init(acct) != 0)
		goto out_fail;

	if (acct->proto->init != NULL && acct->proto->init(acct) != 0)
		goto out_fail2;

	if (acct->proto->read_config != NULL && acct->proto->read_config(acct) != 0)
		goto out_fail2;

	acct->can_connect = 1;
	acct->refnum = acct_last_refnum++;

	blist_init(acct);

	time(&acct->last_input);
	return (acct);

out_fail:
	buddy_destroy(acct);

out_fail2:
	free(acct->username);
	free(acct);
	return (NULL);
}
