/*
 Copyright (C) 2011 Avi Romanoff <aviromanoff@gmail.com>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Elementary {

public class Entry : Gtk.Entry {

    public string hint_string;

    public Entry (string hint_string="") {

        this.hint_string = hint_string;

        this.hint ();
        this.focus_in_event.connect (on_focus_in);
        this.focus_out_event.connect (on_focus_out);
        this.drag_data_received.connect (on_drag_data_received);

    }

    private bool on_focus_in () {

        if (get_text () == "") {
            unhint ();
        }
        return false;

    }

    private bool on_focus_out () {

        if (get_text () == "") {
            hint ();
        }
        return false;

    }

    private void on_drag_data_received (Gdk.DragContext context, int x, int y,
        Gtk.SelectionData selection, uint info, uint timestamp) {

        on_focus_in ();

    }

    protected void hint () {

        this.text = this.hint_string;
        grey_out ();

    }

    protected void unhint () {

        this.text = "";
        reset_font ();

    }

    private void grey_out () {

        var color = Gdk.Color ();
        Gdk.Color.parse ("#999", out color);
        this.modify_text (Gtk.StateType.NORMAL, color);
        this.modify_font (Pango.FontDescription.from_string ("italic"));

    }

    private void reset_font () {

        var color = Gdk.Color ();
        Gdk.Color.parse ("#444", out color);
        this.modify_text (Gtk.StateType.NORMAL, color);
        this.modify_font (Pango.FontDescription.from_string ("normal"));

    }

    public new string get_text () {

        if (text == this.hint_string) {
            return "";
        }
        else {
            return text;
        }

    }

    public new void set_text(string text) {

        if (text == "") {
            hint();
        } else {
            unhint();
        }

        this.text = text;
    }
}

public class SearchEntry : Entry {

    bool is_searching;

    public SearchEntry (string hint_string="") {

        base(hint_string);
        this.set_icon_from_stock(Gtk.EntryIconPosition.PRIMARY, Gtk.STOCK_FIND);
        this.changed.connect (manage_icon);
        this.focus_in_event.connect (on_focus_in);
        this.focus_out_event.connect (on_focus_out);
        this.icon_press.connect (icon_pressed);
        setup_clear_icon ();
        this.is_searching = true;

    }

    private void setup_clear_icon () {

        var stock_item = Gtk.StockItem ();
        stock_item.stock_id = "edit-clear-symbolic";
        stock_item.label = null;
        stock_item.modifier = 0;
        stock_item.keyval = 0;
        stock_item.translation_domain = Gtk.STOCK_CLEAR;
        var factory = new Gtk.IconFactory ();
        var icon_set = new Gtk.IconSet ();
        var icon_source = new Gtk.IconSource ();
        icon_source.set_icon_name (Gtk.STOCK_CLEAR);
        icon_set.add_source (icon_source);
        icon_source.set_icon_name ("edit-clear-symbolic");
        icon_set.add_source (icon_source);
        factory.add ("edit-clear-symbolic", icon_set);
        Gtk.stock_add ({stock_item});
        factory.add_default ();

    }

    private new void hint () {

        this.is_searching = false;
        this.set_icon_from_stock (Gtk.EntryIconPosition.SECONDARY, null);
        base.hint ();

    }

    private new bool on_focus_in () {

        if (!this.is_searching) {
            this.unhint ();
            this.is_searching = true;
        }
        return false;

    }

    private new bool on_focus_out () {

        if (this.get_text() == "") {
            this.hint ();
            this.is_searching = false;
        }
        return false;

    }

    private void manage_icon () {

        if (this.text != "") {
            this.set_icon_from_stock (Gtk.EntryIconPosition.SECONDARY, "edit-clear-symbolic");
        }
        else {
            this.set_icon_from_stock (Gtk.EntryIconPosition.SECONDARY, null);
        }

    }

    private void icon_pressed (Gtk.EntryIconPosition icon_position) {

        if (icon_position == Gtk.EntryIconPosition.SECONDARY) {
            this.text = "";
            if (!is_focus)
                this.hint ();
            this.set_icon_from_stock(Gtk.EntryIconPosition.SECONDARY, null);
        }
        else {
            if (!this.is_focus && this.text == "") {
                this.is_searching = false;
                this.hint ();
            }
        }

    }
}

}
