/*
 Copyright (C) 2010 Christian Dywan <christian@twotoasts.de>

 This library is free software; you can redistribute it and/or
 modify it under the terms of the GNU Lesser General Public
 License as published by the Free Software Foundation; either
 version 2.1 of the License, or (at your option) any later version.

 See the file COPYING for the full license text.
*/

namespace Postler {
    const string STOCK_ACCOUNT_NEW = "contact-new";
    const string STOCK_ADDRESSBOOK = "stock_addressbook";
    const string STOCK_ARCHIVE = "gnome-mime-application-x-archive";
    const string STOCK_ARCHIVE_INSERT = "archive-insert";
    const string STOCK_EDIT_CLEAR_SYMBOLIC = "edit-clear-symbolic";
    const string STOCK_EDIT_FIND_SYMBOLIC = "edit-find-symbolic";
    const string STOCK_EMBLEM_DRAFT = "emblem-draft";
    const string STOCK_EMBLEM_IMPORTANT = "emblem-important";
    const string STOCK_STARRED = "starred";
    const string STOCK_NOT_STARRED = "not-starred";
    const string STOCK_FACE_SMILE_BIG = "face-smile-big";
    const string STOCK_FACE_SAD = "face-sad";
    const string STOCK_FACE_WINK = "face-wink";
    const string STOCK_FOLDER_SAVED_SEARCH = "folder-saved-search";
    const string STOCK_IMAGE = "image-x-generic";
    const string STOCK_INTERNET_MAIL = "internet-mail";
    const string STOCK_MAIL_ATTACHMENT = "mail-attachment";
    const string STOCK_MAIL_FORWARD = "mail-forward";
    const string STOCK_MAIL_FORWARDED = "mail-forwarded";
    const string STOCK_MAIL_SEND = "mail-send";
    const string STOCK_MAIL_SEND_RECEIVE = "mail-send-receive";
    const string STOCK_MAIL_MARK_IMPORTANT = "mail-mark-important";
    const string STOCK_MAIL_MARK_JUNK = "mail-mark-junk";
    const string STOCK_MAIL_MARK_NOT_JUNK = "mail-mark-not-junk";
    const string STOCK_MAIL_MARK_UNREAD = "mail-mark-unread";
    const string STOCK_MAIL_MESSAGE_NEW = "mail-message-new";
    const string STOCK_MAIL_REPLIED = "mail-replied";
    const string STOCK_MAIL_REPLY_SENDER = "mail-reply-sender";
    const string STOCK_MAIL_REPLY_ALL = "mail-reply-all";
    const string STOCK_MAIL_UNREAD = "mail-unread";
    const string STOCK_INBOX = "mail-inbox";
    const string STOCK_OUTBOX = "mail-outbox";
    const string STOCK_SENT_MAIL = "mail-sent";
    const string STOCK_USER_TRASH = "user-trash";
    const string STOCK_REPORT_BUG = "lpi-bug";
}

public class Postler.App : Unique.App {
    Bureau? bureau = null;

    public static string argv0 { get; set; }

    public App () {
        GLib.Object (name: "org.elementary.PostlerApp", startup_id: null);
    }

    public new void watch_window (Bureau bureau) {
        this.bureau = bureau;
    }

    public override Unique.Response message_received (int command,
        Unique.MessageData data, uint timestamp) {
        if (command == Unique.Command.ACTIVATE) {
            return_val_if_fail (bureau != null, Unique.Response.INVALID);
            bureau.present ();
            bureau.folders.select_folder (data.get_text ());
        }
        return Unique.Response.OK;
    }

    const Gtk.StockItem[] stock_items = {
        { STOCK_ACCOUNT_NEW, N_("New _Account"), 0, 0, "list-add" },
        { STOCK_ADDRESSBOOK, N_("_Addressbook") },
        { STOCK_ARCHIVE },
        { STOCK_ARCHIVE_INSERT, N_("Archi_ve"), 0, 0, STOCK_ARCHIVE },
        { STOCK_EDIT_CLEAR_SYMBOLIC, null, 0, 0, Gtk.STOCK_CLEAR },
        { STOCK_EDIT_FIND_SYMBOLIC, null, 0, 0, Gtk.STOCK_FIND },
        { STOCK_EMBLEM_DRAFT },
        { STOCK_EMBLEM_IMPORTANT },
        { STOCK_STARRED },
        { STOCK_NOT_STARRED },
        { STOCK_FACE_SMILE_BIG },
        { STOCK_FACE_SAD },
        { STOCK_FACE_WINK },
        { STOCK_FOLDER_SAVED_SEARCH },
        { STOCK_MAIL_ATTACHMENT, null, 0, 0, "stock_attach" },
        { STOCK_MAIL_FORWARD, N_("_Forward") },
	{ STOCK_MAIL_FORWARDED, "mail-forwarded", 0, 0, STOCK_MAIL_FORWARD },
        { STOCK_MAIL_MARK_IMPORTANT, N_("Mark as _Important"), 0, 0, STOCK_EMBLEM_IMPORTANT },
        { STOCK_MAIL_MARK_JUNK, N_("Mark as Junk"), 0, 0, null },
        { STOCK_MAIL_MARK_NOT_JUNK, N_("Mark as Not Junk"), 0, 0, null },
        { STOCK_MAIL_MARK_UNREAD, N_("Mark as _Unread"), 0, 0, Gtk.STOCK_NEW },
        { STOCK_MAIL_MESSAGE_NEW, N_("New _Message") },
	{ STOCK_MAIL_REPLIED, "mail-replied", 0, 0, STOCK_MAIL_REPLY_SENDER },
        { STOCK_MAIL_REPLY_SENDER, N_("Reply To _Sender") },
        { STOCK_MAIL_REPLY_ALL, N_("Reply to _All") },
        { STOCK_MAIL_SEND, N_("S_end"), 0, 0, "stock_mail-send" },
        { STOCK_MAIL_SEND_RECEIVE, N_("_Receive Mail") },
	{ STOCK_MAIL_UNREAD, "mail-unread", 0, 0, Gtk.STOCK_NEW },
        { STOCK_IMAGE, null, 0, 0, "gnome-mime-image" },
        { STOCK_INBOX, null, 0, 0, "stock_inbox" },
        { STOCK_OUTBOX, null, 0, 0, "stock_outbox" },
        { STOCK_REPORT_BUG, null, 0, 0, "bug-buddy" },
        { STOCK_SENT_MAIL, null, 0, 0, "stock_sent-mail" },
        { STOCK_USER_TRASH }
    };

    public static void register_stock_items () {
        /* We use translation_domain as a fallback icon name. We need to copy
           the array and reset the domain, otherwise translations break. */
        var factory = new Gtk.IconFactory ();
        Gtk.StockItem[] stock_items_new = {};
        foreach (var item in stock_items) {
            var icon_set = new Gtk.IconSet ();
            var icon_source = new Gtk.IconSource ();
            if (item.translation_domain != null) {
                icon_source.set_icon_name (item.translation_domain);
                item.translation_domain = null;
                icon_set.add_source (icon_source);
            }
            icon_source.set_icon_name (item.stock_id);
            icon_set.add_source (icon_source);
            factory.add (item.stock_id, icon_set);
            stock_items_new += item;
        }
        Gtk.stock_add (stock_items_new);
        factory.add_default ();
    }

    public static bool show_uri (Gdk.Screen screen, string uri) {
        string real_uri = uri;
        if (!("://" in uri))
            return spawn_module ("compose", uri);

        try {
            Gtk.show_uri (screen, real_uri, Gtk.get_current_event_time ());
        }
        catch (GLib.Error error) {
            try {
                string[] handlers = {
                    "xdg-open %u",
                    "exo-open %u",
                    "gnome-open %u"
                };
                foreach (var handler in handlers) {
                    GLib.AppInfo info;
                    info = GLib.AppInfo.create_from_commandline (handler, "", 0);
                    var uris = new List<string>();
                    uris.prepend (real_uri);
                    if (info.launch_uris (uris, new GLib.AppLaunchContext ()))
                        return true;
                }
            }
            catch (GLib.Error launch_error) {
                var error_dialog = new Gtk.MessageDialog (null, 0,
                    Gtk.MessageType.ERROR, Gtk.ButtonsType.OK,
                    _("Failed to launch external application."));
                error_dialog.format_secondary_text (launch_error.message);
                error_dialog.response.connect ((dialog, response)
                    => { dialog.destroy (); });
                error_dialog.show ();
            }
        }
        return true;
    }

    public static bool execute_command (string command) {
        try {
            var info = GLib.AppInfo.create_from_commandline (command, "", 0);
            if (info.launch (null, null))
                return true;
        }
        catch (GLib.Error error) {
            var error_dialog = new Gtk.MessageDialog (null, 0,
                Gtk.MessageType.ERROR, Gtk.ButtonsType.OK,
                _("Failed to execute external command."));
            error_dialog.format_secondary_text (error.message);
            error_dialog.response.connect ((dialog, response)
                => { dialog.destroy (); });
            error_dialog.show ();
        }
        return true;
    }

    public static bool spawn_module (string name,
        string? arg1=null, string? arg2=null) {

        string command = argv0 + " --module " + name;
        if (arg1 != null)
            command += " " + Shell.quote (arg1);
        if (arg2 != null)
            command += " " + Shell.quote (arg2);
        try {
            /* Can't use GLib.AppInfo as it would mangle the arguments */
            if (Process.spawn_command_line_async (command))
                return true;
        }
        catch (GLib.Error error) {
            var error_dialog = new Gtk.MessageDialog (null, 0,
                Gtk.MessageType.ERROR, Gtk.ButtonsType.OK,
                _("Failed to open module %s."), name);
            error_dialog.format_secondary_text (error.message);
            error_dialog.response.connect ((dialog, response)
                => { dialog.destroy (); });
            error_dialog.show ();
        }
        return true;
    }

    static Notify.Notification? notification = null;
    public static void send_notification (string message) {
        try {
            if (!Notify.is_initted ()) {
                if (!Notify.init ("Postler"))
                    throw new FileError.FAILED (_("Failed to initialize."));
            }
            if (notification == null) {
                notification = (Notify.Notification)GLib.Object.new (
                    typeof (Notify.Notification),
                    "summary", GLib.Environment.get_application_name (),
                    "body", message,
                    "icon-name", STOCK_INTERNET_MAIL);

                unowned List<string> caps = Notify.get_server_caps ();
                foreach (string cap in caps) {
                    if (cap == "actions") {
                        notification.add_action ("default", _("Open"), (n, a) => {
                            Postler.App.spawn_module ("bureau");
                        });
                    }
                    else if (cap == "sound")
                        notification.set_hint_uint32 ("suppress-sound", 1);
                }
            }
            else
                notification.set ("body", message);
            notification.show ();
        } catch (Error error) {
            GLib.warning (_("Failed to send notification: %s"), error.message);
        }
    }

    static Canberra.Context sound_context = null;
    public static void play_sound (string sound) {
        if (sound_context == null)
            Canberra.Context.create (out sound_context);
        sound_context.play (0, Canberra.PROP_EVENT_ID, sound);
    }

    public static void button_menu_position (Gtk.Menu menu, ref int x, ref int y,
        ref bool push_in) {
        /* Position the menu right below the widget */
        var widget = menu.get_attach_widget ();
        Gtk.Allocation allocation = widget.allocation;
        int widget_x, widget_y;
        widget.window.get_position (out widget_x, out widget_y);
        Gtk.Requisition requisition;
        widget.size_request (out requisition);
        int widget_height = requisition.height;
        x = widget_x + allocation.x;
        y = widget_y + allocation.y + widget_height;
        push_in = true;
    }
}

