/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include "event_management.h"
#include "event.h"
#include "dbus_server.h"
#include "cpufreq_management.h"
#include "globals.h"

#include <string>
#include <signal.h>

using namespace Powersave::Globals;

EventManagement::EventManagement()
{
	id_count = 0;
}

EventManagement::~EventManagement()
{
}

int EventManagement::registerEvent(Event & event, time_t timeout)
{
	pDebug(DBG_INFO, "registering event no.'%d'", id_count);
	// give the event an unique id
	event.setID(id_count);
	// the next event will get a higher id
	id_count++;
	// set the timeout in milliseconds
	event.setTimeout(timeout * 1000);
	// start the counter which looks if an event timed out
	event.start();

	// append the event to the list of pending events
	Events.push_back(event);

	return event.ID();
}

bool EventManagement::remove(int id)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		if (id == it->ID()) {
			Events.erase(it);
			return 1;
		}
	}

	return 0;
}

int EventManagement::count()
{
	return Events.size();
}

const string EventManagement::name(int id)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		if (id == it->ID()) {
			return it->name();
		}
	}

	return "";
}

const string EventManagement::cur_prog(int id)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {
		if (id == it->ID()) {
			if (it->_current_program_2_execute != it->end()) {
				return *(it->_current_program_2_execute);
			} else {
				return "";
			}
		}
	}

	return "";
}

int EventManagement::moveOn(int id)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		if (id == it->ID()) {
			it->_current_program_2_execute++;
			if (it->_current_program_2_execute == it->end())
				return 0;

			return 1;
		}
	}

	return -1;
}

Event* EventManagement::event(int id)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		if (id == it->ID()) {
			return &(*it);
		}
	}

	return NULL;
}

void EventManagement::adjustTimeouts(unsigned long sleep_time)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		it->increaseSleeptime(sleep_time);
	}
}

int EventManagement::eventIsRunning(const std::string &eventName)
{
	for (std::list < Event >::iterator it = Events.begin();
	     it != Events.end(); ++it) {

		if (eventName == it->name()) {
			return 1;
		}
	}

	return 0;
}


void EventManagement::checkScriptReturn(int id, int returnValue, string &comment)
{

	// get the program name
	string prog = cur_prog(id);

	// if a program matching an event was found
	if (!prog.empty() || id == -1) {
		pDebug(DBG_INFO, "SCRIPT returned: Name: '%s', Return value: %d, "
				 "Comment: '%s'", prog.c_str(), returnValue, comment.c_str());
	} else {
		pDebug(DBG_DIAG, "Unknown SCRIPT returned. Return value %d, text: '%s'", returnValue,
		       comment.c_str());
		// allow popup windows also from unknown processes?
		// if (returnValue !=2)
		return;
	}

	// enable the buttons when we come back from restore
	if (prog.find("restore_after_s") == 0) {
		pm->ignore_buttons(false);
	}
	// evaluate the return code
	switch (returnValue) {
		// script succeeded
	case 0:{
			// set the event to the next script
			int i = moveOn(id);

			// no scripts left, so finished successfully
			if (!i) {
				const string name = this->name(id);
				pDebug(DBG_INFO, "SCRIPT Event %s finished successfully", name.c_str());
				// notify clients here
				this->remove(id);
			} else {
				continueEvent(this->event(id), "");
			}
			break;
		}
		// script requests a graphical notification from the daemon
	case 2:
		pm->x_notification(comment);
		break;
		// script requests a xlock/screensaver from the daemon
	case 3:
		pm->screenlock(comment);
		break;
	case 4:
		DBus_Server::emitSignal("Progress", comment.c_str());
		break;
		// script has failed
	case 1:
		const string event_name = this->name(id);
		pDebug(DBG_INFO, "SCRIPT %s failed to execute", event_name.c_str());

		// if the suspend/standby event has failed --> enable buttons
		if ((event_name.find("global.suspend") == 0)
		    || (event_name.find("global.standby") == 0)) {
			pm->ignore_buttons(false);
			pm->resumeNetworkManager();
		}
		// notify clients here
		// x_notification(comment);
		// remove the event from the list of pending events
		this->remove(id);
		break;
	}
}

bool EventManagement::continueEvent(Event *event, string acpi_event_line)
{
	if (event->_programs_2_execute.empty())
		return false;

	if (event->isInternalAction()) {
		if (!executeInternalAction(event, event->currentAction())) {
			pDebug(DBG_DIAG,
			       "Internal action %s returned an error",
			       event->currentAction().c_str());
			string c = "internal action failed: ";
			c.append(event->currentAction());
			checkScriptReturn(event->ID(), 1, c);
		}
		else {
			pDebug(DBG_INFO,
			       "Internal action %s executed",
			       event->currentAction().c_str());
			string c = "internal action successfull: ";
			c.append(event->currentAction());
			checkScriptReturn(event->ID(), 0, c);
		}
	}
	else {
		if (!event->execute_program(acpi_event_line)) {
			pDebug(DBG_WARN,
			       "Could not execute program %s for event %s: %s",
			       PUB_SCRIPT_DIR, event->name().c_str(), strerror(errno));
			string c = "Cannot access or execute: ";
			c.append(event->currentAction());
			checkScriptReturn(event->ID(), 1, c);
		}
	}

	return true;
}

bool EventManagement::executeInternalAction(Event *event, const string action)
{
	bool ret = true;

	pDebug(DBG_INFO, "Executing internal action: %s", action.c_str());

	if (!action.compare("ignore")) {
		/* ... */
	} else if (!action.compare("throttle")) {
		pm->throttle();
	} else if (!action.compare("dethrottle")) {
		pm->dethrottle();
	}
	/* really do the suspend/standby */
	else if (!action.compare("do_suspend_to_disk")) {
		pm->sleep_wrapper("suspend2disk");
		pm->executeScript(ACPI_SLEEP_SCRIPT, "suspend2disk");
		pm->sleep_wrapper("suspend2disk");
	} else if (!action.compare("do_suspend_to_ram")) {
		pm->sleep_wrapper("suspend2ram");
		pm->suspend_to_ram();
		pm->sleep_wrapper("suspend2ram");
	} else if (!action.compare("do_standby")) {
		pm->sleep_wrapper("standby");
		pm->standby();
		pm->sleep_wrapper("standby");
	} else if (!action.compare("screen_saver")) {
		pm->screenlock(event->name());
	}
	else if (!action.compare("suspend_networkmanager")) {
		pm->suspendNetworkManager();
	} else if (!action.compare("resume_networkmanager")) {
		pm->resumeNetworkManager();
	}
	/* throw a suspend/standby event (will trigger other needed
	   actions for proper suspend/standby */
	else if (!action.compare("suspend_to_disk")) {
		if (pm->haveSuspend2())
			executeEvent("global.suspend2disk.other");
		else
			executeEvent("global.suspend2disk");
	} else if (!action.compare("suspend_to_ram")) {
		executeEvent("global.suspend2ram");
	} else if (!action.compare("standby")) {
		executeEvent("global.standby");
	} else if (!action.compare("reread_cpu_capabilities")) {
		if (cpufreq->isSupported()) {
			cpufreq->rereadFrequencies();
		}
	} else
		ret = false;

	return ret;
}

bool EventManagement::executeEvent(const string &eventName, const char *acpi_event_line)
{
	/* this is only for acpi events... */
	Event execEvent = config_obj->current_scheme->getEvent(eventName);
	int eventId;

	if (execEvent.name() == "global.suspend2disk" ||
	    execEvent.name() == "global.suspend2ram" ||
	    execEvent.name() == "global.standby") {

		if (eventIsRunning(execEvent.name())) {
			pDebug(DBG_WARN,
			       "Event %s already running. Ignoring this one",
			       execEvent.name().c_str());
			return true;
		}
	}

	if (execEvent.name() == "default") {
		pDebug(DBG_ERR, "Tried to execute non-existing event.");
		return false;
	}

	DBus_Server::emitSignal("PowersaveEvent", eventName.c_str());

	// append the event to the list of pending events. Timeout is set to 120 seconds.
	// are 2 minutes enough?
	eventId = registerEvent(execEvent, 120);

	// do not ignore child if daemon terminates
	// this is the only event where main process has
	// to wait until proxy is finished
	if (execEvent.name() == "daemon.terminate")
		signal(SIGCHLD, SIG_DFL);

	// execute the first script/internal action of the event
	continueEvent(&execEvent, acpi_event_line);

	return true;
}

void EventManagement::checkEventTimeouts()
{
	// go through all events and check if one of them timed out
	std::list < Event >::iterator it;
	for (it = Events.begin(); it != Events.end(); it++) {

		if (it->timeElapsed() > it->getTimeout()) {
			pDebug(DBG_WARN, "Timeout for event %s with id %d reached, "
			       "removing", it->name().c_str(), it->ID());
			Events.erase(it);
			/* deleting the object may result in invokation
			   of a non existing event (it->timeElapsed() above)
			   -> start reiterating from the beginning of the list
			 */
			it = Events.begin();
		}
	}
}

