/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef POWERSAVE_PM_INTERFACE_H
#define POWERSAVE_PM_INTERFACE_H

#ifdef CPUFREQ_MEASURE
#define MEASURE_LOG_FILE "/tmp/cpufreq.log"
#endif

#define ACPI_SLEEP_SCRIPT "do_acpi_sleep"

#include <errno.h>
#include "throttle_interface.h"
#include "config_pm.h"
#include "powersave_dbus.h"
#include "battery.h"
#include "device_management.h"

class MainLoop;
class EventManagement;
class Brightness;
class ClientConnection;
class CpufreqManagement;

typedef struct PM_STATE {
	// THERM_MODE comes from library libpower.h
	THERM_MODE TEMP_STATE;
	AC_ADAPTER_STATE AC_STATE;
	 PM_STATE() {
		TEMP_STATE = OK;
		AC_STATE = AC_UNKNOWN;
}} PM_STATE;

/** sleep states requested by user, only used for APM standby/suspend
   workarounds. This should go somehow to APM_Interface later */
enum PSD_SLEEP_STATE_REQUESTS { PSD_NO_SLEEP_REQUEST = 0, PSD_SUSPEND2DISK, PSD_SUSPEND2RAM, PSD_STANDBY };

/** @brief global object for abstraction of acpi and apm
 *
 * mainly an abstraction of apm/acpi but contains other things like
 * ac adapter  handling which maybe moves to own class in future.
 */
class PM_Interface {
public:
	/** @brief constructor setting up the object */
	PM_Interface();
	
	/** @brief destructor freeing all members */
	virtual ~ PM_Interface();
	
	/** @brief get the current state the battery is in
	 *
	 * @return the current battery state
	 */
	BATTERY_STATES batteryState();

	/** @brief internal action which is executed directly before and after
	 * do_suspend_* should go into this function
	*/
	void sleep_wrapper(const string &);

	/** @brief used to execute scripts in INT_SCRIPT_PATH */
	int executeScript(const string &script, const string &param);

	/** @brief triggers a screenlock either from a client or script
	 *
	 * emits a screenlock signal over the bus and checks if a client
	 * with capability screenlock is connected. If not, execute script
	 * to lock the screen
	*/
	int screenlock(const string &param);

	/** @brief triggers a notification for the user
	 *
	 * emits a notification signal over the bus and checks if a client
	 * with capability notifications is connected. If not, execute
	 * script to display the notification
	 */
	int x_notification(const string &param);

	/** @brief sets if we ignore button events */
	void ignore_buttons(bool);

	/** @brief starts the pm_object
	 *
	 * throws appropriate ac adapter/battery events and initializes
	 * the active scheme
	 */
	virtual int start();

	/** @brief checks for temperature events and executes appropriate
	 * events */
	virtual void checkTemperatureStateChanges();

	/** @brief Checks for Battery State changes (when AC adapter is
	 *         not plugged in) and throws appropriate event
	 *
	 * Be careful in ACPI you have to check whether the battery alarm also needs
	 * to be set
	 */
	virtual void checkBatteryStateChanges();

	/** @brief Checks for AC state changes and throws appropraite events
	 *
	 * if AC changes to offline, also battery state changes will be checked
	 * @return integer with state of AC 
	 * @retval 0 if AC state has not changed
	 * @retval 1 if AC state has changed
	 * @retval -1 if an error occur
	 */
	virtual int checkACStateChanges();

	/** @brief update temperature state
	 *
	 * @return integer with state of temperature
	 * @retval -1 on failure
	 * @retval 0 when state did not change
	 * @retval 1 when a more critical temperature level is reached
	 * @retval 2 when a less critical battery level is reached
	 */
	virtual int updateTemperatureState();

	/** @brief updates ACState
	 *
	 * @return integer with result of check for changes
	 * @retval 0 if AC state did not change
	 * @retval 1 else 
	 */
	virtual int updateACState();

	/** @brief updates the cpu state and throws appropriate events */
	virtual bool checkThrottling();

	/** @brief check if a removed/added device belongs to us
	 *
	 * calls the appropriate functions out of DeviceManagement
	 *
	 * @param udi the device udi which got removed/added
	 * @param added if the device was added
	 */
	void checkDevice(char *udi, bool added);

	/** @brief switches settings to a possibly new scheme
	 *         and activates them
	 * @return integer with result whether the scheme could (not) be set
	 *         or is already set.
	 * @retval -1 if scheme does not exist
	 * @retval  0 if scheme is already set
	 * @retval  1 if scheme has been successfully set
	 */
	virtual int activateScheme(const string &scheme_name);

	/** @brief switches scheme according to ac_state
	 *
	 * @param ac_state any ac adapter status
	 * @return integer with result whether the scheme could (not) be set
	 *         or is already set.
	 * @retval -1 if scheme does not exist
	 * @retval  0 if scheme is already set
	 * @retval  1 if scheme has been successfully set
	 */
	virtual int activateScheme(AC_ADAPTER_STATE ac_state);

	/** @brief Activate scheme settings 
	 *
	 * Inheriting functions in APM_Interface and ACPI_Interface
	 * set them specifically for APM/ACPI
	 *	    
	 * Add additional code that is needed after activating a new scheme
	 * or after re-reading configurations here (e.g. brightness?)
	 *
	 * Be aware that first the inhereting functions in the ACPI/APM_Interface
	 * layer are called
	 */
	virtual void activateSettings();

	/** @brief reread config files. */
	virtual int rereadConfig();

	/** @brief throttles the cpu
	 * 
	 * Only throttle if temperature is in unknown, ok or active state.
	 * Otherwise kernel may have throttled ...
	 */
	virtual void throttle(int percent = -1);

	/** @brief dethrottles the cpu
	 *
	 * Only dethrottle if temperature is in unknown, ok or active state.
	 * Otherwise kernel may have throttled ...
	 */
	virtual void dethrottle();

	/** @brief called when coming back from suspend
	 *
	 * updates ac state, battery state and adjust speeds
	 */
	virtual void resume();

	/** @brief handles a dbus request
	 *
	 * @param req the dbus message containing the request
	 * @param reply dbus message which should be filled with a reply
	 * @param type one of REQUEST_MESSAGE, ACTION_MESSAGE or
	 *        MANAGER_MESSAGE
	 *
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleDBusRequest(DBusMessage *req, DBusMessage *reply,
				     POWERSAVE_MESSAGE_TYPE type);

	/** @brief triggers suspend to ram
	 *
	 * has to be implemented in derived classes (apm/acpi)
	 */
	virtual int suspend_to_ram() = 0;

	/** @brief triggers standby
	 *
	 * has to be implemented in derived classes (apm/acpi)
	 */
	virtual int standby() = 0;

	/** @brief opens the event fd
	 *
	 * has to be implemented in derived classes (apm/acpi)
	 */
	virtual int openHWEventFD() = 0;

	/** @brief opens the acpid socket
	 *
	 * has to be implemented in derived classes (only acpi)
	 */
	virtual int openAcpidSocket() = 0;

	/** @brief handles a request on the event socket/fd 
	 *
	 * has to be implemented in derived classes (only acpi)
	 */
	virtual int handleHWEventRequest(int fd = -1) = 0;

	/** @brief sends a sleep signal to the networkmanager
	 *
	 * it only sends the signal if the NetworkManager can be found
	 */
	void suspendNetworkManager();

	/** @brief sends a wake signal to the networkmanager
	 *
	 * it only sends the signal if the NetworkManager can be found
	 */
	void resumeNetworkManager();

	/** @brief check whether we have a suspend2 enabled kernel
	 *
	 * @return true if suspend2 is enabled, false otherwise
	 */
	bool haveSuspend2();

	/** @brief object to manage clients connected to the daemon */
	ClientConnection *_cC;

	/** @brief object to manage running events */
	EventManagement *_eM;

	/** @brief object implementing brightness functionality on some
	 * laptops */
	Brightness *_brightness;

protected:
	/** @brief defines whether cooling mode is supported */
	bool _cooling_mode_supported;

	/** @brief defines whether thermal trip points are supported */
	bool _thermal_trip_points_supported;

	/** @brief the current pm state contianing e.g. battery, thermal
	 * and cpu info */
	PM_STATE _cur_state;

	/** @brief defines the supported sleeping states */
	int64_t _supported_sleeping_states;

	/** @brief used for workaround for apm to remembemer triggered
	   sleep state, for ACPI to discard button events during resume */
	static PSD_SLEEP_STATE_REQUESTS _requested_sleep_state;

	/** @brief defines whether sleep was triggered before */
	static bool _sleep_triggered;

	/** @brief object caring about throttling */
	ThrottleInterface _throttleInterface;

	/** @brief battery object represention one single battery */
	Battery _battery;

private:
	/** @brief handles a message on the manager interface
	 * @param msg the dbus message received
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleManagerMessage(DBusMessage * msg);
 
	/** @brief handles a message on the events interface
	 * @param msg the dbus message received
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleScriptsMessage( DBusMessage *msg );

	/** @brief handles a message on the request interface
	 * @param msg the dbus message received
	 * @param reply the dbus reply to send back
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleRequestMessage(DBusMessage * msg, DBusMessage * reply);

	/** @brief handles a message on the request interface
	 * @param msg the dbus message received
	 * @return DBUS_ERROR
	 */
	DBUS_ERROR handleActionMessage(DBusMessage * msg);

	/** @brief object caring about powermanagable devices */
	DeviceManagement _devices;
};

#endif // POWERSAVE_PM_INTERFACE_H
