/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005 SUSE Linux Products GmbH               *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include <getopt.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>

#include "powerlib.h"
#include "config.h"

#include "powersave_dbus.h"

#include <string>
#include <iostream>

#define MAX_OPT_LEN 20
#define MAX_PARAM_LENGTH 50

using namespace std;

static int option = -1;
static int verbose = 0;
static char schemeName[MAX_PARAM_LENGTH + 1] = "";

static char dpmDeviceClass[MAX_PARAM_LENGTH + 1] = "";

// toDo: set modes to specification ...
static int BATT_INFO = 0;
static int BAT_STATE = 0;
static int THROT_INFO = 0;
static int SET_THROT_PERC = 0;
static int AC_INFO = 0;
static int APM_ACPI = 0;
static int CPUFREQ_INFO = 0;
static int REAL_CPU_SPEED = 0;
static int EXT_BATT_INFO = 0;
static int THERMAL_INFO = 0;
static int FAN_INFO = 0;

static int PERFORMANCE_SPEED = 0;
static int POWERSAVE_SPEED = 0;
static int DYNAMIC_SPEED = 0;

static int SUSPEND_TO_DISK = 0;
static int SUSPEND_TO_RAM = 0;
static int STANDBY = 0;
static int VERSION_INFO = 0;

static int LIST_SCHEMES = 0;
static int SET_ACTIVE_SCHEME = 0;
static int GET_SCHEME_DESCRIPTION = 0;

static int GET_BRIGHTNESS = 0;
static int GET_BRIGHTNESS_LEVELS = 0;
static int SET_BRIGHTNESS = 0;

static int DPM_SUSPEND = 0;
static int DPM_RESUME = 0;
static int DPM_NUMBER = 0;
static int DPM_GET_DEVICES = 0;
static int DPM_GET_CLASSES = 0;

static int GET_CLIENTS = 0;

void general_error()
{
	fprintf(stderr, "An error occured. Make sure that the powersave daemon is\n"
		"running and use -v to increase verbose level and have a look into syslog.\n");
}

void no_connect_no_rights_error()
{
	fprintf(stderr, "Could not connect to powersaved because of missing permissions."
		" Are you priviledged to connect to the powersaved?\n");
}

void no_connect_no_daemon_error()
{
	fprintf(stderr, "Could not connect to powersaved. Is the powersaved running?\n");
}

void no_cpufreq_error()
{
	fprintf(stderr, "Speedstepping is not supported.\n");
}

void usage()
{
	fprintf(stderr,
		"Usage: powersave {-f|-l|-A} {-u|-U|-m} -[crbBsSaTFVtxKL] [-p percent] [-v level] [-e scheme_name] -k [brightness level]\n"
		"                 [-h|--help] -i [device class] -I [device class] -J [device class] -j [device number]\n\n"
		" Suspend/Standby:\n"
		"   -U, --suspend-to-disk                  set machine into suspend-to-disk (ACPI S4/APM suspend)\n"
		"   -u, --suspend-to-ram                   set machine into suspend-to-ram  (ACPI S3/APM suspend)\n"
		"   -m, --standby                          set machine into standby         (ACPI S1/APM standby)\n\n"
		" CPUFreq modes:\n"
		"   -f, --performance-speed                set cpufreq to performance mode\n"
		"   -l, --powersave-speed                  set cpufreq to powersave mode\n"
		"   -A, --dynamic-speed                    set cpufreq to dynamic mode\n"
		"   -c, --cpufreq-state-info               print out the current cpufreq policy\n\n"
		" Throttling:\n"
		"   -p <x>, --set-throttling-percent <x>   throttles processors of machine by X percent\n"
		"   -t, --throttling-info                  prints out current throttling state\n\n"
		" Schemes:\n"
		"   -x, --list-schemes                     show all available schemes\n"
		"   -X, --get-scheme-description           show the description of scheme\n"
		"   -e <x>, --set-active-scheme <x>        switch currently active scheme\n\n"
		" Brightness:\n"
		"   -k, --set-brightness <x>               set current brightness level\n"
		"   -K, --get-brightness                   get current brightness level\n"
		"   -L, --get-brightness-levels            get number of brightness levels\n\n"
		" Runtime powermanagement: (experimental)\n"
		"   -i --dpm-suspend <device class>       set devices of a device class into D3 power save mode\n"
		"   -I --dpm-resume <device class>        set devices of a device class back to D0 power mode\n"
		"   -j --dpm-device-number <x>            only suspend/resume device number <x>\n"
		"   -J --dpm-get-devices <device class>   List devices from a specific device class\n"
		"   -o --dpm-get-classes                  List available device classes\n\n"
		" Print information:\n"
		"   -s, --battery-state-info               print out the current battery state\n"
		"   -b, --battery-info                     general battery info\n"
		"   -B, --detailed-battery-info            info of each battery\n"
		"   -S, --apm-acpi                         prints out whether APM or ACPI is supported\n"
		"   -a, --ac-status-info                   power supply info (AC/Battery)\n"
		"   -T, --get-thermal-info                 thermal Info\n"
		"   -F, --get-fan-info                     fan Info\n"
		"   -r, --calc-CPU-frequency               current speed (see manpage)\n"
		"   -C, --clients                          get all connected clients and their capabilities (only for debugging)\n"
		"   -v <x>, --verbose <x>                  Verbose level (see manpage for values)\n"
		"   -V, --version                          current compiled version\n\n");
	exit(EXIT_FAILURE);
}

void checkArgs()
{
	int info =
	    BATT_INFO + BAT_STATE + THROT_INFO + SET_THROT_PERC + AC_INFO + APM_ACPI + CPUFREQ_INFO + REAL_CPU_SPEED +
	    EXT_BATT_INFO + THERMAL_INFO + FAN_INFO + GET_BRIGHTNESS + GET_BRIGHTNESS_LEVELS;
	int cpufreq = PERFORMANCE_SPEED + POWERSAVE_SPEED + DYNAMIC_SPEED;
	int sleep = SUSPEND_TO_DISK + SUSPEND_TO_RAM;

	if (sleep > 1 || cpufreq > 1) {
		usage();
		exit(EXIT_FAILURE);
	}
	if (sleep == 1 && (info > 0 || cpufreq > 0)) {
		usage();
		exit(EXIT_FAILURE);
	}
	return;
}

void getArgs(int argc, char **argv)
{

	int option_index = 0;
	char *endptr[MAX_PARAM_LENGTH + 1];

	struct option opts[] = {
		{"performance-speed", 0, 0, 'f'},
		{"powersave-speed", 0, 0, 'l'},
		{"dynamic-speed", 0, 0, 'A'},
		{"cpufreq-state-info", 0, 0, 'c'},
		{"calc-CPU-frequency", 0, 0, 'r'},
		{"apm-acpi", 0, 0, 'S'},
		{"throttling-info", 0, 0, 't'},
		{"set-throttling-percent", 1, 0, 'p'},
		{"suspend-to-disk", 0, 0, 'U'},
		{"suspend-to-ram", 0, 0, 'u'},
		{"standby", 0, 0, 'm'},
		{"get-thermal-info", 0, 0, 'T'},
		{"get-fan-info", 0, 0, 'F'},
		{"battery-info", 0, 0, 'b'},
		{"detailed-battery-info", 0, 0, 'B'},
		{"battery-state-info", 0, 0, 's'},
		{"ac-status-info", 0, 0, 'a'},
		{"list-schemes", 0, 0, 'x'},
		{"set-active-scheme", 1, 0, 'e'},
		{"get-scheme-description", 1, 0, 'X'},
		{"set-brightness", 1, 0, 'k'},
		{"get-brightness", 0, 0, 'K'},
		{"get-brightness-levels", 0, 0, 'L'},
		{"dpm-suspend", 1, 0, 'i'},
		{"dpm-resume", 1, 0, 'I'},
		{"dpm-device-number", 1, 0, 'j'},
		{"dpm-get-devices", 1, 0, 'J'},
		{"dpm-available-classes", 0, 0, 'o'},
		{"verbose", 0, 0, 'v'},
		{"version", 0, 0, 'V'},
		{"clients", 0, 0, 'C'},
		{"help", 0, 0, 'h'},
		{NULL, 0, 0, 0},
	};
	while (1) {
		int i = getopt_long(argc, argv, "v:p:e:crbBsSaTFVxuUmhtlfAk:KLi:I:j:J:oX:C", opts, &option_index);
		if (argc <= 1) {
			usage();
			exit(0);
			break;
		}
		if (i == -1) {
			break;
		}
		if (optarg && strlen(optarg) >= MAX_PARAM_LENGTH) {
			fprintf(stderr, "Too long parameter\n");
			exit(EXIT_FAILURE);
		}
		switch (i) {
		case 0:
			/*
			   if (optarg && strlen(optarg) >= MAX_PARAM_LENGTH){
			   fprintf(stderr, "Too long parameter\n");
			   exit(EXIT_FAILURE);
			   }
			 */
			break;

		case 'c':
			CPUFREQ_INFO = 1;
			break;
		case 'b':
			BATT_INFO = 1;
			break;
		case 'B':
			EXT_BATT_INFO = 1;
			break;
		case 'r':
			REAL_CPU_SPEED = 1;
			break;
		case 'S':
			APM_ACPI = 1;
			break;
		case 'u':
			SUSPEND_TO_RAM = 1;
			break;
		case 'm':
			STANDBY = 1;
			break;
		case 't':
			THROT_INFO = 1;
			break;
		case 'f':
			PERFORMANCE_SPEED = 1;
			break;
		case 'l':
			POWERSAVE_SPEED = 1;
			break;
		case 'A':
			DYNAMIC_SPEED = 1;
			break;
		case 'U':
			SUSPEND_TO_DISK = 1;
			break;
		case 'a':
			AC_INFO = 1;
			break;
		case 'T':
			THERMAL_INFO = 1;
			break;
		case 'e':
			SET_ACTIVE_SCHEME = 1;
			strcpy(schemeName, optarg);
			break;
		case 'x':
			LIST_SCHEMES = 1;
			break;
		case 'X':
			GET_SCHEME_DESCRIPTION = 1;
			strcpy(schemeName, optarg);
			break;
		case 'F':
			FAN_INFO = 1;
			break;
		case 'p':
			SET_THROT_PERC = 1;
			option = strtol(optarg, endptr, 10);
			if (**endptr != '\0' || option < 0 || option > 100) {
				printf("Wrong throttling in percent value,"
				       " must be between 0-100: %s\n", strerror(errno));
				exit(EXIT_FAILURE);
			}
			break;
		case 's':
			BAT_STATE = 1;
			break;
		case 'k':
			SET_BRIGHTNESS = 1;
			if (*optarg == 'u')
				option = -1;	// up
			else if (*optarg == 'd')
				option = -2;	// down
			else
				option = strtol(optarg, endptr, 10);
			break;
		case 'K':
			GET_BRIGHTNESS = 1;
			break;
		case 'L':
			GET_BRIGHTNESS_LEVELS = 1;
			break;
		case 'J':
			DPM_GET_DEVICES = 1;
			strcpy(dpmDeviceClass, optarg);
			break;
		case 'j':
			DPM_NUMBER = 1;
			option = strtol(optarg, endptr, 10);
			break;
		case 'i':
			DPM_SUSPEND = 1;
			strcpy(dpmDeviceClass, optarg);
			break;
		case 'I':
			DPM_RESUME = 1;
			strcpy(dpmDeviceClass, optarg);
			break;
		case 'o':
			DPM_GET_CLASSES = 1;
			break;
		case 'C':
			GET_CLIENTS = 1;
			break;
		case 'v':
			verbose = strtol(optarg, endptr, 10);
			if (**endptr != '\0' || verbose < 0 || verbose > 36) {
				printf("Wrong verbose(-v, debug) paramater: %s\n", optarg);
				exit(EXIT_FAILURE);
			} else {
				setDebugLevel(verbose);
				//fprintf(stderr, "Debug Level set to %d\n", verbose);
			}
			break;
		case 'V':
			VERSION_INFO = 1;
			break;
		case 'h':
			usage();
			exit(0);
			break;
		}
	}

}

int get_bat_info(int ext)
{
	BatteryGeneral bg;
	int x, bat_num = 0, ac_state = 0;

	// test of general battery information:
	if (getBatteriesInfo(&bg) < 0) {
		fprintf(stderr, "Could not get battery information.");
		return -1;
	} else {
		ac_state = getACAdapterStatus();
		if (bg.remaining_percent >= 0) {
			fprintf(stdout, "Battery: %d %%", bg.remaining_percent);

			if (ac_state == AC_ONLINE && bg.remaining_minutes > 0) {
				fprintf(stdout, ", %d minutes until fully charged", bg.remaining_minutes);
			} else if (ac_state == AC_OFFLINE && bg.remaining_minutes > 0) {
				fprintf(stdout, ", %d minutes remaining", bg.remaining_minutes);
			}

		        if (bg.charging_state & CHARG_STATE_CHARGING) {
                                fprintf(stdout, " charging");
                        }
                        if (bg.charging_state & CHARG_STATE_DISCHARGING) {
                                fprintf(stdout, " discharging");
			}

			fprintf(stdout, "\n");
		}		

		if (ac_state == AC_ONLINE)
			fprintf(stdout, "AC is online.\n");
		else 
			fprintf(stdout, "AC is offline.\n");
	}
	// Test of detailed battery information
	if (ext) {
		if (checkACPI() == ACPI || checkACPI() == APM) {			
			for (x = 0; x < numBatteries(); x++) {

				int ret = getBatteryInfo(x, &bg);

				if (ret < 0) {
					fprintf(stderr, "Battery error. Maybe hal is not running?\n");
					exit(EXIT_FAILURE);
				}

				bat_num++;

				if (!ret) {
					fprintf(stdout, "Battery%d: not present\n", bat_num);
				}
				else {
					fprintf(stdout, "Battery%d: %d %%", bat_num, bg.remaining_percent);
					if (bg.remaining_minutes >= 0 ) {
						fprintf(stdout, ", %d minutes", bg.remaining_minutes);
					}
					if (bg.charging_state & CHARG_STATE_CHARGING) {
						fprintf(stdout, " charging");
					}
					if (bg.charging_state & CHARG_STATE_DISCHARGING) {
						fprintf(stdout, " discharging");
					}
					fprintf(stdout,"\n");
				}
			}
		} else
			fprintf(stdout, "No acpi or apm system\n");
	}
	return 1;
}

/** @brief main function
 *
 * errors:
 *     - -1 for parameter specific error
 *     - -2 if user has no rights to connect to daemon
 *     - -3 if socket conn could not be established (no daemon...)
*/
int main(int argc, char **argv)
{
	int ret = 0;
	int states, current, err_code;
	unsigned int min, max;
	float ret_freq;
	ThermalDev *thermal_dev;
	min = max = 0;
	getArgs(argc, argv);
	checkArgs();
	int y, x = 0;

	DBusMessage *reply;
	DBusError error;
	dbus_error_init(&error);

	string result;
	char *dummy = "";

	if (SUSPEND_TO_RAM) {
		if (getuid() == 0)
			err_code = dbusSendSimpleMessage(ADMIN_MESSAGE, "SuspendToRam");
		else
			err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "SuspendToRam");
		switch (err_code) {
		case REPLY_SUCCESS:
			break;
		case REPLY_DISABLED:
			fprintf(stderr, "Suspend-To-RAM has been disabled by administrator.\n");
			ret = -1;
			break;
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Suspend-To-RAM is not supported.\n");
			ret = -1;
			break;
		case REPLY_DBUS_ERROR:
			fprintf(stderr, "Cannot connect to powersaved. Is the daemon running? (%s)\n",
				DBus_Error_Array[err_code]);
			break;
		default:
			fprintf(stderr, "Error(%d): %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	}
	if (STANDBY) {
		if (getuid() == 0)
			err_code = dbusSendSimpleMessage(ADMIN_MESSAGE, "Standby");
		else 
			err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "Standby");
		switch (err_code) {
		case REPLY_SUCCESS:
			break;
		case REPLY_DISABLED:
			fprintf(stderr, "Standby has been disabled by administrator.\n");
			ret = -1;
			break;
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Standby is not supported.\n");
			ret = -1;
			break;
		case REPLY_DBUS_ERROR:
			fprintf(stderr, "Cannot connect to powersaved. Is the daemon running? (%s)\n",
				DBus_Error_Array[err_code]);
			break;
		default:
			fprintf(stderr, "Error(%d): %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	} else if (SUSPEND_TO_DISK) {
		if (getuid() == 0)
			err_code = dbusSendSimpleMessage(ADMIN_MESSAGE, "SuspendToDisk");
		else
			err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "SuspendToDisk");
		switch (err_code) {
		case REPLY_SUCCESS:
			break;
		case REPLY_DISABLED:
			fprintf(stderr, "Suspend-To-disk has been disabled by administrator.\n");
			ret = -1;
			break;
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Suspend-To-disk is not supported.\n");
			ret = -1;
			break;
		case REPLY_DBUS_ERROR:
			fprintf(stderr, "Cannot connect to powersaved. Is the daemon running? (%s)\n",
				DBus_Error_Array[err_code]);
			break;
		default:
			fprintf(stderr, "Error(%d): %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	}
	if (PERFORMANCE_SPEED) {
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "CpufreqPerformance");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			no_cpufreq_error();
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Speed already set to maximum \n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Speed set to maximum \n");
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
			ret = -1;
		}
	} else if (POWERSAVE_SPEED) {
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "CpufreqPowersave");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			no_cpufreq_error();
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Speed already set to minimum.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Speed set to minimum.\n");
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
			ret = -1;
		}
	} else if (DYNAMIC_SPEED) {
		err_code = dbusSendSimpleMessage(ACTION_MESSAGE, "CpufreqDynamic");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			no_cpufreq_error();
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Speed already set to dynamic.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Speed set to dynamic.\n");
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
			ret = -1;
		}
	}
	if (CPUFREQ_INFO) {
		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply, "CpufreqPolicy");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			no_cpufreq_error();
			ret = -1;
			break;
		case REPLY_SUCCESS:
			if (!dbusGetMessageString(reply, &dummy, 0) && dummy != NULL) {
				result = dummy;	//string(parsed_reply_message._strings[0]);
				if (result == "performance") {
					ret = 1;
					cout << "PERFORMANCE" << endl;
				} else if (result == "powersave") {
					ret = 2;
					cout << "POWERSAVE" << endl;
				} else if (result == "dynamic") {
					ret = 3;
					cout << "DYNAMIC" << endl;
				} else {
					ret = -1;
					cout << "Error: unknown reply '" << result << "' from daemon." << endl;
				}
			} else {
				ret = -1;
				general_error();
			}
			dbus_message_unref(reply);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			printf("Error: %d - %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	}
	if (BAT_STATE) {
		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply, "BatteryState");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			cerr << "No Battery." << endl;
			ret = -1;
			break;
		case REPLY_SUCCESS:
			if (!dbusGetMessageString(reply, &dummy, 0) && dummy != NULL) {
				result = dummy;
				if (result == "normal") {
					ret = 1;
					cout << "NORMAL" << endl;
				} else if (result == "warning") {
					ret = 2;
					cout << "WARNING" << endl;
				} else if (result == "low") {
					ret = 3;
					cout << "LOW" << endl;
				} else if (result == "critical") {
					ret = 4;
					cout << "CRITICAL" << endl;
				}
			} else {
				ret = -1;
				general_error();
			}
			dbus_message_unref(reply);
			break;
		default:
			cerr << "unknown reply code to 'BatteryState' request" << endl;
			ret = -1;
		}
	}
	if (SET_ACTIVE_SCHEME) {
		if (strlen(schemeName) != 0) {
			dummy = schemeName;

			if (!strcmp(schemeName, "AdvancedPowersave")) {
				int a;
				fprintf(stdout, "About to activate AdvancedPowersave scheme. This scheme "
					"is highly experimental. Really activate it? (y/n) ");

				a = getchar();
				
				if (tolower(a) != 'y') {
					fprintf(stdout, "Scheme AdvancedPowersave not set\n");
					exit(EXIT_SUCCESS);
				}

			}


			err_code = dbusSendMessage(ACTION_MESSAGE,
						   "SchemesSet", DBUS_TYPE_STRING, &dummy, DBUS_TYPE_INVALID);
			switch (err_code) {
			case REPLY_SUCCESS:
				printf("Scheme %s successfully activated\n", schemeName);
				break;
			case REPLY_ALREADY_SET:
				printf("Scheme %s already active\n", schemeName);
				break;
			case REPLY_INVALID_PARAM:
				printf("Scheme %s does not exist\n", schemeName);
				ret = -1;
				break;
			default:
				printf("Error: %s\n", DBus_Error_Array[err_code]);
				general_error();
			}
		}
	}
	if (LIST_SCHEMES) {
		char *_schemes;
		int active;
		int battery;
		int ac_power;

		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply, "SchemesGet");
		if (err_code == REPLY_SUCCESS) {
			if (!dbusGetMessageInteger(reply, &active, 0)
			    && !dbusGetMessageInteger(reply, &battery, 1)
			    && !dbusGetMessageInteger(reply, &ac_power, 2))
				if (err_code < 0) {
					fprintf(stderr, "Could not connect with daemon\n");
				} else {
					for (int x = 0;
					     !dbusGetMessageString(reply, &_schemes, x) && dummy != NULL; x++) {
						printf("%s%s%s%s\n",
						       _schemes,
						       ((active == x) ? "\tactive" : ""),
						       ((battery == x) ? "\tbattery_default_scheme" : ""),
						       ((ac_power == x) ? "\tAC_default_scheme" : ""));
					}
			} else {
				fprintf(stderr, "Could not assign schemes to current/battery/ac_power\n");
				ret = -1;
			}
		} else {
			general_error();
			ret = -1;
		}
	}
	if (GET_SCHEME_DESCRIPTION) {
		dummy = schemeName;
		char *description;

		err_code = dbusSendMessageWithReply(REQUEST_MESSAGE, &reply, "SchemesGetDescription",
					   DBUS_TYPE_STRING, &dummy, DBUS_TYPE_INVALID);

		switch (err_code) {
		case REPLY_INVALID_PARAM:
			printf("Error: Scheme %s does not exist\n", schemeName);
			ret = -1;
			break;
		case REPLY_SUCCESS:
			if (!dbusGetMessageString(reply, &description, 0)) {
				printf("%s\n", description);
			}
			else {
				general_error();
			}
			break;
		default:
			printf("Error: %s\n", DBus_Error_Array[err_code]);
			general_error();
		}
	}
	if (APM_ACPI) {
		switch (checkACPI()) {
		case APM:
			fprintf(stdout, "APM\n");
			ret = 2;
			break;
		case ACPI:
			fprintf(stdout, "ACPI\n");
			ret = 1;
			break;
		default:
			fprintf(stdout, "No ACPI/APM support \n");
			ret = -1;
		}
	}
	if (BATT_INFO) {
		if (get_bat_info(0) < 0) {
			fprintf(stderr, "Could not read out battery info properly\n");
			ret = -2;
		}
	}
	if (EXT_BATT_INFO) {
		if (get_bat_info(1) < 0) {
			fprintf(stderr, "Could not read out battery info properly\n");
			ret = -2;
		}
	}
	if (AC_INFO) {
		states = getACAdapterStatus();
		if (states == AC_ONLINE) {
			fprintf(stdout, "ONLINE\n");
			ret = 2;
		} else if (states == AC_OFFLINE) {
			fprintf(stdout, "OFFLINE\n");
			ret = 1;
		} else {
			fprintf(stderr, "AC state is unknown, assume ONLINE\n");
			ret = -1;
		}
	}
	if (THROT_INFO) {
		for (x = 0; getThrottlingInfoCPU(x, &states, &current) > 0; x++) {
			fprintf(stdout, "CPU: %d\nThrottling states: %d\nCurrent state: %d\n", x, states, current);
		}
		if (x == 0)
			fprintf(stderr, "Throttling not supported \n");
	}
	if (SET_THROT_PERC) {
		if (getThrottlingInfo(&states, &current) < 0) {
			fprintf(stderr, "Throttling not supported\n");
			ret = -1;
		} else {
			for (x = 0; getThrottlingInfoCPU(x, &states, &current) > 0; x++) {
				fprintf(stdout, "CPU %d:\nThrottling states : %d\nCurrent state: %d\n", x, states,
					current);
				if (setThrottlingPercentCPU(x, option) < 0) {
					fprintf(stderr, "Cannot set throttling state to %d percent for cpu %d\n", option,
						x);
					ret = -1;
					break;
				}
			}
			usleep(100);
			fprintf(stdout, "Changed to:\n");
			for (y = 0; getThrottlingInfoCPU(y, &states, &current) > 0; y++) {
				fprintf(stdout, "CPU %d:\nThrottling states : %d\nCurrent state: %d\n", y, states,
					current);
			}
		}
	}
	if (THERMAL_INFO) {
		err_code = getThermalZonesNum();
		if (err_code < 0) {
			fprintf(stderr, "No thermal devices available: %d\n", err_code);
			ret = -1;
		}
		thermal_dev = (ThermalDev *) malloc(sizeof(ThermalDev));
		for (x = 0; x < err_code && !(getThermalZone(x, thermal_dev) < 0); x++) {
			fprintf(stdout, "Thermal Device no. %d:\n", x);
			if (thermal_dev->temperature > 0)
				fprintf(stdout, "Temperature: %d\n", thermal_dev->temperature);
			if (thermal_dev->state > 0) {
				fprintf(stdout, "State: ");
				switch (thermal_dev->state) {
				case HOT:
					fprintf(stdout, "HOT\n");
					break;
				case CRITICAL:
					fprintf(stdout, "CRITICAL\n");
					break;
				case PASSIVE:
					fprintf(stdout, "PASSIVE\n");
					break;
				case ACTIVE:
					fprintf(stdout, "ACTIVE\n");
					break;
				default:
					fprintf(stdout, "UNDEFINED\n");
				}
			}
			if (thermal_dev->critical > 0)
				fprintf(stdout, "Critical: %d\n", thermal_dev->critical);
			if (thermal_dev->hot > 0)
				fprintf(stdout, "Hot: %d\n", thermal_dev->hot);
			if (thermal_dev->passive > 0)
				fprintf(stdout, "Passive: %d\n", thermal_dev->passive);
			for (y = 0; y < MAX_THERMAL_ACTIVE; y++) {
				if (thermal_dev->active[y] > 0)
					fprintf(stdout, "Active %d: %d\n", y, thermal_dev->active[y]);
			}
		}
		free(thermal_dev);
	}
	if (FAN_INFO) {
		err_code = getFanNum();
		if (err_code <= 0) {
			fprintf(stderr, "No fan devices available.\n");
			ret = -1;
		} else {
			for (x = 0; x < err_code && !((current = getFanStatus(x)) < 0); x++)
				printf("Fan no %d: %s\n", x, current ? "off" : "on");
		}
	}
	if (REAL_CPU_SPEED) {
		ret_freq = getRealProcessorSpeed();
		if (ret_freq == -2) {
			fprintf(stderr, "0\nMachine's architecture is not supported or feature has"
				" not been compiled for this machine's architecture(see manpage).\n");
			ret = -1;
		} else if (ret_freq < 0) {
			fprintf(stderr, "0\nCould not calculate CPU frequency\n");
			ret = -1;
		} else
			fprintf(stdout, "%5f MHz\n", ret_freq);
	}
	if (SET_BRIGHTNESS) {
		char *method = "BrightnessSet";
		switch (option) {
		case -1:
			method = "BrightnessUp";
			break;
		case -2:
			method = "BrightnessDown";
			break;
		default:
			break;
		}
		err_code = dbusSendMessage(ACTION_MESSAGE, method, DBUS_TYPE_INT32, &option, DBUS_TYPE_INVALID);

		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Setting of brightness not supported.\n");
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Brightness level already set.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Brightness level set to %d.\n", option);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			general_error();
			ret = -1;
		}
	}
	if (GET_BRIGHTNESS) {
		int level = -1;
		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply, "BrightnessGet");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Brightness not supported.\n");
			ret = -1;
			break;
		case REPLY_SUCCESS:
			if (!dbusGetMessageInteger(reply, &level, 0)) {
				printf("Current brightness level: %d.\n", level);
			} else {
				ret = -1;
				general_error();
			}
			dbus_message_unref(reply);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			printf("Error: %d - %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	}
	if (GET_BRIGHTNESS_LEVELS) {
		int levels = -1;
		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply, "BrightnessLevelsGet");
		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Brightness not supported.\n");
			ret = -1;
			break;
		case REPLY_SUCCESS:
			if (!dbusGetMessageInteger(reply, &levels, 0)) {
				printf("Available brightness levels: %d.\n", levels);
			} else {
				ret = -1;
				general_error();
			}
			dbus_message_unref(reply);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		default:
			printf("Error: %d - %s\n", err_code, DBus_Error_Array[err_code]);
			ret = -1;
		}
	}
	if (DPM_GET_DEVICES) {
		dummy = dpmDeviceClass;

		err_code = dbusSendMessageWithReply(REQUEST_MESSAGE, &reply, "DpmDevicesGet",
						    DBUS_TYPE_STRING, &dummy,
						    DBUS_TYPE_INVALID);

		if (err_code != REPLY_SUCCESS) {
			switch (err_code) {
			case REPLY_DISABLED:
				fprintf(stderr, "Device power management has been disabled by administrator.\n");
				break;
			default:
				general_error();
			}
			ret = -1;
		}

		char *device;
		int state = -1;
		int i = 0;
		while (!dbusGetMessageString(reply, &device, i)
		       && !dbusGetMessageInteger(reply, &state, i)) {
			printf("Device [%d]: %s\n"
			       "            power state: %d\n", i, device, state);
			i++;
		}
	
	}
	if (DPM_GET_CLASSES) {
		err_code = dbusSendMessageWithReply(REQUEST_MESSAGE, &reply, "DpmClassesGet",
						    DBUS_TYPE_INVALID);

		if (err_code != REPLY_SUCCESS) {
			switch (err_code) {
			case REPLY_DISABLED:
				fprintf(stderr, "Device power management has been disabled by administrator.\n");
				break;
			default:
				general_error();
			}
			ret = -1;
		}
		
		char *c;
		int i = 0;
		while (!dbusGetMessageString(reply, &c, i)) {
			printf("%s\n", c);
			i++;
		}
	}
	if (DPM_SUSPEND) {
		dummy = dpmDeviceClass;

		err_code = dbusSendMessage(ACTION_MESSAGE, "DpmDevicesSuspend",
					   DBUS_TYPE_STRING, &dummy,
					   DBUS_TYPE_INT32, &option,
					   DBUS_TYPE_INVALID);


		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Runtime powermanagement not supported.\n");
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Already suspended.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Suspended %s.\n", dpmDeviceClass);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		case REPLY_DISABLED:
			fprintf(stderr, "Device power management has been disabled by administrator.\n");
			break;
		default:
			general_error();
			ret = -1;
		}
	}
	if (DPM_RESUME) {
		dummy = dpmDeviceClass;
		err_code = dbusSendMessage(ACTION_MESSAGE, "DpmDevicesResume",
					   DBUS_TYPE_STRING, &dummy,
					   DBUS_TYPE_INT32, &option,
					   DBUS_TYPE_INVALID);

		switch (err_code) {
		case REPLY_HW_NOT_SUPPORTED:
			fprintf(stderr, "Runtime powermanagement not supported.\n");
			ret = -1;
			break;
		case REPLY_ALREADY_SET:
			fprintf(stderr, "Already resumed.\n");
			//noerror
			break;
		case REPLY_SUCCESS:
			fprintf(stderr, "Resumed %s.\n", dpmDeviceClass);
			break;
		case REPLY_NO_RIGHTS:
			no_connect_no_rights_error();
			ret = -2;
			break;
		case REPLY_DISABLED:
			fprintf(stderr, "Device power management has been disabled by administrator.\n");
			break;
		default:
			general_error();
			ret = -1;
		}
	}
	if (GET_CLIENTS) {
		err_code = dbusSendSimpleMessageWithReply(REQUEST_MESSAGE, &reply,
							  "ClientsGet");

		if (err_code != REPLY_SUCCESS) {
			general_error();
			ret = -1;
		}

		char *name;
		int cap;
		int i = 0;
		while (!dbusGetMessageString(reply, &name, i)) {
			printf("Client name: %s", name);
			if (!dbusGetMessageInteger(reply, &cap, i))
				printf(", Capabilities: %d", cap);
			printf("\n");
			i++;
		}
		
		if (!i)
			printf("No clients connected\n");
	}
	if (VERSION_INFO) {
#ifdef VERSION
		printf("Powersave package version: %s\n", VERSION);
#else
		printf("Version has not been defined by VERSION at compile time.\n");
#endif
	}
	if (ret == -1)
		exit(EXIT_FAILURE);
	else
		exit(ret);
}
