/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *          Copyright (C) 2004,2005,2006 SUSE Linux Products GmbH          *
 *                                                                         *
 *               Author(s): Holger Macht <hmacht@suse.de>                  *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#ifndef CLIENTCONNECTION_H
#define CLIENTCONNECTION_H

#include <list>
#include <string>

class CPUFreq_Interface;
struct PM_STATE;

/** @brief This struct represents a connected client */
typedef struct cNode {
	/** the clients name (dbus name)*/
	std::string name;
	/** The capabilities to which the message belongs @ref client_capabilities */
	int capabilities;
} cNode;

/** @brief  Base class for administrate clients
 *
 * Base class for administrate clients which connect to a server socket
 * provided by powersaved. Everytime one of the above defined states changes, 
 * all client which requested this state will be notified. Additionally it is
 * also possible that acpi_events occurring on /proc/acpi/events will be 
 * forwarded to the clients instead of changes.
 *
 * The class clientStorage is a list which holds all the registered clients.
*/
class ClientConnection {
public:
	/** @brief destructor clearing the class */
	~ClientConnection();

	/** @brief check if we have a connected client with a specific capability
	 *
	 * @param capability the capability to check for
	 *
	 * @return integer value representing result of function call
	 * @retval 1 on success
	 * @retval 0 otherwise
	*/
	bool checkClientCapabilities(int capability);

	/** @brief get number of clients connected
	 *
	 * @return integer with number of clients connected to the daemon
	 *
	 * invokes count() from class clientStorage and so returns the
	 * number of clients which are connected to the daemon
	 */
	int count();

	/** @brief adds a client to the list of connected clients
	 *
	 * @param name the dbus name of the client
	 * @param modes the modes the client is aware of
	 *
	 * @return true on success, false otherwise
	 */
	bool addClient(const char *name, int modes);

	/** @brief check if given client should get removed 
	 *
	 * @param name the dbus name of the client
	 *
	 * @return true when the client got removed, false otherwise
	 */
	bool checkClientRemoval(const char *name);

	/** @brief get all connected clients with their capabilities
	 *
	 * @param names stl list that gets filled with client names
	 * @param caps slt list that gets filled with client capabilities
	 */
	void getClients(std::list< std::string > &names, std::list< int > &caps);

private:
	/** list for storing all connected clients */
	std::list < cNode > _clients;
};

#endif // CLIENTCONNECTION_H
