/***************************************************************************
 *                                                                         *
 *                         Powersave Daemon                                *
 *                                                                         *
 *            Copyright (C) 2005 SUSE Linux Products GmbH                  *
 *                                                                         *
 *             Author(s): Holger Macht <hmacht@suse.de>                    *
 *                                                                         *
 * This program is free software; you can redistribute it and/or modify it *
 * under the terms of the GNU General Public License as published by the   *
 * Free Software Foundation; either version 2 of the License, or (at you   *
 * option) any later version.                                              *
 *                                                                         *
 * This program is distributed in the hope that it will be useful, but     *
 * WITHOUT ANY WARRANTY; without even the implied warranty of              *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU       *
 * General Public License for more details.                                *
 *                                                                         *
 * You should have received a copy of the GNU General Public License along *
 * with this program; if not, write to the Free Software Foundation, Inc., *
 * 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA                  *
 *                                                                         *
 ***************************************************************************/

#include "config.h"
#include "powersave_hal.h"

#include <string.h>

DBusError dbus_error;
LibHalContext *hal_ctx = NULL;
DBusConnection *dbus_connection = NULL;

int ps_hal_init()
{
	if (hal_ctx != NULL) {
		return 1;
	}

	dbus_error_init(&dbus_error);

	dbus_connection = dbus_connection_open_private(DBUS_SYSTEM_BUS_SOCKET, &dbus_error);
	if (dbus_connection == NULL || dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_INFO, "could not open connection to system bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}

	/* register our own connection */
	dbus_bus_register(dbus_connection, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_INFO, "could not register connection to system bus: %s", dbus_error.message);
		dbus_error_free(&dbus_error);
		return 0;
	}

	int hal_is_ready = dbus_bus_name_has_owner(dbus_connection,
						   "org.freedesktop.Hal",
						   &dbus_error);
	if (!hal_is_ready) {
		pDebug(DBG_INFO, "hal not ready. We will try later... ");

		if ( dbus_error_is_set( &dbus_error ) ) {
			pDebug(DBG_INFO, "Error checking if hal service exists: %s",
			       dbus_error.message );
			dbus_error_free( &dbus_error );
		}

		ps_hal_free();

		return 0;
	}

	hal_ctx = libhal_ctx_new();

	/* setup dbus connection for hal */
	if (!libhal_ctx_set_dbus_connection(hal_ctx, dbus_connection)) {
		pDebug(DBG_INFO, "could not set up connection to dbus for hal");
		ps_hal_free();
		return 0;
	}

	/* init the hal library */
	if (!libhal_ctx_init(hal_ctx, &dbus_error)) {
		pDebug(DBG_INFO, "could not init hal library: %s", dbus_error.message);
		ps_hal_free();
		return 0;
	}

	return 1;
}

void ps_hal_free()
{
	if ( dbus_connection != NULL &&
	     dbus_connection_get_is_connected( dbus_connection ) ) {
		dbus_connection_close( dbus_connection );
		dbus_connection_unref( dbus_connection );
		dbus_connection = NULL;
	}
	
	if ( hal_ctx != NULL ) {
		libhal_ctx_free( hal_ctx );
		hal_ctx = NULL;
	}

	dbus_error_free( &dbus_error );
}

LibHalContext* ps_hal_context()
{
	return hal_ctx;
}

DBusError* ps_hal_dbus_error()
{
	return &dbus_error;
}


int ps_hal_get_property_int(char *device, char *property)
{
	if (!ps_hal_init()) {
		return HAL_ERROR;
	}

	if (!libhal_device_property_exists(hal_ctx, device, property, &dbus_error)) {
		pDebug(DBG_INFO, "property '%s' does not exist.", property);
		return -1;
	}

	int val = libhal_device_get_property_int(hal_ctx, device,
						 property, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_WARN, "Error fetching property '%s': '%s'", property, dbus_error.message);
		dbus_error_free(&dbus_error);
		return -1;
	}

	return val;
}

int ps_hal_get_property_bool(char *device, char *property)
{
	if (!ps_hal_init()) {
		return HAL_ERROR;
	}

	if (!libhal_device_property_exists(hal_ctx, device, property, &dbus_error)) {
		pDebug(DBG_INFO, "property '%s' does not exist.", property);
		return -1;
	}

	int val = libhal_device_get_property_bool(hal_ctx, device,
						  property, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_WARN, "Error fetching property '%s': '%s'", property, dbus_error.message);
		dbus_error_free(&dbus_error);
		return -1;
	}

	return val;
}

char* ps_hal_get_property_string(char *device, char *property)
{
	if (!ps_hal_init()) {
		return NULL;
	}

	if (!libhal_device_property_exists(hal_ctx, device, property, &dbus_error)) {
		pDebug(DBG_INFO, "property '%s' does not exist.", property);
		return NULL;
	}

	char *val = libhal_device_get_property_string(hal_ctx, device,
						      property, &dbus_error);

	if (dbus_error_is_set(&dbus_error)) {
		pDebug(DBG_WARN, "Error fetching property '%s': '%s'", property, dbus_error.message);
		dbus_error_free(&dbus_error);
		return NULL;
	}

	return val;
}

int ps_hal_query_capability(char *device, char *capability)
{
	if (!ps_hal_init()) {
		return HAL_ERROR;
	}

	int ret = libhal_device_query_capability(hal_ctx, device,
						 capability, &dbus_error);
	
	if (!ret) {
		if (dbus_error_is_set(&dbus_error)) {
			pDebug(DBG_DIAG, "Error querying for capability '%s': '%s'",
			       capability, dbus_error.message);
			dbus_error_free(&dbus_error);
			return -1;
		}
		return 0;
	}
	
	return 1;
}
